/*
   Copyright (C) 2009  Stephane Pion
   This file is part of Intifada.

    Intifada is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Intifada is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Intifada.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <ichecker.hxx>

#include <cstring>
#include <intifada/Configuration_XML_Reader.hxx>
#include <intifada/Configuration_Internal_Reader.hxx>

#include <intifada/Record_Repository.hxx>
#include <intifada/Message.hxx>


#include <fstream>
#include <Message_Database.hxx>
#include <intifada/Logger.hxx>

Options::Options():
  asterixDescription_()
  ,asterixValidation_()
  ,checks_()
  ,idx_(-1)
  ,source_()
  ,source_type_("INTERNAL")
  ,dump_(false)
  ,dump_type_()
  ,family_("eurocontrol")
  ,internal_description_(false)
  {}

void Options::set_asterix_description(const char* f){asterixDescription_=f;}
const std::string& Options::get_asterix_description()const{return asterixDescription_;}
void Options::set_asterix_validation(const char* f){asterixValidation_=f;}
const std::string& Options::get_asterix_validation()const{return asterixValidation_;}

void Options::add_check(const char* c)
  {

    checks_.insert(c);
  }
bool Options::check_exist(const std::string& c)const
{
  bool found=false;
  desc_const_iterator_type it=checks_.find(c);
  if(it!=checks_.end()){
    found=true;
  }
  return found;
}
void Options::set_idx(const char* i)
  {
    std::istringstream is(i);
    is >> idx_;
  }
int Options::get_idx()const
{
  if(checks_.size()>1){idx_=-1;}
  return idx_;
}
void Options::set_source(const char* s)
  {
    source_=s;
  }
const std::string& Options::get_source()const
{
  return source_;
}

void Options::set_type(const std::string& t)
  {
    source_type_=t;
  }

const std::string& Options::get_type()const
{
  return source_type_;
}
void Options::set_dump(const bool&d)
  {
    dump_=d;
  }
bool Options::get_dump()const
{
  return dump_;
}
void Options::set_dump_type(const std::string& t)
  {
    dump_type_=t;
  }

const std::string& Options::get_dump_type()const
{
  return dump_type_;
}

void Options::set_family(const std::string& t)
  {
    family_=t;
  }

const std::string& Options::get_family()const
{
  return family_;
}

bool Options::get_internal_description()const
{
  return internal_description_;
}
void Options::set_internal_description(const bool& t)
  {
    internal_description_=t;
  }

static Options options;

void help()
  {
    std::cout << "usage: -h -f <asterix description file> -c asterix type -i idx -s source -t source type\n";
    std::cout << "\t-h  short help" << std::endl;
    std::cout << "\t-f asterix description file" << std::endl;
    std::cout << "\t-c asterix type" << std::endl;
    std::cout << "\t-i index if applicable" << std::endl;
    std::cout << "\t-I use internal description" << std::endl;
    std::cout << "\t-s source file" << std::endl;
    std::cout << "\t-t source type (INTERNAL)" << std::endl;
    std::cout << "\t-F family if applicable (for PURE for example)" << std::endl;
    std::cout << "\t-d ALL|NOTNULL dump (if NOTNULL no null values are printed)" << std::endl;
    std::cout << "\tif -t and -s aren't specified, messages are taken from internal" << std::endl;
    std::cout << "\tmessage database" << std::endl << std::endl;
    std::cout << "source type:INTERNAL. index is applicable"<<std::endl;
    std::cout << "\tPURE. index isn't applicable, but family is" << std::endl;
    std::cout << "\tassociated asterix type:001-002|034-048|ARTAS|STRODS|MAESTRO|062-065" << std::endl;



  }
int handle_options(int argc, char**argv)
  {
    int opt;
    while ((opt = getopt(argc, argv, "hf:D:c:i:Is:t:d:F:")) != -1) {
      switch (opt) {
      case 'h':
        help();
        exit(EXIT_SUCCESS);
        break;
      case 'f':
        options.set_asterix_description(optarg);
        break;
      case 'D':
        options.set_asterix_validation(optarg);
        break;
      case 'c':
        options.add_check(optarg);
        break;
      case 'i':
        options.set_idx(optarg);
        break;
      case 'I':
        options.set_internal_description(true);
        break;
      case 's':
        options.set_source(optarg);
        break;
      case 't':
        options.set_type(optarg);
        break;
      case 'd':
        options.set_dump(true);
        options.set_dump_type(optarg);
        break;
      case 'F':
        options.set_family(optarg);
        break;
      default: /* ���?��� */
        help();
        exit(EXIT_FAILURE);
      }
    }
    return 0;
  }


Check_Item_Iterator::Check_Item_Iterator():record_(NULL){

}


int Check_Item_Iterator::operator()(const intifada::Stream::size_type&,const intifada::Stream::size_type&,const intifada::Path_Name& full_name,intifada::Type& t)
    {
      clog("Check_Item_Iterator") << "-->" << cendl;
      intifada::Path_Name path_name=full_name.get_path();

      try{
        (*record_)(full_name)=t;
        //record_->set_value(full_name,t);
        record_->set_presence(path_name,true);
      }catch(intifada::Description_Field_Unknow_Exception& ex){
        std::cerr << "FATAL:EDASTERIX: unknow field:full name:<"<<full_name<<">" << std::endl;
        std::cerr << "EDASTERIX: unknow field:path name:<"<<path_name<<">(path size="<< full_name.get_name_size() << ")" << std::endl;
        exit(EXIT_FAILURE);
      }
      clog("Check_Item_Iterator") << full_name << "=" << t.get_to_string() << cendl;
      clog("Check_Item_Iterator") << path_name << " presence=true" << cendl;

      clog("Check_Item_Iterator") << "<--" << cendl;
      return 0;
    }

void Check_Item_Iterator::set_record(intifada::Record* r){record_=r;}

Dump_Item_Iterator::Dump_Item_Iterator():strip_null_(true){

}

void Dump_Item_Iterator::dump_all()
  {
    strip_null_=false;
  }

int Dump_Item_Iterator::operator()(
    const intifada::Stream::size_type&stream_pos
    ,const intifada::Stream::size_type& stream_size
    ,const intifada::Path_Name& full_name
    ,intifada::Type& t)
                  {
                    clog("Check_Item_Iterator") << "-->" << cendl;
                    intifada::Path_Name path_name=full_name.get_path();

                    try{
                      std::string value=t.get_to_string();
                      // don't print if strip_null is true and value=0 s . (v-)
                      // print if s- + v
                      if(strip_null_!=true || (value != "0" && value !="")){
                        std::cout << "\t\t("<< stream_pos << ","<< stream_size << "):"<<full_name << "=" << value << std::endl;
                      }
                    }catch(intifada::Description_Field_Unknow_Exception& ex){
                      std::cerr << "FATAL:EDASTERIX: unknow field:full name:<"<<full_name<<">" << std::endl;
                      std::cerr << "EDASTERIX: unknow field:path name:<"<<path_name<<">(path size="<< full_name.get_name_size() << ")" << std::endl;
                      exit(EXIT_FAILURE);
                    }
                    return 0;
                  }

Check_Iterator::Check_Iterator(intifada::Message& msg):it_(),msg_(msg),block_(NULL){}

int Check_Iterator::operator()(uint8_t c)
                  {
                    // Get a new block to insert
                    block_=msg_.get_new_block(c);
                    // Push back it
                    msg_.push_back(block_);

                    // std::cout << block_ << " cat=" << (uint32_t) c << std::endl;
                    return 0;
                  }

int Check_Iterator::operator()(
    intifada::Record_Iterator::record_type& i,
    const intifada::Message::block_list_size_t& /*b*/,
    const intifada::Block::record_list_size_t& /*r*/)
                      {
                        //->DEBUG
                        //intifada::Type *t = i->get_value("380.ADR");
                        //assert(t!=NULL);
                        //std::cout << t->get_to_string() << std::endl;
                        //exit(0);
                        //<-DEBUG
                        // std::cout << "block=" << b << " record=" << r << std::endl;
                        // Handle all items record
                        // Create a freshly record, fill it using functor and push it to the current block
                        intifada::Record *new_rec=block_->get_new_record();
                        it_.set_record(new_rec);
                        i.foreach(it_,intifada::Path_Name());
                        block_->push_back(new_rec);
                        return 0;
                      }

Dump_Iterator::Dump_Iterator():it_(){}

void Dump_Iterator::dump_all()
  {
    it_.dump_all();
  }
/// Called when a block began
int Dump_Iterator::operator()(uint8_t c)
                  {
                    std::cout << "cat=" << (uint32_t) c << std::endl;
                    return 0;
                  }

int Dump_Iterator::operator()(
    intifada::Record_Iterator::record_type& i,
    const intifada::Message::block_list_size_t& b,
    const intifada::Block::record_list_size_t& r)
                      {
                        std::cout << "\tblock=" << b << " record=" << r << std::endl;
                        i.foreach(it_,intifada::Path_Name());
                        return 0;
                      }

Operation::Operation(intifada::Record_Repository* rep):rep_(rep){}
Operation::~Operation(){}


Check_Operation::Check_Operation(intifada::Record_Repository* rep):inherited(rep){}

int Check_Operation::operator()(const std::string& family,int midx,const uint8_t*m, uint16_t msg_size)
throw(intifada::Parsing_Input_Length_Exception)
      {

        intifada::Message msg(rep_,family);
        intifada::Stream::size_type s = msg.set_stream(m,0,msg_size);
        // Expected cloned message
        intifada::Message ecmsg(rep_,family);
        Check_Iterator cit(ecmsg);

        intifada::Record_Iterator_List ril;
        ril.register_iterator(cit);
        msg.foreach(ril);

        intifada::Stream::size_type est=ecmsg.size();
        uint8_t *t=new uint8_t[est];

        intifada::Stream::size_type st=ecmsg.get_stream(t,0,est);
        if(s!=est){
          std::cerr << midx << " Error computing size src->dest (source size=" << s
          << " dest size=" << est << ")" << std::endl;
        }
        if(st!=est){
          std::cerr << midx << " Error computing size calculated dest->real dest (calculated dest size=" << est
          << " real size=" << st << ")" << std::endl;
        }
        unsigned int minsize=std::min(s,st);

        for(unsigned int i=0;i<minsize;++i){
          if(m[i]!=t[i]){
            std::cerr << std::dec << midx << ":Error ["<< i <<"], 0x" << std::hex
            << static_cast<uint16_t>(m[i]) << "->0x" << static_cast<uint16_t>(t[i])
            << std::endl;
          }
        }
        delete[]t;

        return 0;
      }


Dump_Operation::Dump_Operation(intifada::Record_Repository* rep,const std::string& dump_type):inherited(rep),dump_type_(dump_type){}

int Dump_Operation::operator()(const std::string& family,int,const uint8_t*m, uint16_t msg_size)
throw(intifada::Parsing_Input_Length_Exception)
              {

                intifada::Message msg(rep_,family);

                intifada::Stream::size_type s;
                s = msg.set_stream(m,0,msg_size);
                Dump_Iterator cit;
                if(dump_type_=="ALL"){
                  cit.dump_all();
                }

                intifada::Record_Iterator_List ril;
                ril.register_iterator(cit);
                msg.foreach(ril);
                return 0;
              }

void check_internal_database(Operation& op_asterix)
throw(intifada::Parsing_Input_Length_Exception)
        {
          if(options.check_exist("001-002")){
            int idx=options.get_idx();
            if(idx >=0){
              const uint8_t*m=cat_001_002_message[idx];
              uint16_t msg_size=cat_001_002_message_size[idx];
              op_asterix("eurocontrol",idx,m,msg_size);
              exit(0);
            }

            for(unsigned int midx=0;midx<cat_001_002_total_messages;++midx){
              // std::cerr << "message I001/I002:" << midx << std::endl;
              op_asterix("eurocontrol",midx,cat_001_002_message[midx],cat_001_002_message_size[midx]);
            }
          }

          if(options.check_exist("034-048")){
            int idx=options.get_idx();
            if(idx >=0){
              const uint8_t*m=mode_s_message[idx];
              uint16_t msg_size=mode_s_message_size[idx];
              op_asterix("eurocontrol",idx,m,msg_size);
              exit(0);
            }
            for(unsigned int midx=0;midx<mode_s_total_messages;++midx){
              op_asterix("eurocontrol",midx,mode_s_message[midx],mode_s_message_size[midx]);
            }
          }
          if(options.check_exist("ARTAS")){
            int idx=options.get_idx();
            if(idx >=0){
              const uint8_t*m=eurocontrol_30_message[idx];
              uint16_t msg_size=eurocontrol_30_message_size[idx];
              op_asterix("eurocontrol",idx,m,msg_size);
              exit(0);
            }
            for(unsigned int midx=0;midx<eurocontrol_30_total_messages;++midx){
              op_asterix("eurocontrol",midx,eurocontrol_30_message[midx],eurocontrol_30_message_size[midx]);
            }
          }
          if(options.check_exist("STRODS")){
            int idx=options.get_idx();
            if(idx >=0){
              const uint8_t*m=eurocontrol_30_message[idx];
              uint16_t msg_size=eurocontrol_30_message_size[idx];
              op_asterix("strods",idx,m,msg_size);
              exit(0);
            }
            for(unsigned int midx=0;midx<eurocontrol_30_total_messages;++midx){
              op_asterix("strods",midx,eurocontrol_30_message[midx],eurocontrol_30_message_size[midx]);
            }
          }
          if(options.check_exist("MAESTRO")){
            int idx=options.get_idx();
            if(idx >=0){
              const uint8_t*m=cat_030_252_approach_message[idx];
              uint16_t msg_size=cat_030_252_approach_message_size[idx];
              op_asterix("strods",idx,m,msg_size);
              exit(0);
            }
            for(unsigned int midx=0;midx<cat_030_252_approach_total_messages;++midx){
              op_asterix("strods",midx,cat_030_252_approach_message[midx],cat_030_252_approach_message_size[midx]);
            }
          }
          if(options.check_exist("062-065")){
            int idx=options.get_idx();
            if(idx >=0){
              const uint8_t*m=cat_062_065[idx];
              uint16_t msg_size=cat_062_065_size[idx];
              op_asterix("eurocontrol",idx,m,msg_size);
              exit(0);
            }
            for(unsigned int midx=0;midx<cat_062_065_total_messages;++midx){
              op_asterix("eurocontrol",midx,cat_062_065[midx],cat_062_065_size[midx]);
            }
          }

        }

int
main(int argc, char**argv)
  {
    int idx=0;
    try{
      // clog.insert_mask("Condition");
      // clog.insert_mask("Configuration_XML_Reader");
      // clog.insert_mask("Configuration_Reader");
      // clog.insert_mask("Record");
      // clog.insert_mask("Foreach_Field_Iterator");
      // clog.insert_mask("Configuration_Reader");
      // clog.insert_mask("Field_Specification_Data");
      // clog.insert_mask("Field_Iterator_Size");
      // clog.insert_mask("Check_Item_Iterator");
      // clog.insert_mask("Repetitive_Length_Data_Field");
      // clog.insert_mask("Extended_Length_Data_Field");
      // clog.insert_mask("size debug");
      // clog.insert_mask("stream pos");
      // clog.insert_mask("String_Type");
      // clog.insert_mask("Memory_Allocation");
      // clog.insert_mask("Configuration_Internal_Reader");
      // clog.insert_mask("Data_Field_Characteristics_List");
      handle_options(argc,argv);

      std::string asterix_description=INTIFADA_TOP_SRCDIR;
      std::string asterix_validation=INTIFADA_TOP_SRCDIR;
      if(options.get_internal_description()==true)
        {
          asterix_description+="/test.kv";
        }
      else
        {
          asterix_description +="/asterix.xml";
          asterix_validation +="/asterix.dtd";
        }
      std::string alternate_asterix_description=options.get_asterix_description();
      std::string alternate_asterix_validation=options.get_asterix_validation();
      if(alternate_asterix_description.empty()!=true){
        asterix_description=alternate_asterix_description;
      }
      if(alternate_asterix_validation.empty()!=true){
        asterix_validation=alternate_asterix_validation;
      }


      intifada::Configuration_Reader *p=NULL;
      // Reading asterix description
      if(options.get_internal_description()==true)
        {
          p= new intifada::Configuration_Internal_Reader(asterix_description);
        }
      else
        {
          p= new intifada::Configuration_XML_Reader(asterix_description,asterix_validation);
        }


      int opened=p->open();
      if(opened!=0){
        std::cerr << "Error opening asterix description (" << asterix_description << ")" << std::endl;
        exit(EXIT_FAILURE);
      }
      // Loading repository with asterix description
      intifada::Record_Repository* rep=intifada::Record_Repository::instance();
      p->parse(rep);
      Operation *op_asterix=NULL;
      if(options.get_dump()==false){
        op_asterix = new Check_Operation(rep);
      }else{
        op_asterix = new Dump_Operation(rep,options.get_dump_type());
      }
      if(options.get_type()=="INTERNAL"){
        check_internal_database(*op_asterix);
      }else if(options.get_type()=="PURE"){
        std::ifstream ifile(options.get_source().c_str());
        idx=0;
        while(ifile){
          char *astmsg=NULL;
          uint16_t size;
          char asthead[3];
          ifile.read(asthead,3);
          if(ifile){
            size=((asthead[1]& 0xff) << 8) | (asthead[2] & 0xff);
            astmsg=new char[size];
            ::memcpy(astmsg,asthead,3);
            char *ap=astmsg;
            ap+=3;
            ifile.read(ap,size-3);
          }
          if(ifile){
            // handle asterix msg here
            //std::cout <<"/ handle asterix msg here\n";
            const uint8_t*m=reinterpret_cast<const uint8_t*>(astmsg);
            uint16_t msg_size=size;
            (*op_asterix)(options.get_family(),idx,m,msg_size);
          }
          if(astmsg!=NULL){
            delete[]astmsg;
          }
          ++idx;
        }
        ifile.close();
      }else{
        std::cout << "type:"<<options.get_type()<< " not yet implemented" << std::endl;
      }
      exit(EXIT_SUCCESS);
    }catch(intifada::Description_Field_Unknow_Exception & ex){
      std::cerr << "Unknow field:"<<ex.get_name()<<std::endl;
      exit(EXIT_FAILURE);
    }catch(intifada::Parsing_Input_Length_Exception & ex){
      std::cerr << "Invalid size:expected:"<<ex.get_waited() << " bytes, got:"<< ex.get_received() << " bytes for message:" << idx <<std::endl;
      exit(EXIT_FAILURE);
    }
  }
