#
# Copyright (C) 2010 Alexander Taler <dissent@0--0.org>
#

# This file is part of hsh.

# hsh is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# hsh is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with hsh.  If not, see <http://www.gnu.org/licenses/>.

######################################################################

# CursesDisplay is the central object defining the visual state of hsh.
# It is a singleton which contains references to multiple View objects, as well
# as curses window objects.  The view objects are mapped to curses window
# objects.  Additionally, it contains the main input loop, reading curses input
# and processing it.

# CursesDisplayThread is a separate thread which performs the actual drawing
# and calls to the curses library.  It interacts with the View objects by
# reading their data to decide what to draw.

import curses
import curses.ascii
import datetime
import logging
import os
import signal
import sys
import termios
import threading
import time
import traceback

import hsh.aliases
import hsh.config
import hsh.debug
import hsh.display
import hsh.jobs
import hsh.welcome

from hsh.exceptions import *

from view import View
from input_view import InputView
from job_view import JobView
from list_view import ListView
from message_view import MessageView
from keyvalue_view import KeyValueView
from search_view import SearchView

def init_curses():
    stdscr = curses.initscr()
    curses.noecho()
    curses.raw()
    stdscr.keypad(1)
    return stdscr

def deinit_curses(stdscr):
    stdscr.keypad(0)
    curses.noraw()
    curses.echo()
    curses.endwin()

def run():
    if not sys.stdin.isatty():
        print "Standard input is not a tty, aborting."
        sys.exit(1)

    tsettings = None
    try:
        # Set up job control

        # Hsh must be the foreground job, wait if not.
        while os.getpgrp() != os.tcgetpgrp(sys.stdin.fileno()):
            os.kill(os.getpid(), signal.SIGTTIN)

        # Ignore job control signals since we are doing job control
        signal.signal(signal.SIGINT, signal.SIG_IGN)
        signal.signal(signal.SIGQUIT, signal.SIG_IGN)
        signal.signal(signal.SIGTSTP, signal.SIG_IGN)
        signal.signal(signal.SIGTTIN, signal.SIG_IGN)
        signal.signal(signal.SIGTTOU, signal.SIG_IGN)

        # If this process isn't the group leader, put it in a new group.
        if os.getpid() != os.getsid(0):
            os.setpgid(os.getpid(), os.getpid())
        # Put the new group into the foreground.
        os.tcsetpgrp(sys.stdin.fileno(), os.getpgrp())
        # Remember terminal settings.
        tsettings = termios.tcgetattr(sys.stdin.fileno())

        logging.debug("Job control initialized.")

        stdscr = init_curses()
        logging.debug("Curses initialized.")

        hsh.display_obj = CursesDisplay(stdscr, tsettings)
        hsh.display_obj.run()

    finally:
        try:
            deinit_curses(stdscr)
        except:
            pass
        if tsettings:
            termios.tcsetattr(sys.stdin.fileno(), termios.TCSANOW, tsettings)

class CursesDisplay(hsh.display.HshDisplay, hsh.jobs.JobManagerListener):

    # Private variables
    #    scr: a curses window object for the whole screen
    #    full_screen_job: If a job has been granted control of the terminal,
    #                     this will be its job object.  If not, it is None.
    #    views: a dict storing all view objects indexed by their name.
    #    layout: a list of view objects which are visible, starting from the
    #            top of the screen.  layout[0] is the main view.
    #    focus: a View object which currently has the cursor

    def __init__(self, stdscr, tsettings):
        self.scr = stdscr
        self.tsettings = tsettings
        self.full_screen_job = None
        self.input_reassigned = False # True means a job is reading input

        self.initialize_faces()

        # A dict mapping job ids to job view objects.
        # Use get_job_view() to get a view.
        self.job_view_index = dict()

        self.views = {}
        self.views["JobView"] = None
        for v in [InputView(self),
                  SearchView(self),
                  MessageView(self, "Welcome", hsh.welcome.message),
                  MessageView(self, "TooSmall","Help! The screen is too small"),
                  MessageView(self, "Alert", "", can_focus=False),
                  MessageView(self, "CrashLog", "Nothing so far!"),
                  KeyValueView(self, "Aliases", hsh.aliases.manager),
                  KeyValueView(self, "Environment", os.environ)
                  ]:
            self.views[v.get_name()] = v

        # Load job list views
        for newviewconf in hsh.config.listviews:
            newview = ListView(self, **newviewconf)
            self.views[newview.get_name()] = newview

        self.layout = [self.views["Welcome"],
                       self.views["InputView"],
                       self.views["Alert"]]

        self.focus = None
        self.give_focus(self.views["InputView"])

        self.displaythread = CursesDisplayThread(self)
        self.displaythread.setDaemon(True)

        hsh.jobs.manager.register_listener(self)

        self.escape = False  # Used for handling escapes in input stream
        self.ctrlz = False   # Used for handling ctrl-zs in input stream
        # For the basic cut and paste implementation.
        self.last_copy = ''

    def initialize_faces(self):
        if curses.has_colors():
            curses.start_color()
            logging.debug("COLORS: %d  COLOR_PAIRS: %d" %
                          (curses.COLORS, curses.COLOR_PAIRS))
            logging.debug("can_change_color = %s" % curses.can_change_color())
        self._faces = {}
        # Assign a colour pair number to each specified text face.  This
        # restricts us to COLOR_PAIRS different faces.  If this is insufficient
        # a numbering scheme for mapping pairs to numbers could be created, but
        # don't forget that 0 is white on black. (8-fg) * (1+bg) - 1
        cnum = 1
        for faceinfo in hsh.config.faces.items():
            curses.init_pair(cnum, faceinfo[1][0], faceinfo[1][1])
            self._faces[faceinfo[0]] = curses.color_pair(cnum)
            logging.debug("name=%s cnum=%s attr=%s fi[0]=%s fi[1]=%s" %
                          (faceinfo[0], cnum, self._faces[faceinfo[0]],
                           faceinfo[1][0], faceinfo[1][1]))
            cnum += 1

    def get_face(self, viewname, facename):
        vn = viewname + "." + facename
        if vn in self._faces:
            return self._faces[vn]
        if facename in self._faces:
            return self._faces[facename]
        logging.debug("No face named " + facename)
        return self._faces["default"]

    def get_size(self):
        """Return a pair with the size of the display, (height, width)."""
        return self.scr.getmaxyx()

    def get_job_view(self, job):
        if job.jid not in self.job_view_index:
            self.job_view_index[job.jid] = JobView(self, job)
        return self.job_view_index[job.jid]

    def get_main_view(self):
        return self.layout[0]

    def give_focus(self, newfocus):
        if newfocus not in self.layout:
            raise Exception("Cannot give focus to invisible view")
        if self.focus == newfocus:
            return
        if self.focus is not None:
            self.focus.set_focus(has_focus=False)
        self.focus = newfocus
        if not self.focus.set_focus(has_focus=True):
            raise Exception("View refused focus: " + self.focus.get_name())

    def handle_crash(self, exc):
        # Clear the input area, in case that was the cause of the issue
        try:
            inptext = str(self.views["InputView"].command_text)
        except:
            inptext = "<Exception fetching input text>"
        logging.debug(str(exc))
        logging.debug("input: " + inptext)
        logging.debug(traceback.format_exc())
        self.views["InputView"]._clear_text()
        cl = self.views["CrashLog"]
        cl.set_message(str(datetime.datetime.now())+"\n" +
                       str(exc) + "\n" +
                       "input: " + inptext + "\n" +
                       traceback.format_exc())
        self.set_main(cl)
        self.show_alert("Exception!")

    def run(self):
        # All input is read using the curses getch() function, even in the case
        # where a full screen job is accepting the input.

        # Full screen job input used to be done using sys.stdin.read(), which
        # was problematic because it required timeouts for a smooth transition
        # between full screen jobs and regular input, but timeouts on curses
        # functions prevented screen size changes from being communicated
        # correctly, and using select() for timeouts reading stdin didn't work
        # correctly when curses keystrokes starting with ESC were received.

        self.displaythread.start()

        while True:
            try:
                while self.input_reassigned:
                    time.sleep(0.1)
                ch = self.scr.getch()

                fsj = self.full_screen_job
                if fsj is not None:
                    self.handle_raw_input(fsj, ch)
                else:
                    self.handle_curses_input(ch)
            except QuitException:
                # Clean up remaining jobs.
                hsh.jobs.manager.terminate()
                break
            except Exception, e:
                self.handle_crash(e)

    def handle_raw_input(self, fsj, ch):
        if (not self.ctrlz) and ch == ord(curses.ascii.ctrl('Z')):
            logging.debug("received ctrlz")
            self.ctrlz = True
            return
        if self.ctrlz:
            logging.debug("follows ctrlz: %s" % ch)
            self.ctrlz = False
            if chr(ch) == '\n':
                # "suspend" the job by giving focus to the input view
                self.give_focus(self.views["InputView"])
            else:
                pass
                # default just ignores the earlier ctrl-z
        try:
            chr(ch)
        except ValueError:
            # This happens when the first character of a full-screen #piped job
            # is exteneded, like the arrow key, but subsequent characters are
            # handled just fine.  So just ignore it.
            logging.info("Attempt to send extended character as raw")
            return
        fsj.job.send_input(chr(ch))

    def handle_curses_input(self, ch):
        if ch == -1:
            # Timeout or screen resize or similar, no character to process.
            return
        if not self.escape and ch == curses.ascii.ESC:
            self.escape = True
            return
        if self.escape and ch == 0x5B:
            # This is the beginning of an ANSI escape sequence, one that
            # ncurses apparently isn't converting for me.  bah.  Ignore it for
            # now, support would ideally be added at ncurses level, but upgrade
            # is an issue.
            # http://en.wikipedia.org/wiki/ANSI_escape_code
            # http://www.xfree86.org/current/ctlseqs.html
            seq = ""
            self.escape = False
            while True:
                ch = self.scr.getch()
                seq += chr(ch)
                if 0x40 <= ch and ch <= 0x7e:
                    break

            logging.debug("control seq: %s" % seq)
            return

        self.focus.putch(ch, esc=self.escape)
        self.escape = False

    def next_window(self):
        """Move focus to the next view."""
        # Rotate layout into a list of candidates for accepting focus.
        cands = list(self.layout)
        if self.focus in cands:
            while cands[-1] != self.focus:
                cands.append(cands.pop(0))

        # Try to change focus
        self.focus.set_focus(has_focus=False)
        for view in cands:
            if view.set_focus(has_focus=True):
                self.focus = view
                return

    def set_main(self, newmain, give_focus=False):
        # Show the provided view object in the main area.
        if newmain is None:
            newmain = self.views["Welcome"]
        logging.debug("Switching display to %s" % newmain.get_name())
        self.show_alert("")
        if self.focus == self.layout[0]:
            give_focus = True
        self.layout[0] = newmain
        if give_focus:
            self.give_focus(newmain)

    def display_job(self, job, show=None, focus=False):
        """Set the job in the current Job View.

        @param job: hsh.jobs.Job or JobView object; -1 to move back a job; +1 to
        move forward a job; or 0 to mean the current one is gone, and
        a new one needs to be chosen.

        @param show: True means show the new job; False means don't change the
        view; None (default) means show the new job if JobView is visible.

        @param focus: boolean indicating if jobview should receive focus."""
        if show == None:
            show = self.views["JobView"] == self.layout[0]
        if type(job) == int:
            jv = self.views["JobView"]
            sl = self.views["SessionList"]
            if jv is None:
                job = None
            if job == -1:
                job = sl.get_predecessor(jv.job)
                if job is None: # Don't fall off the end
                    job = jv
            elif job == 1:
                job = sl.get_successor(jv.job)
                if job is None: # Don't fall off the end
                    job = jv
            elif job == 0:
                job = sl.get_successor(jv.job)
                if job is None:
                    job = sl.get_predecessor(jv.job)

        if isinstance(job, hsh.jobs.Job):
            job = self.get_job_view(job)

        # job is now a JobView or None
        self.views["JobView"] = job
        if show:
            self.set_main(job, give_focus=focus)
            jobid = job and job.job or None
            for view in self.views.values():
                if view is not None:
                    view.set_visible_job(jobid)
                    return

    def display_sessionlist(self):
        "Show the session list when input has created a new job."
        self.set_main(self.views["SessionList"])

    def display_search(self, show=None):
        """Display the search window per user request, boolean argument
        indicates if search should be displayed or hidden, default or None
        means toggle."""
        sv = self.views["SearchView"]
        if show == None:
            show = not sv in self.layout
        if show == False and sv in self.layout:
            self.layout.remove(sv)
            if self.focus == sv:
                if self.presearch_focus in self.layout:
                    self.focus = self.presearch_focus
                else:
                    self.focus = self.layout[0]
        if show == True:
            if sv not in self.layout:
                self.layout.insert(1, sv)
                self.presearch_focus = self.focus
            self.focus = sv

    def relinquish_terminal(self, job):
        # Deinitialize curses so that a subprocess can take control of the
        # terminal.  If the process is being brokered through a tty or pipe,
        # then curses is left initialized, but if the process is being given
        # exclusive control over the terminal, then curses is deinitialized.
        self.full_screen_job = job
        self.scr.clear()
        self.scr.refresh()
        if job.wants_terminal() == 1:
            logging.debug("relinquishing terminal: full-screen mode")
            self.scr.keypad(0)
        else:
            logging.debug("relinquishing terminal: exclusive mode")
            self.input_reassigned = True
            deinit_curses(self.scr)
            if self.tsettings:
                termios.tcsetattr(sys.stdin.fileno(), termios.TCSANOW,
                                  self.tsettings)

    def claim_terminal(self):
        # The subprocess is done with the terminal, so take it back.
        if self.full_screen_job.wants_terminal() == 1:
            logging.debug("reclaiming terminal: full-screen mode")
            curses.noecho()
            curses.raw()
            self.scr.keypad(1)
        else:
            logging.debug("reclaiming terminal: exclusive mode")
            # Restore the very original terminal settings in case the process
            # broke things.
            if self.tsettings:
                termios.tcsetattr(sys.stdin.fileno(), termios.TCSANOW,
                                  self.tsettings)
            self.scr = init_curses()
            self.input_reassigned = False
        self.full_screen_job = None
        self.scr.clear()
        for view in self.layout:
            view.set_dirty()
        curses.flushinp()

    ######################################################################
    # JobManagerListener interface

    def on_remove_job(self, job):
        """If the removed job is the currently displayed one, update."""
        if self.views["JobView"] and self.views["JobView"].job == job:
            self.display_job(0)

    ######################################################################
    # HshDisplay Interface

    def show_alert(self, message):
        self.views["Alert"].set_message(message)

    def get_visible_job(self):
        """Return whichever job the user would think is the current one, based
        on the visible display."""
        return self.layout[0].current_job()

    def show_view(self, view_name):
        """Show the view of the given name to the user."""
        if view_name in self.views:
            self.set_main(self.views[view_name], give_focus=True)
        else:
            self.show_alert("No view with that name")

    def get_view(self, view_name):
        """Return a view object for the named view."""
        if view_name in self.views:
            return self.views[view_name]
        else:
            self.show_alert("No view with that name")

    def list_views(self):
        """Return a list of view names."""
        return self.views.keys()


class CursesDisplayThread(threading.Thread):
    # A thread for drawing the curses display.

    def __init__(self, display):
        super(CursesDisplayThread, self).__init__()
        self.display = display

    def _ensure_size(self, layout, h, w):
        """Check that the screen is large enough to be useful.  If it's too
        small, show a message and wait for it to be bigger.  Return true if it
        was big enough first time, and false otherwise."""
        min_w = max(map(lambda x: x.min_width(), layout))
        min_h = reduce(lambda x,y: x+y, map(lambda x: x.min_height(), layout))

        (h, w) = self.display.scr.getmaxyx()
        if (h >= min_h) and (w >= min_w):
            return True

        # Screen is too small, so wait for it to be bigger
        old_focus = self.display.focus
        old_layout = self.display.layout

        while (h < min_h) or (w < min_w):
            self.display.layout = [self.display.views["TooSmall"]]
            self.display.give_focus(self.display.views["TooSmall"])
            self.display.views["TooSmall"].draw(self.display.scr.subwin(0, 0))
            time.sleep(0.1)
            (h, w) = self.display.scr.getmaxyx()

        self.display.layout = old_layout
        self.display.give_focus(old_focus)

        return False

    def _partition_screen(self, layout, h, w):
        """Break the screen up into several windows of the necessary sizes."""
        scr = self.display.scr
        curses_wins = []
        for view in reversed(layout[1:]):
            vh = view.min_height()
            h -= vh
            curses_wins.insert(0, scr.subwin(vh, w, h, 0))
        curses_wins.insert(0, scr.subwin(h, w, 0, 0))

        return curses_wins

    def run(self):
        disp = self.display

        # Remember the view layout and curses size from the last draw cycle.
        prev_layout = []
        (prev_h, prev_w) = (None, None)
        curses_wins = []  # curses windows corresponding to layout.

        while True:
            stdscr = disp.scr
            try:
                force_redraw = False  # If changes force a redraw this time.
                drawn = False         # If something was drawn this time.

                # Check for layout and size changes.
                while True:
                    # Make copies of layout and size so nothing changes mid-draw
                    layout = list(disp.layout)
                    (h, w) = disp.scr.getmaxyx()
                    focus = disp.focus

                    if not self._ensure_size(layout, h, w):
                        # Had to wait for a size change, so previous size is
                        # now wrong.
                        (prev_h, prev_w) = (None, None)
                        # Continue loop to pick up layout and size changes.
                        continue

                    break

                # Handle full screen jobs
                if focus.wants_terminal() != 0:
                    # The focus view has requested the full screen
                    disp.relinquish_terminal(focus)
                    focus.set_has_terminal(True)
                    while (disp.focus == focus and
                           focus.wants_terminal()):
                        time.sleep(0.1)
                    focus.set_has_terminal(False)
                    disp.claim_terminal()
                    (prev_h, prev_w) = (None, None) # force a redraw
                    continue

                # Recalculate curses windows if necessary
                if (prev_h,prev_w) != (h,w) or prev_layout != layout:
                    force_redraw = True
                    curses_wins = self._partition_screen(layout, h, w)

                prev_layout = layout
                (prev_h, prev_w) = (h, w)

                # If search is active and has changed since last time then
                # force a redraw.
                sv = None
                if self.display.views["SearchView"] in layout:
                    sv = self.display.views["SearchView"]
                    if sv.is_dirty():
                        force_redraw = True

                for i in range(len(layout)):
                    d1 = layout[i].draw(curses_wins[i], force_redraw, search=sv)
                    sv = None  # Only highlight searches in first view
                    if d1:
                        drawn = True

                if drawn:
                    # Draw the cursor in case a broken drawing routine moved it.
                    for i in range(len(layout)):
                        if layout[i] == focus:
                            layout[i].draw_cursor(curses_wins[i])
                else:
                    # Take a pause if nothing needed to be drawn.
                    time.sleep(0.1)

            except Exception, e:
                self.display.handle_crash(e)
                # Don't spin when dealing with an exception
                time.sleep(1)
