#
# Copyright (C) 2010 Alexander Taler <dissent@0--0.org>
#

# This file is part of hsh.

# hsh is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# hsh is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with hsh.  If not, see <http://www.gnu.org/licenses/>.

######################################################################

import curses
import logging

from hsh.exceptions import *
from hsh.lib import *

import hsh.content.text
import hsh.content.text_viewport

from view import View

class ContentView(View):
    """An abstract subclass of view with additional functionality for
    displaying content provided as a Text object.

    It provides a draw() method, formatting options including linefolding, and
    maintains a viewport and cursor on the provided Text.

    Formatting, viewport and cursor tracking are provided by a TextViewport.

    Limited editing functionality is provided.
    """

    def __init__(self, display, text, name=None):
        """Initialize the new ContentView.

        @param display: Where it will be drawn
        @type display: L{CursesDisplay}
        @param text: What to draw
        @type text: L{Text} or an array of strings
        @param name: Optional unique name, defaulting to class name.
        @type name: string or None
        """
        super(ContentView, self).__init__(display, name)
        self.text = text
        self.textf = hsh.content.text_viewport.TextViewport(text)
        self.textf.set_format(face_view_name = self.get_name())

        # editable: boolean, if this content view supports editing commands
        self.editable = False

        # The last_* variables store values as they were the last time the view
        # was drawn.  They are modified exclusively by the draw() method, but
        # may be of interest to other methods.
        self.cur_height = 1
        self.last_display_pos = None
        self.last_cursor_pos = None
        self.last_drawn = None

    def insert_text(self, text):
        if not self.editable:
            return
        (cy, cx) = self.textf.cursor_pos
        cl = self.text[cy]
        self.text[cy] = cl[0:cx] + text + cl[cx:]
        self.textf.cursor_pos[1] += len(text)
        self.set_dirty()

    def is_dirty(self):
        if (self.textf.display_pos != self.last_display_pos):
            return "all"
        return super(ContentView, self).is_dirty()

    def draw_cursor(self, win):
        cy, cx = self.textf.get_cursor_pos()
        # I want to use win.move(cy, cx), but it doesn't work for me.
        (py, px) = win.getparyx()

        (height, width) = win.getmaxyx()
        cy = clamp(0, cy + self.header_size, height)
        cx = clamp(0, cx, width)

        self.display.scr.move(py + cy, px + cx)

        curses.curs_set(2)  # Make the cursor visible again.
        self.display.scr.refresh()

    def draw(self, win, force_redraw=False, search=None):
        """Draw this view on the provided curses window.  If force_redraw is
        True, then do a full redraw, otherwise just an update is enough."""

        if force_redraw:
            self.set_dirty()

        if not self.is_dirty():
            if self.textf.cursor_pos != self.last_cursor_pos:
                self.last_cursor_pos = list(self.textf.cursor_pos)
                return True
            else:
                return False

        # It's dirty, so do some drawing.

        dirty = self.is_dirty()
        self.set_dirty(None)
        self.last_display_pos = list(self.textf.display_pos)
        self.last_cursor_pos = list(self.textf.cursor_pos)

        self.draw_header(win)

        # Recalculate sizes in case the header changed size.
        cwin = self._content_win(win)
        cwin.leaveok(1)

        (height, width) = cwin.getmaxyx()
        self.cur_height = height

        self.textf.set_size(height, width)

        # Get the necessary formatted text.
        f_c = self.textf.get_lines(search)

        lni = 0

        # In append mode, no need to overwrite unchanged lines.
        if dirty == "append":
            while (lni < len(f_c) and f_c[lni].line_number < self.last_drawn):
                lni += 1
        else:
            cwin.clear()

        # Draw the remaining lines.
        while (lni < height and lni < len(f_c)):
            self.draw_line(f_c[lni], lni, cwin)
            lni += 1

        if lni > 0 and len(f_c) >= lni:
            self.last_drawn = f_c[lni-1].line_number
        else:
            self.last_drawn = None

        cwin.refresh()
        cwin.leaveok(0)
        return True

    ######################################################################

    def move_search(self, pattern, forward=True, from_end=False):
        """Update the display position so the next match of the given pattern
        is visible."""
        self.set_dirty()
        self.textf.move_search(pattern, forward, from_end)

    ######################################################################
    # Functions bound to key strokes

    def move_left(self, ki):
        if self.editable:
            self.textf.move_cursor_left(ki.num)
        else:
            self.textf.move_display_left(ki.num)

    def move_right(self, ki):
        if self.editable:
            self.textf.move_cursor_right(ki.num)
        else:
            self.textf.move_display_right(ki.num)

    def move_up(self, ki):
        if self.editable:
            self.textf.move_cursor_up(ki.num)
        else:
            self.textf.move_display_up(ki.num)

    def move_down(self, ki):
        if self.editable:
            self.textf.move_cursor_down(ki.num)
        else:
            self.textf.move_display_down(ki.num)

    def move_start(self, ki):
        if self.editable:
            self.textf.move_cursor_start()
        else:
            self.textf.move_display_start()

    def move_end(self, ki):
        if self.editable:
            self.textf.move_cursor_end()
        else:
            self.textf.move_display_end()

    def move_top(self, ki):
        self.textf.move_top()

    def move_bottom(self, ki):
        self.textf.move_bottom()

    def page_up(self, ki):
        ki.num *= self.textf.height - 1
        self.move_up(ki)

    def page_down(self, ki):
        ki.num *= self.textf.height - 1
        self.move_down(ki)

    def delete_left(self, ki):
        if not self.editable:
            return
        (cy, cx) = self.textf.cursor_pos
        if (cy, cx) == (0, 0): return
        self.textf.move_cursor_left(1)
        cl = self.text[cy]
        if cx > 0:
            self.text[cy] = cl[0:cx-1] + cl[cx:]
        elif cy > 0:
            self.text[cy-1] += self.text[cy]
            del self.text[cy:cy+1]
        self.set_dirty()

    def delete_right(self, ki):
        if not self.editable:
            return
        (cy, cx) = self.textf.cursor_pos
        cl = self.text[cy]
        if len(cl) > cx:
            self.text[cy] = cl[0:cx] + cl[cx+1:]
        elif len(self.text) > cy + 1:
            self.text[cy] = cl + self.text.pop(cy+1)
        self.set_dirty()

    def delete_line(self, ki):
        if not self.editable:
            return
        (cy, cx) = self.textf.cursor_pos
        if len(self.text) > cy:
            self.display.last_copy = self.text[cy][cx:]
            self.text[cy] = self.text[cy][0:cx]
        self.set_dirty()

    def insert(self, ki):
        return self.insert_text(ki.ch())

    def paste(self, ki):
        return self.insert_text(self.display.last_copy)

    def toggle_wrap(self, ki):
        self.textf.set_format(wrap = not self.textf.get_format('wrap'))
