#
# Copyright (C) 2010 Alexander Taler <dissent@0--0.org>
#

# This file is part of hsh.

# hsh is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# hsh is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with hsh.  If not, see <http://www.gnu.org/licenses/>.

######################################################################

import logging

from text_formatter import TextFormatter
from text import Text

class TextViewport(TextFormatter):
    """Extends TextFormatter with functionality for tracking a displayed area
    within the formatted text, tracking a cursor, and regular expression search
    through the text."""

    def __init__(self, text):
        """Create a viewport on the given Text object positioned at [0,0]

        @param text: The text to view.
        @type text: L{Text}
        """
        super(TextViewport, self).__init__(text)
        # display_pos and cursor_pos are coordinates in unformatted content.
        # height is the number of lines in the viewport.
        # When tail is true the viewport will track the bottom of the content.
        self.display_pos = [0,0]
        self.cursor_pos = [0,0]
        self.height = 0
        self.tail = False

        # search_pos tracks the last search, to handle multiple matches per line
        self.search_pos = [[], []]

    def get_tail(self):
        return self.tail

    def set_tail(self, tail):
        """Instruct this viewport to start or stop tailing the content.
        @param tail: If True, always show the end of the content.
        @type tail: boolean
        """
        if tail == self.tail: return

        self.tail = tail
        self.set_format(reverse = tail)
        if not tail:
            self.coord_up_fline(self.display_pos, clamp=True, count=self.height)
            self.cursor_pos[:] = self.display_pos

    def set_size(self, height, width):
        """Set the size of this viewport, adjusting position to ensure that the
        cursor is visible."""
        self.set_format(width=width)
        self.height = height
        self._align_display()

    def _align_display(self):
        """If necessary, adjust display_pos to ensure that cursor_pos is within
        the viewport.  This is necessary when the viewport changes size, or
        after cursor movement.
        """
        if self.tail:
            self.display_pos = [self.text.end()[0] + 1, 0]
            self.cursor_pos[:] = self.display_pos
            return
        cv = self.coord_diff(self.cursor_pos, self.display_pos)
        if self.format['wrap']:
            while cv[0] < 0:
                cv[0] += 1
                if not self.coord_up_fline(self.display_pos, clamp=True):
                    logging.warn("misaligned display")
                    break
            while cv[0] >= self.height:
                cv[0] -= 1
                if not self.coord_down_fline(self.display_pos, clamp=True):
                    logging.warn("misaligned display")
                    break
        else:
            if cv[0] <= 0: self.display_pos[0] += cv[0]
            if cv[0] >= self.height: self.display_pos[0] += cv[0] - self.height

            # Horizontal adjustments are done using the trim format.
            if self.cursor_pos[1] < self._trim:
                self._trim = max(self.cursor_pos[1], 0)
            if self.cursor_pos[1] > self._trim + self.text_width():
                self._trim = self.cursor_pos[1] - self.text_width()

    def _len_curline(self):
        """Return the length of the line containing the cursor."""
        if self.cursor_pos[0] >= len(self.text):
            return 0
        else:
            return len(str(self.text[self.cursor_pos[0]]).strip('\n'))

    def get_lines(self, search=None):
        """Get formatted lines appropriate for drawing to a UI window.

        @param search: Optional regexp to be highlighted in results.
        @type search: regular expression"""
        self._align_display()
        return super(TextViewport, self).get_lines(self.height,
                                                   self.display_pos,
                                                   search=search)

    # Private property to simplify access to trim format.
    def _get_trim(self):
        "Easy access to the trim format."
        return self.get_format('trim')
    def _set_trim(self, nt):
        return self.set_format(trim=nt)
    _trim = property(_get_trim, _set_trim)

    def get_cursor_pos(self):
        """Get the cursor position relative to the top left corner of the
        output from get_lines().

        @return: cursor position
        @rtype: Coordinates"""
        if not self.tail:
            origin = [self.display_pos[0], max(self.display_pos[1], self._trim)]
        else:
            origin = [self.display_pos[0], self.display_pos[1]]
            self.coord_up_fline(origin, clamp=True, count=self.height)

        return self.coord_visi(self.cursor_pos, origin)

    def get_end_pos(self):
        """Get the text end position relative to the top left corner of the
        output from get_lines().

        @return: cursor position
        @rtype: Coordinates"""
        if not self.tail:
            origin = [self.display_pos[0], max(self.display_pos[1], self._trim)]
        else:
            origin = [self.display_pos[0], self.display_pos[1]]
            self.coord_up_fline(origin, clamp=True, count=self.height)

        return self.coord_visi(self.text.end(), origin)

    def move_cursor_left(self, count):
        self.set_tail(False)
        for i in range(count):
            if self.cursor_pos[1] > self._trim:
                self.cursor_pos[1] -= 1
            elif self.cursor_pos[1] > 0:
                self.cursor_pos[1] -= 1
                self._trim -= 1
            elif self.cursor_pos[0] > 0:
                self.cursor_pos[0] -= 1
                self.cursor_pos[1] = self._len_curline()

    def move_display_left(self, count):
        self.set_tail(False)
        for i in range(count):
            if self._trim > 0:
                self._trim -= 1
            if self.display_pos[1] > self._trim:
                self.display_pos[1] -= 1
                self.cursor_pos[1] = self.display_pos[1]

    def move_cursor_right(self, count):
        self.set_tail(False)
        for i in range(count):
            if self.cursor_pos[1] < self._len_curline():
                self.cursor_pos[1] += 1
            elif self.cursor_pos[0] + 1 < len(self.text):
                self.cursor_pos[0] += 1
                self.cursor_pos[1] = self._trim

    def move_display_right(self, count):
        self.set_tail(False)
        for i in range(count):
            if self._trim < self.text.max_line_length() - 1:
                self._trim += 1
                if self.display_pos[1] < self._trim:
                    self.display_pos[1] += 1
                    self.cursor_pos[1] = self.display_pos[1]

    def move_cursor_up(self, count):
        self.set_tail(False)
        self.coord_up_fline(self.cursor_pos, clamp=True, count=count)

    def move_display_up(self, count):
        self.set_tail(False)
        self.coord_up_fline(self.display_pos, clamp=False, count=count)
        self.cursor_pos[:] = self.display_pos

    def move_cursor_down(self, count):
        self.set_tail(False)
        self.coord_down_fline(self.cursor_pos, clamp=True, count=count)

    def move_display_down(self, count):
        self.set_tail(False)
        self.coord_down_fline(self.display_pos, clamp=False, count=count)
        self.cursor_pos[:] = self.display_pos

    def move_cursor_start(self):
        self.set_tail(False)
        self.cursor_pos[1] = 0

    def move_display_start(self):
        self.set_tail(False)
        self._trim = 0
        self.display_pos[1] = 0
        self.cursor_pos[1] = self.display_pos[1]

    def move_cursor_end(self):
        self.set_tail(False)
        self.cursor_pos[1] = self._len_curline()

    def move_display_end(self):
        self.set_tail(False)
        ll = self._len_curline()
        self._trim = max(0, ll - self.text_width())
        self.display_pos[1] = self._trim
        self.cursor_pos[1] = self.display_pos[1]

    def move_top(self):
        self.set_tail(False)
        self.cursor_pos[:] = [0, self._trim]

    def move_bottom(self):
        self.cursor_pos[:] = self.text.end()
        self.cursor_pos[1] += 1 # Cursor can always point past end of line

    def move_search(self, pattern, forward=True, from_end=False):
        """Update the display and cursor positions to the next match of the
        given pattern.

        @param pattern: Pattern to search for
        @type pattern: compiled regexp
        @param forward: True to search forward, False for backwards
        @param from_end: True means start searching from the beginning/end of
                         the text instead of the display position
        """
        if pattern is None: return
        self.set_tail(False)

        # Determine where to start search from
        if from_end:
            from_pos = forward and [0,0] or self.text.end()
        else:
            from_pos = self.cursor_pos

        while True:
            # search_pos tracks the bounds of the previous search match, so it
            # can be skipped over.  it has two elements:
            #   position of beginning of last match
            #   position of end of last match.
            if self.search_pos[0] == from_pos:
                from_pos = forward and self.search_pos[1] or self.search_pos[0]

            match = self.search(pattern, from_pos, forward)

            if match is None:
                return False

            self.search_pos = match

            # If match is on current visible line, try again
            if self.coord_diff(match[0], self.display_pos)[0] == 0:
                from_pos = match[0]
                continue

            # Update display for a successful match.
            self.cursor_pos[:] = match[0]
            self.display_pos[0] = match[0][0]
            self.display_pos[1] = match[0][1] - match[0][1] % self.text_width()
            return True
