#
# Copyright (C) 2010 Alexander Taler <dissent@0--0.org>
#

# This file is part of hsh.

# hsh is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# hsh is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with hsh.  If not, see <http://www.gnu.org/licenses/>.

######################################################################

import logging

import hsh
import hsh.content.text

from hsh.lib import *

def pos_cmp(pos1, pos2):
    """Compare two vector positions.
    @param pos1: vector pair
    @param pos2: vector pair
    @return: -1 if pos1 is before pos2, 0 if they are equal, 1 if it's after.
    """
    if pos1[0] > pos2[0]: return 1
    if pos1[0] < pos2[0]: return -1
    if pos1[1] > pos2[1]: return 1
    if pos1[1] < pos2[1]: return -1
    return 0

class TextFormatter(object):
    """Formats a given Text object to make it appropriate for presentation to
    the user, most importantly supporting line-folding for fixed width displays.
    Details of formatting are configurable through assigning of format options.

    Formatting includes assigning faces, such as colours for curses displays,
    to regions within the formatted output.  In addition to faces used by the
    formatter itself (eg. wrap markers and search highlighting) input text can
    be tagged with a source, which is used to determine a face.  The formatter
    uses its association with a specific view to choose specific faces.

    Coordinates as defined in L{Text} are heavily utilized, including functions
    for manipulating them from the perspective of the visible formatting.
    """

    def __init__(self, text):
        """Create a formatter for the given text with default formatting.

        @param text: The text to be formatted.
        @type text: Text
        """
        self.text = text
        self.format = {
                       'face_default': "text",
                       'face_prefix': "prefix",
                       'face_search': "search",
                       'face_view_name' : "",
                       'face_wrap': "wrapmark",
                       'prefix' : '',
                       'reverse' : False,
                       'trim' : 0,
                       'truncate' : None,
                       'width' : 80,
                       'wrap' : True,
                       }

    def get_format(self, key):
        """Return the format option for the given key."""
        return self.format[key]

    def set_format(self, **kwargs):
        """Set formatting options.  All parameters are optional, and only
        specified parameters are modified.

        @keyword face_default: Name of the default face.
        @type face_default: string
        @keyword face_prefix: Name of the face to use on the line prefix.
        @type face_prefix: string
        @keyword face_search: Name of the face to show search matches.
        @type face_search: string
        @keyword face_view_name: Name of the view showing this text, to assist
                                 with locating the named faces.
        @type face_view_name: string
        @keyword face_wrap: Name of the face to use on wrap markers
        @type face_wrap: string
        @keyword prefix: Text to be prepended to each line before formatting.
        @type prefix: string
        @keyword reverse: If True, formatting goes backwards from display_pos.
        @type reverse: Boolean
        @keyword trim: Number of characters to trim from the beginning of each
                       line before formatting.
        @type trim: integer >= 0
        @keyword truncate: Return at most this number of lines, and include a
                           truncation marker if some lines were skipped.
        @type truncate: integer or None
        @keyword width: The number of columns available for formatting text.
        @type width: integer > 0
        @keyword wrap: True to wrap text onto subsequent lines if it exceeds
                       width, False to clip it at width.
        @type wrap: Boolean
        """
        for k in self.format.keys():
            if k in kwargs.keys():
                if self.format[k] != kwargs[k]:
                    self._clear_cache()
                    self.format[k] = kwargs[k]

    def get_text(self):
        """@returns: Text being formatted.
        @rtype: Text
        """
        return self.text

    def set_text(self, text):
        """@param text: New text to be formatted.
        @type text: Text
        """
        self._clear_cache()
        self.text = text

    def _clear_cache(self):
        """Forget any cached formatting work because something has changed."""
        pass

    def _get_face(self, formatname=None, facename=None):
        """Return a face for this specific view, specified either by its format
        name or face name.

        @param formatname: key into format options which identifies a facename
        @param facename: a face name as specified in display configuration
        """
        if formatname is not None:
            facename = self.format[formatname]
        return hsh.display_obj.get_face(self.format['face_view_name'], facename)

    def _set_face(self, region):
        """Set a face for a region, either using the region's source
        (eg. stderr) if specified, any face already assigned, or a default."""
        if "source" in region.__dict__:
            region.face = self._get_face(facename=region.source)
        if "face" not in region.__dict__:
            region.face = self._get_face(formatname="face_default")

    def char_at(self, coord):
        """Return the character at the given raw coordinates, or None."""
        if coord[0] != clamp(0, coord[0], len(self.text) - 1): return
        line = self.text[coord[0]]
        if coord[1] != clamp(0, coord[1], len(line) - 1): return
        return line[coord[1]]

    ######################################################################
    # Coordinate calculations

    def text_width(self):
        """The actual width for drawing text, accounting for wrap columns at
        either end and a prefix."""
        return self.format['width'] - 2 - len(self.format['prefix'])

    def coord_line_start(self, coord_src):
        """Modify the given raw coordinates to the beginning of the line."""
        coord_src[1] -= coord_src[1] % self.format['width']

    def coord_diff(self, coord_src, coord_origin):
        """Given two sets of coordinates in raw text, determine the
        difference between them in formatted coordinates.

        Typically this is used to find the location of the cursor relative to
        the top left corner of the display, in which case the cursor
        coordinates would be passed as the first parameter, and the display
        position as the second.

        @param coord_src: The raw coordinates to locate
        @type coord_src: Coordinates
        @param coord_origin: The position relative to which the result is given
        @type coord_origin: Coordinates
        @return: difference between positions in formatted coordinates
        @rtype: Coordinates
        """
        if len(self.text) == 0: return [0,0]

        ret = [0,0]
        c_ori = list(coord_origin)
        # Move vertically until src and ori are on the same line.
        while coord_src[0] < c_ori[0]:
            ret[0] -= 1
            if not self.coord_up_fline(c_ori, clamp=False, beyond=1):
                logging.warn("misaligned display %s" %[c_ori, coord_src])
                break
        while coord_src[0] > c_ori[0]:
            ret[0] += 1
            if not self.coord_down_fline(c_ori, clamp=False, beyond=1):
                logging.warn("misaligned display %s" %[c_ori, coord_src])
                break

        # Move horizontally.
        width = self.text_width()
        if self.format['wrap']:
            if coord_src[1] < c_ori[1]:
                ret[0] -= (c_ori[1] - coord_src[1]) / width + 1
                ret[1] = width - (c_ori[1] - coord_src[1]) % width
            elif coord_src[1] > c_ori[1]:
                ret[0] += (coord_src[1] - c_ori[1]) / width
                ret[1] = (coord_src[1] - c_ori[1]) % width
        else:
            ret[1] = coord_src[1] - c_ori[1]

        return ret

    def coord_visi(self, coord_src, coord_origin):
        """Given source and origin coordinates in raw text, with origin
        representing the top left displayed position, return the coordinates of
        source in the visible area, adjusted for formatting, wrap columns and
        line prefix.
        """
        ret = self.coord_diff(coord_src, coord_origin)
        ret[1] += 1 # Wrap marker column
        if (not self.format['wrap']) or (coord_src[1] < self.text_width()):
            ret[1] += len(self.format['prefix'])
        return ret

    def coord_up_fline(self, coord, clamp, count=1, beyond=0):
        """Given raw coordinates, modify them so that they refer to a cursor
        position in a formatted line above.

        @param coord: Raw coordinates
        @param clamp: If True, adjust coord[1] so that the coords refer to
                      a place that exists, possibly outside the visible area.
        @param count: how many lines up to go
        @param beyond: If 1, allow coord to go one line beyond the top
        @return: The number of rows moved up
        @rtype: int
        """
        width = self.text_width()
        trim = self.format['trim']
        mc = 0
        while mc < count:
            if self.format['wrap']:
                if coord[1] >= trim + width:
                    coord[1] -= width
                elif coord[0] > 0:
                    coord[0] -= 1
                    # adjust coord[1] to be at the end of the new line.
                    if coord[0] < len(self.text):
                        ll = max(0, len(self.text[coord[0]]) - trim)
                    else:
                        ll = 0
                    vis_offset = (coord[1] - trim) % width
                    coord[1] = trim + (ll - ((ll - 1) % width + 1)) + vis_offset
                    # Ignore newlines which have been wrapped onto a line all
                    # by themselves.
                    if self.char_at(coord) == '\n':
                        if coord[1] > 0 and (coord[1] % width) == 0:
                            coord[1] -= width
                elif coord[0] > 0 - beyond:
                    coord[0] -= 1
                    coord[1] = 0
                else:
                    break
            else:
                if coord[0] <= 0 - beyond:
                    break
                coord[0] -= 1
            mc += 1

        if clamp and coord[0] < len(self.text):
            coord[1] = min(coord[1], len(self.text[coord[0]]))
        return mc

    def coord_down_fline(self, coord, clamp, count=1, beyond=0):
        """Given raw coordinates, modify them so that they refer to the cursor
        position in the formatted line directly below.

        @param coord: Coordinates
        @param clamp: If True, adjust coord[1] so that the coords refer to
                      a place that exists, possibly outside the visible area.
        @param count: how many lines down to go
        @param beyond: If 1, allow coord to go one line beyond the bottom
        @return: The number of rows moved down
        @rtype: int
        """
        width = self.text_width()
        trim = self.format['trim']
        mc = 0
        while mc < count:
            if self.format['wrap']:
                if coord[0] >= len(self.text):
                    break
                # Subtract 1 to ignore the terminating \n
                if coord[1] + width < len(self.text[coord[0]]) - 1:
                    coord[1] += width
                elif coord[0] < len(self.text) - 1 + beyond:
                    coord[0] += 1
                    coord[1] = trim + (coord[1] - trim) % width
                else:
                    break
            else:
                if coord[0] >= len(self.text) - 1 + beyond:
                    break
                coord[0] += 1
            mc += 1

        if clamp and coord[0] < len(self.text):
            coord[1] = min(coord[1], len(self.text[coord[0]]))
        return mc

    ######################################################################
    # Line formatting

    def get_lines(self, height, display_pos=[0,0], search=None, text=None,
                  more=None):
        """Get formatted lines suitable for drawing to a fix-width UI window.

        Return the requested number of lines, or fewer if that's not possible.

        @param height: The number of lines to return
        @type height: positive integer
        @param display_pos: Where to start drawing from.  If reverse formatting
                            is set, drawing goes upwards.
        @type display_pos: Coordinates
        @param search: Optional regexp to be highlighted in results.
        @type search: regular expression or None
        @param text: Optional text to be formatted, replacing current text.
        @type text: Text
        @param more: Optional list which will have a boolean placed in it
                     indicating if formatting stopped before the end.
        @type more: list or None

        @return: Appropriately formatted text, preserving any formatting
            regions of the original source.  Columns containing line
            continuation and clipping markers are added to either side of the
            output.  Each contained L{TextLine} its C{line_number} variable set
            to the number of the line from which it originated, and regions are
            updated to include face information based on input channel.
        @rtype: L{MutableText}
        """
        if text is not None:
            self._clear_cache()
            self.text = text

        text_fmt = hsh.content.text.MutableText()

        display_pos = list(display_pos) # so it can be modified
        trunc = self.get_format('truncate')
        reverse = self.get_format('reverse')

        if reverse and trunc is not None:
            # If the truncation point is higher, everything fits so draw from
            # the top.
            trunc_pos = [0,0]
            self.coord_down_fline(trunc_pos, clamp=False, count=trunc)
            if pos_cmp(trunc_pos, display_pos) == -1:
                display_pos = [0,0]
                reverse = False

        if reverse:
            # Move display_pos backwards and then format forwards like usual.
            # First move to the beginning of the line.
            if self.char_at(display_pos) == '\n':
                display_pos[1] -= 1
            display_pos[1] -= display_pos[1] % self.text_width()
            tc = self.coord_up_fline(display_pos, clamp=False, count=height)
            height = min(height, tc)

        lnum = display_pos[0]

        linecount = height
        if trunc is not None and trunc < linecount:
            linecount = trunc

        while len(text_fmt) < linecount and lnum < len(self.text):
            line = self.text[lnum]

            # Create a new TextLine object which we are free to modify
            if isinstance(line, hsh.content.text.TextLine):
                line = hsh.content.text.TextLine(line)
            else:
                line = hsh.content.text.TextLine(str(line).strip('\n') + '\n')

            # Modify line so that strings matching the search are highlighted.
            if search:
                search_face = self._get_face(formatname='face_search')
                line = search.highlight_regions(line, search_face)

            # Handle special characters
            self._clean_control_characters(line)

            # Format the line, possibly resulting in multiple lines
            ltrim = self.format['trim']
            prewrap = 0
            if lnum == display_pos[0]:
                prewrap = display_pos[1]

            line_fmt = self._format_line(line, ltrim, prewrap, lnum)

            text_fmt.extend(line_fmt)
            lnum += 1

        truncated = (lnum < len(self.text)) or (len(text_fmt) > linecount)

        if more is not None:
            more[:] = [(lnum < len(self.text) or
                        len(text_fmt) > height or
                        trunc and truncated and linecount == height)]

        # Ensure the right number of lines are returned
        if len(text_fmt) > linecount:
            text_fmt[linecount:] = []

        if trunc and truncated:
            face = self._get_face(formatname="face_wrap")
            text_fmt.append_region(' ...\n', attrs={"face":face})

        return text_fmt

    def _clean_control_characters(self, line):
        """Handle ASCII control characters to prevent drawing corruption.

        This support is still very limited: tabs are replaced with spaces, and
        carriage returns are removed.

        @param line: the input to clean - will be modified
        @type line: TextLine
        """
        tw = hsh.config.tab_width
        i = 0
        while i < len(line):
            if line[i] == '\r':
                line[i:i+1] = ''
            elif line[i] == '\t':
                sc = (i / tw + 1) * tw - i
                line[i:i+1] = " " * sc
                i += sc
            else:
                i += 1

    def _format_line(self, line, trim, prewrap, lnum):
        face_default = self._get_face(formatname='face_default')
        wrapface     = self._get_face(formatname='face_wrap')
        wrap_width = self.format['width'] - 1

        line_fmt = hsh.content.text.MutableText()

        self._format_line_start(line_fmt, wrapface, lnum, trim, prewrap)
        trim = max(prewrap, trim)
        for region in line.regions:
            if len(region) <= trim:
                trim -= len(region)
                continue

            rg = line_fmt.append_region(region)
            self._set_face(rg)
            if trim > 0:
                rg[0:trim] = []
                trim = 0

            if line_fmt[-1][-1] == '\n':
                line_fmt[-1][-1:] = []
            while len(line_fmt[-1]) > wrap_width:
                spill = "".join(line_fmt[-1][wrap_width:])
                line_fmt[-1][wrap_width:] = []
                if self.format['wrap']:
                    self._format_line_wrap(line_fmt, wrapface, lnum)
                    rg = line_fmt.append_region(spill)
                    rg.set_attrs(region)
                    self._set_face(rg)
                else:
                    self._format_line_clip(line_fmt, wrapface)
                    break

        self._format_line_end(line_fmt)

        return line_fmt

    def _format_line_wrap(self, line_fmt, wrapface, lc):
        "Internal function for _format_line()"
        line_fmt.append_region('\\\n', {"face" : wrapface})
        rg = line_fmt.append_region('/', {"face" : wrapface})
        rg.get_line().line_number = lc

    def _format_line_clip(self, line_fmt, wrapface):
        "Internal function for _format_line()"
        line_fmt.append_region('_', {"face" : wrapface})

    def _format_line_start(self, line_fmt, wrapface, lc, trim, prewrap):
        "Internal function for _format_line()"
        fc = (trim > 0) and '-' or (prewrap > 0) and '/' or ' '
        rg = line_fmt.append_region(fc, {"face" : wrapface})
        rg.get_line().line_number = lc
        if self.format['prefix']:
            rg = line_fmt.append_region(self.format['prefix'])
            rg.face = self._get_face(formatname='face_prefix')

    def _format_line_end(self, line_fmt):
        "Internal function for _format_line()"
        rg = line_fmt.append_region('\n')


    ######################################################################
    # Search

    def search(self, pattern, from_pos=[0,0], forward=True):
        """Search for a pattern within the raw text.

        Searching stops at the beginning or end of the text.  As a special case
        if forward is False and search is from the beginning, it will start
        searching at the end.

        @param pattern: Pattern to search for
        @type pattern: compiled regexp
        @param from_pos: Position to start search from
        @type from_pos: coordinates
        @param forward: True to search forward, False for backwards
        @return: Pair of raw coordinates, the beginning and end of the match.
                 None if there was no match.
        @rtype: (coordinates,coordinates) or None
        """
        if pattern is None: return

        ######## ****** Hmmm ***** 
        # Special case is breaking list search, is it important?
        if from_pos == [0,0] and forward == False:
            # from_pos = self.text.end()
            logging.debug("warning: is this scenario broken?")

        # The beginning of the line after the end is equivalent to the end
        if from_pos == [len(self.text), 0]:
            from_pos = self.text.end()

        # loop through lines to do the search
        match = None
        while from_pos[0] == clamp(0, from_pos[0], len(self.text) - 1):
            sline = str(self.text[from_pos[0]])
            if forward:
                match = pattern.search(sline, from_pos[1])
                if match:
                    match_s = match.start()
                    match_e = match.end()
            else:
                matches = pattern.findall(sline, 0, from_pos[1])
                if len(matches) > 0:
                    match = matches[-1]
                    match_s = sline[0:from_pos[1]].rfind(match)
                    match_e = match_s + len(match)
            if match:
                return ([from_pos[0],match_s], [from_pos[0],match_e])

            # No match, so advance to the next line
            if forward:
                from_pos = [from_pos[0] + 1, 0]
            else:
                from_pos = [from_pos[0] - 1, len(self.text[from_pos[0] - 1])]
