#
# Copyright (C) 2010 Alexander Taler <dissent@0--0.org>
#

# This file is part of hsh.

# hsh is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# hsh is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with hsh.  If not, see <http://www.gnu.org/licenses/>.

######################################################################

import curses
import logging

import hsh
import hsh.content

from hsh.exceptions import *

import keybinds

# Each View has associated keybindings, although not all accept
# input.  There are many layers of keybindings in order to allow user
# customization and shared configuration (although they are assembled into
# fewer maps at runtime).  User specified bindings are searched first, followed
# by displayable specific bindings, followed by general keybindings.  Within
# each layer there are two data structures, the first is a hash from key code
# to function, the second is a list of predicate/function pairs, if the first
# returns True, it's considered a match, and the search stops.

class KeyInput(object):
    """Represents a single key input from the user, which may consist of
    multiple keystrokes, such as Esc-M.  Count arguments will be implemented in
    the future."""
    def __init__(self, key, meta=False, num=1):
        self.key = key
        self.meta = meta
        self.num = num

    def ch(self):
        """Return the key value as a single character string, or raise an
        exception if it's not a character."""
        return chr(self.key)

class View(object):
    """An abstract superclass for anything that can be displayed in a simple
    curses window and can receive input.

    Input keystrokes are bound to functions on the instance using a keybinding
    mechanism and configuration file.  An instance can be named by setting the
    name member variable, which will affect keybinding assignment.

    The default draw() method includes support for drawing a header, and
    skipping the draw entirely if it's not needed.  The *_dirty() methods
    manage skipping the draw.  The header is defined by the header_fmt member
    and the header_info() method, or just overriding the draw_header() method.
    """

    @staticmethod
    def format_lines(display, lines, height, width, dpos=[0,0], wrap=True,
                     face=None, wrapface=None, trim=0):
        """Format the provided lines into lines of appropriate size for display.
          lines is a TextContent or a list of NL terminated strings
          wrap  is a boolean, if True, lines should be wrapped
          dpos  is a pair of ints indicating where the display should start
                in the provided lines.  The first is the vertical position,
                from the top line down, the second is the horizontal position
                from left to right.
          width is the width to make the lines
          height is the number of output lines to generate.  If the last line
                needs to be wrapped, more than height lines will be returned
                so that the final input line will be completely included.
          face is the face to use for output lines.  If None and lines is a
               TextContent, then the face from lines will be used, otherwise
               no face will be set.
          trim is the number of characters to trim from the left of each line.

        A TextContent object is returned, preserving any formatting regions of
        the original source.  Additionally, each TextLine within the content
        has its "line_number" variable set to the number of the line from which
        it originated.

        A blank column is included at the beginning and end of each formatted
        line, and line continuation or truncation markers are put in that
        column, using the specified wrapface.
        """

        content_fmt = hsh.content.TextContent()
        wrap_width = width - 1

        if not wrapface:
            wrapface=display.faces["wrapmark"]

        def wrap_line(lc):
            rg = content_fmt.append_region('\\\n')
            rg.face = wrapface
            rg = content_fmt.append_region('/')
            rg.face = wrapface
            rg.get_line().line_number = lc

        def trunc_line():
            rg = content_fmt.append_region('_')
            rg.face = wrapface

        def start_line(lc):
            if trim > 0:
                rg = content_fmt.append_region('-')
            else:
                rg = content_fmt.append_region(' ')
            rg.face = wrapface
            rg.get_line().line_number = lc

        def end_line():
            rg = content_fmt.append_region('\n')

        loff = dpos[1]  # Drop this much from the first line.
        lc = dpos[0] - 1
        while lc + 1 < len(lines):
            lc += 1
            line = lines[lc]
            lt = max(trim - loff, 0)

            # Break up the lines, but preserve region info
            if not isinstance(line, hsh.content.TextLine):
                line = hsh.content.TextLine(str(line).strip('\n') + '\n')

            start_line(lc)
            for region in line.regions:
                if len(region) <= loff:
                    loff -= len(region)
                    continue
                if len(region) <= lt:
                    lt -= len(region)
                    continue

                rg = content_fmt.append_region(region)
                if loff > 0:
                    rg[0:loff] = []
                    loff = 0
                if lt > 0:
                    rg[0:lt] = []
                    lt = 0

                if face:
                    rg.face = face
                if content_fmt[-1][-1] == '\n':
                    content_fmt[-1][-1:] = []
                while len(content_fmt[-1]) > wrap_width:
                    spill = "".join(content_fmt[-1][wrap_width:])
                    content_fmt[-1][wrap_width:] = []
                    if wrap:
                        wrap_line(lc)
                        rg = content_fmt.append_region(spill)
                        if face:
                            rg.face = face
                        rg.set_attrs(region)
                    else:
                        trunc_line()
                        break

            end_line()
            loff = 0
            # no point in formatting content that won't fit on the screen
            if len(content_fmt) >= height:
                break

        return content_fmt

    def __init__(self, display, name=None):
        self.display = display

        # Space requirements for this window
        self.min_height_main = 1
        self.min_width_main = 5

        self.has_focus = False
        self.dirty = None

        # The header is a section at the top of display showing information
        # about the view, typically in a different colour, and drawn by the
        # draw_header().
        self.header_fmt = ""
        self.header_wrap = False
        self.header_face = None
        self.header_wrapface = None
        self.header_size = 0 # Set by draw_header()

        if name is not None:
            self.name = name

        # Build up the keybindings for this displayable.  Start with the
        # default bindings, and then override with customizations.

        # Bindings may be represented as strings to indicate method names on
        # this object, or as function objects to be called directly.
        def bind_to_func(bindval):
            args = []
            kwargs = {}
            if bindval is None:
                return None
            if type(bindval) in [list, tuple]:
                if len(bindval) == 0:
                    return None
                if len(bindval) > 1:
                    args = bindval[1]
                if len(bindval) > 2:
                    kwargs = bindval[2]
                bindval = bindval[0]
            if type(bindval).__name__ == 'function':
                return (bindval, args, kwargs)
            if type(bindval) == str:
                bindfunc = self.__getattribute__(bindval)
                if type(bindfunc).__name__ == 'instancemethod':
                    return (bindfunc, args, kwargs)
                raise ConfigException("Binding is not a function: " + bindval)
            raise ConfigException("Binding to wrong type: %s" % bindval)

        # Key bindings
        self.keys = dict(keybinds.keys_default)

        class_keys_name = "keys_" + self.get_name()
        if class_keys_name in keybinds.__dict__:
            self.keys.update(keybinds.__dict__[class_keys_name])

        for (k, v) in self.keys.items():
            self.keys[k] = bind_to_func(self.keys[k])

        # For predicates, form a list with the highest priority first
        self.preds = list(keybinds.preds_default)

        class_preds_name = "preds_" + self.get_name()
        if class_preds_name in keybinds.__dict__:
            self.preds[0:0] = keybinds.__dict__[class_preds_name]

        for i in range(len(self.preds)):
            self.preds[i] = (self.preds[i][0], bind_to_func(self.preds[i][1]))

    def putch(self, ch, esc=False):
        # Keys can be bound to functions or instance methods.
        def call(f):
            args = []
            kwargs = {}
            if type(f) in [list, tuple]:
                args = f[1]
                kwargs = f[2]
                f = f[0]
            if f is None:
                return
            if type(f).__name__ == 'instancemethod':
                f(ki, *args, **kwargs)
            else:  # Must be a function.
                f(self, ki, *args, **kwargs)

        # Check keybindings first, and predicates second.
        ki = KeyInput(ch, esc)
        if (ch, esc) in self.keys:
            call(self.keys[(ch, esc)])
        else:
            for pred in self.preds:
                if pred[0](self, ki):
                    call(pred[1])
                    break
        self.set_dirty()

    def get_name(self):
        if 'name' in dir(self):
            return self.name
        else:
            return self.__class__.__name__

    # Some view subclasses may want exclusive terminal access.
    def has_terminal(self):
        return False

    def wants_terminal(self):
        """Return 0 if the terminal is not required, 1 if the terminal is
        required with curses still enabled, 2 if curses should be
        deinitialized."""
        return 0

    def set_has_terminal(self, newval):
        pass

    def min_height(self):
        return self.header_size + self.min_height_main

    def min_width(self):
        return self.min_width_main

    def set_focus(self, has_focus):
        # Called when a displayable receives or loses focus
        if self.has_focus != has_focus:
            self.set_dirty()
        self.has_focus = has_focus

    def header_info(self):
        """Return a dict containing details to include in the header."""
        return {}

    def is_dirty(self):
        return self.dirty

    def set_dirty(self, newdirty="all"):
        """dirty can be None, "append", "all"."""
        if self.dirty == "all" and newdirty == "append":
            return
        self.dirty = newdirty

    def draw_cursor(self, win):
        """Default implementation does not draw a cursor."""
        try:
            win.move(0,0)
            curses.curs_set(1)
            curses.curs_set(0)
        except:
            pass
        self.display.scr.refresh()
        return

    def draw_header(self, win):
        """Draw the header into the provided window.  Return the unused portion
        of the window.  The header is arbitrarily limited to 5 lines, a
        restriction which should be lifted by providing a way to switch between
        header wrap and trim displays."""
        (height, width) = win.getmaxyx()

        if len(self.header_fmt) == 0:
            # Empty header, nothing to draw.
            self.header_size = 0
            return

        # Format the header into flines
        rawh = (self.header_fmt % self.header_info()).split('\n')
        flines = View.format_lines(self.display, rawh, 5, width,
                                   wrap=self.header_wrap, face=self.header_face,
                                   wrapface=self.header_wrapface)

        # If the header changed size, the rest of the window needs a redraw.
        new_header_size = min(height - 1, 5, len(flines))
        if new_header_size != self.header_size:
            self.set_dirty()
        self.header_size = new_header_size

        # Write the relevant output.
        hwin = win.derwin(self.header_size, width, 0, 0)
        hwin.leaveok(1)
        hwin.clear()
        hwin.bkgd(' ', self.header_face)
        for i in range(self.header_size):
            self.draw_line(flines[i], i, hwin)
        hwin.refresh()
        hwin.leaveok(0)

    def set_face(self, region):
        """Given a region, set its face.  This default implementation sets
        the default face from the display."""
        region.face = self.display.faces["default"]

    # Internal function for drawing each line, already formatted for width.
    def draw_line(self, line, yoff, win):
        try:
            xoff = 0
            for region in line.regions:
                if "face" not in region.__dict__:
                    self.set_face(region)
                face = region.face
                win.addstr(yoff, xoff, str(region).strip('\n'), face)
                xoff += len(region)
        except Exception, e:
            # curses returns ERR if it writes all the way to the end of
            # the last line of the window, so ignore any errors.
            if str(e) == 'addstr() returned ERR':
                (h, w) = win.getmaxyx()
                if (len(str(line).strip('\n')) != w or yoff+1 != h):
                    logging.debug("%s", e)
                    logging.debug("%s %s", w, len(str(line).strip('\n')))
                    logging.debug("%s %s", h, yoff)
            else:
                raise

    def _content_win(self, win):
        """Given the window for the whole view, derive one for the non-header
        portion."""
        return win.derwin(self.header_size, 0)

    def draw(self, win, force_redraw):
        """This default implementation of draw() is purely illustrative, and
        only draws the header.  Subclasses will certainly override it.  The
        method returns a boolean, True meaning something was drawn, used by the
        main drawing loop to decide whether it's appropriate to pause."""
        # Unfortunately this code will be duplicated in other draw() methods,
        # because the difficulty of sharing it is too great.
        if force_redraw:
            self.set_dirty()

        if not self.is_dirty():
            return False

        self.set_dirty(None)

        self.draw_header(win)

        cwin = self._content_win(win)
        cwin.leaveok(1)
        # Actual drawing is done here, using draw_line()
        cwin.refresh()
        cwin.leaveok(0)
        return True

    ######################################################################

    def current_job(self):
        """Return the current job associated with this view, or None if there
        is nothing appropriate."""
        return None

    ######################################################################
    # Commands bound to key strokes.

    # Most of these do nothing, and are available only for subclasses which
    # choose to ignore them.  They are part of the default keybindings.

    def next_window(self, ki):
        # Move focus to the next window.
        self.display.next_window()

    def next_job(self, ki):
        pass

    def prev_job(self, ki):
        pass

    def restart_job(self, ki):
        pass

    def change_view(self, ki, name):
        if name:
            self.display.show_view(name)

    def quit(self, ki):
        raise QuitException()

    def delete_job(self, ki):
        pass

    def move_left(self, ki):
        pass

    def move_right(self, ki):
        pass

    def move_up(self, ki):
        pass

    def move_down(self, ki):
        pass

    def move_start(self, ki):
        pass

    def move_end(self, ki):
        pass

    def page_up(self, ki):
        pass

    def page_down(self, ki):
        pass

    def delete_left(self, ki):
        pass

    def delete_right(self, ki):
        pass

    def delete_line(self, ki):
        pass

    def insert(self, ki):
        pass

    def paste(self, ki):
        pass
