#
# Copyright (C) 2010 Alexander Taler <dissent@0--0.org>
#

# This file is part of hsh.

# hsh is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# hsh is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with hsh.  If not, see <http://www.gnu.org/licenses/>.

######################################################################

import curses
import logging

import hsh.content

from view import View

class KeyValueView(View):
    """An abstract superclass for views that display and permit editing of
    key-value type information.  It will be used by the alias and environment
    views.

    Subclasses need to provide self.content_dict with a mutable dictionary that
    this class will modify.

    The view location and current key are tracked by self.curkey and
    self.dispkey.  The position of the cursor on the line is tracked by
    self.cursor_pos, a 3-tuple: first element is 0 if the cursor is in the key
    or 1 if it's in the value; the second is the cursor position in the string;
    the third is the first character from the string to draw.
    """

    def __init__(self, display, name, content_dict):
        if name:
            self.name = name
        self.content_dict = content_dict
        super(KeyValueView, self).__init__(display)

        self.kw = 20 # Width of space for drawing keys.  Values start at kw+1.
        self.min_width_main = 25

        self.curkey = 0
        self.dispkey = 0
        self.cursor_pos = [0,0,0]

        # For tracking values at the last draw.
        self.last_curkey = None
        self.last_cursor_pos = None
        self.cur_height = 1
        self.cur_width = 1

        self.header_fmt = "    %s\n" % name
        self.header_face = self.display.faces["list.header"]

        # To handle editing, the modified text is stored separately, and only
        # written to the dict when editing is finished.
        self.edit_buf = None

    ######################################################################
    # Editing functionality

    def start_editing(self):
        if self.edit_buf is None:
            self.edit_buf = self.get_focus_text()

    def get_focus_text(self):
        """Return a string with the text of the key or value with focus."""
        if self.edit_buf is not None:
            return self.edit_buf
        ks = list(self.content_dict.keys())
        ks.sort()
        if self.cursor_pos[0]:
            return self.content_dict[ks[self.curkey]]
        else:
            return ks[self.curkey]

    def insert_text(self, text):
        self.start_editing()
        eb = self.edit_buf
        self.edit_buf = eb[0:self.cursor_pos[1]] +text+ eb[self.cursor_pos[1]:]
        self.cursor_pos[1] += len(text)
        self.set_dirty()

    ######################################################################
    # Drawing functionality

    def align_cursor(self):
        if len(self.content_dict) == 0:
            return

        # Vertical adjustments to make sure the cursor is visible.
        if self.dispkey > self.curkey:
            self.dispkey = self.curkey
        if self.dispkey + self.cur_height - 1 < self.curkey:
            self.dispkey = self.curkey - self.cur_height + 1

        # Realign the cursor x position after changes to y.
        cp1max = len(self.get_focus_text())
        if self.cursor_pos[0] == 0:
            cp1wid = self.kw - 2
        else:
            cp1wid = self.cur_width - self.kw - 2 - 1
        self.cursor_pos[1] = min(self.cursor_pos[1], cp1max)
        # Check if cursor is visible.
        if self.cursor_pos[2] > self.cursor_pos[1]:
            self.cursor_pos[2] = self.cursor_pos[1]
        elif self.cursor_pos[2] + cp1wid - 1 < self.cursor_pos[1]:
            self.cursor_pos[2] = self.cursor_pos[1] - cp1wid + 1
        self.cursor_pos[2] = min(self.cursor_pos[2], cp1max - 1)
        # Don't hide more than necessary
        if cp1max - self.cursor_pos[2] + 1 < cp1wid:
            self.cursor_pos[2] = max(0, cp1max - cp1wid)

    def draw_cursor(self, win):
        cy = self.curkey - self.dispkey
        cx = self.cursor_pos[0] and (self.kw + 1)
        cx += 1 # Wrap marker column
        cx += self.cursor_pos[1] - self.cursor_pos[2]

        # I want to use win.move(cy, cx), but it doesn't work for me.
        (py, px) = win.getparyx()
        self.display.scr.move(py + cy + self.header_size, px + cx)
        curses.curs_set(2)  # Make the cursor visible again.
        self.display.scr.refresh()

    def draw(self, win, force_redraw):
        
        if (not force_redraw and not self.is_dirty() and 
            self.curkey == self.last_curkey):
            return False

        self.set_dirty(None)
        self.last_curkey = self.curkey
        
        self.draw_header(win)

        cwin = self._content_win(win)
        cwin.leaveok(1)
        cwin.clear()

        (self.cur_height, self.cur_width) = (h, w) = cwin.getmaxyx()
        fface = None
        if self.edit_buf is not None:
            fface = self.display.faces["kv.mod_buff"]

        def append_txt(tl, txt, width, dp, face=None):
            """Append the string txt to the TextLine tl, using at most width
            characters, starting from character dp."""
            rg = tl.append_region(dp > 0 and "_" or " ")
            rg.face = self.display.faces["wrapmark"]
            txta = txt[dp:dp+width - 2]
            if len(txta) > 0:
                rg = tl.append_region(txta)
                if face:
                    rg.face = face
            rg = tl.append_region((len(txt) - dp >= width - 1) and "_" or " ")
            rg.face = self.display.faces["wrapmark"]
            if (width-2) - len(txta) > 0:
                tl.append_region(" "*((width-2) - len(txta)))

        # Draw the keys and values
        dmin = self.dispkey or 0
        dmax = min(dmin + h, len(self.content_dict))
        ks = list(self.content_dict.keys())
        ks.sort()
        for i in range(dmin,dmax):
            k = ks[i]
            v = self.content_dict[k]
            tl = hsh.content.TextLine()
            if self.curkey == i and self.cursor_pos[0] == 0:
                t = self.edit_buf
                if t is None: t = k
                append_txt(tl, t, self.kw, self.cursor_pos[2], face=fface)
            else:
                append_txt(tl, k, self.kw, 0)
            tl.append_region(" ")
            if self.curkey == i and self.cursor_pos[0] == 1:
                t = self.edit_buf
                if t is None: t = v
                append_txt(tl, t, self.cur_width - self.kw - 1,
                           self.cursor_pos[2], face=fface)
            else:
                append_txt(tl, v, self.cur_width - self.kw - 1, 0)

            self.draw_line(tl, i - dmin, cwin)

        cwin.refresh()
        cwin.leaveok(0)
        return True

    ######################################################################
    # Commands bound to key strokes.

    def move_left(self, ki):
        for i in range(ki.num):
            if self.cursor_pos[1] > 0:
                self.cursor_pos[1] -= 1
        self.align_cursor()
        pass

    def move_right(self, ki):
        for i in range(ki.num):
            self.cursor_pos[1] += 1
        self.align_cursor()
        pass

    def move_up(self, ki):
        for i in range(ki.num):
            if self.curkey > 0:
                self.curkey -= 1
        self.edit_buf = None
        self.align_cursor()

    def move_down(self, ki):
        for i in range(ki.num):
            if self.curkey + 1 < len(self.content_dict):
                self.curkey += 1
        self.edit_buf = None
        self.align_cursor()

    def move_start(self, ki):
        self.cursor_pos[1] = self.cursor_pos[2] = 0

    def move_end(self, ki):
        self.cursor_pos[1] = len(self.get_focus_text())
        self.align_cursor()

    def page_up(self, ki):
        ki.num *= self.cur_height - 1
        self.move_up(ki)

    def page_down(self, ki):
        ki.num *= self.cur_height - 1
        self.move_down(ki)

    def cycle_field(self, ki):
        self.cursor_pos[0] = (self.cursor_pos[0] + 1) % 2
        self.cursor_pos[1] = 0
        self.cursor_pos[2] = 0
        self.edit_buf = None
        self.align_cursor()

    def save_field(self, ki):
        ks = list(self.content_dict.keys())
        ks.sort()
        k = ks[self.curkey]
        v = self.content_dict[k]
        if self.cursor_pos[0] == 0:
            # Change to the key, so add a new one and delete the old one.
            if len(self.edit_buf) > 0:
                self.content_dict[self.edit_buf] = v
            del self.content_dict[k]
        else:
            self.content_dict[k] = self.edit_buf
        self.edit_buf = None
        self.set_dirty()
        self.align_cursor()

    def delete_left(self, ki):
        self.start_editing()
        if self.cursor_pos[1] > 0:
            eb = self.edit_buf
            self.edit_buf = eb[0:self.cursor_pos[1]-1]+ eb[self.cursor_pos[1]:]
            self.cursor_pos[1] -= 1
            self.set_dirty()
            self.align_cursor()

    def delete_right(self, ki):
        self.start_editing()
        if self.cursor_pos[1] < len(self.edit_buf):
            eb = self.edit_buf
            self.edit_buf = eb[0:self.cursor_pos[1]]+ eb[self.cursor_pos[1]+1:]
            self.set_dirty()
            self.align_cursor()

    def delete_line(self, ki):
        self.start_editing()
        if self.cursor_pos[1] < len(self.edit_buf):
            eb = self.edit_buf
            self.display.last_copy = eb[self.cursor_pos[1]:]
            self.edit_buf = eb[0:self.cursor_pos[1]]
            self.set_dirty()
            self.align_cursor()

    def insert(self, ki):
        self.insert_text(ki.ch())
        self.align_cursor()

    def paste(self, ki):
        self.insert_text(self.display.last_copy)
        self.align_cursor()

