#
# Copyright (C) 2010 Alexander Taler <dissent@0--0.org>
#

# This file is part of hsh.

# hsh is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# hsh is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with hsh.  If not, see <http://www.gnu.org/licenses/>.

######################################################################

# The format of the user configuration file is to be determined.

import logging
import os
import sys
import time
import traceback

import hsh.command
import hsh.config
import hsh.jobs

from hsh.exceptions import *
from view import View
from content_view import ContentView

class JobView(ContentView, hsh.jobs.JobListener):

    def __init__(self, display, job):
        super(JobView, self).__init__(display)
        self.job = job
        self.content = job.get_output()

        self.header_fmt = hsh.config.header_job
        self.header_face = self.display.faces["job.header"]
        self.header_wrapface = self.display.faces["job.headerwrap"]
        self.header_wrap = True
        # header_size will change, but this is a good start.
        self.header_size = len(self.header_fmt.split('\n'))

        self.job.register_listener(self)

        self.raw_output = False
        self.output_tail = False

        if self.job.can_send_input() or self.job.state == "Initializing":
            self.input_mode = "silent"
        else:
            self.input_mode = "closed"

        # input_buffer is a region of the job content used in
        # line oriented input mode.
        self.input_buffer = None

        self.set_dirty()

    def draw(self, *args, **kwargs):
        if self.is_dirty() and self.input_buffer is not None:
            self._align_inbuff()
            self.move_bottom(None)
        return super(JobView, self).draw(*args, **kwargs)

    ######################################################################
    # View interface

    def header_info(self):
        hi = {'username' : self.job.username,
              'hostname' : self.job.hostname,
              'cwd'      : self.job.cwd,
              'cmd_line' : self.job.cmdline,
              'expanded_cmd' : hsh.command.format_cmd(self.job.cmd),
              'display_line' : self.display_pos[0] + 1,
              'total_lines'  : len(self.content),
              'jobid'        : self.job.jid,
              'pid'          : self.job.pid,
              'input_mode'   : self.input_mode,
              'output_tail'  : self.output_tail and "tailing" or "",
              }
        hi['state'] = self.job.get_state()
        if self.job.get_retcode():
            hi['state'] += "(%s)" % self.job.get_retcode()
        return hi

    def draw_raw(self, text, channel):
        # For raw output, full control of the terminal is needed.  This should
        # ideally be granted by the main thread, so set a flag here asking for
        # it and then wait.
        self.raw_output = True
        self.input_mode = "fullscreen"
        while not self.has_terminal():
            time.sleep(0.1)

        outfile = channel == "stdout" and sys.stdout or sys.stderr
        outfile.write(text)
        outfile.flush()

    def set_focus(self, has_focus):
        if self.has_focus == has_focus:
            return
        self.set_dirty()
        self.has_focus = has_focus
        if self.wants_terminal() != 2:
            return
        # Exclusive jobs require more handling when switching focus
        if self.has_focus:
            # Exclusive job is receiving focus, so start it running.
            self.job.do_exclusive(start_proc=False)
            if self.job.get_state() == "Suspended":
                # The job didn't terminate, so switch focus away from it
                self.display.next_window()
        else:
            # Exclusive job is losing focus, resume regular operation.  This
            # call can't come from the main thread, because exclusive jobs
            # block it.  Presumably it's an error in another thread.
            self.job.suspend()

    def set_face(self, region):
        """Given a region, set its face."""
        super(JobView, self).set_face(region)
        if "source" in region.__dict__:
            region.face = self.display.faces["job." + region.source]

    def current_job(self):
        return self.job

    def has_terminal(self):
        return self.job.has_terminal

    def wants_terminal(self):
        """Return 0 if the job doesn't want special access to the terminal, 1
        if it wants to write curses controls on the terminal, and 2 if it wants
        exclusive access to input also."""
        if self.job.wants_terminal:
            return 2
        elif self.raw_output:
            return 1
        else:
            return 0

    def set_has_terminal(self, newval):
        self.job.has_terminal = newval

    ######################################################################
    # jobs.JobListener interface

    def on_job_output(self, job):
        self.set_dirty("append")
        if self.output_tail:
            self.move_bottom(None)

    def on_job_terminate(self, job):
        self.input_mode = "closed"
        self.job.wants_terminal = False
        self.raw_output = False
        self.set_dirty()

    def on_job_raw_output(self, job, text, channel):
        self.draw_raw(text, channel)

    ######################################################################
    # JobView specific stuff

    def restart(self):
        """Restart this job."""
        self.cursor_pos = [0,0]
        self.display_pos = [0,0]
        if "exclusive" in self.job.get_directives():
            # For exclusive jobs, start() blocks until the job is visible.
            self.display.display_job(self.job, focus=True)
        self.job.start()
        if self.job.get_state() == "Suspended":
            self.display.next_window()
        if self.job.can_send_input():
            self.input_mode = "silent"
        self.set_dirty()

    # Input modes

    # self.input_mode can have these values:
    #   silent:       input is not currently being sent to the process
    #   character:    input is sent to the process one character at a time
    #   line:         input is buffered and sent to the process a line at a time
    #   busy:         process is actively receiving input from another source
    #   fullscreen:   process uses curses and requires the full terminal screen
    #   closed:       the input pipe has been closed

    # Output modes

    # self.output_tail is a boolean, if True then the display will be updated to
    #                  always show the bottom line of output.

    def _align_inbuff(self):
        # Move the line input buffer to the end of the display
        if self.input_buffer is not None:
            buf_txt = str(self.input_buffer)
            self.input_buffer.delete()
            self.input_buffer = self.content.append_region(buf_txt)
            self.input_buffer.source = "in_buff"

    # Commands bound to keystrokes

    def cycle_input(self, ki):
        """Cycle through the various input options."""
        if self.input_mode == "silent":
            self.input_mode = "line"
        elif self.input_mode == "line":
            if self.input_buffer is not None:
                self.input_buffer.delete()
                self.input_buffer = None
            self.input_mode = "character"
        elif self.input_mode == "character":
            self.input_mode = "silent"
        elif self.input_mode == "closed" and self.job.can_send_input():
            self.input_mode = "silent"
        # Other input modes can't be changed

    def close_input(self, ki):
        """Close standard input on the process."""
        if self.input_mode == "line" and self.input_buffer is not None:
            self.input_buffer.delete()
            self.input_buffer = None
        self.job.close_input()
        self.input_mode = "closed"

    def toggle_tail(self, ki):
        self.output_tail = not self.output_tail

    def insert(self, ki):
        if self.input_mode == "character":
            inch = self.content.append_region(ki.ch())
            inch.source = "stdin"
            self.job.send_input(ki.ch())
        elif self.input_mode == "line":
            if self.input_buffer is None:
                self.input_buffer = self.content.append_region(ki.ch())
                self.input_buffer.source = "in_buff"
            else:
                self._align_inbuff()
                self.input_buffer.extend(ki.ch())
        self.set_dirty("append")
        self.move_bottom(ki)

    def input_line(self, ki):
        if self.input_mode == "character":
            self.insert(ki)
        elif self.input_mode == "line":
            if self.input_buffer is None:
                self.input_buffer = self.content.append_region(ki.ch())
            else:
                self._align_inbuff()
                self.input_buffer.extend(ki.ch())
            self.input_buffer.source = "stdin"
            self.job.send_input(str(self.input_buffer))
            self.input_buffer = None
        self.set_dirty("append")
        self.move_bottom(ki)

    def delete_left(self, ki):
        if self.input_mode == "line" and self.input_buffer is not None:
            self.input_buffer[-1:] = ""
            if len(self.input_buffer) == 0:
                self.input_buffer = None
        self.move_bottom(ki)

    def delete_job(self, ki):
        # Delete this job
        if self.job.get_state() == "Running":
            logging.info("Can't delete an active job")
            return
        hsh.jobs.manager.forget_job(self.job)

    def next_job(self, ki):
        self.display.display_job(1)

    def prev_job(self, ki):
        self.display.display_job(-1)

    def restart_job(self, ki):
        self.restart()

