#
# Copyright (C) 2010 Alexander Taler <dissent@0--0.org>
#

# This file is part of hsh.

# hsh is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# hsh is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with hsh.  If not, see <http://www.gnu.org/licenses/>.

######################################################################

import curses
import logging

import hsh.content

from hsh.exceptions import *

from view import View

class ContentView(View):
    """An abstract subclass of view with additional functionality for storing
    content as a TextContent object or an array of strings.

    The draw() method draws the content member variable.

    Internally the content needs to be addressable both in its raw format as
    (essentially) an array of newline terminated strings for tasks like
    editing, and in its wrapped/truncated format as displayed on the screen.
    This requirement is addressed by storing the position of the cursor
    (self.cursor_position) and the position of the top left visible corner
    (self.display_position) as indices into the content array.  Methods are
    then provided for transforming between coordinates in the self.content
    array, and visible coordinates defined as having the origin at the top left
    corner of the display which maps to self.display_pos.
    """

    def __init__(self, display, name=None):
        super(ContentView, self).__init__(display, name)

        # content: an array of strings for the default draw() method.
        # cursor_pos: position of cursor within content
        # display_pos: position of top left corner of screen within content.
        # display_off: number of columns on the left which aren't drawn
        # display_wrap: true to wrap lines, false to truncate them.
        self.content = [""]
        self.cursor_pos = [0,0]
        self.display_pos = [0,0]
        self.display_off = 0
        self.display_wrap = True
        self.editable = False

        # The last_* variables store values as they were the last time the view
        # was drawn.  They are modified exclusively by the draw() method, but
        # may be of interest to other methods.
        self.cur_width = 1
        self.cur_height = 1
        self.last_display_pos = None
        self.last_cursor_pos = None
        self.last_drawn = None

    def _len_line(self, lnum=None):
        """Return the length of the specified line, or by default the current
        line containing the cursor or zero.  If the line is past the end of the
        content, return 0."""
        if lnum is None:
            lnum = self.cursor_pos[0]
        if lnum >= len(self.content):
            return 0
        else:
            return len(str(self.content[lnum]).strip('\n'))

    def insert_text(self, text):
        if not self.editable:
            return
        (cy, cx) = self.cursor_pos
        cl = self.content[cy]
        self.content[cy] = cl[0:cx] + text + cl[cx:]
        self.cursor_pos[1] += len(text)
        self.set_dirty()

    def is_dirty(self):
        if (self.display_pos != self.last_display_pos):
            return "all"
        return super(ContentView, self).is_dirty()

    def _coord_up_vline(self, coord, clamp):
        """Modify the given raw form coords to specify the visible point
        directly above.  Return True if the move succeeded, False if it hit the
        top.  If the line above is too short, realign the cursor to the end.
        If clamp is True, adjust coord[1] so it doesn't point past the end of
        the line, although this may put it outside of the visible area if the
        line is shorter than display_off."""
        if self.display_wrap:
            if coord[1] >= self.cur_width - self.display_off:
                coord[1] -= self.cur_width
            elif coord[0] > 0:
                coord[0] -= 1
                ll = max(0, self._len_line(coord[0]) - self.display_off)
                coord[1] = self.display_off + ll - ll % self.cur_width +coord[1]
            else:
                return False
        else:
            if coord[0] <= 0:
                return False
            coord[0] -= 1

        if clamp:
            coord[1] = min(coord[1], self._len_line(coord[0]))
        return True

    def _coord_down_vline(self, coord, clamp):
        """Like _coord_up_vline but going down."""
        if self.display_wrap:
            if coord[1] + self.cur_width < self._len_line(coord[0]):
                coord[1] += self.cur_width
            elif coord[0] < len(self.content) - 1:
                coord[0] += 1
                coord[1] = (self.display_off +
                            (coord[1] - self.display_off) % self.cur_width)
            else:
                return False
        else:
            if coord[0] >= len(self.content) - 1:
                return False
            coord[0] += 1

        if clamp:
            coord[1] = min(coord[1], self._len_line(coord[0]))
        return True

    def _coord_raw_to_visi(self, coord, clamp):
        """Return the visible coordinates corresponding to raw coordinate
        coord.  Negative values are possible in either result coordinate,
        unless clamp is True in which case the closest value on the screen will
        be returned.  See class description for more details about
        coordinates."""
        if len(self.content) == 0:
            return [0,0]

        ret = [0,0]
        wc = list(coord)
        wd = list(self.display_pos)
        # Move up or down visible lines until wc and wd are on the same line.
        while wc[0] < wd[0]:
            ret[0] -= 1
            if not self._coord_up_vline(wd, clamp=False):
                logging.warn("misaligned display")
                break
        while wc[0] > wd[0]:
            ret[0] += 1
            if not self._coord_down_vline(wd, clamp=False):
                logging.warn("misaligned display")
                break
        # Adjust ret horizontally.
        if self.display_wrap:
            if wc[1] < wd[1]:
                ret[0] -= (wd[1] - wc[1]) / self.cur_width + 1
                ret[1] = self.cur_width - (wd[1] - wc[1]) % self.cur_width
            elif wc[1] > wd[1]:
                ret[0] += (wc[1] - wd[1]) / self.cur_width
                ret[1] = (wc[1] - wd[1]) % self.cur_width
        else:
            ret[1] = wc[1] - wd[1] - self.display_off

        if clamp:
            ret[0] = max(0, ret[0])
            ret[0] = min(self.cur_height - 1, ret[0])
            ret[1] = max(0, ret[1])
            ret[1] = min(self.cur_width, ret[1])

        return ret

    def _coord_visi_to_raw(self, coord, clamp):
        """Return the raw coordinates corresponding to visible coordinate
        coord.  Negative values are permitted in coord.  The result may be a
        position not within the content, unless clamp is True in which case the
        closest position in content will be returned.  See class description
        for more details about coordinates."""
        if len(self.content) == 0:
            return [0,0]

        ret = list(self.display_pos)
        vwc = list(coord)  # Working coord in visible coordinates
        vwd = [0,0]        # Working display pos in visible coordinates
        # Move up or down visible lines until vwc and vwd are on the same line.
        while vwc[0] < vwd[0]:
            vwd[0] -= 1
            if not self._coord_up_vline(ret, clamp=False):
                logging.warn("misaligned display")
                break
        while vwc[0] > vwd[0]:
            vwd[0] += 1
            if not self._coord_down_vline(ret, clamp=False):
                logging.warn("misaligned display")
                break
        # Adjust ret horizontally.
        if self.display_wrap:
            if vwc[1] < vwd[1]:
                ret[0] -= 1
                ret[1] = self.cur_width - (vwd[1] - vwc[1])
            elif vwc[1] > vwd[1]:
                ret[1] = vwc[1] - vwd[1]
        else:
            ret[1] = vwc[1] - vwd[1]

        if clamp:
            ret[0] = max(0, ret[0])
            ret[0] = min(len(self.content), ret[0])
            ret[1] = max(0, ret[1])
            ret[1] = min(self._len_line(ret[0]), ret[1])

        return ret

    def draw_cursor(self, win):
        cy, cx = self._coord_raw_to_visi(self.cursor_pos, clamp=True)
        # I want to use win.move(cy, cx), but it doesn't work for me.
        (py, px) = win.getparyx()
        # Account for the header size and left wrap column.
        self.display.scr.move(py + cy + self.header_size, px + cx + 1)

        curses.curs_set(2)  # Make the cursor visible again.
        self.display.scr.refresh()

    def draw(self, win, force_redraw):
        """Draw this view on the provided curses window.  If force_redraw is
        True, then do a full redraw, otherwise just an update is enough."""

        if force_redraw:
            self.set_dirty()

        if not self.is_dirty():
            if self.cursor_pos != self.last_cursor_pos:
                self.last_cursor_pos = list(self.cursor_pos)
                return True
            else:
                return False

        # It's dirty, so do some drawing.

        dirty = self.is_dirty()
        self.set_dirty(None)
        self.last_display_pos = list(self.display_pos)
        self.last_cursor_pos = list(self.cursor_pos)

        self.draw_header(win)

        # Recalculate sizes in case the header changed size.
        cwin = self._content_win(win)
        cwin.leaveok(1)

        (height, width) = cwin.getmaxyx()

        self.align_display(height, width)

        # Completely recalculate the formatted content.  This isn't necessary
        # in all cases and could be optimized.
        f_c = View.format_lines(self.display, self.content,
                                height, width, self.display_pos,
                                wrap=self.display_wrap, trim=self.display_off)

        fci = wi = 0

        # In append mode, no need to overwrite unchanged lines.
        if dirty == "append":
            while (fci < len(f_c) and f_c[fci].line_number < self.last_drawn):
                fci += 1
                wi += 1
        else:
            cwin.clear()

        # Draw the remaining lines.
        while (wi < height and fci < len(f_c)):
            self.draw_line(f_c[fci], wi, cwin)
            fci += 1
            wi += 1

        if fci > 0 and len(f_c) >= fci:
            self.last_drawn = f_c[fci-1].line_number
        else:
            self.last_drawn = None

        cwin.refresh()
        cwin.leaveok(0)
        return True

    def align_display(self, height, width):
        """Realign display_pos so that the cursor is on the screen"""
        self.cur_height = height
        self.cur_width = width - 2
        dv = self._coord_raw_to_visi(self.display_pos, clamp=False)
        cv = self._coord_raw_to_visi(self.cursor_pos, clamp=False)
        while dv[0] > cv[0]:
            dv[0] -= 1
            if not self._coord_up_vline(self.display_pos, clamp=True):
                logging.warn("misaligned display")
                break
        while dv[0] + height <= cv[0]:
            dv[0] += 1
            if not self._coord_down_vline(self.display_pos, clamp=True):
                logging.warn("misaligned display")
                break

    ######################################################################
    # Functions bound to key strokes

    def move_left(self, ki):
        for i in range(ki.num):
            if self.editable:
                if self.cursor_pos[1] > self.display_off:
                    self.cursor_pos[1] -= 1
                elif self.cursor_pos[0] > 0:
                    self.cursor_pos[0] -= 1
                    self.cursor_pos[1] = self._len_line()
            else:
                if self.display_off > 0:
                    self.display_off -= 1
                if self.display_pos[1] > self.display_off:
                    self.display_pos[1] -= 1
                    self.cursor_pos[1] = self.display_pos[1]

    def move_right(self, ki):
        for i in range(ki.num):
            if self.editable:
                if self.cursor_pos[1] < self._len_line():
                    self.cursor_pos[1] += 1
                elif self.cursor_pos[0] + 1 < len(self.content):
                    self.cursor_pos[0] += 1
                    self.cursor_pos[1] = self.display_off
            else:
                if self.display_off < self._len_line():
                    self.display_off += 1
                if self.display_pos[1] < self.display_off:
                    self.display_pos[1] += 1
                    self.cursor_pos[1] = self.display_pos[1]

    def move_up(self, ki):
        if self.editable:
            for i in range(ki.num - 1):
                self._coord_up_vline(self.cursor_pos, clamp=False)
            self._coord_up_vline(self.cursor_pos, clamp=True)
        else:
            for i in range(ki.num):
                if self.display_pos[0] > 0:
                    self.display_pos[0] -= 1
                    self.cursor_pos[0] = self.display_pos[0]
                

    def move_down(self, ki):
        if self.editable:
            for i in range(ki.num - 1):
                self._coord_down_vline(self.cursor_pos, clamp=False)
            self._coord_down_vline(self.cursor_pos, clamp=True)
        else:
            for i in range(ki.num):
                if self.display_pos[0] < len(self.content):
                    self.display_pos[0] += 1
                    self.cursor_pos[0] = self.display_pos[0]

    def move_start(self, ki):
        if self.editable:
            self.cursor_pos[1] = self.display_off
        else:
            self.display_off = 0
            self.display_pos[1] = 0
            self.cursor_pos[1] = self.display_pos[1]

    def move_end(self, ki):
        if self.editable:
            if len(self.content) > self.cursor_pos[0]:
                self.cursor_pos[1] = self._len_line()
        else:
            self.display_off = max(0, self._len_line() - self.cur_width)
            self.display_pos[1] = self.display_off
            self.cursor_pos[1] = self.display_pos[1]

    def move_top(self, ki):
        self.cursor_pos = [0,self.display_off]

    def move_bottom(self, ki):
        self.cursor_pos[0] = max(len(self.content) -1, 0)
        self.move_end(ki)

    def page_up(self, ki):
        ki.num *= self.cur_height - 1
        self.move_up(ki)

    def page_down(self, ki):
        ki.num *= self.cur_height - 1
        self.move_down(ki)

    def delete_left(self, ki):
        if not self.editable:
            return
        cy = self.cursor_pos[0]
        cx = self.cursor_pos[1]
        cl = self.content[cy]
        if cx > 0:
            self.content[cy] = cl[0:cx-1] + cl[cx:]
            self.cursor_pos[1] = max(self.cursor_pos[1] - 1, self.display_off)
        elif cy > 0:
            self.content[cy-1] += self.content[cy]
            del self.content[cy:cy+1]
        self.set_dirty()

    def delete_right(self, ki):
        if not self.editable:
            return
        cy = self.cursor_pos[0]
        cx = self.cursor_pos[1]
        cl = self.content[cy]
        if len(cl) > cx:
            self.content[cy] = cl[0:cx] + cl[cx+1:]
        elif len(self.content) > cy + 1:
            self.content[cy] = cl + self.content.pop(cy+1)
        self.set_dirty()

    def delete_line(self, ki):
        if not self.editable:
            return
        if len(self.content) > self.cursor_pos[0]:
            (cy, cx) = self.cursor_pos
            self.display.last_copy = self.content[cy][cx:]
            self.content[cy] = self.content[cy][0:cx]
        self.set_dirty()

    def insert(self, ki):
        return self.insert_text(ki.ch())

    def paste(self, ki):
        return self.insert_text(self.display.last_copy)
