#
# Copyright (C) 2010 Alexander Taler <dissent@0--0.org>
#

# This file is part of hsh.

# hsh is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# hsh is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with hsh.  If not, see <http://www.gnu.org/licenses/>.

######################################################################

import glob
import logging
import os
import pwd
import re

import hsh
import aliases
import builtins
import config
import jobs

# A command is an instruction issued to hsh by the user, typically causing the
# creation of a subprocess executing the specified program.  It is represented
# externally as a string similar to what you type on a typical shell's command
# line.  Internally it is a tree of tokens.

class ParseException(Exception):
    """Raised when there's an error in parsing."""

######################################################################
# Static functions.

# re which matches strings that require glob expansion.
glob_match = re.compile(r"(^|[^\\])([\*\?\[\]])")
# re which matches characters to escape to prevent glob expansion, including \.
glob_chars = re.compile(r"([\*\?\[\]\\])")
# re to unescape everything except escaped glob symbols and \.
unescaper_nonglobs = re.compile(r"\\([^\*\?\[\]\\])")
# re to unescape a string of all backslash escapes.
unescaper = re.compile(r"\\(.)")

def escape_glob_chars(text):
    """Given a string, return a version of it with the special glob characters
    escaped.  Backslashes are also escaped to prevent ambiguity."""
    return glob_chars.sub(r"\\\1", text)

whitespace = re.compile(r"\s")
def format_cmd(args):
    """Given a list of strings format them to be readable when printed."""
    res = ""
    for arg in args:
        if whitespace.search(arg):
            res += "'" + arg + "' "
        else:
            res += arg + " "
    return res

######################################################################
# Command classes.

class Token(object):
    """Part of a command representing a syntactic token.

    The top of the token class hierarchy.  Subclasses need to implement methods
    defined here.

    Additionally, subclasses may wish to override these class variables:

    commencer_ch = None # Character indicating the start of the token
    dquotable = False   # True to permit the token within double quotes.
    """

    commencer_ch = None
    dquotable = False

    def __init__(self):
        self.terminator = re.compile("()(\s)")
        if self.__class__.commencer_ch is not None:
            self.commencer = re.compile(re.escape(self.__class__.commencer_ch))
        else:
            self.commencer = re.compile("\S")

    def __str__(self):
        """Return the raw text entered by the user.  There is no default
        implementation of this method."""
        raise Exception("Not Implemented")

    def __len__(self):
        "There is no default implementation of this method."
        raise Exception("Not Implemented")

    def modify(self, txt="", begin=0, end=None):
        """Modify this token by replacing the text between begin and end with
        the given text.  begin defaults to the beginning of the token, end to
        the end, and txt to the empty string.

        Returns a 3-tuple:
          1] string: text which wasn't used by this token and spills over to
             the next token.
          2] boolean: True means that this token may absorb some characters
             from the next token.  Spill over is not possible in this case.
          3] boolean: True means that this token needs to be deleted.  May
             include spillover.

        The default implementation checks the insertion range, raising an
        exception in case of error, but does not perform the actual modify.
        """
        if end is None:
            end = self.__len__()
        if begin > end:
            raise ParseException("Bad insertion range. (begin > end)")
        if end > self.__len__():
            raise ParseException("Bad insertion range. (end > len())")
        if self.terminated() and begin == self.__len__():
            return (txt, False, False)

    def expand(self):
        """Return the expanded version of this token.  Return value is a list
        of strings in which any special shell syntax is replaced with its
        expanded value, appropriate for passing as an arg in a process
        invocation.

        Arg tokens will perform glob expansion on the results of this value, so
        other tokens should protect their response using escape_glob_chars() if
        appropriate.

        The default implementation returns an array with one element containing
        the raw token.
        """
        return [self.__str__()]

    def completions(self, cloc):
        """Return a list of completions for this token from the specified
        cursor location.  If no possible completions exist, [] is returned.
        If possible completions exist, a list of strings is returned.  The
        first item is the root of the completions, the rest are possible
        extensions of the root.  Most tokens will return None if the cursor is
        not at the end.  The default implementation returns nothing."""
        return []

    def expand_directives(self):
        """Return the list of directives which are expanded from this token.
        The HashDirective token will return something here, but most other
        tokens use the default implementation which returns an empty list."""
        return []

    def commences(self, txt):
        """Return True if the beginning of the provided text can validly start
        this token.

        The default implementation returns True iff txt matches the regexp
        self.commencer.  The default value of self.commencer is \\S to match
        non-whitespace."""
        return self.commencer.match(txt) is not None

    def terminates(self, txt):
        """Return a pair of strings if the first character of the provided text
        would terminate this token.  The first string are the characters to be
        included in the token, the second are the characters which spill over
        to the next token.  Return None if the text would not terminate the
        token.

        The token mustn't end in an escape character (\) because that would
        make the test incorrect.

        The default implementation matches txt against the regexp
        self.terminator.  On a match it returns match groups 1 and 2, otherwise
        it returns None.  The default value of self.terminator is ()(\\s) to
        terminate on whitespace, but not accept it."""
        match = self.terminator.match(txt)
        if match:
            return match.group(1), match.group(2) + txt[match.end():]
        return None

    def terminated(self):
        """Return True if this token will not accept any more characters at the
        end because it has terminated.  The default implementation returns
        False."""
        return False

    def dump(self):
        """Return a representation of the token tree as a list of strings.  The
        default implementation returns an empty list."""
        return []

class AggregateToken(Token):
    # A token which contains no text directly, and is instead an aggregate of
    # other tokens.

    # Subclasses must implement create_token().

    def __init__(self):
        super(AggregateToken, self).__init__()
        self.tokens = []

    def __str__(self):
        return "".join(map(str, self.tokens))

    def expand(self):
        """Expansions of subtokens are "multiplied" together.
        e.g. "a{u,v}b{w,x}" expands to ["aubw", "aubx", "avbw", "avbx"]."""
        def expand_multiply(x, y):
            if type(x) != list:
                x = x.expand()
            if type(y) != list:
                y = y.expand()
            prod = []
            for xe in x:
                for ye in y:
                    prod.append(str(xe) + str(ye))
            return prod
        return reduce(expand_multiply, self.tokens, [''])

    def __len__(self):
        return reduce(lambda a,b: a+b, map(len, self.tokens), 0)

    def modify(self, txt="", begin=0, end=None):
        check = super(AggregateToken, self).modify(txt, begin, end)
        if check is not None:
            return check

        if end is None:
            end = self.__len__()

        # Determine if this function should return hungry.
        ret_hungry = False
        was_terminated = self.terminated()

        # Find the subtokens in which the modify begins and ends.  If the
        # boundaries fall between tokens, treat it as the end of the earlier
        # token.
        begin_tok, begin_tok_off = (0, 0)
        end_tok, end_tok_off = (0, 0)
        trav_len = 0
        for i in range(len(self.tokens)):
            if trav_len < begin:
                begin_tok = i
                begin_tok_off = begin - trav_len
            if end <= trav_len + len(self.tokens[i]):
                end_tok = i
                end_tok_off = end - trav_len
                break
            trav_len += len(self.tokens[i])

        # Delete tokens as necessary.
        if end_tok > begin_tok:
            extra_txt = str(self.tokens[end_tok])[end_tok_off:]
            self.tokens[begin_tok + 1:end_tok + 1] = []
            txt = txt + extra_txt
            end_tok = begin_tok
            end_tok_off = len(self.tokens[end_tok])

        # If the last character is being changed, then the next token may be
        # absorbed.
        if (end_tok == len(self.tokens) or (end_tok == len(self.tokens) - 1 and
            end_tok_off == len(self.tokens[end_tok]))):
            ret_hungry = True

        # If an existing subtoken is being modified, pass the call to it.
        (hungry, delete) = (False, False)
        if end_tok_off > 0:
            (txt, hungry, delete) = \
                self.tokens[begin_tok].modify(txt, begin_tok_off, end_tok_off)

            # Delete the token if necessary.
            if delete:
                del self.tokens[begin_tok]
                if len(self.tokens) == 0 and not txt:
                    return ("", False, True)
            else:
                begin_tok += 1

        # As long as there is text remaining, create new tokens to hold it, and
        # then dismantle following tokens if the new ones are hungry.

        while txt or hungry:

            # Creating tokens for unused text.
            while txt:
                new_tok = self.create_token(txt, begin_tok)
                if new_tok is None:
                    # This token itself is terminated and can't accept txt.
                    # Strip off any remaining tokens and return them.
                    txt += "".join(map(str, self.tokens[begin_tok:]))
                    self.tokens[begin_tok:] = []
                    return (txt, False, False)
                self.tokens.insert(begin_tok, new_tok)
                (txt, hungry, delete) = new_tok.modify(txt, 0, 0)
                begin_tok += 1

            # If the last token (either the modified or created one) is hungry,
            # then start eating tokens.
            while hungry:
                if len(self.tokens) == begin_tok:
                    # No more tokens
                    return ("", True, False)
                else:
                    eat = str(self.tokens.pop(begin_tok))
                    ht = self.tokens[begin_tok - 1]
                    (txt, hungry, delete) = ht.modify(eat, ht.__len__())

            # Loop back to spill over code if needed.

        # Everything is handled, so return.
        if not self.terminated() and was_terminated:
            ret_hungry = True
        return ("", ret_hungry, False)

    def create_token(self, txt, loc):
        # Create a new token to take characters from the beginning of the
        # provided string, which will be inserted at the given location within
        # this aggregate token.  Implemented by subclasses.  If no token can be
        # created starting with that character, then None is returned.  The
        # newly created token will be empty, and expects insert() to be called
        # with identical text to get it started.
        pass

    def completions(self, cloc):
        if cloc == self.__len__() and len(self.tokens) > 0:
            return self.tokens[-1].completions(len(self.tokens[-1]))

    def dump(self):
        ret = ["%s %s\n" % (str(self), self.__class__.__name__)]
        for tok in self.tokens:
            ret.extend(map(lambda x: "   " + x, tok.dump()))
        return ret

class LeafToken(Token):
    # A token which contains text, and does not contain any other tokens.

    # Subclasses must implement terminates() or set self.terminator as a
    # regexp, and commences() or set self.commencer as a regexp.  Also
    # self.terminated() to indicate if additional characters can be appended.

    def __init__(self):
        super(LeafToken, self).__init__()
        self.repr = ""  # String content of this token.

    def __str__(self):
        return self.repr

    def __len__(self):
        return len(self.repr)

    def modify(self, txt="", begin=0, end=None):
        check = super(LeafToken, self).modify(txt, begin, end)
        if check is not None:
            return check

        if end is None:
            end = self.__len__()

        # Info about whether this token will now be hungry
        hungry = False
        was_terminated = self.terminated()
        if end == self.__len__():
            # If the last character changed, the next token must be checked.
            hungry = True

        # Append end of token to inserted text, moving any trailing backslashes
        # to appended text.
        txt = txt + self.repr[end:]
        self.repr = self.repr[0:begin]
        while len(self.repr) > 0 and self.repr[-1] == "\\":
            self.repr = self.repr[0:-1]
            txt = "\\" + txt

        # Make sure the new first character is valid to commence this token.
        # If it's not valid, call for this token's deletion.
        if begin == 0:
            if len(txt) == 0:
                return txt, False, True
            if txt[0] == "\\":
                first = txt[0:2]
            else:
                first = txt[0]
            if not self.commences(first):
                return txt, False, True

        # Add txt character by character, checking for termination.
        while len(txt) > 0:
            if txt[0] == '\\':
                next = txt[0:2]
                txt = txt[2:]
            else:
                next = txt[0:1]
                txt = txt[1:]

            term = self.terminates(next)
            if term:
                self.repr += term[0]
                return term[1] + txt, False, len(self.repr) == 0

            self.repr += next

        # All text was added, but the token may now be hungry.
        if not self.terminated() and was_terminated:
            hungry = True
        return "", hungry, False

    def dump(self):
        return ["%s %s\n" % (str(self), self.__class__.__name__)]

class CommandText(AggregateToken):
    # The root of the token tree, contains a list of Args and Arg separators.

    def __init__(self, do_aliases=True):
        """Create a new CommandText token.  If aliases is False, do not expand
        aliases in this command, to prevent unbounded recursion."""
        super(CommandText, self).__init__()
        self.do_aliases = do_aliases

    def insert(self, loc, txt):
        """Insert the provided text at the given location, raising an exception
        in case of error."""
        self.modify(txt, begin=loc, end=loc)

    def delete(self, loc, count=1):
        """Delete count characters at the given location, raising an exception
        in case of error."""
        self.modify("", begin=loc, end=loc+count)

    def create_token(self, txt, loc):
        if re.match("\s", txt):
            return ArgSeparator()
        elif loc == 0:
            return Prog(self.do_aliases)
        elif txt[0] == "#":
            return HashDirective()
        else:
            return Arg()

    def expand(self):
        """Command expansion is to concatenate the results of expansions of the
        Args into a list of parameters.  ArgSeparators disappear since they
        expand to the empty list."""
        res = []
        for tok in self.tokens:
            res.extend(tok.expand())
        if (len(res) > 0) and not isinstance(self.tokens[0], Prog):
            # No Prog token is present, so include an empty string so the first
            # Arg won't be mistaken for the program.
            res[0:0] = ('', )
        return res

    def completions(self, cloc):
        # Completions will be shown at the end of any arg
        for tok in self.tokens:
            if cloc <= len(tok):
                return tok.completions(cloc)
            else:
                cloc -= len(tok)

    def expand_directives(self):
        """Like parameter expansion, directives are concatenated into a list.
        Directives which don't make sense together are filtered so that only
        the last one is preserved."""
        res = []
        for tok in self.tokens:
            res.extend(tok.expand_directives())
        return HashDirective.canonify(res)

class ArgOrProg(AggregateToken):
    # Intermediate class for implementation shared by Arg and Prog classes.

    def __init__(self):
        super(ArgOrProg, self).__init__()
        self.escaper = re.compile("(["+ re.escape(commencers + "*?[]") +"])")

    def create_token(self, txt, loc):
        if re.match("\s", txt):
            # Whitespace is a terminator, so return None
            return None
        for tt in arg_subtoks:
            if txt.startswith(tt.commencer_ch):
                return tt()
        # No special token matches, so it's raw text
        return RawText()

class Arg(ArgOrProg):
    # A top level token representing a single command argument, as passed to
    # the subprocesses.  It is terminated by whitespace.

    def expand(self):
        """When args are expanded, glob matching for files must be done.  The
        resulting completions have all escape characters removed."""
        # Parent takes responsibility for all of the subtokens, which is done
        # prior to glob expansion.
        parexp = super(Arg, self).expand()

        # Results of glob expansion
        globbedexp = []

        for unglobbed in parexp:
            # Unglobbed may contain \ escaped glob characters.  Unfortunately
            # the glob library does not support escaping glob characters, so
            # this will cause bugs.

            if not glob_match.search(unglobbed):
                # No globbing symbols in there, so skip glob expansion, but
                # unescape any escaped glob characters.
                globbedexp.append(unescaper.sub("\\1", unglobbed))
            else:
                # Do glob expansion.  Unfortunately the results are ambiguous
                # if an escaped glob character was in the original pattern or
                # in the results.
                globbed = glob.glob(unglobbed)
                globbed.sort()
                globbedexp.extend(globbed)

        return globbedexp

    def completions(self, cloc):
        """Completions are filename completions, and only work if the arg
        expands to a single filename."""
        check = super(Arg, self).completions(cloc)
        if check:
            return check
        if cloc != self.__len__(): # Cursor must be at the end
            return []

        expansions = self.expand()
        text = self.__str__()
        if len(expansions) > 1 or (len(expansions) == 0 and len(text) > 0):
            return []
        # Use globbing to get completions, but escape globbing symbols first.
        # This implementation of glob doesn't handle escapes properly, a
        # drawback that needs to be rectified, and makes this code broken.
        root = escape_glob_chars(expansions[0])
        comps = glob.glob(root + '*')
        comps = map(lambda x: os.path.isdir(x) and x + os.sep or x, comps)
        comps = map(lambda x: x[len(expansions[0]):], comps)
        comps = map(lambda x: self.escaper.sub(r"\\\1", x), comps)
        comps.sort()
        return text.split(os.sep)[-1:] + comps

class Prog(ArgOrProg):
    """
    Prog specifies what to run.  It must be the first token of a command.
    
    Prog objects behave differently depending on if they are specifying a
    program, builtin or alias.  It would be desirable to use subclasses for the
    three, but isn't straightforward because as text is entered the type would
    change, resulting in a lot reparsing, and risk of infinite loops.

    The rules for choosing what to run are:

     - Supported syntax elements are expanded: single and double quoted tokens,
       environment variables, named directories, job references.  Glob
       expansion is not permitted in Progs.

     - If the resultant text starts with '+' then a builtin is being explicitly
       chosen, and if there's no such builtin, then nothing is run.

     - If the resultant text starts with '/', './' or '../' then an external
       program is specified.  The program will be found absolutely or relative
       to the current directory, and the PATH will not be consulted.

     - Alias list is searched, and if the text matches, that alias is expanded.
       The alias text is expanded as a full command, except that the first
       token will not do alias expansion.

     - Builtin list is searched, and if the text matches, that builtin is used.

     - Finally, the text will be searched for relative to all entries on
       the PATH.

    These rules depart from those of other shells (bash, zsh, ksh, tcsh) in the
    case of a command name with a '/' in it.  hsh will search for such commands
    in subdirectories of path entries, while other shells will run programs
    relative to the current directory.  With a well-structured path this does
    not represent a weakening of security, and in fact tightens security in
    same cases.  The traditional usage is still available by prepending "./".
    Finally this allows namespacing of groups of commands, for instance, all of
    the ImageMagick programs could be moved to a subdirectory, so that their
    generic names would not be so confusing, or the foomatic programs could
    have simplified names and go in their own subdirectory.

    As per shell tradition, external programs are searched only from the path
    unless they start with './', '../', '/'.  This is so that the user isn't
    tricked into running something in the local directory by mistake.

    Glob expansion is not supported in Prog because multiple expansions just
    opens risk of error, has non-obvious interaction with path search, and does
    not seem particularly useful anyway.  Other syntax which may result in
    multiple expansion may also be precluded.  This differs from other shells
    which will look in the current directory when glob syntax is used.
    """
    # re which matches explicitly specified programs.
    expl_cmd_re = re.compile(r"\.?\.?/")

    def __init__(self, do_aliases):
        super(Prog, self).__init__()
        self.do_aliases = do_aliases

    def _parexp(self):
        """Return the result of the parent expanding the subtokens.  Remove
        escapes because Prog does not do glob expansion.  Raise an exception if
        the parent expands it to multiple tokens."""
        parexp = super(Prog, self).expand()

        if len(parexp) > 1:
            raise ParseException("Command not permitted as multiple expansion.")

        if len(parexp) == 0:
            return ""

        if glob_match.search(parexp[0]):
            raise ParseException("Glob symbols not permitted in command.")

        return unescaper.sub("\\1", parexp[0])

    def expand(self):
        """The command is expanded, with any glob characters causing an
        exception.  \ escapes are removed.  If the command doesn't exist an
        empty string is returned."""

        expnd = self._parexp()

        if Prog.expl_cmd_re.match(expnd):
            # Program is specified explicitly.
            if os.path.exists(expnd):
                return [expnd]
            else:
                return ['']
        elif expnd.startswith(config.builtin_commencer):
            # Explicitly stated builtin
            if (expnd[len(config.builtin_commencer):] in jobs.loaded_builtins):
                return [expnd]
            else:
                return ['']

        elif expnd in aliases.manager and self.do_aliases:
            return aliases.manager.get_parsed_value(expnd).expand()

        elif expnd in jobs.loaded_builtins:
            # Implicit builtin
            return ['+' + expnd]

        else:
            # Treat as a relative specification, so search the path.
            expnd_mtch = []

            for pathe in os.environ["PATH"].split(os.path.pathsep):
                ea = os.path.join(pathe, expnd)
                if os.path.exists(ea):
                    expnd_mtch.append(ea)

            if len(expnd_mtch) == 0:
                return ['']
            if len(expnd_mtch) > 1:
                hsh.display_obj.show_alert('Multiple commands found on path')

            return expnd_mtch[0:1]

    def expand_directives(self):
        """Only aliases can include directives."""

        expnd = self._parexp()

        if expnd in aliases.manager and self.do_aliases:
            return aliases.manager.get_parsed_value(expnd).expand_directives()
        else:
            return []

    def completions(self, cloc):
        """Completions are filename or built-in completions, and are searched
        for across the entire path."""

        # The parent checks for subtoken completions
        check = super(Prog, self).completions(cloc)
        if check:
            return check
        if cloc != self.__len__(): # Cursor must be at the end
            return

        expnd = self._parexp()

        comps = []

        if expnd.startswith(config.builtin_commencer):
            # expand from the list of builtins
            bimatch = expnd[len(config.builtin_commencer):]
            for biname in jobs.loaded_builtins.keys():
                if biname.startswith(bimatch):
                    comps.append(biname[len(bimatch):])

        else:
            # Form a list of filenames on which to do glob expansion
            if Prog.expl_cmd_re.match(expnd):
                # Explicit specification so don't use $PATH
                srchpaths = [expnd]
            else:
                srchpaths = map(lambda x: os.path.join(x, expnd),
                                os.environ["PATH"].split(os.path.pathsep))

            for sp in srchpaths:
                gres = glob.glob(sp + "*")
                # Append / to directories
                gres = map(lambda x: os.path.isdir(x) and x+os.sep or x, gres)
                # Trim the common bit off.
                gres = map(lambda x: x[len(sp):], gres)
                comps.extend(gres)

            # Add any matching builtins
            for biname in jobs.loaded_builtins.keys():
                if biname.startswith(expnd):
                    comps.append(biname[len(expnd):])

            # Add any matching aliases
            for alname in aliases.manager:
                if alname.startswith(expnd) and self.do_aliases:
                    comps.append(alname[len(expnd):])

        # Sort and remove duplicates.
        comps.sort()
        i = len(comps) - 1
        while i > 0:
            if comps[i] == comps[i-1]:
                del comps[i]
            i -= 1
        # Escape syntax characters.
        comps = map(lambda x: self.escaper.sub(r"\\\1", x), comps)
        # Don't include directories in the common bit, to reduce clutter.
        return self.__str__().split(os.sep)[-1:] + comps

class HashDirective(LeafToken):
    """A directive to the shell about how to launch the job, identified by a
    leading # character."""

    commencer_ch = "#"
    # Grouped into mutually-exclusive sets.
    directives_g = [["exclusive", "piped", "pty"],
                    ["hide", "hideonsuccess", "show"]]
    directives = reduce(lambda x,y: x+y, directives_g)

    @staticmethod
    def canonify(dlist):
        """Given a list of directive names, remove incompatible ones and
        duplicates, preferring the last of each group."""
        for dg in HashDirective.directives_g:
            dlist =  (filter(lambda x: x not in dg, dlist) +
                      filter(lambda x: x in dg, dlist)[-1:])
        return dlist

    def expand(self):
        """Hash directives are not included in the final command line."""
        return []

    def expand_directives(self):
        return [self.__str__()[1:]]

    def completions(self, cloc):
        if cloc != self.__len__(): # Cursor must be at the end
            return
        root = self.__str__()[1:]
        comps = filter(lambda x: x.startswith(root),HashDirective.directives)
        comps = map(lambda x: x[len(root):] + " ", comps)
        comps.sort()
        return [root] + comps

class ArgSeparator(LeafToken):
    # A token consisting of whitespace which separates Args.
    def __init__(self):
        super(ArgSeparator, self).__init__()
        self.terminator = re.compile("()(\S)")
        self.commencer = re.compile("\s")

    def expand(self):
        """They expand to nothing."""
        return []

class RawText(LeafToken):
    # RawText part of a token which is not subjected to any modification
    def __init__(self):
        super(RawText, self).__init__()
        # Terminates whenever any more specialized token starts.
        self.terminator = re.compile("()(\s|["+ re.escape(commencers) +"])")
        self.commencer = re.compile("\S")

    def expand(self):
        # Unescape everything except globbing stuff.
        return [unescaper_nonglobs.sub("\\1", self.__str__())]

class SingleQuotedToken(LeafToken):

    commencer_ch = "'"

    def terminates(self, ch):
        # Terminates on a single quote other than the one at the beginning.
        if self.__len__() > 0 and ch[0:1] == "'":
            return ch[0:1], ch[1:]
        return None

    def terminated(self):
        return self.__len__() > 1 and self.repr[-1]=="'" and self.repr[-2]!="\\"

    def expand(self):
        res = self.repr
        # drop single quotes from the beginning and end
        if self.terminated():
            res = res[0:-1]
        if len(res) > 0 and res[0] == "'":
            res = res[1:]
        return [escape_glob_chars(res)]

class DoubleQuote(LeafToken):
    """A leaf token which contains a single double quote character.  Used
    exclusively within the DoubleQuotedToken token as a commencer and
    terminator."""
    def __init__(self):
        super(DoubleQuote, self).__init__()
        self.terminator = re.compile('(")()')
        self.commencer = re.compile('"')

    def terminated(self):
        # It terminates after the first character.
        return self.__len__() > 0

    def expand(self):
        "Double quotes are dropped on expansion."
        return ['']

class DoubleQuotedText(LeafToken):
    """The text between the double quotes, may include whitespace characters."""

    def __init__(self):
        super(DoubleQuotedText, self).__init__()
        # Terminates whenever any more specialized token starts.
        self.terminator = re.compile("()([\""+ re.escape(commencers_dq) +"])")
        self.commencer = re.compile('[^"]')

    def expand(self):
        """Double quotes suppress glob/re stuff."""
        return [escape_glob_chars(self.repr)]

class DoubleQuotedToken(AggregateToken):
    """Text enclosed in double quotes.  Special shell syntax like env variables
    are expanded within double quoted text, so it is an AggregateToken with
    DoubleQuote tokens at the beginning and end.
    """

    commencer_ch = "\""

    def terminates(self, ch):
        # Terminates on a double quote other than the one at the beginning.
        if self.__len__() > 0 and ch[0:1] == '"':
            return ch[0:1], ch[1:]
        return None

    def terminated(self):
        """While modifying the token it's possible to have double quotes other
        than at the end, so search all subtokens for a terminating quote."""
        for tok in self.tokens[1:]:
            if tok.__class__ == DoubleQuote:
                return True
        return False

    def create_token(self, txt, loc):
        if self.terminated():
            return None
        elif txt.startswith('"'):
            return DoubleQuote()

        for tt in arg_subtoks_dq:
            if txt.startswith(tt.commencer_ch):
                return tt()

        # No special token matches, so it's text
        return DoubleQuotedText()

class BracedToken(LeafToken):
    """Super class for tokens which can be enclosed in braces, after a
    commencing character like $ or ~."""

    def _braced(self):
        return self.__len__() > 1 and self.repr[1:2] == '{'

    def _text(self):
        # The token "text" without braces or the commencer
        if self._braced():
            if self.repr[-1] == '}':
                return self.repr[2:-1]
            else:
                return self.repr[2:]
        else:
            return self.repr[1:]

    def terminates(self, ch):
        # If an open brace is used, terminates only on a closed brace,
        # otherwise terminates on any non-alphanumeric.
        if self.__len__() == 0:
            return None
        if self.__len__() == 1:
            if ch[0:1] == '{':
                return None
        if self._braced():
            if ch[0:1] == "}":
                return ch[0:1], ch[1:]
            else:
                return None
        else:
            return super(BracedToken, self).terminates(ch)

    def terminated(self):
        return self._braced() and self.repr[-1:] == '}'

    def _completions(self, cloc, possibilities, root=None, brace=True):
        if cloc != self.__len__(): # Cursor must be at the end
            return
        # Search possibilities for matching options.
        if root is None:
            root = self._text()
        comps = filter(lambda x: x.startswith(root), possibilities)
        comps = map(lambda x: x[len(root):], comps)
        if brace and self._braced():
            comps = map(lambda x: x + '}', comps)
        comps.sort()
        return [root] + comps

class EnvVariable(BracedToken):
    """An environment variable reference, starting with a $, with optional
    braces {}."""

    dquotable = True
    commencer_ch = "$"
    
    def __init__(self):
        super(EnvVariable, self).__init__()
        self.terminator = re.compile('()(\W)')

    def expand(self):
        try:
            return [escape_glob_chars(os.environ[self._text()])]
        except KeyError:
            return ['']

    def completions(self, cloc):
        return self._completions(cloc, os.environ.keys())

class NamedDir(BracedToken):
    """A named directory reference, starting with a ~, with optional
    braces {}."""
    
    dquotable = True
    commencer_ch = "~"
    
    def __init__(self):
        super(NamedDir, self).__init__()
        self.terminator = re.compile('()(\W)')

    def expand(self):
        return [escape_glob_chars(os.path.expanduser('~' + self._text()))]

    def completions(self, cloc):
        return self._completions(cloc, map(lambda x: x.pw_name, pwd.getpwall()))

class JobReference(BracedToken):
    """A reference to some feature of a job, starting with a % and optionally
    using braces to surround the value."""

    dquotable = True
    commencer_ch = "%"
    parsere = re.compile('^([0-9]*)([a-zA-Z,]*)$')

    def __init__(self):
        super(JobReference, self).__init__()
        self.terminator = re.compile('()(\s)')
        self.jobid = None
        self.features = []

    def _parse(self):
        """Break up the string into component parts."""
        match = JobReference.parsere.match(self._text())
        if match is None:
            self.jobid = None
            self.features = []
        else:
            self.jobid = match.group(1)
            self.features = match.group(2).split(",")

    feature_list = ["p", "fo", "fe", "fi", "to", "te", "ti"]

    def _expand_feature(self, job, feature):
        if feature == "p":
            return [str(job.pid) or ""]
        if feature[0:1] == "f":
            fname = self._expand_feature_filetype(feature[1:2])
            if fname is not None:
                return [os.path.join(config.jobs_dir, str(job.jid), fname)]

        if feature[0:1] == "t":
            fname = self._expand_feature_filetype(feature[1:2])
            res = []
            if fname is not None:
                try:
                    jf = open(os.path.join(config.jobs_dir,str(job.jid),fname))
                    res = map(lambda x: x.strip(), jf.readlines())
                    jf.close()
                except:
                    pass
            return res
        return []

    def _expand_feature_filetype(self, ftype):
        if ftype == "o":
            return "stdout"
        if ftype == "e":
            return "stderr"
        if ftype == "i":
            return "stdin"

    def expand(self):
        self._parse()
        if not self.jobid:
            job = hsh.display_obj.get_visible_job() 
        else:
            job = jobs.manager.get_job(int(self.jobid))
        if job is None:
            return []
        # List of lists of expanded feature 
        expsl = map(lambda x: self._expand_feature(job, x), self.features)
        exps = reduce(lambda x,y: x+y, expsl, [])
        return map(escape_glob_chars, exps)

    def completions(self, cloc):
        # If no feature text is present, collect job id completions.
        if len(self.features) == 0:
            return []
        if len(self.features) == 1 and len(self.features[0]) == 0:
            comps = self._completions(cloc, jobs.manager.get_jobid_list(),
                                      brace=False)
            if self.jobid in jobs.manager.get_jobid_list():
                comps += JobReference.feature_list
            return comps
        else:
            return self._completions(cloc, JobReference.feature_list,
                                     root=self.features[-1])

######################################################################

# Lists of token classes needed in parsing.

# All token types which can be instantiated within an arg.  Excludes
# superclasses which are not instantiated, like ArgOrProg, and special purpose
# tokens, like DoubleQuote.  See Token class for explanation of class variables
# commencer_ch, dquotable, etc.

arg_subtoks = [ SingleQuotedToken, DoubleQuotedToken,
                EnvVariable, NamedDir, JobReference,
              ]

# A string consisting of all commencer characters.
commencers = "".join(map(lambda x: x.commencer_ch, arg_subtoks))

# Token types which can appear in double quotes.
arg_subtoks_dq = filter(lambda x: x.dquotable, arg_subtoks)

# A string consisting of all commencer characters which can appear in quotes
commencers_dq = "".join(map(lambda x: x.commencer_ch, arg_subtoks_dq))
