#
# Copyright (C) 2010 Alexander Taler <dissent@0--0.org>
#

# This file is part of hsh.

# hsh is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# hsh is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with hsh.  If not, see <http://www.gnu.org/licenses/>.

######################################################################

# Test the command module

import os
import unittest
import hsh.command

class CommandTester(unittest.TestCase):

    def setUp(self):
        self.ct = hsh.command.CommandText()

    def test_basic(self):
        self.assertEqual(str(self.ct), "")

        # Insert new characters at the three possible positions in a token.
        self.ct.insert(0, "a")
        self.assertEqual(str(self.ct), "a")
        self.ct.insert(0, "b")
        self.assertEqual(str(self.ct), "ba")
        self.ct.insert(2, "c")
        self.assertEqual(str(self.ct), "bac")

        self.assertEqual(len(self.ct.tokens), 1)

        self.ct.delete(0, 0)
        self.assertEqual(str(self.ct), "bac")
        self.ct.delete(1)
        self.assertEqual(str(self.ct), "bc")
        self.ct.delete(0, 2)
        self.assertEqual(str(self.ct), "")

    def test_whitespace(self):
        # Create new tokens
        self.ct.insert(0, "bac")
        self.ct.insert(3, " ")
        self.assertEqual(str(self.ct), "bac ")
        self.assertEqual(len(self.ct.tokens), 2)
        self.ct.insert(3, " ")
        self.assertEqual(str(self.ct), "bac  ")
        self.assertEqual(len(self.ct.tokens), 2)
        self.ct.insert(2, " ")
        self.assertEqual(str(self.ct), "ba c  ")
        self.assertEqual(len(self.ct.tokens), 4)
        self.ct.insert(5, "d")
        self.assertEqual(str(self.ct), "ba c d ")
        self.assertEqual(len(self.ct.tokens), 6)

        self.ct.delete(0, 2)
        self.assertEqual(str(self.ct), " c d ")
        self.assertEqual(len(self.ct.tokens), 5)
        self.ct.delete(1)
        self.assertEqual(str(self.ct), "  d ")
        self.assertEqual(len(self.ct.tokens), 3)

    def test_singlequoted(self):
        self.ct.insert(0, "111")
        self.assertEqual(str(self.ct), "111")
        self.assertEqual(len(self.ct.tokens), 1)
        self.ct.insert(3, "'")
        self.assertEqual(str(self.ct), "111'")
        self.assertEqual(len(self.ct.tokens[0].tokens), 2)
        self.ct.insert(4, " ")
        self.assertEqual(str(self.ct), "111' ")
        self.assertEqual(len(self.ct.tokens[0].tokens), 2)
        self.ct.insert(5, "'")
        self.assertEqual(str(self.ct), "111' '")
        self.assertEqual(len(self.ct.tokens[0].tokens), 2)
        self.ct.insert(6, "2")
        self.assertEqual(str(self.ct), "111' '2")
        self.assertEqual(str(self.ct.tokens[0].tokens[1]), "' '")
        self.assertEqual(len(self.ct.tokens[0].tokens), 3)

        self.ct.delete(1)
        self.assertEqual(str(self.ct), "11' '2")
        self.assertEqual(len(self.ct.tokens[0].tokens), 3)
        self.ct.delete(2)
        self.assertEqual(str(self.ct), "11 '2")
        self.assertEqual(str(self.ct.tokens[0]), "11")
        self.assertEqual(str(self.ct.tokens[1]), " ")
        self.assertEqual(str(self.ct.tokens[2]), "'2")
        self.assertEqual(len(self.ct.tokens), 3)

    def test_doublequoted(self):
        self.ct.insert(0, '"')
        self.assertEqual(str(self.ct), '"')
        self.assertEqual(self.ct.tokens[0].tokens[0].__class__,
                         hsh.command.DoubleQuotedToken)
        self.assertEqual(len(self.ct.tokens[0].tokens[0].tokens), 1)
        self.ct.insert(1, 'u u')
        self.assertEqual(str(self.ct), '"u u')
        self.assertEqual(len(self.ct.tokens[0].tokens[0].tokens), 2)
        self.assertEqual(str(self.ct.tokens[0].tokens[0].tokens[1]), 'u u')
        self.ct.insert(4, '"')
        self.assertEqual(str(self.ct), '"u u"')
        self.assertEqual(len(self.ct.tokens[0].tokens[0].tokens), 3)

        self.ct.delete(3)
        self.assertEqual(str(self.ct), '"u "')
        self.assertEqual(len(self.ct.tokens[0].tokens[0].tokens), 3)

        self.ct.delete(3)
        self.assertEqual(str(self.ct), '"u ')
        self.assertEqual(len(self.ct.tokens[0].tokens[0].tokens), 2)

    def test_environ(self):
        self.ct.insert(0, '$')
        self.assertEqual(self.ct.tokens[0].tokens[0].__class__,
                         hsh.command.EnvVariable)
        self.ct.insert(1, '{')
        self.assertEqual(str(self.ct.tokens[0].tokens[0]), "${")
        self.ct.insert(2, 'HOME}')
        self.assertEqual(str(self.ct.tokens[0].tokens[0]), "${HOME}")
        self.ct.insert(1, 'USER')
        self.assertEqual(str(self.ct.tokens[0].tokens[0]), "$USER")
        self.assertEqual(len(self.ct.tokens[0].tokens), 2)

        self.ct.delete(5)
        self.assertEqual(str(self.ct.tokens[0].tokens[0]), "$USERHOME")
        self.assertEqual(str(self.ct), "$USERHOME}")
        self.assertEqual(len(self.ct.tokens[0].tokens), 2)
        
    def test_nameddir(self):
        self.ct.insert(0, ' ~')
        self.assertEqual(self.ct.tokens[1].tokens[0].__class__,
                         hsh.command.NamedDir)
        self.assertEqual(self.ct.expand(), [os.path.expanduser("~")])
        self.ct.insert(2, 'root')
        self.assertEqual(self.ct.expand(), [os.path.expanduser("~root")])
        self.ct.insert(2, '{root}')
        self.assertEqual(str(self.ct), " ~{root}root")
        self.assertEqual(self.ct.expand(),
                         [os.path.expanduser("~root") + "root"])
        
    def test_environ_in_dq(self):
        self.ct.insert(0, '"${HOME}XXX"')
        self.assertEqual(len(self.ct.tokens[0].tokens[0].tokens), 4)
        self.assertEqual(str(self.ct.tokens[0].tokens[0].tokens[1]), "${HOME}")

    def test_expand(self):
        self.ct.insert(0, " aa")
        self.assertEqual(self.ct.tokens[1].__class__, hsh.command.Arg)
        self.assertEqual(self.ct.expand(), ["aa"])
        self.ct.insert(1, "'bb'")
        self.assertEqual(str(self.ct), " 'bb'aa")
        self.assertEqual(len(self.ct.tokens), 2)
        self.assertEqual(self.ct.expand(), ["bbaa"])
        self.ct.insert(1, '"cc"')
        self.assertEqual(self.ct.expand(), ["ccbbaa"])
        self.assertEqual(self.ct.tokens[1].tokens[0].__class__,
                         hsh.command.DoubleQuotedToken)
        self.assertEqual(str(self.ct.tokens[1].tokens[0]), '"cc"')
        os.environ['HSH_TEST'] = 'hsh_test_val'
        self.ct.insert(1, '$HSH_TEST')
        self.assertEqual(str(self.ct.tokens[1].tokens[1]), '"cc"')
        self.assertEqual(len(self.ct.tokens[1].tokens), 4)
        self.assertEqual(self.ct.tokens[1].tokens[0].__class__,
                         hsh.command.EnvVariable)
        self.assertEqual(self.ct.expand(), ["hsh_test_valccbbaa"])
        self.ct.insert(10, ' ')
        self.assertEqual(self.ct.expand(), ["hsh_test_val", "ccbbaa"])

    def test_expand_escapes(self):
        self.ct.insert(0, ' .')
        self.assertEqual(self.ct.expand(), ["."])
        self.ct.insert(1, '-')
        self.assertEqual(self.ct.expand(), ["-."])
        self.ct.insert(1, '\\*')
        self.assertEqual(str(self.ct), " \\*-.")
        self.assertEqual(self.ct.expand(), ["*-."])

    def test_escapes(self):
        self.ct.insert(0, '\\')
        self.ct.insert(1, '\'')
        self.assertEqual(len(self.ct.tokens), 1)
        self.ct.insert(2, 'x')
        self.assertEqual(len(self.ct.tokens), 1)
        self.ct.insert(3, ' ')
        self.assertEqual(len(self.ct.tokens), 2)
        self.assertEqual(str(self.ct.tokens[0].tokens[0]), "\\'x")
        self.ct.delete(0)
        self.assertEqual(str(self.ct.tokens[0].tokens[0]), "'x ")
        self.assertEqual(len(self.ct.tokens), 1)

    def test_prog(self):
        self.ct.insert(0, 'ls')
        self.assertEqual(type(self.ct.tokens[0]).__name__, 'Prog')

class GlobTest(unittest.TestCase):

    def setUp(self):
        self.ct = hsh.command.CommandText()
        try:
            os.mkdir("/tmp/hsh-test-dir")
            os.mkdir("/tmp/hsh-test-dir/dir1")
            os.mkdir("/tmp/hsh-test-dir/dir2")
            os.mkdir("/tmp/hsh-test-dir/dir3")
        except:
            pass

    def tearDown(self):
        os.rmdir("/tmp/hsh-test-dir/dir1")
        os.rmdir("/tmp/hsh-test-dir/dir2")
        os.rmdir("/tmp/hsh-test-dir/dir3")
        os.rmdir("/tmp/hsh-test-dir")    

    def test_globbing(self):
        self.ct.insert(0, ' /')
        self.assertEqual(self.ct.expand(), ["/"])
        self.ct.insert(2, 'tm?')
        self.assertEqual(self.ct.expand(), ["/tmp"])
        self.ct.insert(5, '/hsh-test-*')
        self.assertEqual(self.ct.expand(), ["/tmp/hsh-test-dir"])
        self.ct.insert(16, '/*')
        self.assertEqual(str(self.ct), " /tm?/hsh-test-*/*")
        self.assertEqual(self.ct.expand(), ["/tmp/hsh-test-dir/dir1",
                                            "/tmp/hsh-test-dir/dir2",
                                            "/tmp/hsh-test-dir/dir3"])

    def test_completions(self):
        self.ct.insert(0, '/tmp/hsh-test-dir/d')
        self.assertEqual(self.ct.tokens[0].completions(19),
                         ['d', 'ir1/', 'ir2/', 'ir3/'])

class CompletionTest(unittest.TestCase):

    def test_env_completions(self):
        ev = hsh.command.EnvVariable()
        os.environ['HSH_TEST_1'] = 'hsh_val_1'
        os.environ['HSH_TEST_2'] = 'hsh_val_2'
        os.environ['HSH_TEST_3'] = 'hsh_val_3'
        ev.modify('$HSH_TEST_', 0)
        self.assertEqual(ev.completions(10), ['HSH_TEST_', '1', '2', '3'])

    def test_nameddir_completions(self):
        ev = hsh.command.NamedDir()
        ev.modify('~r', 0)
        self.assertTrue('oot' in ev.completions(2))

    def test_prog_completions(self):
        prg = hsh.command.Prog()
        prg.modify('ls', 0)
        self.assertEqual(['/bin/ls'], prg.expand())
        prg.modify('xxxxxxxxxxxxls', 0, 2)
        self.assertEqual([''], prg.expand())
        prg.modify('l*', 0, 14)
        self.assertRaises(hsh.command.ParseException, prg.expand)
        prg.modify('l', 0, 2)
        self.assertEqual('l', prg.completions(1)[0])
        self.assertTrue('s' in prg.completions(1))

if __name__ == '__main__':
    unittest.main()
