#
# Copyright (C) 2010 Alexander Taler <dissent@0--0.org>
#

# This file is part of hsh.

# hsh is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# hsh is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with hsh.  If not, see <http://www.gnu.org/licenses/>.

######################################################################

# TextContent is an annotated text class.  Provided to curses displayable as a
# representation of the text to be shown along with additional face
# information.  Used for providing job output/input/error to the display.

# Content consists of an array of TextLines which are newline-terminated arrays
# of characters with defined subregions.

# TextLine's regions have a specified length and are stored as an ordered
# array.  They are manipulated through TextLine's array methods.

class TextLineRegion(object):
    """Details about a region within a TextLine.  Each TextLine is broken into
    regions which are substrings of that specific line.  Regions may not
    overlap, and all characters in the line are a part of a region.  Users of
    the region may attach arbitrary data to the region in the form of
    attributes, to record information such as origin of the Text, error
    conditions or rendering instructions.

    Regions are not created directly and do not provide an interface to modify
    their text, use methods of TextLine for those purposes."""

    def __init__(self, line, length):
        """Create a new region for the given line.  This constructor should not
        be called directly, use TextLine's interface instead."""
        self._line = line
        self._length = length

    def _pindex(self):
        """Find the index inside the parent array where this region starts."""
        loc = 0
        for region in self._line.regions:
            if region == self:
                return loc
            loc += region._length
        raise Exception("Region has bad parent.")

    def __str__(self):
        loc = self._pindex()
        return "".join(self._line[loc:loc+self._length])

    def set_attrs(self, tlr):
        """Given another TextLineRegion, copy its attributes to this one."""
        for key in tlr.__dict__.keys():
            if not key.startswith('_'):
                self.__dict__[key] = tlr.__dict__[key]

    def __len__(self):
        return self._length

    def delete(self):
        loc = self._pindex()
        self._line[loc:loc + self._length] = ""

    # Array interface

    def __setslice__(self, i, j, seq):
        if i < 0: i = 0
        if j > self._length: j = self._length
        if i > j:
            raise Exception("bad parameters to __setslice__()")
        self._line.__setslice__(self._pindex() + i, self._pindex() + j, seq)

    def __setitem__(self, key, val):
        if type(key) == slice:
            raise Exception("slice parameter to __setitem__() not supported")
        self.__setslice__(key, key+1, [val])

    def append(self, val):
        self.__setslice__(self.__len__(), self.__len__(), [val])

    def extend(self, vals):
        self.__setslice__(self.__len__(), self.__len__(), vals)

    def insert(self, ind, val):
        self.__setslice__(ind, ind, [val])


class TextLine(list):
    """A single line of text within the content, which may have at most one new
    line as its last character.  The line is broken into regions, with each
    character belonging in exactly one region."""

    def __init__(self, init=""):
        self._chk_slice(0, len(init), init)
        super(TextLine, self).__init__(init)
        self.regions = []
        if len(init) > 0:
            self.regions.append(TextLineRegion(self, len(init)))

    def __str__(self):
        return "".join(self)

    def _update_regions(self, i, j, seq):
        """Update the regions of this TextLine for the inserted text."""

        # Traverse the regions, and update them.
        to_delete = list()
        loc = 0
        for region in self.regions:
            region_end = loc + region._length
            if region_end < i:
                # Region is before the changed area
                pass
            elif loc <= i:
                # Changed area starts in region
                region._length = i - loc + len(seq)
                if region_end > j:
                    region._length += region_end - j
                    break
                if region._length == 0:
                    to_delete.append(region)
            elif region_end <= j:
                # Region falls entirely within the removed area, delete it.
                to_delete.append(region)
            else:
                # Changed area ends in region
                region._length = region_end - j
                break
            loc = region_end

        for region in to_delete:
            self.regions.remove(region)

        if len(self) > 0 and len(self.regions) == 0:
            self.regions.append(TextLineRegion(self, len(self)))

    def _chk_slice(self, i, j, seq):
        # Ensure that they are only characters.
        for c in seq:
            if type(c) is not str or len(c) != 1:
                raise TypeError("Elements must be characters")
        # Ensure that a single line terminator only appears at the end.
        for c in seq[:-1]:
            if c == '\n':
                raise Exception('TextLine may only have a newline at the end.')
        if self.__len__() == 0:
            return
        if ((i >= self.__len__() and self[-1] == '\n') or
            (j < self.__len__() and len(seq) > 0 and seq[-1] == '\n')):
                raise Exception('TextLine may only have a newline at the end.')

    def __setslice__(self, i, j, seq):
        self._chk_slice(i, j, seq)
        super(TextLine, self).__setslice__(i, j, seq)
        self._update_regions(i, j, seq)

    def __setitem__(self, key, val):
        if type(key) == slice:
            self._chk_slice(key.start, key.stop, val)
            self._update_regions(key.start, key.stop, val)
        else:
            self._chk_slice(key, key + 1, [val])
            self._update_regions(key, key + 1, [val])
        super(TextLine, self).__setitem__(key, val)

    def append(self, val):
        self.__setslice__(self.__len__(), self.__len__(), [val])

    def extend(self, vals):
        self.__setslice__(self.__len__(), self.__len__(), vals)

    def insert(self, ind, val):
        self.__setslice__(ind, ind, [val])

    def append_region(self, text):
        """Append a new region to the TextLine, initializing it with the
        provided text.  text may be a string or a TextLineRegion."""
        # When inserting between regions, __setslice__() always appends to the
        # left region, but we want a new region in this case, so we attach to
        # the old one, shrink it, and add a new one.
        self.__setslice__(self.__len__(), self.__len__(), str(text))
        self.regions[-1]._length -= len(text)
        if len(self.regions[-1]) == 0:
            del self.regions[-1]
        self.regions.append(TextLineRegion(self, len(text)))
        if isinstance(text, TextLineRegion):
            self.regions[-1].set_attrs(text)
        return self.regions[-1]

    def is_terminated(self):
        return len(self) > 0 and self[-1] == '\n'

class TextContent(list):
    """TextContent is a representation of Text suitable for being displayed to
    the user, with additional interesting annotated information.  It is
    presented as an array of TextLines.  Each TextLine except the last must be
    terminated with a newline."""

    def __str__(self):
        return "".join(map(str, self))

    @staticmethod
    def _chk_elem(elem):
        # Ensure that a provided element is a TextLine, converting if needed.
        if isinstance(elem, TextLine):
            return elem
        if isinstance(elem, str):
            return TextLine(elem)
        raise TypeError("TextContent may only contain lines of text")

    def _chk_slice(self, i, j, seq):
        for c in seq[:-1]:
            if c != '\n':
                raise Exception('TextContent elements must end with newlines.')
        if self.__len__() == 0:
            return
        if i >= self.__len__() and not self[-1].is_terminated():
            raise Exception('TextContent elements must end with newlines.')
        if j < self.__len__() and seq[-1][-1] != '\n':
            raise Exception('TextContent elements must end with newlines.')

    def __setslice__(self, i, j, seq):
        self._chk_slice(i, j, seq)
        super(TextContent, self).__setslice__(i, j, map(self._chk_elem, seq))

    def __setitem__(self, key, val):
        if type(key) == slice:
            self._chk_slice(key.start, key.stop, val)
            val = map(self._chk_elem, val)
        else:
            self._chk_slice(key, key + 1, [val])
            val = self._chk_elem(val)
        super(TextContent, self).__setitem__(key, val)

    def append(self, val):
        self.__setslice__(self.__len__(), self.__len__(), [val])

    def extend(self, vals):
        self.__setslice__(self.__len__(), self.__len__(), vals)

    def insert(self, ind, val):
        self.__setslice__(ind, ind, [val])

    def append_region(self, text):
        """Append a new region to the TextContent, initializing it with the
        provided content.  It is added to the final TextLine or a new TextLine
        is created as needed."""
        if len(text) == 0:
            return None
        if len(self) == 0 or self[-1].is_terminated():
            self.append(TextLine(""))
        return self[-1].append_region(text)
