#
# Copyright (C) 2010 Alexander Taler <dissent@0--0.org>
#

# This file is part of hsh.

# hsh is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# hsh is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with hsh.  If not, see <http://www.gnu.org/licenses/>.

######################################################################

# A job is an external process invoked by hsh.  Various information like output
# is stored in a directory assigned to the job.

# Internally a Job object is created for each job, which contains a process
# object, and two threads for reading the job's output, (stdout and stderr).
# The main thread handles input from the terminal, so there's no need for a
# thread for stdin, although it will be required in the future to handle pipes
# from other processes or files.

# Normally output is accumulated in memory and on disk and then presented to
# the user in a curses window.  However, this doesn't work for processes which
# send terminal control sequences, like curses applications, who want raw
# access to the terminal.  Some processes check if stdin and stdout are
# attached to a terminal (using ioctl() presumably) and adjust their behaviour
# accordingly.  For obvious reasons hsh doesn't want to cede control of stdin
# and stdout to its subprocesses, so they are created with pipes by default.
# Unfortunately once the process is started, it's too late to switch the fds,
# so some processes may be left inconsolably miserable with a couple of pipes.
# A command line syntax needs to be defined for launching the process with
# direct access to the terminal, or possibly virtual terminals on Linux.

# stdin is left open so that the display can provide mechanisms to the user for
# providing input to the subprocess.

import curses
import curses.ascii
import errno
import fcntl
import logging
import os
import subprocess
import time
import thread
import threading
import traceback

import config
import system
import content

# A mapping from command names to builtin classes.
loaded_builtins = dict()

# A mapping from classnames of builtins to the classes themselves.
builtin_classes = dict()

def register_builtin(name, biclass):
    global loaded_builtins, builtin_classes
    if name in loaded_builtins:
        logging.info("Overriding existing builtin: %s" % name)
    loaded_builtins[name] = biclass

    builtin_classes[biclass.__name__] = biclass

class JobManager(object):
    # Provides access to all known jobs.  Used to create jobs, and assign them
    # ids.  Will read jobs from directories on disk, as well as clean up disk.

    # self.all_jobs is an array which indexes all jobs on disk.  (Except there
    # is currently no mechanism for reading jobs created by other processes
    # after this process started.)  The array is indexed using job id, and
    # contains None if the job was deleted, a string with a filepath if the job
    # has not yet been read from disk, and a job object if the Job was already
    # read from disk.

    # The use of an array for all_jobs will not be appropriate for large
    # numbers of jobs, especially if there are large sparse areas.  Something
    # like blist from pypi would be more appropriate.

    def __init__(self):
        self.listeners = list()

        # Read jobs from disk so that they can be loaded later.
        job_ids = []
        jobdir_contents = os.listdir(config.jobs_dir)
        for jd in jobdir_contents:
            try:
                job_ids.append(int(jd))
            except ValueError, ve:
                logging.warn('Invalid directory in jobs dir: %s' % 
                             os.path.join(config.jobs_dir, jd))
        array_len = max(job_ids) + 1

        self.all_jobs = [None] * array_len
        for jd in job_ids:
            self.all_jobs[jd] = os.path.join(config.jobs_dir, str(jd))

    def register_listener(self, listener):
        if not isinstance(listener, JobManagerListener):
            raise Exception("Listeners must be of correct type.")
        if listener not in self.listeners:
            self.listeners.append(listener)

    def unregister_listener(self, listener):
        if listener in self.listeners:
            self.listeners.remove(listener)

    def create_job(self, cmd, cmdline, start):
        """Given a command as an array of args, create and return a new job to
        execute it.  cmdline is the string entered by the user to generate the
        command."""
        if cmd[0].startswith(config.builtin_commencer):
            biname = cmd[0][len(config.builtin_commencer):]
            job = loaded_builtins[biname].create_job(cmd, cmdline, start)
        else:
            job = ExternalJob.create_job(cmd, cmdline, start)
        # Extend the all_jobs array to include the newly created job.
        while len(self.all_jobs) < job.jid:
            njdir = os.path.join(config.jobs_dir, str(len(self.all_jobs)))
            if os.path.exists(njdir):
                self.all_jobs.append(self.load_job(njdir))
            else:
                self.all_jobs.append(None)
        self.all_jobs.append(job)
        for listener in self.listeners:
            listener.on_add_job(job)
        return job

    def load_job(self, datadir):
        job = Job.load_job(datadir)
        for listener in self.listeners:
            listener.on_add_job(job)
        return job

    def forget_job(self, job):
        """Forget the job from disk."""
        for listener in self.listeners:
            listener.on_remove_job(job)
        # Clear out all the files and delete the job directory.
        self.all_jobs[job.jid] = None
        try:
            for jfile in os.listdir(job.jdir):
                os.remove(os.path.join(job.jdir, jfile))
            os.rmdir(job.jdir)
        except Exception, e:
            logging.warn("Ignorning error cleaning up job: %s" % e)

    def get_job(self, jobid):
        """Fetch the job with the given id, or None if there's no job with that
        id."""
        if jobid >= len(self.all_jobs) or self.all_jobs[jobid] is None:
            return None
        if type(self.all_jobs[jobid]) == str:
            self.all_jobs[jobid] = self.load_job(self.all_jobs[jobid])
        return self.all_jobs[jobid]

    def get_prev_job(self, jobid):
        """Given a job object, return the job immediately before it, or None if
        it's the first job."""
        jobid = min(jobid, len(self.all_jobs))
        while jobid > 0:
            jobid -= 1
            if self.all_jobs[jobid] is not None:
                return self.get_job(jobid)
        return None

    def get_last_job(self):
        return self.get_prev_job(len(self.all_jobs))

    def get_jobid_list(self):
        """Return a list of all jobids as strings."""
        return map(str, filter(lambda x: self.all_jobs[x] is not None, 
                               range(len(self.all_jobs))))

    def terminate(self):
        "Called when the program is exiting, so that the manager can clean up."
        for job in self.all_jobs:
            if isinstance(job,Job) and job.is_new and job.state == "Running":
                job.state = "Orphaned"
                job._write_job_state()

# Global shared manager instance.
manager = JobManager()

class JobManagerListener(object):
    """An interface for objects that are interested in changes to loaded
    jobs."""

    def on_add_job(self, job):
        """A job has been loaded into the manager."""
        pass

    def on_remove_job(self, job):
        """A job has been removed from the manager."""
        pass

class JobListener(object):
    """An "interface" for objects interested in changes to a job.  Particularly
    useful to an object displaying details of a job."""

    def on_job_output(self, job, amount):
        # Called when output has been added.  amount can be "line" or "char"
        # indicating how much output has been appended.
        pass

    def on_job_raw_output(self, job, text, channel):
        # Called when control sequences have been identified and includes text
        # generated by the process.  Once this is called, all output will be
        # provided in this way.  channel is one of "stdout" or "stderr"
        # indicating where the subprocess wrote the output.
        pass

    def on_job_terminate(self, job):
        # Called when the process has terminated.
        pass

# Each job has multiple threads (one per pipe) because it appears Python
# doesn't (easily) support non-blocking IO on pipes, and definitely not on
# Windows.

class JobOutputThread(threading.Thread):
    # A thread for managing a job's output, it reads from the pipe and stores
    # its output in the job record.  channel is either "stdout" or "stderr" and
    # is used to determine where to store the output.
    def __init__(self, job, pipe, channel):
        super(JobOutputThread, self).__init__()
        self.job = job
        # Put pipe into non-blocking mode, so that character-by-character
        # output can be done.
        fc = fcntl.fcntl(pipe, fcntl.F_GETFL)
        fcntl.fcntl(pipe, fcntl.F_SETFL, fc | os.O_NONBLOCK)
        self.pipe = pipe
        self.channel = channel
        self.outfile = file(os.path.join(self.job.jdir, channel), "w")

    def output_normal(self, text):
        nr = self.job.output.append_region(text)
        nr.source = self.channel
        self.outfile.write(text)
        self.outfile.flush()
        for listener in self.job.listeners:
            listener.on_job_output(self.job, "line")

    def output_raw(self, text):
        for listener in self.job.listeners:
            listener.on_job_raw_output(self.job, text, self.channel)

    def run(self):
        try:
            ps_code = None
            raw_mode = False

            # Buffer output to make it easy to scan for newlines.

            buf = ""
            bufsize = 1024

            while True:
                if len(buf) < bufsize:
                    try:
                        buf += self.pipe.read(bufsize)
                    except IOError, ioe:
                        # errno 11 - resource unavailable happens when no data
                        # is ready.
                        if ioe.errno != 11:
                            raise

                if len(buf) == 0:
                    # Don't stop reading until the process is done, it might
                    # just not be ready to send more output.  If it is done, do
                    # one more loop to be sure.
                    if ps_code is not None:
                        logging.info("Job terminated (%d %s)" %
                                     (ps_code, self.channel))
                        break
                    else:
                        ps_code = self.job.ps.poll()
                        if ps_code is None:
                            time.sleep(0.1)

                elif raw_mode:
                    self.output_raw(buf)
                    buf = ""

                else:
                    split = buf.find("\n")
                    if split == -1:
                        l = buf
                        buf = ""
                    else:
                        l = buf[0:split+1]
                        buf = buf[split+1:]

                    # If there are curses escape sequences, then use raw mode.
                    if l.find("\x1b[") >= 0:
                        raw_mode = True
                        buf = l + buf
                    else:
                        self.output_normal(l)

        except Exception, e:
            logging.error(str(e))
            logging.error(traceback.format_exc())

        logging.info("waiting for process")
        self.job.ps_code = self.job.ps.wait()
        if self.job.ps_code == 0:
            self.job.state = "Succeeded"
        else:
            self.job.state = "Failed"
        self.job._write_job_state()
        for listener in self.job.listeners:
            listener.on_job_terminate(self.job)
        self.outfile.close()


class Job():

    @classmethod
    def create_job(cls, cmd, cmdline, start):
        """Return a new job object created from the given command."""
        job = cls()
        job.cmd = cmd
        job.cmdline = cmdline
        job.cwd = os.getcwd()
        job.username = system.get_username()
        job.hostname = system.get_hostname()
        job.is_new = True

        # All jobs need an id, so get one for this job.  This is done by
        # finding the job directory with the highest number and adding one.
        # The os.mkdir() call returns an error if the directory exists, to
        # resolve collisions.
        while True:
            job.jid = max([len(manager.all_jobs)]
                          + map(int, os.listdir(config.jobs_dir))) + 1
            try:
                job.jdir = os.path.join(config.jobs_dir, str(job.jid))
                os.mkdir(job.jdir, 0700)
                break
            except OSError, ose:
                if ose.errno != errno.EEXIST:
                    raise

        job._write_job_context()
        job._write_job_state()
        if start:
            job.start()
        return job

    @staticmethod
    def load_job(datadir):
        """Return a new job object loaded from the given directory."""
        global builtin_classes

        clsfile = file(os.path.join(datadir, 'classname'), "r")
        clsname = clsfile.readline().strip()
        clsfile.close()
        if clsname in builtin_classes:
            job = builtin_classes[clsname]()
        else:
            job = eval(clsname)()
        job.jdir = datadir
        job._load_params('creation_context',
                         ("cmd", "cmdline",
                          "cwd", "username", "hostname", "jid"))
        job._load_params('job_state', ("state", "ps_code"))
        job.is_new = False
        return job

    def __init__(self):
        self.cmd = None
        self.cmdline = None
        self.output = content.TextContent()
        self.cwd = None
        self.username = None
        self.hostname = None
        self.state = "NotRun"
        self.ps = None
        self.ps_code = None
        self.pid = None
        self.jid = None
        self.jdir = None
        self.listeners = list()
        self.show_output_var = True

    def _write_job_context(self):
        self._write_params('creation_context',
                           ("cmd", "cmdline",
                            "cwd", "username", "hostname", "jid"))
        clsfile = file(os.path.join(self.jdir, 'classname'), "w")
        clsfile.write(self.__class__.__name__)

    def _write_job_state(self):
        self._write_params('job_state',
                           ("state", "pid", "ps_code"))        

    def _write_params(self, name, attrs):
        # Store the job details on disk
        cxtfile = file(os.path.join(self.jdir, name), "w")
        cxtfile.write("%s = {\n" % name)
        for i in attrs:
            cxtfile.write("    %s: %s,\n" % (repr(i), repr(self.__dict__[i])))
        cxtfile.write("}\n")
        cxtfile.close()

    def _load_params(self, name, attrs):
        # Read job details from disk
        ll = dict()
        execfile(os.path.join(self.jdir, name), dict(), ll)
        for i in attrs:
            try:
                self.__dict__[i] = ll[name][i]
            except KeyError, ke:
                logging.info("Key missing from job info: %s, %s" % (name, i))

    def get_retcode(self):
        return self.ps_code

    def get_state(self):
        return self.state

    def get_output(self):
        return self.output

    def show_output(self):
        return self.show_output_var

    def register_listener(self, listener):
        if not isinstance(listener, JobListener):
            raise Exception("Listeners must be of correct type.")
        if listener not in self.listeners:
            self.listeners.append(listener)

    def unregister_listener(self, listener):
        if listener in self.listeners:
            self.listeners.remove(listener)
    

class ExternalJob(Job):
    """A Job which executes an external command."""

    def start(self):
        if self.get_state() == "Running":
            logging.info("Can't retstart a running job")
            return

        self.state = "Running"
        self.output[0:] = []
        self.ps_code = None
        self.ps = subprocess.Popen(self.cmd, cwd=self.cwd,
                                   stdin=subprocess.PIPE,
                                   stdout=subprocess.PIPE,
                                   stderr=subprocess.PIPE)
        self.pid = self.ps.pid
        # self.ps.stdin.close()

        self.stderrthread = JobOutputThread(self, self.ps.stderr, "stderr")
        self.stderrthread.setDaemon(True)
        self.stderrthread.start()

        self.stdoutthread = JobOutputThread(self, self.ps.stdout, "stdout")
        self.stdoutthread.setDaemon(True)
        self.stdoutthread.start()

        self._write_job_state()

class BuiltInJob(Job):
    """A Job which executes a builtin command.  Actual builtins are classes
    which inherit from this one.  More complex ones override start() while
    simpler ones implement do_start()."""

    def start(self):
        if self.get_state() == "Running":
            logging.info("Can't retstart a running job")
            return

        self.state = "Running"
        self.show_output_var = False
        self.output[0:] = []
        try:
            self.do_start()
            self.state = "Succeeded"
            self._write_job_state()
        except:
            self.state = "Failed"
            self._write_job_state()
            raise
