#
# Copyright (C) 2010 Alexander Taler <dissent@0--0.org>
#

# This file is part of hsh.

# hsh is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# hsh is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with hsh.  If not, see <http://www.gnu.org/licenses/>.

######################################################################

import logging

import hsh.content

from hsh.exceptions import *

import keybinds

# Each CursesDisplayable has associated keybindings, although not all accept
# input.  There are many layers of keybindings in order to allow user
# customization and shared configuration (although they are assembled into
# fewer maps at runtime).  User specified bindings are searched first, followed
# by displayable specific bindings, followed by general keybindings.  Within
# each layer there are two data structures, the first is a hash from key code
# to function, the second is a list of predicate/function pairs, if the first
# returns True, it's considered a match, and the search stops.

class KeyInput(object):
    """Represents a single key input from the user, which may consist of
    multiple keystrokes, such as Esc-M, or more when counts are supported."""
    def __init__(self, key, meta=False, num=1):
        self.key = key
        self.meta = meta
        self.num = num

    def ch(self):
        """Return the key value as a single character string, or raise an
        exception if it's not a character."""
        return chr(self.key)

class CursesDisplayable(object):
    # An abstract superclass for anything that can be displayed in a simple
    # curses window.  Subclasses must implement draw(self).

    # Includes functionality for handling input and editing 

    # Includes a draw() method.  To take advantage of this, the text to draw
    # should be stored in self.content, an array of strings.

    # A name may optionally be specified for the object, in which case keybinds
    # assigned to that name will be loaded.

    def __init__(self, display):
        # The curses window object this is drawn in
        self.win = None
        self.display = display

        # Space requirements for this window
        self.min_height_main = 1
        self.min_width_main = 5

        # Optional place to store an array of strings containing the lines of
        # text associated with the display.  This is used by the drawing
        # functionality in draw() and the editing functionality in putch().
        # content are the strings, cursor_pos and display_pos are the positions
        # of the cursor and top left corner of the display, as 2 tuples
        # specifying the line in content and character in that line.
        self.content = [""]
        self.cursor_pos = [0,0]
        self.display_pos = [0,0]
        self.editable = False
        self.has_focus = False

        # The header is a section at the top of display showing information
        # about the view, typically in a different colour, and drawn by the
        # draw_header() method of subclasses.
        self.header_fmt = ""
        self.header_wrap = False
        self.header_size = 0 # Set by draw_header()

        # Build up the keybindings for this displayable.  Start with the
        # default bindings, and then override with customizations.

        # Bindings may be represented as strings to indicate method names on
        # this object, or as function objects to be called directly.
        def bind_to_func(bindval):
            if type(bindval).__name__ == 'function':
                return bindval
            if type(bindval) == str:
                bindfunc = self.__getattribute__(bindval)
                if type(bindfunc).__name__ == 'instancemethod':
                    return bindfunc
                raise ConfigException("Binding is not a function: " + bindval)
            raise ConfigException("Binding to wrong type: %s" % bindval)

        # Key bindings
        self.keys = dict(keybinds.keys_default)

        class_keys_name = "keys_" + self.get_name()
        if class_keys_name in keybinds.__dict__:
            self.keys.update(keybinds.__dict__[class_keys_name])

        for (k, v) in self.keys.items():
            self.keys[k] = bind_to_func(self.keys[k])

        # For predicates, form a list with the highest priority first
        self.preds = list(keybinds.preds_default)

        class_preds_name = "preds_" + self.get_name()
        if class_preds_name in keybinds.__dict__:
            self.preds[0:0] = keybinds.__dict__[class_preds_name]

        for i in range(len(self.preds)):
            self.preds[i] = (self.preds[i][0], bind_to_func(self.preds[i][1]))

    def get_name(self):
        if 'name' in dir(self):
            return self.name
        else:
            return self.__class__.__name__

    def format_lines(self, lines, dpos=[0,0], width=None, height=None,
                     wrap=True, trunc_end="...", wrap_end="\\", wrap_begin="\\"):
        # Format the provided lines so that they will fit into into lines of
        # the appropriate length to display in this curses window.
        #  lines is a list of strings (typically terminated with a \n
        #  wrap  is a boolean, if true, lines should be wrapped
        #  dpos  is a pair of ints indicating where the display should start
        #        in the provided lines.  The first is the vertical position,
        #        from the top line down, the second is the horizontal position
        #        from left to right.
        #  width is the width to make the lines, defaulting to window width
        #  height is the number of output lines to generate, defaulting to
        #        window height.  If the last line needs to be wrapped, more
        #        than height lines will be returned so that the final input
        #        line will be completely included.
        #  trunc_end is a string placed at the end of truncated lines
        #  wrap_end is a string placed at the end of wrapped lines
        #  wrap_begin is a string placed at the begining of continuation line

        # A pair is returned, the first element is a hsh.content.TextContent
        # formatted to be displayed.  The second is a list of integers
        # indicating where the original lines start in the formatted output.
        # The first element of this list will always be 0, indicating that the
        # dpos[0] input line starts at line 0 in the formatted output.  The
        # second element will be larger, and is the line in the formatted
        # output where dpos[0]+1 line from the input starts in the output.

        content_fmt = hsh.content.TextContent()
        line_starts = []
        width = width or self.get_main_size()[1]
        height = height or self.get_main_size()[0]
        if wrap:
            wrap_width = width - len(wrap_end)
        else:
            wrap_width = width - len(trunc_end)
        for line in lines[dpos[0]:]:
            line_starts.append(len(content_fmt))
            # Break up the lines, but preserve regions
            if isinstance(line, hsh.content.TextLine):
                for region in line.regions:
                    content_fmt.append_region(region)
                    while len(content_fmt[-1]) > width + 1:
                        spill = "".join(content_fmt[-1][wrap_width:])
                        content_fmt[-1][wrap_width:] = []
                        if wrap:
                            content_fmt[-1].append_region(wrap_end + '\n')
                            rg = content_fmt.append_region(wrap_begin + spill)
                            rg.set_attrs(region)
                        else:
                            content_fmt[-1].append_region(trunc_end + '\n')
            else:
                line = line.strip('\n')[dpos[1]:]
                while len(line) > width:
                    if wrap:
                        content_fmt.append(line[0:wrap_width] + wrap_end + '\n')
                        line = wrap_begin + line[wrap_width:]
                    else:
                        line = line[0:wrap_width] + trunc_end
                content_fmt.append(line + '\n')
                
            # no point in formatting content that won't fit on the screen
            if len(content_fmt) >= height:
                break

        return content_fmt, line_starts

    def min_height(self):
        return self.header_size + self.min_height_main

    def min_width(self):
        return self.min_width_main

    def _len_current_line(self):
        """Return the length of the current line containing the cursor or zero
        if the cursor is passed the end."""
        if self.cursor_pos[0] >= len(self.content):
            return 0
        else:
            return len(self.content[self.cursor_pos[0]])

    def set_focus(self, has_focus):
        # Called when a displayable receives or loses focus
        self.has_focus = has_focus
        if self.has_focus:
            self.draw()

    def set_win(self, win):
        self.win = win
        self.draw()

    def rem_win(self, win):
        self.win.clear()
        self.win.move(0,0)
        self.win = None

    def get_main_size(self):
        # return a tuple with the size of the main drawing area.
        (height, width) = self.win.getmaxyx()
        return height - self.header_size, width

    def insert_text(self, text):
        if not self.editable:
            return
        (cy, cx) = self.cursor_pos
        cl = self.content[cy]
        self.content[cy] = cl[0:cx] + text + cl[cx:]
        self.cursor_pos[1] += len(text)

    def header_info(self):
        """Return a dict containing details to include in the header."""
        return {}

    def draw_header(self):
        # Arbitrarily limit the header to 5 lines
        hlim = 5
        if len(self.header_fmt) == 0:
            self.header_size = 0
            return
        rawh = (self.header_fmt % self.header_info()).split('\n')
        flines, ls = self.format_lines(rawh, height=hlim, wrap=self.header_wrap)
        self.header_size = min(hlim, len(flines))
        (h, w) = self.get_main_size()
        if h < 1:
            self.header_size -= 1 - h
        if self.win is not None:
            for i in range(self.header_size):
                self.win.addstr(i, 0, str(flines[i]).strip('\n'), self.header_face)

    def set_face(self, region):
        """Given a region, set its face."""
        region.face = self.display.faces["default"]

    # Internal function for drawing each line, already formatted for width.
    def _draw_line(self, line, yoff):
        try:
            xoff = 0
            for region in line.regions:
                if "face" not in region.__dict__:
                    self.set_face(region)
                face = region.face
                self.win.addstr(yoff, xoff, str(region).strip('\n'), face)
                xoff += len(region)
        except Exception, e:
            # curses returns ERR if it writes all the way to the end of
            # the last line of the window, so ignore any errors.
            pass

    def draw(self, append=None, tail=False):
        # append can be None to redraw the whole screen, "line" to indicate a
        #    line has been appended, or "byte" for a single byte.
        # tail means output should be scrolled to always show the bottom

        # If we're tailing, then move the cursor to the end and update display
        # position.
        if tail:
            # append mode can't work if the display position has changed
            append = None
            # the cursor should always be at the end
            self.cursor_pos[0] = max(len(self.content) -1, 0)
            if len(self.content) == 0:
                self.cursor_pos[1] = 0
            else:
                self.cursor_pos[1] = len(self.content[-1])
            self.align_display()

        if not self.win:  # Not currently visible
            return

        self.win.leaveok(1)

        if self.display.full_screen_job is not None: # Job has control
            return

        (height, width) = self.get_main_size()

        # Format the content into lines of the appropriate length
        content_fmt, line_starts = self.format_lines(self.content,
                                                     dpos=self.display_pos)

        # Display the formatted content
        if append == "byte":
            self.draw_header()
            # Only draw the last byte of the display
            if len(content_fmt[-1]) == 0:
                return # Just a boring newline, nothing to do
            if (len(content_fmt) == height and
                len(content_fmt[height - 1]) == width):
                # curses returns ERR if writes all the way to the end of
                # the last line of the window, so drop the final character.
                return
            # If we've gone past the end of the screen, nothing to draw
            if len(content_fmt) > height:
                return
            self.win.addch(len(content_fmt) - 1 + self.header_size,
                           len(content_fmt[-1]) - 1,
                           ord(content_fmt[-1][-1]))
        elif append == "line":
            # Draw the last line of the display
            self.draw_header()
            # The actual line to draw starts at this content line:
            lstart = line_starts[-1]
            if lstart > height:
                # Gone past the end of the screen, nothing to draw.
                return

            while lstart < min(height, len(content_fmt)):
                self._draw_line(content_fmt[lstart], lstart + self.header_size)
                lstart += 1

        else:
            # Redraw the entire display
            self.win.clear()

            self.draw_header()

            for i in range(min(height, len(content_fmt))):
                self._draw_line(content_fmt[i], i + self.header_size)

        if self.has_focus:
            self.win.move(self.cursor_pos[0] - self.display_pos[0] + self.header_size,
                          self.cursor_pos[1] - self.display_pos[1])
        self.win.refresh()
        self.win.leaveok(0)

    def putch(self, ch, esc=False):
        # Keys can be bound to functions or instance methods.
        def call(f):
            if type(f).__name__ == 'instancemethod':
                f(ki)
            else:  # Must be a function.
                f(self, ki)

        # Check keybindings first, and predicates second.
        ki = KeyInput(ch, esc)
        if (ch, esc) in self.keys:
            call(self.keys[(ch, esc)])
        else:
            for pred in self.preds:
                if pred[0](self, ki):
                    call(pred[1])
                    break
        self.align_display()
        self.draw()
        self.display.set_cursor()

    def align_display(self):
        """Realign display_pos so that the cursor is on the screen"""
        if not self.win:
            return
        display_max = self.get_main_size()
        if self.cursor_pos[1] < self.display_pos[1]:
            self.display_pos[1] = self.cursor_pos[1]
        if self.cursor_pos[0] < self.display_pos[0]:
            self.display_pos[0] = self.cursor_pos[0]
        if self.cursor_pos[1] >= self.display_pos[1] + display_max[1]:
            self.display_pos[1] = self.cursor_pos[1] - display_max[1] + 1
        if self.cursor_pos[0] >= self.display_pos[0] + display_max[0]:
            self.display_pos[0] = self.cursor_pos[0] - display_max[0] + 1
        if self.display_pos[1] < 0:
            self.display_pos[1] = 0
        if self.display_pos[0] < 0:
            self.display_pos[0] = 0

    ######################################################################

    def current_job(self):
        return None

    ######################################################################
    # Commands bound to key strokes.

    def move_left(self, ki):
        for i in range(ki.num):
            if self.cursor_pos[1] > 0:
                self.cursor_pos[1] -= 1
            elif self.cursor_pos[0] > 0:
                self.cursor_pos[0] -= 1
                self.cursor_pos[1] = self._len_current_line()

    def move_right(self, ki):
        for i in range(ki.num):
            if self.cursor_pos[1] < self._len_current_line():
                self.cursor_pos[1] += 1
            elif self.cursor_pos[0] + 1 < len(self.content):
                self.cursor_pos[0] += 1
                self.cursor_pos[1] = 0

    def move_up(self, ki):
        for i in range(ki.num):
            if self.cursor_pos[0] > 0:
                self.cursor_pos[0] -= 1
            if self.cursor_pos[1] > self._len_current_line():
                self.cursor_pos[1] = self._len_current_line()

    def move_down(self, ki):
        if len(self.content) == 0:
            return
        for i in range(ki.num):
            if self.cursor_pos[0] + 1 < len(self.content):
                self.cursor_pos[0] += 1
            if self.cursor_pos[1] > self._len_current_line():
                self.cursor_pos[1] = self._len_current_line()

    def move_start(self, ki):
        self.cursor_pos[1] = 0

    def move_end(self, ki):
        if len(self.content) > self.cursor_pos[0]:
            self.cursor_pos[1] = self._len_current_line()

    def move_top(self, ki):
        self.cursor_pos = [0,0]
        self.align_display()

    def move_bottom(self, ki):
        self.cursor_pos[0] = max(len(self.content) -1, 0)
        self.move_end(ki)
        self.align_display()

    def page_up(self, ki):
        pagesize = self.get_main_size()[0] - 1
        ki.num *= pagesize
        self.move_up(ki)
        self.display_pos[0] -= pagesize

    def page_down(self, ki):
        pagesize = self.get_main_size()[0] - 1
        ki.num *= pagesize
        self.move_down(ki)
        self.display_pos[0] += pagesize

    def delete_left(self, ki):
        if not self.editable:
            return
        cy = self.cursor_pos[0]
        cx = self.cursor_pos[1]
        cl = self.content[cy]
        if cx > 0:
            self.content[cy] = cl[0:cx-1] + cl[cx:]
            self.cursor_pos[1] -= 1
        elif cy > 0:
            self.content[cy-1] += self.content[cy]
            del self.content[cy:cy+1]

    def delete_right(self, ki):
        if not self.editable:
            return
        cy = self.cursor_pos[0]
        cx = self.cursor_pos[1]
        cl = self.content[cy]
        if len(cl) > cx:
            self.content[cy] = cl[0:cx] + cl[cx+1:]
        elif len(self.content) > cy + 1:
            self.content[cy] = cl + self.content.pop(cy+1)

    def delete_line(self, ki):
        if not self.editable:
            return
        if len(self.content) > self.cursor_pos[0]:
            (cy, cx) = self.cursor_pos
            self.display.last_copy = self.content[cy][cx:]
            self.content[cy] = self.content[cy][0:cx]

    def insert(self, ki):
        return self.insert_text(ki.ch())

    def paste(self, ki):
        return self.insert_text(self.display.last_copy)

    def next_window(self, ki):
        # Move focus to the next window.
        self.display.next_window()

    def next_job(self, ki):
        pass

    def prev_job(self, ki):
        pass

    def restart_job(self, ki):
        pass

    def change_view(self, ki):
        self.display.show_view(ki.key)

    def quit(self, ki):
        raise QuitException()

    def delete_job(self, ki):
        pass
