#
# Copyright (C) 2010 Alexander Taler <dissent@0--0.org>
#

# This file is part of hsh.

# hsh is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# hsh is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with hsh.  If not, see <http://www.gnu.org/licenses/>.

######################################################################

# CursesDisplay is the central object defined here.  It interfaces with the
# curses library by holding the standard screen object and receiving input.
# It contains several other CursesDisplayable objects, which are wrappers
# around curses windows with additional functionality.  These are subclassed to
# specific purposes for each part of the display, such as the input window, job
# output window, joblist window, etc.

import curses
import curses.ascii
import datetime
import logging
import select
import sys
import traceback

import hsh.config
import hsh.jobs
import hsh.welcome

from hsh.exceptions import *

from displayable import CursesDisplayable
from displayable_input import CursesInput
from displayable_job import CursesJobDisplay
from displayable_joblist import CursesListDisplay
from displayable_message import CursesMessage

class CursesDisplay(hsh.jobs.JobManagerListener):

    # Private variables
    #    scr: a curses window object for the whole screen
    #    maxyx: a 2 tuple for the size of the whole screen last time the
    #           windows were defined.
    #    full_screen_job: If a job has been granted control of the terminal,
    #                     this will be its job object.  If not, it is None.

    # Several CursesDisplayable objects encapsulating and extending curses
    # windows.

    #    cd_job: Displays output of current job.
    #    cd_sessionlist: For managing jobs from this session.
    #    cd_input: an input object which receives curses input
    #    cd_welcome: a friendly message to show at start up
    #    cd_toosmall: a message to show when the screen is too small.
    #    cd_alert: a 1 line window at the screen bottom for messages.
    #    cd_crashlog: a main window message with details of "crashes"

    #    focus: a CursesDisplayable object which currently has the cursor
    #    main: A CursesDisplayable object which is currently in the main view

    def __init__(self, stdscr):
        self.scr = stdscr
        self.maxyx = None
        self.full_screen_job = None

        self.initialize_faces()

        self.cd_job = None
        self.cd_input = CursesInput(self)
        self.cd_welcome = CursesMessage(self, hsh.welcome.message)
        self.cd_toosmall = CursesMessage(self, "Help! The screen is too small")
        self.cd_alert = CursesMessage(self, "", face="alert")
        self.cd_crashlog = CursesMessage(self, "Nothing so far!")

        # A dict mapping job ids to job display objects.
        # Use get_job_display() to get a display.
        self.job_display_index = dict()

        # Load job list views
        self.cd_lists = []
        self.listkeys = { curses.ascii.BS: self.cd_welcome,
                          curses.KEY_F1: self.cd_welcome,
                          curses.KEY_F2: "cd_job",
                          curses.KEY_F10: self.cd_crashlog }
        for newview in hsh.config.listviews:
            (name, key, filt, format) = newview
            self.cd_lists.append(CursesListDisplay(self, name, filt, format))
            if name == "SessionList":
                self.cd_sessionlist = self.cd_lists[-1]
            if name == "CommandHistory":
                self.cd_cmdhistory = self.cd_lists[-1]
            self.listkeys[key] = self.cd_lists[-1]

        self.focus = None
        self.give_focus(self.cd_input)
        self.main = self.cd_welcome

        self.set_windows()

        hsh.jobs.manager.register_listener(self)

        self.escape = False  # Used for handling escapes in input stream
        # For the basic cut and paste implementation.
        self.last_copy = ''

    def initialize_faces(self):
        if curses.has_colors():
            curses.start_color()
            logging.debug("COLORS: %d  COLOR_PAIRS: %d" %
                          (curses.COLORS, curses.COLOR_PAIRS))
            logging.debug("can_change_color = %s" % curses.can_change_color())
        self.faces = {}
        # Assign a colour pair number to each specified text face.  This
        # restricts us to COLOR_PAIRS different faces.  If this is insufficient
        # a numbering scheme for mapping pairs to numbers could be created, but
        # don't forget that 0 is white on black. (8-fg) * (1+bg) - 1
        cnum = 1
        for faceinfo in hsh.config.faces.items():
            curses.init_pair(cnum, faceinfo[1][0], faceinfo[1][1])
            self.faces[faceinfo[0]] = curses.color_pair(cnum)
            logging.debug("name=%s cnum=%s attr=%s fi[0]=%s fi[1]=%s" %
                          (faceinfo[0], cnum, self.faces[faceinfo[0]],
                           faceinfo[1][0], faceinfo[1][1]))
            cnum += 1
        logging.debug(str(self.faces))

    def get_job_display(self, job):
        if job.jid not in self.job_display_index:
            self.job_display_index[job.jid] = CursesJobDisplay(self, job)
        return self.job_display_index[job.jid]

    def give_focus(self, displayable):
        if self.focus == displayable:
            return
        if self.focus is not None:
            self.focus.set_focus(has_focus=False)
        self.focus = displayable
        self.focus.set_focus(has_focus=True)

    def run(self):
        # I would like to specify a timeout to curses input like this:
        # self.scr.timeout(100)

        # Because it provides for smoother transition between full screen jobs,
        # like vi, and normal mode of hsh.  Unfortunately specifying this
        # timeout tickles a bug in curses where it won't update screen size
        # information (scr.getmaxyx()) until a real input character (anything
        # but -1) arrives which means the screen won't redraw on size change
        # until the next input character arrives.

        while True:
            try:
                # Both of the handle_*() methods return after a brief wait if
                # no input is ready, in case the "owner" of the terminal (hsh
                # or the job) has changed, so that input can be handled in the
                # correct way.
                if self.full_screen_job is not None:
                    self.handle_raw_input()
                else:
                    self.handle_curses_input()
            except QuitException:
                # Clean up remaining jobs.
                hsh.jobs.manager.terminate()
                break
            except Exception, e:
                # Clear the input area, in case that was the cause of the issue
                try:
                    inptext = str(self.cd_input.command_text)
                except:
                    inptext = "<Exception fetching input text>"
                self.cd_input._clear_text()
                self.cd_crashlog.set_message(str(datetime.datetime.now())+"\n" +
                                             str(e) + "\n" + 
                                             "input: " + inptext + "\n" +
                                             traceback.format_exc())
                self.set_main(self.cd_crashlog)
                self.cd_alert.set_message("Exception!")
                logging.debug(str(e))
                logging.debug("input: " + inptext)
                logging.debug(traceback.format_exc())
                self.set_cursor()

    def handle_raw_input(self):
        # If no input is ready, just return after doing nothing.
        (infd, outfd, errfd) = select.select([sys.stdin], [], [], 0.1)
        if len(infd) == 0:
            return

        nb = sys.stdin.read(1)

        message = "raw: %o 0x%02x" % (ord(nb),ord(nb))
        if curses.ascii.isprint(ord(nb)):
            message += "  printable (%s)" % nb
        logging.debug(message)
        # If the job terminated while we were waiting for input, then
        # full_screen_job will be None, and we have no choice but to discard
        # the input.
        if self.full_screen_job is not None:
            self.full_screen_job.job.ps.stdin.write(nb)
            self.full_screen_job.job.ps.stdin.flush()

    def handle_curses_input(self):
        ch = self.scr.getch()
        if self.maxyx != self.scr.getmaxyx():
            self.set_windows()
        if ch == -1:
            # Timeout or screen resize or similar, no character to process.
            return
        if not self.escape and ch == curses.ascii.ESC:
            self.escape = True
            return
        if self.escape and ch == 0x5B:
            # This is the beginning of an ANSI escape sequence, one that
            # ncurses apparently isn't converting for me.  bah.  Ignore it for
            # now, support would ideally be added at ncurses level, but upgrade
            # is an issue.
            # http://en.wikipedia.org/wiki/ANSI_escape_code
            # http://www.xfree86.org/current/ctlseqs.html
            seq = ""
            self.escape = False
            while True:
                ch = self.scr.getch()
                seq += chr(ch)
                if 0x40 <= ch and ch <= 0x7e:
                    break

            logging.debug("control seq: %s" % seq)
            return

        message = "%o 0x%02x" % (ch,ch)
        if curses.ascii.isprint(ch):
            message += "  printable (%s)" % chr(ch)
        if self.escape:
            message += "  escaped"
        logging.debug(message)

        self.focus.putch(ch, esc=self.escape)
        self.escape = False
                
    # Calculate and set the sizes of the visible windows.  Should be called
    # when screen changes size, or different windows need to be shown.

    def set_windows(self):
        # Remember the size of window we have configured for.
        self.maxyx = self.scr.getmaxyx()
        w = self.maxyx[1]
        h = self.maxyx[0]

        if (w < max(self.focus.min_width(), self.cd_input.min_width()) or
            h < (self.focus.min_height() + self.cd_input.min_height()
                 + self.cd_alert.min_height())):
            # Much too small to do anything
            self.cd_toosmall.set_win(self.scr.subwin(h, w, 0, 0))
            if self.focus != self.cd_toosmall:
                self.old_focus = self.focus
                self.give_focus(self.cd_toosmall)
                self.cd_toosmall.min_height_main = self.old_focus.min_height()
                self.cd_toosmall.min_width_main = self.old_focus.min_width()
        else:
            # The alert window appears at the very bottom of the screen, the
            # input window above it, and the rest is for the main display.
            ah = self.cd_alert.min_height()
            h = h - ah
            self.cd_alert.set_win(self.scr.subwin(ah, w, h, 0))

            inh = self.cd_input.min_height()
            h = h - inh
            self.cd_input.set_win(self.scr.subwin(inh, w, h, 0))

            self.main.set_win(self.scr.subwin(h, w, 0, 0))

            if self.focus == self.cd_toosmall:
                self.give_focus(self.old_focus)
        self.set_cursor()

    def next_window(self):
        # Move focus to the next window
        if self.focus == self.cd_input:
            self.give_focus(self.main)
        else:
            self.give_focus(self.cd_input)
        self.set_cursor()

    def set_main(self, cd, refocus=False):
        # Set the provided displayable object as the main window.
        if cd is None:
            self.cd_alert.set_message("Nothing to display")
            logging.warn("Switching display to None")
            return
        logging.debug("Switching display to %s" % cd.get_name())
        self.cd_alert.set_message("")
        if self.focus == self.main or refocus:
            self.give_focus(cd)
        if cd != self.main:
            cd.set_win(self.main.win)
            cd.align_display()
            self.main.win = None
            self.main = cd

    def get_visible_job(self):
        """Return whichever job the user would think is the current one, based
        on the visible display."""
        return self.main.current_job()

    def display_job(self, job, show=True):
        """Set the job being displayed, if show is False, then the job
        displayable is updated, but the job is not shown.  job may be a
        hsh.jobs.Job object, a CursesJobDisplay object, -1 to mean move back a
        job, +1 to mean move forward a job, or 0 to mean the current one is
        gone, and a new one needs to be chosen."""
        if type(job) == int:
            if self.cd_job is None:
                job = None
            if job == -1:
                job = self.cd_sessionlist.get_predecessor(self.cd_job.job)
                if job is None: # Don't fall of the end
                    job = self.cd_job
            elif job == 1:
                job = self.cd_sessionlist.get_successor(self.cd_job.job)
                if job is None: # Don't fall of the end
                    job = self.cd_job
            elif job == 0:
                job = self.cd_sessionlist.get_successor(self.cd_job.job)
                if job is None:
                    job = self.cd_sessionlist.get_predecessor(self.cd_job.job)

        if isinstance(job, hsh.jobs.Job):
            job = self.get_job_display(job)

        # job is now a CursesJobDisplay or None
        self.cd_job = job
        if show:
            self.set_main(self.cd_job)

    def show_view(self, key):
        """Show the view associated with the given key."""
        if key in self.listkeys:
            view = self.listkeys[key]
            if type(view) == str:
                view = self.__getattribute__(view)
            self.set_main(view, refocus=True)

    def set_cursor(self):
        # Put the cursor in the correct place in the window with focus.  Someone
        # more skilled with curses might know why I need to use the screen,
        # rather than the subwindow like this: self.focus.win.move(0,0)

        (oy, ox) = self.focus.win.getparyx()
        self.scr.move(oy + self.focus.cursor_pos[0] - self.focus.display_pos[0] + self.focus.header_size,
                      ox + self.focus.cursor_pos[1] - self.focus.display_pos[1])
        self.scr.refresh()

    def relinquish_terminal(self, job):
        # (Mostly) deinitialize curses so that a subprocess can take control of
        # the terminal, but leaving it in raw mode so we can receive character
        # input one at a time.  Another possibility would be to use
        # curses.endwin() followed by some tty commands to enable raw mode.
        # set_prog_mode() may also be useful here.
        self.full_screen_job = job
        self.scr.clear()
        self.scr.refresh()
        self.scr.keypad(0)
        # curses.endwin()  

    def reclaim_terminal(self):
        # The subprocess is done with the terminal, so take it back.
        # curses.initscr() would be the thing to use if we had used endwin().
        self.full_screen_job = None
        self.scr.keypad(1)
        self.scr.clear()
        self.main.draw()
        self.cd_input.draw()
        curses.flushinp()

    ######################################################################
    # JobManagerListener interface

    def on_remove_job(self, job):
        """If the removed job is the currently displayed one, update."""
        if self.cd_job is not None and self.cd_job.job == job:
            self.display_job(0, show = self.cd_job == self.main)

