#
# Copyright (C) 2010 Alexander Taler <dissent@0--0.org>
#

# This file is part of hsh.

# hsh is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# hsh is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with hsh.  If not, see <http://www.gnu.org/licenses/>.

######################################################################

import glob
import logging
import os
import pwd
import re

import hsh
import jobs
import builtins
import config

# A command is an instruction issued to hsh by the user, typically causing the
# creation of a subprocess executing the specified program.  It is represented
# externally as a string similar to what you type on a typical shell's command
# line.  Internally it is a tree of tokens.

class ParseException(Exception):
    """Raised when there's an error in parsing."""

######################################################################
# Static functions.

# re which matches strings that require glob expansion.
glob_match = re.compile(r"(^|[^\\])([\*\?\[\]])")
# re which matches characters to escape to prevent glob expansion, including \.
glob_chars = re.compile(r"([\*\?\[\]\\])")
# re to unescape everything except escaped glob symbols and \.
unescaper_nonglobs = re.compile(r"\\([^\*\?\[\]\\])")
# re to unescape a string of all backslash escapes.
unescaper = re.compile(r"\\(.)")

def escape_glob_chars(text):
    """Given a string, return a version of it with the special glob characters
    escaped.  Backslashes are also escaped to prevent ambiguity."""
    return glob_chars.sub(r"\\\1", text)

whitespace = re.compile(r"\s")
def format_cmd(args):
    """Given a list of strings format them to be readable when printed."""
    res = ""
    for arg in args:
        if whitespace.search(arg):
            res += "'" + arg + "' "
        else:
            res += arg + " "
    return res

######################################################################
# Command classes.

class Token(object):
    # Part of a command representing a syntactic token.

    # This classes is a token superclass.  Subclasses need to implement these
    # methods: __str__(), __len__(), modify().  Default implementation, useful
    # to some subclasses, of these are provided: terminates(), terminated(),
    # commences(). A version of modify() is provided which does a couple of
    # useful checks and can be called with super().

    # If dquotable is True, the the token may appear inside double quotes.
    dquotable = False

    def __init__(self):
        self.terminator = re.compile("()(\s)")
        self.commencer = re.compile("\S")

    def modify(self, txt="", begin=0, end=None):
        """Modify this token by replacing the text between begin and end with
        the given text.  begin defaults to the beginning of the token, end to
        the end, and txt to the empty string.

        Returns a 3-tuple:
          1] string: text which wasn't used by this token and spills over to
             the next token.
          2] boolean: True means that this token may absorb some characters
             from the next token.  Spill over is not possible in this case.
          3] boolean: True means that this token needs to be deleted.  May
             include spillover.
        """
        if end is None:
            end = self.__len__()
        if begin > end:
            raise ParseException("Bad insertion range. (begin > end)")
        if end > self.__len__():
            raise ParseException("Bad insertion range. (end > len())")
        if self.terminated() and begin == self.__len__():
            return (txt, False, False)

    def expand(self):
        """Return the expanded version of this token.  Return value is a list
        of strings in which any special shell syntax is replaced with its
        expanded value, appropriate for passing as an arg in a process
        invocation.

        Arg tokens will perform glob expansion on the results of this value, so
        other tokens should protect their response using escape_glob_chars() if
        appropriate.

        The default implementation returns an array with one element containing
        the raw token.
        """
        return [self.__str__()]

    def completions(self, cloc):
        """Return a list of completions for this token from the specified
        cursor location.  If no possible completions exist, [] is returned.
        If possible completions exist, a list of strings is returned.  The
        first item is the root of the completions, the rest are possible
        extensions of the root.  Most tokens will return None if the cursor is
        not at the end.  The default implementation returns None."""
        return []

    def commences(self, txt):
        """Return True if the beginning of the provided text can validly start
        this token.

        The default implementation returns True iff txt matches the regexp
        self.commencer.  The default value of self.commencer is \\S to match
        non-whitespace."""
        return self.commencer.match(txt) is not None

    def terminates(self, txt):
        """Return a pair of strings if the first character of the provided text
        would terminate this token.  The first string are the characters to be
        included in the token, the second are the characters which spill over
        to the next token.  Return None if the text would not terminate the
        token.

        The token mustn't end in an escape character (\) because that would
        make the test incorrect.

        The default implementation matches txt against the regexp
        self.terminator.  On a match it returns match groups 1 and 2, otherwise
        it returns None.  The default value of self.terminator is ()(\\s) to
        terminate on whitespace, but not accept it."""
        match = self.terminator.match(txt)
        if match:
            return match.group(1), match.group(2) + txt[match.end():]
        return None

    def terminated(self):
        """Return True if this token will not accept any more characters at the
        end because it has terminated.  The default implementation returns
        False."""
        return False

    def dump(self):
        """Return a representation of the token tree as an array of strings."""
        return []

class AggregateToken(Token):
    # A token which contains no text directly, and is instead an aggregate of
    # other tokens.

    # Subclasses must implement create_token().

    def __init__(self):
        super(AggregateToken, self).__init__()
        self.tokens = []

    def __str__(self):
        return "".join(map(str, self.tokens))

    def expand(self):
        """Expansions of subtokens are "multiplied" together.
        e.g. "a{u,v}b{w,x}" expands to ["aubw", "aubx", "avbw", "avbx"]."""
        def expand_multiply(x, y):
            if type(x) != list:
                x = x.expand()
            if type(y) != list:
                y = y.expand()
            prod = []
            for xe in x:
                for ye in y:
                    prod.append(str(xe) + str(ye))
            return prod
        return reduce(expand_multiply, self.tokens, [''])

    def __len__(self):
        return reduce(lambda a,b: a+b, map(len, self.tokens), 0)

    def modify(self, txt="", begin=0, end=None):
        check = super(AggregateToken, self).modify(txt, begin, end)
        if check is not None:
            return check

        if end is None:
            end = self.__len__()

        # Determine if this function should return hungry.
        ret_hungry = False
        was_terminated = self.terminated()

        # Find the subtokens in which the modify begins and ends.  If the
        # boundaries fall between tokens, treat it as the end of the earlier
        # token.
        begin_tok, begin_tok_off = (0, 0)
        end_tok, end_tok_off = (0, 0)
        trav_len = 0
        for i in range(len(self.tokens)):
            if trav_len < begin:
                begin_tok = i
                begin_tok_off = begin - trav_len
            if end <= trav_len + len(self.tokens[i]):
                end_tok = i
                end_tok_off = end - trav_len
                break
            trav_len += len(self.tokens[i])

        # Delete tokens as necessary.
        if end_tok > begin_tok:
            extra_txt = str(self.tokens[end_tok])[end_tok_off:]
            self.tokens[begin_tok + 1:end_tok + 1] = []
            txt = txt + extra_txt
            end_tok = begin_tok
            end_tok_off = len(self.tokens[end_tok])

        # If the last character is being changed, then the next token may be
        # absorbed.
        if (end_tok == len(self.tokens) or (end_tok == len(self.tokens) - 1 and
            end_tok_off == len(self.tokens[end_tok]))):
            ret_hungry = True

        # If an existing subtoken is being modified, pass the call to it.
        (hungry, delete) = (False, False)
        if end_tok_off > 0:
            (txt, hungry, delete) = \
                self.tokens[begin_tok].modify(txt, begin_tok_off, end_tok_off)

            # Delete the token if necessary.
            if delete:
                del self.tokens[begin_tok]
                if len(self.tokens) == 0 and not txt:
                    return ("", False, True)
            else:
                begin_tok += 1

        # As long as there is text remaining, create new tokens to hold it, and
        # then dismantle following tokens if the new ones are hungry.

        while txt or hungry:

            # Creating tokens for unused text.
            while txt:
                new_tok = self.create_token(txt, begin_tok)
                if new_tok is None:
                    # This token itself is terminated and can't accept txt.
                    # Strip off any remaining tokens and return them.
                    txt += "".join(map(str, self.tokens[begin_tok:]))
                    self.tokens[begin_tok:] = []
                    return (txt, False, False)
                self.tokens.insert(begin_tok, new_tok)
                (txt, hungry, delete) = new_tok.modify(txt, 0, 0)
                begin_tok += 1

            # If the last token (either the modified or created one) is hungry,
            # then start eating tokens.
            while hungry:
                if len(self.tokens) == begin_tok:
                    # No more tokens
                    return ("", True, False)
                else:
                    eat = str(self.tokens.pop(begin_tok))
                    ht = self.tokens[begin_tok - 1]
                    (txt, hungry, delete) = ht.modify(eat, ht.__len__())

            # Loop back to spill over code if needed.

        # Everything is handled, so return.
        if not self.terminated() and was_terminated:
            ret_hungry = True
        return ("", ret_hungry, False)

    def create_token(self, txt, loc):
        # Create a new token to take characters from the beginning of the
        # provided string, which will be inserted at the given location within
        # this aggregate token.  Implemented by subclasses.  If no token can be
        # created starting with that character, then None is returned.  The
        # newly created token will be empty, and expects insert() to be called
        # with identical text to get it started.
        pass

    def completions(self, cloc):
        if cloc == self.__len__() and len(self.tokens) > 0:
            return self.tokens[-1].completions(len(self.tokens[-1]))

    def dump(self):
        ret = ["%s %s\n" % (str(self), self.__class__.__name__)]
        for tok in self.tokens:
            ret.extend(map(lambda x: "   " + x, tok.dump()))
        return ret

class LeafToken(Token):
    # A token which contains text, and does not contain any other tokens.

    # Subclasses must implement terminates() or set self.terminator as a
    # regexp, and commences() or set self.commencer as a regexp.  Also
    # self.terminated() to indicate if additional characters can be appended.

    def __init__(self):
        super(LeafToken, self).__init__()
        self.repr = ""  # String content of this token.

    def __str__(self):
        return self.repr

    def __len__(self):
        return len(self.repr)

    def modify(self, txt="", begin=0, end=None):
        check = super(LeafToken, self).modify(txt, begin, end)
        if check is not None:
            return check

        if end is None:
            end = self.__len__()

        # Info about whether this token will now be hungry
        hungry = False
        was_terminated = self.terminated()
        if end == self.__len__():
            # If the last character changed, the next token must be checked.
            hungry = True

        # Append end of token to inserted text, moving any trailing backslashes
        # to appended text.
        txt = txt + self.repr[end:]
        self.repr = self.repr[0:begin]
        while len(self.repr) > 0 and self.repr[-1] == "\\":
            self.repr = self.repr[0:-1]
            txt = "\\" + txt

        # Make sure the new first character is valid to commence this token.
        # If it's not valid, call for this token's deletion.
        if begin == 0:
            if len(txt) == 0:
                return txt, False, True
            if txt[0] == "\\":
                first = txt[0:2]
            else:
                first = txt[0]
            if not self.commences(first):
                return txt, False, True

        # Add txt character by character, checking for termination.
        while len(txt) > 0:
            if txt[0] == '\\':
                next = txt[0:2]
                txt = txt[2:]
            else:
                next = txt[0:1]
                txt = txt[1:]

            term = self.terminates(next)
            if term:
                self.repr += term[0]
                return term[1] + txt, False, len(self.repr) == 0

            self.repr += next

        # All text was added, but the token may now be hungry.
        if not self.terminated() and was_terminated:
            hungry = True
        return "", hungry, False

    def dump(self):
        return ["%s %s\n" % (str(self), self.__class__.__name__)]

class CommandText(AggregateToken):
    # The root of the token tree, contains a list of Args and Arg separators.

    def insert(self, loc, txt):
        """Insert the provided text at the given location, raising an exception
        in case of error."""
        self.modify(txt, begin=loc, end=loc)

    def delete(self, loc, count=1):
        """Delete count characters at the given location, raising an exception
        in case of error."""
        self.modify("", begin=loc, end=loc+count)

    def create_token(self, txt, loc):
        if re.match("\s", txt):
            return ArgSeparator()
        elif loc == 0:
            return Prog()
        else:
            return Arg()

    def expand(self):
        """Args are not concatenated on expansion and separators are
        dropped since they return an empty list."""
        res = []
        for tok in self.tokens:
            res.extend(tok.expand())
        return res

    def completions(self, cloc):
        # Completions will be shown at the end of any arg
        for tok in self.tokens:
            if cloc <= len(tok):
                return tok.completions(cloc)
            else:
                cloc -= len(tok)

class ArgOrProg(AggregateToken):
    # Intermediate class for implementation shared by Arg and Prog classes.

    def __init__(self):
        super(ArgOrProg, self).__init__()
        esc_chars = "".join(map(lambda x: x.commencer_ch, arg_subtoks))
        esc_chars += "*?[]"
        self.escaper = re.compile("(["+ re.escape(esc_chars) +"])")

    def create_token(self, txt, loc):
        if re.match("\s", txt):
            # Whitespace is a terminator, so return None
            return None
        for tt in arg_subtoks:
            if txt.startswith(tt.commencer_ch):
                return tt()
        # No special token matches, so it's raw text
        return RawText()

class Arg(ArgOrProg):
    # A top level token representing a single command argument, as passed to
    # the subprocesses.  It is terminated by whitespace.

    def expand(self):
        """When args are expanded, glob matching for files must be done.  The
        resulting completions have all escape characters removed."""
        # Parent takes responsibility for all of the subtokens, which is done
        # prior to glob expansion.
        parexp = super(Arg, self).expand()

        # Results of glob expansion
        globbedexp = []

        for unglobbed in parexp:
            # Unglobbed may contain \ escaped glob characters.  Unfortunately
            # the glob library does not support escaping glob characters, so
            # this will cause bugs.

            if not glob_match.search(unglobbed):
                # No globbing symbols in there, so skip glob expansion, but
                # unescape any escaped glob characters.
                globbedexp.append(unescaper.sub("\\1", unglobbed))
            else:
                # Do glob expansion.  Unfortunately the results are ambiguous
                # if an escaped glob character was in the original pattern or
                # in the results.
                globbed = glob.glob(unglobbed)
                globbed.sort()
                globbedexp.extend(globbed)

        return globbedexp

    def completions(self, cloc):
        """Completions are filename completions, and only work if the arg
        expands to a single filename."""
        check = super(Arg, self).completions(cloc)
        if check:
            return check
        if cloc != self.__len__(): # Cursor must be at the end
            return []

        expansions = self.expand()
        text = self.__str__()
        if len(expansions) > 1 or (len(expansions) == 0 and len(text) > 0):
            return []
        # Use globbing to get completions, but escape globbing symbols first.
        # This implementation of glob doesn't handle escapes properly, a
        # drawback that needs to be rectified, and makes this code broken.
        root = escape_glob_chars(expansions[0])
        comps = glob.glob(root + '*')
        comps = map(lambda x: os.path.isdir(x) and x + os.sep or x, comps)
        comps = map(lambda x: x[len(expansions[0]):], comps)
        comps = map(lambda x: self.escaper.sub(r"\\\1", x), comps)
        comps.sort()
        return text.split(os.sep)[-1:] + comps

class Prog(ArgOrProg):
    # Special case Arg specifying the program or builtin to be run.

    # Inherits basic mechanics of expansion from Arg, but has these additional
    # restrictions:

    #  - It can only expand to a single value, otherwise an error results.
    #  - Expansion is done from $PATH, not from the current directory.
    #  - It may only exist as the very first token of a command.

    # When executing programs there are security concerns to consider.  The
    # user should not be "tricked" into running the wrong program.  Existing
    # shells have restrictions in place to avoid this scenario.  The rules used
    # by hsh are:

    #  - If the program name starts with '/', './' or '../' then the PATH will
    #    not be consulted and normal file location rules will apply.  This
    #    includes cases where tilde or environment expansion results in a
    #    leading '/', './' or '../'.
    #  - Otherwise, the program will be searched for relative to all entries on
    #    the PATH, and in builtins.

    # This represents a departure from other shells (bash, zsh, ksh, tcsh) in
    # the case of a command name with a '/' in it.  hsh will search for such
    # commands in subdirectories of path entries, while other shells will run
    # prgrams relative to the current directory.  With a well-structured path
    # this does not represent a weakening of security, and in fact tighten
    # security in same cases.  The old usage is still available by prepending
    # "./" in that case.  Finally this allows namespacing of groups of
    # commands, for instance, all of the ImageMagick programs could be moved to
    # a subdirectory, so that their generic names would not be so confusing,
    # and the foomatic programs could have simplified names in its own
    # subdirectory.

    # Glob expansion for progs is not supported.  Other shells do glob
    # expansion from the current directory, so for instance 'ech? foo' will not
    # call echo, but will instead look for a file in the current directory
    # which matches the glob pattern.  hsh could search the path for progs
    # matching the glob pattern, but that seems like a marginal feature.

    # re which matches commands not requiring PATH search
    abs_cmd_re = re.compile(r"\.?\.?/")

    def expand(self, check_exists=True):
        """The command is expanded, with any glob characters causing an
        exception.  \ escapes are removed.  If the command doesn't exist an
        empty string is returned, unless check_exists is False."""
        # The parent takes responsibility for all of the subtokens, which
        # is done prior to glob expansion.
        parexp = super(Prog, self).expand()

        # An expression which expands to multiple tokens is not specific enough
        # to be safe.
        if len(parexp) > 1:
            raise ParseException("Command not permitted as multiple expansion.")

        if len(parexp) == 0:
            return []

        if glob_match.search(parexp[0]):
            raise ParseException("Glob symbols not permitted in command.")

        expanded = [unescaper.sub("\\1", parexp[0])]

        if Prog.abs_cmd_re.match(expanded[0]):
            # Program is specified absolutely, so no need to use $PATH
            if check_exists and not os.path.exists(expanded[0]):
                return ['']
        elif (expanded[0].startswith(config.builtin_commencer) or
              expanded[0] in jobs.loaded_builtins.keys()):
            # Builtin command
            if expanded[0].startswith(config.builtin_commencer):
                biname = expanded[0][len(config.builtin_commencer):]
            else:
                biname = expanded[0]
            if biname in jobs.loaded_builtins.keys():
                return [config.builtin_commencer + biname]
            elif check_exists:
                return ['']
            else:
                return expanded[0:1]
        else:
            # Relative specification, so search the path.
            expanded_abs = []

            for pathe in os.environ["PATH"].split(os.path.pathsep):
                ea = os.path.join(pathe, expanded[0])
                if os.path.exists(ea) or not check_exists:
                    expanded_abs.append(ea)
                
            if len(expanded_abs) == 0:
                return ['']
            if check_exists and len(expanded_abs) > 1:
                # Multiple commands with the same name appear on the path.
                # The user should be informed of this irregularity.
                expanded_abs = expanded_abs[:1]

            # And add a builtin too if needed.
            if not check_exists or expanded[0] in jobs.loaded_builtins.keys():
                expanded_abs[0:0] = [config.builtin_commencer + expanded[0]]

            expanded = expanded_abs

        return expanded

    def completions(self, cloc):
        """Completions are filename or built-in completions, and are searched
        for across the entire path."""

        # The parent checks for subtoken completions
        check = super(Prog, self).completions(cloc)
        if check:
            return check
        if cloc != self.__len__(): # Cursor must be at the end
            return

        # Fetch expansions without checking that they exist
        expansions = self.expand(check_exists=False)
        text = self.__str__()

        comps = []

        for exp in expansions:
            # Use globbing to get external completions
            if not exp.startswith(config.builtin_commencer):
                ec = glob.glob(exp + '*')
                # Append / to directories
                ec = map(lambda x: os.path.isdir(x) and x+os.sep or x, ec)
                # Trim root from completions
                ec = map(lambda x: x[len(exp):], ec)
                comps.extend(ec)
                bimatch = exp
            else:
                # Trim the leading '+' from the text
                bimatch = exp[len(config.builtin_commencer):]
            for biname in jobs.loaded_builtins.keys():
                if biname.startswith(bimatch):
                    comps.append(biname[len(bimatch):])

        comps.sort()
        i = len(comps) - 1
        while i > 0:
            if comps[i] == comps[i-1]:
                del comps[i]
            i -= 1
        comps = map(lambda x: self.escaper.sub(r"\\\1", x), comps)
        return text.split(os.sep)[-1:] + comps

class ArgSeparator(LeafToken):
    # A token consisting of whitespace which separates Args.
    def __init__(self):
        super(ArgSeparator, self).__init__()
        self.terminator = re.compile("()(\S)")
        self.commencer = re.compile("\s")

    def expand(self):
        """They expand to nothing."""
        return []

class RawText(LeafToken):
    # RawText part of a token which is not subjected to any modification
    def __init__(self):
        super(RawText, self).__init__()
        # Terminates whenever any more specialized token starts.
        term_chars = "".join(map(lambda x: x.commencer_ch, arg_subtoks))
        self.terminator = re.compile("()(\s|["+ re.escape(term_chars) +"])")
        self.commencer = re.compile("\S")

    def expand(self):
        # Unescape everything except globbing stuff.
        return [unescaper_nonglobs.sub("\\1", self.__str__())]

class SingleQuotedToken(LeafToken):

    commencer_ch = "'"

    def __init__(self):
        super(SingleQuotedToken, self).__init__()
        # Uses custom terminate methods rather than the re
        self.commencer = re.compile("'")

    def terminates(self, ch):
        # Terminates on a single quote other than the one at the beginning.
        if self.__len__() > 0 and ch[0:1] == "'":
            return ch[0:1], ch[1:]
        return None

    def terminated(self):
        return self.__len__() > 1 and self.repr[-1]=="'" and self.repr[-2]!="\\"

    def expand(self):
        res = self.repr
        # drop single quotes from the beginning and end
        if self.terminated():
            res = res[0:-1]
        if len(res) > 0 and res[0] == "'":
            res = res[1:]
        return [escape_glob_chars(res)]

class DoubleQuote(LeafToken):
    """A leaf token which contains a single double quote character.  Used
    exclusively within the DoubleQuotedToken token as a commencer and
    terminator."""
    def __init__(self):
        super(DoubleQuote, self).__init__()
        self.terminator = re.compile('(")()')
        self.commencer = re.compile('"')

    def terminated(self):
        # It terminates after the first character.
        return self.__len__() > 0

    def expand(self):
        "Double quotes are dropped on expansion."
        return ['']

class DoubleQuotedText(LeafToken):
    """The text between the double quotes, may include whitespace characters."""

    def __init__(self):
        super(DoubleQuotedText, self).__init__()
        # Terminates whenever any more specialized token starts.
        term_chars = "".join(map(lambda x: x.commencer_ch,
                                 filter(lambda x: x.dquotable, arg_subtoks)))
        self.terminator = re.compile("()([\""+ re.escape(term_chars) +"])")
        self.commencer = re.compile('[^"]')

    def expand(self):
        """Double quotes suppress glob/re stuff."""
        return [escape_glob_chars(self.repr)]

class DoubleQuotedToken(AggregateToken):
    """Text enclosed in double quotes.  Special shell syntax like env variables
    are expanded within double quoted text, so it is an AggregateToken with
    DoubleQuote tokens at the beginning and end.
    """

    commencer_ch = "\""

    def __init__(self):
        super(DoubleQuotedToken, self).__init__()
        # Uses custom terminate methods rather than the re
        self.commencer = re.compile('"')

    def terminates(self, ch):
        # Terminates on a double quote other than the one at the beginning.
        if self.__len__() > 0 and ch[0:1] == '"':
            return ch[0:1], ch[1:]
        return None

    def terminated(self):
        """While modifying the token it's possible to have double quotes other
        than at the end, so search all subtokens for a terminating quote."""
        for tok in self.tokens[1:]:
            if tok.__class__ == DoubleQuote:
                return True
        return False

    def create_token(self, txt, loc):
        if self.terminated():
            return None
        elif txt.startswith('"'):
            return DoubleQuote()

        for tt in arg_subtoks:
            if tt.dquotable and txt.startswith(tt.commencer_ch):
                return tt()

        # No special token matches, so it's text
        return DoubleQuotedText()

class BracedToken(LeafToken):
    """Super class for tokens which can be enclosed in braces, after a
    commencing character like $ or ~."""

    def _braced(self):
        return self.__len__() > 1 and self.repr[1:2] == '{'

    def _text(self):
        # The token "text" without braces or the commencer
        if self._braced():
            if self.repr[-1] == '}':
                return self.repr[2:-1]
            else:
                return self.repr[2:]
        else:
            return self.repr[1:]

    def terminates(self, ch):
        # If an open brace is used, terminates only on a closed brace,
        # otherwise terminates on any non-alphanumeric.
        if self.__len__() == 0:
            return None
        if self.__len__() == 1:
            if ch[0:1] == '{':
                return None
        if self._braced():
            if ch[0:1] == "}":
                return ch[0:1], ch[1:]
            else:
                return None
        else:
            return super(BracedToken, self).terminates(ch)

    def terminated(self):
        return self._braced() and self.repr[-1:] == '}'

    def _completions(self, cloc, possibilities, root=None, brace=True):
        if cloc != self.__len__(): # Cursor must be at the end
            return
        # Search possibilities for matching options.
        if root is None:
            root = self._text()
        comps = filter(lambda x: x.startswith(root), possibilities)
        comps = map(lambda x: x[len(root):], comps)
        if brace and self._braced():
            comps = map(lambda x: x + '}', comps)
        comps.sort()
        return [root] + comps

class EnvVariable(BracedToken):
    """An environment variable reference, starting with a $, with optional
    braces {}."""

    dquotable = True
    commencer_ch = "$"
    
    def __init__(self):
        super(EnvVariable, self).__init__()
        self.terminator = re.compile('()(\W)')
        self.commencer = re.compile('\$')

    def expand(self):
        try:
            return [escape_glob_chars(os.environ[self._text()])]
        except KeyError:
            return ['']

    def completions(self, cloc):
        return self._completions(cloc, os.environ.keys())

class NamedDir(BracedToken):
    """A named directory reference, starting with a ~, with optional
    braces {}."""
    
    dquotable = True
    commencer_ch = "~"
    
    def __init__(self):
        super(NamedDir, self).__init__()
        self.terminator = re.compile('()(\W)')
        self.commencer = re.compile('\~')

    def expand(self):
        return [escape_glob_chars(os.path.expanduser('~' + self._text()))]

    def completions(self, cloc):
        return self._completions(cloc, map(lambda x: x.pw_name, pwd.getpwall()))

class JobReference(BracedToken):
    """A reference to some feature of a job, starting with a % and optionally
    using braces to surround the value."""

    dquotable = True
    commencer_ch = "%"
    parsere = re.compile('^([0-9]*)([a-zA-Z,]*)$')

    def __init__(self):
        super(JobReference, self).__init__()
        self.terminator = re.compile('()(\s)')
        self.commencer = re.compile('\%')
        self.jobid = None
        self.features = []

    def _parse(self):
        """Break up the string into component parts."""
        match = JobReference.parsere.match(self._text())
        if match is None:
            self.jobid = None
            self.features = []
        else:
            self.jobid = match.group(1)
            self.features = match.group(2).split(",")

    feature_list = ["p", "fo", "fe", "fi", "to", "te", "ti"]

    def _expand_feature(self, job, feature):
        if feature == "p":
            return [str(job.pid) or ""]
        if feature[0:1] == "f":
            fname = self._expand_feature_filetype(feature[1:2])
            if fname is not None:
                return [os.path.join(config.jobs_dir, str(job.jid), fname)]

        if feature[0:1] == "t":
            fname = self._expand_feature_filetype(feature[1:2])
            res = []
            if fname is not None:
                try:
                    jf = open(os.path.join(config.jobs_dir,str(job.jid),fname))
                    res = map(lambda x: x.strip(), jf.readlines())
                    jf.close()
                except:
                    pass
            return res
        return []

    def _expand_feature_filetype(self, ftype):
        if ftype == "o":
            return "stdout"
        if ftype == "e":
            return "stderr"
        if ftype == "i":
            return "stdin"

    def expand(self):
        self._parse()
        if not self.jobid:
            job = hsh.display_obj.get_visible_job() 
        else:
            job = jobs.manager.get_job(int(self.jobid))
        if job is None:
            return []
        # List of lists of expanded feature 
        expsl = map(lambda x: self._expand_feature(job, x), self.features)
        exps = reduce(lambda x,y: x+y, expsl, [])
        return map(escape_glob_chars, exps)

    def completions(self, cloc):
        # If no feature text is present, collect job id completions.
        if len(self.features) == 0:
            return []
        if len(self.features) == 1 and len(self.features[0]) == 0:
            comps = self._completions(cloc, jobs.manager.get_jobid_list(),
                                      brace=False)
            if self.jobid in jobs.manager.get_jobid_list():
                comps += JobReference.feature_list
            return comps
        else:
            return self._completions(cloc, JobReference.feature_list,
                                     root=self.features[-1])

######################################################################

# List of token classes which can be created within an arg.

# The list is used during parsing to decide which token class to create, and is
# used during object construction to build terminator expressions.  Each class
# in the list needs to specify two class variables:
#   .commencer_ch: A character which identifies the beginning of such a token
#   .dquotable: True to permit token in double quotes.  False by default

# Superclasses which are not instantiated, such as ArgOrProg, and special
# purpose tokens, such as DoubleQuote, are not included.

arg_subtoks = [ SingleQuotedToken, DoubleQuotedToken,
                EnvVariable, NamedDir, JobReference,
              ]
