/*
   Copyright (C)  2000    Daniel A. Atkinson  <DanAtk@aol.com>
   Copyright (C)  2004    Ivano Primi  <ivano.primi@tin.it>    

   This file is part of the HPA Library.

   The HPA Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   The HPA Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the HPA Library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307 USA.
*/

/*
    Test:  cxinv  cxlog   cxlog10  cxlog2  
           cxtan  cxtanh  cxatan   cxatanh

    Uses:  All the tested functions + atocx  xpr_asprint

    Input file:  data/cxgen.dat
*/
#include<stdio.h>
#include<stdlib.h>
#include"cxpre.h"

#define DECD 32

int
getline (char *buffer, int size, FILE * fp)
{
  if ((fgets (buffer, size, fp)))
    {
      int i;

      for (i = 0; buffer[i] != '\0'; i++);
      if (i > 0 && buffer[i - 1] == '\n')
	buffer[--i] = '\0';
      return i;
    }
  else
    return -1;
}

typedef struct cxpr (*pfun) (struct cxpr);
extern int xErrNo;

int
main (int na, char **av)
{
  struct cxpr z, w;
  FILE *fp;
  char num[1024], *re, *im;
  const char *nlist[] = {
    "cxinv", "cxlog", "cxlog10", "cxlog2",
    "cxtan", "cxtanh", "cxatan", "cxatanh",
    NULL
  };
  pfun flist[] = {
    cxinv, cxlog, cxlog10, cxlog2,
    cxtan, cxtanh, cxatan, cxatanh
  };
  int i;

  if (na != 2)
    {
      printf ("para: input_file\n");
      exit (-1);
    }
  fp = fopen (*++av, "r");
  for (i = 0; nlist[i] != NULL; i++)
    {
      printf ("Test of %s\n", nlist[i]);
      while (getline (num, 1024, fp) != -1)
	{
	  printf ("*string in : \t%s\n", num);
	  /* convert ascii string to extended precision complex */
	  z = atocx (num);
	  re = xpr_asprint (z.re, 1, 0, DECD);
	  im = xpr_asprint (z.im, 1, 0, DECD);
	  printf (" z =  \t\t%s  \t%s\n", re, im);
	  free ((void *) re);
	  free ((void *) im);
	  printf (" %s(z) = \t", nlist[i]);
	  xErrNo = 0;
	  w = (*flist[i]) (z);
	  switch (xErrNo)
	    {
	    case XENONE:
	      re = xpr_asprint (w.re, 1, 0, DECD);
	      im = xpr_asprint (w.im, 1, 0, DECD);
	      printf ("%s  \t%s\n", re, im);
	      free ((void *) re);
	      free ((void *) im);
	      break;
	    case XEDIV:
	      printf ("*** Division by zero\n");
	      break;
	    case XEDOM:
	      printf ("*** Out of domain\n");
	      break;
	    case XEBADEXP:
	      printf ("*** Bad exponent\n");
	      break;
	    case XFPOFLOW:
	      printf ("*** Floating point overflow\n");
	      break;
	    default:
	      printf ("*** Unknown error\n");
	    }
	}
      printf ("\n");
      rewind (fp);
    }
  fclose (fp);
  return 0;
}

/*  Test output

Test of cxinv
*string in : 	0
 z =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxinv(z) = 	*** Out of domain
*string in : 	+2.3106e-1571
 z =  		2.31059999999999999999999999999571e-1571  	0.00000000000000000000000000000000e+0
 cxinv(z) = 	4.32788020427594564182463429412274e+1570  	-0.00000000000000000000000000000000e-1791
*string in : 	+8.7792e-20  +9.7732e-20i
 z =  		8.77919999999999999999999999999979e-20  	9.77319999999999999999999999999977e-20
 cxinv(z) = 	5.08674351781565818187866617107985e+18  	-5.66267561375933918160385686887160e+18
*string in : 	+2.3298e-1571i
 z =  		0.00000000000000000000000000000000e+0  	2.32979999999999999999999999999567e-1571
 cxinv(z) = 	0.00000000000000000000000000000000e-1791  	-4.29221392394196926774830457550005e+1570
*string in : 	-4.5592e-19  +8.2219e-18i
 z =  		-4.55919999999999999999999999999990e-19  	8.22189999999999999999999999999983e-18
 cxinv(z) = 	-6.72373973585603016881674533871915e+15  	-1.21253543898567060986564306238847e+17
*string in : 	-3.8932e-1624i
 z =  		0.00000000000000000000000000000000e+0  	-3.89319999999999999999999999999252e-1624
 cxinv(z) = 	0.00000000000000000000000000000000e-1686  	2.56858111579163669988698243090518e+1623
*string in : 	-1.2291e-17  -7.3342e-18i
 z =  		-1.22909999999999999999999999999998e-17  	-7.33419999999999999999999999999984e-18
 cxinv(z) = 	-5.99973140650805087141008841487323e+16  	3.58011797914013072175542026298619e+16
*string in : 	-4.0987e-1617i
 z =  		0.00000000000000000000000000000000e+0  	-4.09869999999999999999999999999216e-1617
 cxinv(z) = 	0.00000000000000000000000000000000e-1700  	2.43979798472686461560982750628248e+1616
*string in : 	+3.9917e-19  -8.2215e-19i
 z =  		3.99169999999999999999999999999991e-19  	-8.22149999999999999999999999999982e-19
 cxinv(z) = 	4.77894914061639884258973527932558e+17  	9.84295672509901121936806588645821e+17
*string in : 	+1.0
 z =  		1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxinv(z) = 	1.00000000000000000000000000000000e+0  	-0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000000986
 z =  		1.00000000000000000009860000000000e+0  	0.00000000000000000000000000000000e+0
 cxinv(z) = 	9.99999999999999999901400000000001e-1  	-0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000004567 +2.43e-18i
 z =  		1.00000000000000000045670000000000e+0  	2.42999999999999999999999999999995e-18
 cxinv(z) = 	9.99999999999999999543300000000001e-1  	-2.42999999999999999778043799999995e-18
*string in : 	+1.0  +3.4506e-1298i
 z =  		1.00000000000000000000000000000000e+0  	3.45059999999999999999999999999470e-1298
 cxinv(z) = 	1.00000000000000000000000000000000e+0  	-3.45059999999999999999999999999470e-1298
*string in : 	+0.999999999999999998736  +4.6677e-1297i
 z =  		9.99999999999999998735999999999976e-1  	4.66769999999999999999999999999284e-1297
 cxinv(z) = 	1.00000000000000000126400000000002e+0  	-4.66770000000000001179994559999306e-1297
*string in : 	+0.999999999999999999754
 z =  		9.99999999999999999753999999999976e-1  	0.00000000000000000000000000000000e+0
 cxinv(z) = 	1.00000000000000000024600000000003e+0  	-0.00000000000000000000000000000000e+0
*string in : 	+0.999999999999999999338  -5.0783e-1145i
 z =  		9.99999999999999999337999999999976e-1  	-5.07829999999999999999999999999312e-1145
 cxinv(z) = 	1.00000000000000000066200000000003e+0  	5.07830000000000000672366919999337e-1145
*string in : 	+1.0000000000000000000    -6.7002e-1301i
 z =  		1.00000000000000000000000000000000e+0  	-6.70019999999999999999999999998969e-1301
 cxinv(z) = 	1.00000000000000000000000000000000e+0  	6.70019999999999999999999999998969e-1301
*string in : 	+1.0000000000000007423    -5.3321e-1344i
 z =  		1.00000000000000074230000000000000e+0  	-5.33209999999999999999999999999153e-1344
 cxinv(z) = 	9.99999999999999257700000000000550e-1  	5.33209999999999208396434000000035e-1344
*string in : 	-1.0
 z =  		-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxinv(z) = 	-1.00000000000000000000000000000000e+0  	-0.00000000000000000000000000000000e+0
*string in : 	-0.99999999999999999998811
 z =  		-9.99999999999999999988109999999973e-1  	0.00000000000000000000000000000000e+0
 cxinv(z) = 	-1.00000000000000000001189000000003e+0  	-0.00000000000000000000000000000000e+0
*string in : 	-0.99999999999999999976632 +2.3344e-15i
 z =  		-9.99999999999999999766319999999974e-1  	2.33439999999999999999999999999996e-15
 cxinv(z) = 	-1.00000000000000000023367999999458e+0  	-2.33440000000000000109100518398736e-15
*string in : 	-1.0  +4.5601e-997i
 z =  		-1.00000000000000000000000000000000e+0  	4.56009999999999999999999999999462e-997
 cxinv(z) = 	-1.00000000000000000000000000000000e+0  	-4.56009999999999999999999999999462e-997
*string in : 	-1.00000000000000000000654 +7.4332e-1532i
 z =  		-1.00000000000000000000654000000000e+0  	7.43319999999999999999999999998655e-1532
 cxinv(z) = 	-9.99999999999999999993460000000001e-1  	-7.43319999999999999990277374398655e-1532
*string in : 	-1.00000000000000000000743
 z =  		-1.00000000000000000000743000000000e+0  	0.00000000000000000000000000000000e+0
 cxinv(z) = 	-9.99999999999999999992570000000001e-1  	-0.00000000000000000000000000000000e+0
*string in : 	-1.00000000000000000085023 -6.9983e-18i
 z =  		-1.00000000000000000085023000000000e+0  	-6.99829999999999999999999999999985e-18
 cxinv(z) = 	-9.99999999999999999149770000000001e-1  	6.99829999999999998809967078199985e-18
*string in : 	-1.0  -5.00098e-1023i
 z =  		-1.00000000000000000000000000000000e+0  	-5.00097999999999999999999999999395e-1023
 cxinv(z) = 	-1.00000000000000000000000000000000e+0  	5.00097999999999999999999999999395e-1023
*string in : 	-0.9999999999999999997643  -6.7321e-16i
 z =  		-9.99999999999999999764299999999974e-1  	-6.73209999999999999999999999999987e-16
 cxinv(z) = 	-1.00000000000000000023569999999957e+0  	6.73210000000000000317351193999717e-16
*string in : 	+1.0i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
 cxinv(z) = 	0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	+5.6098e-2198 +1.0i
 z =  		5.60979999999999999999999999998542e-2198  	1.00000000000000000000000000000000e+0
 cxinv(z) = 	5.60979999999999999999999999998542e-2198  	-1.00000000000000000000000000000000e+0
*string in : 	+4.9987e-18   +1.0000000000000000005421i
 z =  		4.99869999999999999999999999999989e-18  	1.00000000000000000054210000000000e+0
 cxinv(z) = 	4.99869999999999999458040945999989e-18  	-9.99999999999999999457900000000001e-1
*string in : 	+1.000000000000000000003201i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000320100000000e+0
 cxinv(z) = 	0.00000000000000000000000000000000e+0  	-9.99999999999999999996799000000001e-1
*string in : 	-6.5541e-20   +1.0000000000000000080981i
 z =  		-6.55409999999999999999999999999984e-20  	1.00000000000000000809810000000000e+0
 cxinv(z) = 	-6.55409999999999989384848557999985e-20  	-9.99999999999999991901900000000001e-1
*string in : 	-7.0998e-2200 +1.0i
 z =  		-7.09979999999999999999999999998153e-2200  	1.00000000000000000000000000000000e+0
 cxinv(z) = 	-7.09979999999999999999999999998153e-2200  	-1.00000000000000000000000000000000e+0
*string in : 	-6.8432e-17   +0.9999999999999999999864i
 z =  		-6.84319999999999999999999999999986e-17  	9.99999999999999999986399999999974e-1
 cxinv(z) = 	-6.84320000000000000018613504000018e-17  	-1.00000000000000000001360000000002e+0
*string in : 	+0.99999999999999989098732i
 z =  		0.00000000000000000000000000000000e+0  	9.99999999999999890987319999999974e-1
 cxinv(z) = 	0.00000000000000000000000000000000e+0  	-1.00000000000000010901268000000004e+0
*string in : 	+4.3329e-16   +0.9999999999999999997877i
 z =  		4.33289999999999999999999999999992e-16  	9.99999999999999999787699999999974e-1
 cxinv(z) = 	4.33290000000000000183974933999933e-16  	-1.00000000000000000021229999999984e+0
*string in : 	-1.0i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
 cxinv(z) = 	0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
*string in : 	+2.7762e-2111 -1.0i
 z =  		2.77619999999999999999999999999307e-2111  	-1.00000000000000000000000000000000e+0
 cxinv(z) = 	2.77619999999999999999999999999307e-2111  	1.00000000000000000000000000000000e+0
*string in : 	+3.3321e-17   -0.9999999999999999996555i
 z =  		3.33209999999999999999999999999993e-17  	-9.99999999999999999655499999999974e-1
 cxinv(z) = 	3.33210000000000000229581690000010e-17  	1.00000000000000000034450000000003e+0
*string in : 	-0.999999999999999999996999972013i
 z =  		0.00000000000000000000000000000000e+0  	-9.99999999999999999996999972012965e-1
 cxinv(z) = 	0.00000000000000000000000000000000e+0  	1.00000000000000000000300002798704e+0
*string in : 	-7.5541e-17   -0.999999999999999999987231i
 z =  		-7.55409999999999999999999999999985e-17  	-9.99999999999999999987230999999972e-1
 cxinv(z) = 	-7.55410000000000000019291660580023e-17  	1.00000000000000000001276900000002e+0
*string in : 	-2.1156e-2203 -1.0i
 z =  		-2.11559999999999999999999999999449e-2203  	-1.00000000000000000000000000000000e+0
 cxinv(z) = 	-2.11559999999999999999999999999449e-2203  	1.00000000000000000000000000000000e+0
*string in : 	-5.9923e-16   -1.00000000000000000089321i
 z =  		-5.99229999999999999999999999999988e-16  	-1.00000000000000000089321000000000e+0
 cxinv(z) = 	-5.99229999999999998929523543399773e-16  	9.99999999999999999106789999999641e-1
*string in : 	-1.000000000000000000007161i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000716100000000e+0
 cxinv(z) = 	0.00000000000000000000000000000000e+0  	9.99999999999999999992839000000001e-1
*string in : 	+8.6631e-19   -1.00000000000000000008765i
 z =  		8.66309999999999999999999999999981e-19  	-1.00000000000000000008765000000000e+0
 cxinv(z) = 	8.66309999999999999848135856999981e-19  	9.99999999999999999912350000000001e-1
*string in : 	+1.5876e1876
 z =  		1.58759999999999999999999999999648e+1876  	0.00000000000000000000000000000000e+0
 cxinv(z) = 	6.29881582262534643487024439405392e-1877  	-0.00000000000000000000000000000000e+0
*string in : 	-6.4391e1965
 z =  		-6.43909999999999999999999999998503e+1965  	0.00000000000000000000000000000000e+0
 cxinv(z) = 	-1.55301206690375984221397400257800e-1966  	-0.00000000000000000000000000000000e+0
*string in : 	+3.4210e2015i
 z =  		0.00000000000000000000000000000000e+0  	3.42099999999999999999999999999185e+2015
 cxinv(z) = 	0.00000000000000000000000000000000e+0  	-2.92312189418298743057585501315405e-2016
*string in : 	-4.5522e2011i
 z =  		0.00000000000000000000000000000000e+0  	-4.55219999999999999999999999998917e+2011
 cxinv(z) = 	0.00000000000000000000000000000000e+0  	2.19674003778392864988357277799745e-2012
*string in : 	+2.1907e1854  +9.8765e1872i
 z =  		2.19069999999999999999999999999520e+1854  	9.87649999999999999999999999997812e+1872
 cxinv(z) = 	2.24582944783968201794260058413043e-1892  	-1.01250442970687996759985824937984e-1873
*string in : 	-3.2966e1946  +7.5721e1834i
 z =  		-3.29659999999999999999999999999241e+1946  	7.57209999999999999999999999998357e+1834
 cxinv(z) = 	-3.03342838075593035248437784383910e-1947  	-6.96760997449553486077988153592612e-2059
*string in : 	-6.0987e2310  -5.4411e2198i
 z =  		-6.09869999999999999999999999998333e+2310  	-5.44109999999999999999999999998585e+2198
 cxinv(z) = 	-1.63969370521586567629166871628380e-2311  	1.46289166862610830689664988475770e-2423
*string in : 	+7.8895e1854  -3.8699e1943i
 z =  		7.88949999999999999999999999998271e+1854  	-3.86989999999999999999999999999111e+1943
 cxinv(z) = 	5.26805129359351591086138303666690e-2033  	2.58404609938241298224760329724282e-1944
*string in : 	+10.009876543100091234
 z =  		1.00098765431000912340000000000000e+1  	0.00000000000000000000000000000000e+0
 cxinv(z) = 	9.99013320188559245050809813217570e-2  	-0.00000000000000000000000000000000e+0
*string in : 	+9.8980000000006542009 +6.11002230065009876i
 z =  		9.89800000000065420089999999999997e+0  	6.11002230065009876000000000000000e+0
 cxinv(z) = 	7.31544485263629822917788348825009e-2  	-4.51581442602352024120154388961665e-2
*string in : 	+17.9998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+1
 cxinv(z) = 	0.00000000000000000000000000000000e+0  	-5.55558987294654958788507043727429e-2
*string in : 	-12.9822227      +20.87777220192i
 z =  		-1.29822227000000000000000000000000e+1  	2.08777722019200000000000000000000e+1
 cxinv(z) = 	-2.14788291328571257093581789652378e-2  	-3.45418586756914867622652282805959e-2
*string in : 	-17.9877771265
 z =  		-1.79877771265000000000000000000000e+1  	0.00000000000000000000000000000000e+0
 cxinv(z) = 	-5.55933061082226434822844201087784e-2  	-0.00000000000000000000000000000000e+0
*string in : 	-19.09872210981  -32.9800000654i
 z =  		-1.90987221098100000000000000000000e+1  	-3.29800000653999999999999999999999e+1
 cxinv(z) = 	-1.31493908148755359206170442404372e-2  	2.27065930087444780684581424642228e-2
*string in : 	-43.987600182345i
 z =  		0.00000000000000000000000000000000e+0  	-4.39876001823449999999999999999999e+1
 cxinv(z) = 	0.00000000000000000000000000000000e+0  	2.27336793972534812153382764185621e-2
*string in : 	+15.982201192361 -43.2189031229i
 z =  		1.59822011923610000000000000000000e+1  	-4.32189031228999999999999999999999e+1
 cxinv(z) = 	7.52704210860149871587709852116151e-3  	2.03545494001868476459468175295031e-2
*string in : 	+1000.009876543100091234
 z =  		1.00000987654310009123400000000000e+3  	0.00000000000000000000000000000000e+0
 cxinv(z) = 	9.99990123554445048965178381439077e-4  	-0.00000000000000000000000000000000e+0
*string in : 	+989.80000000006542009 +611.002230065009876i
 z =  		9.89800000000065420089999999999996e+2  	6.11002230065009875999999999999998e+2
 cxinv(z) = 	7.31544485263629822917788348825008e-4  	-4.51581442602352024120154388961664e-4
*string in : 	+179.998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+2
 cxinv(z) = 	0.00000000000000000000000000000000e+0  	-5.55558987294654958788507043727429e-3
*string in : 	-1298.22227  +208.7777220192i
 z =  		-1.29822227000000000000000000000000e+3  	2.08777722019199999999999999999999e+2
 cxinv(z) = 	-7.50864891259365593987370069400607e-4  	-1.20752713278693536654582145833162e-4
*string in : 	-1798.777712650981
 z =  		-1.79877771265098099999999999999999e+3  	0.00000000000000000000000000000000e+0
 cxinv(z) = 	-5.55933061081923245424239792765725e-4  	-0.00000000000000000000000000000000e+0
*string in : 	-190.9872210981   -329.800700654i
 z =  		-1.90987221098100000000000000000000e+2  	-3.29800700653999999999999999999999e+2
 cxinv(z) = 	-1.31493490140628799704584503524431e-3  	2.27065690209420215215126222323033e-3
*string in : 	+1598.2201192361  -4321.89031229i
 z =  		1.59822011923609999999999999999999e+3  	-4.32189031228999999999999999999998e+3
 cxinv(z) = 	7.52704210860149871587709852116151e-5  	2.03545494001868476459468175295031e-4

Test of cxlog
*string in : 	0
 z =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxlog(z) = 	*** Out of domain
*string in : 	+2.3106e-1571
 z =  		2.31059999999999999999999999999571e-1571  	0.00000000000000000000000000000000e+0
 cxlog(z) = 	-3.61652367386257898843685163641844e+3  	0.00000000000000000000000000000000e-3362
*string in : 	+8.7792e-20  +9.7732e-20i
 z =  		8.77919999999999999999999999999979e-20  	9.77319999999999999999999999999977e-20
 cxlog(z) = 	-4.34762430459384677476607181152403e+1  	8.38924958292089629849736401676606e-1
*string in : 	+2.3298e-1571i
 z =  		0.00000000000000000000000000000000e+0  	2.32979999999999999999999999999567e-1571
 cxlog(z) = 	-3.61651539866666223001606640007433e+3  	1.57079632679489661923132169163975e+0
*string in : 	-4.5592e-19  +8.2219e-18i
 z =  		-4.55919999999999999999999999999990e-19  	8.22189999999999999999999999999983e-18
 cxlog(z) = 	-3.93381952501562719183299310559455e+1  	1.62619149766846609737886543279789e+0
*string in : 	-3.8932e-1624i
 z =  		0.00000000000000000000000000000000e+0  	-3.89319999999999999999999999999252e-1624
 cxlog(z) = 	-3.73803895958075998935377582630999e+3  	-1.57079632679489661923132169163975e+0
*string in : 	-1.2291e-17  -7.3342e-18i
 z =  		-1.22909999999999999999999999999998e-17  	-7.33419999999999999999999999999984e-18
 cxlog(z) = 	-3.87853702886043822258197184335673e+1  	-2.60359353930614271172250071199188e+0
*string in : 	-4.0987e-1617i
 z =  		0.00000000000000000000000000000000e+0  	-4.09869999999999999999999999999216e-1617
 cxlog(z) = 	-3.72186942552111066666318991842490e+3  	-1.57079632679489661923132169163975e+0
*string in : 	+3.9917e-19  -8.2215e-19i
 z =  		3.99169999999999999999999999999991e-19  	-8.22149999999999999999999999999982e-19
 cxlog(z) = 	-4.15365334098677599742614120643356e+1  	-1.11879993977436025825292922558941e+0
*string in : 	+1.0
 z =  		1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxlog(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000000986
 z =  		1.00000000000000000009860000000000e+0  	0.00000000000000000000000000000000e+0
 cxlog(z) = 	9.85999999999999028997916193240068e-20  	0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000004567 +2.43e-18i
 z =  		1.00000000000000000045670000000000e+0  	2.42999999999999999999999999999995e-18
 cxlog(z) = 	4.56700000000000235358292413058534e-19  	2.42999999999999999889021899999995e-18
*string in : 	+1.0  +3.4506e-1298i
 z =  		1.00000000000000000000000000000000e+0  	3.45059999999999999999999999999470e-1298
 cxlog(z) = 	0.00000000000000000000000000000000e+0  	3.45059999999999999999999999999470e-1298
*string in : 	+0.999999999999999998736  +4.6677e-1297i
 z =  		9.99999999999999998735999999999976e-1  	4.66769999999999999999999999999284e-1297
 cxlog(z) = 	-1.26400000000002530239333998946922e-18  	4.66770000000000000589997279999295e-1297
*string in : 	+0.999999999999999999754
 z =  		9.99999999999999999753999999999976e-1  	0.00000000000000000000000000000000e+0
 cxlog(z) = 	-2.46000000000025114777727616384842e-19  	0.00000000000000000000000000000000e+0
*string in : 	+0.999999999999999999338  -5.0783e-1145i
 z =  		9.99999999999999999337999999999976e-1  	-5.07829999999999999999999999999312e-1145
 cxlog(z) = 	-6.62000000000025539740200611368359e-19  	-5.07830000000000000336183459999325e-1145
*string in : 	+1.0000000000000000000    -6.7002e-1301i
 z =  		1.00000000000000000000000000000000e+0  	-6.70019999999999999999999999998969e-1301
 cxlog(z) = 	0.00000000000000000000000000000000e+0  	-6.70019999999999999999999999998969e-1301
*string in : 	+1.0000000000000007423    -5.3321e-1344i
 z =  		1.00000000000000074230000000000000e+0  	-5.33209999999999999999999999999153e-1344
 cxlog(z) = 	7.42299999999999724850441654385376e-16  	-5.33209999999999604198216999999447e-1344
*string in : 	-1.0
 z =  		-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxlog(z) = 	0.00000000000000000000000000000000e+0  	3.14159265358979323846264338327950e+0
*string in : 	-0.99999999999999999998811
 z =  		-9.99999999999999999988109999999973e-1  	0.00000000000000000000000000000000e+0
 cxlog(z) = 	-1.18900000000281955174677185038234e-20  	3.14159265358979323846264338327950e+0
*string in : 	-0.99999999999999999976632 +2.3344e-15i
 z =  		-9.99999999999999999766319999999974e-1  	2.33439999999999999999999999999996e-15
 cxlog(z) = 	-2.33679999997303308994748520721630e-19  	3.14159265358979090406264338327950e+0
*string in : 	-1.0  +4.5601e-997i
 z =  		-1.00000000000000000000000000000000e+0  	4.56009999999999999999999999999462e-997
 cxlog(z) = 	0.00000000000000000000000000000000e+0  	3.14159265358979323846264338327950e+0
*string in : 	-1.00000000000000000000654 +7.4332e-1532i
 z =  		-1.00000000000000000000654000000000e+0  	7.43319999999999999999999999998655e-1532
 cxlog(z) = 	6.53999999999975488870838461200929e-21  	3.14159265358979323846264338327950e+0
*string in : 	-1.00000000000000000000743
 z =  		-1.00000000000000000000743000000000e+0  	0.00000000000000000000000000000000e+0
 cxlog(z) = 	7.42999999999994813856200361547675e-21  	3.14159265358979323846264338327950e+0
*string in : 	-1.00000000000000000085023 -6.9983e-18i
 z =  		-1.00000000000000000085023000000000e+0  	-6.99829999999999999999999999999985e-18
 cxlog(z) = 	8.50230000000000289928437668489227e-19  	-3.14159265358979323146434338327950e+0
*string in : 	-1.0  -5.00098e-1023i
 z =  		-1.00000000000000000000000000000000e+0  	-5.00097999999999999999999999999395e-1023
 cxlog(z) = 	0.00000000000000000000000000000000e+0  	-3.14159265358979323846264338327950e+0
*string in : 	-0.9999999999999999997643  -6.7321e-16i
 z =  		-9.99999999999999999764299999999974e-1  	-6.73209999999999999999999999999987e-16
 cxlog(z) = 	-2.35699999999800498301067807364444e-19  	-3.14159265358979256525264338327950e+0
*string in : 	+1.0i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
 cxlog(z) = 	0.00000000000000000000000000000000e+0  	1.57079632679489661923132169163975e+0
*string in : 	+5.6098e-2198 +1.0i
 z =  		5.60979999999999999999999999998542e-2198  	1.00000000000000000000000000000000e+0
 cxlog(z) = 	0.00000000000000000000000000000000e+0  	1.57079632679489661923132169163975e+0
*string in : 	+4.9987e-18   +1.0000000000000000005421i
 z =  		4.99869999999999999999999999999989e-18  	1.00000000000000000054210000000000e+0
 cxlog(z) = 	5.42100000000000085236578338630674e-19  	1.57079632679489661423262169163975e+0
*string in : 	+1.000000000000000000003201i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000320100000000e+0
 cxlog(z) = 	3.20099999999942544908209262789253e-21  	1.57079632679489661923132169163975e+0
*string in : 	-6.5541e-20   +1.0000000000000000080981i
 z =  		-6.55409999999999999999999999999984e-20  	1.00000000000000000809810000000000e+0
 cxlog(z) = 	8.09810000000000007345442644509469e-18  	1.57079632679489661929686269163975e+0
*string in : 	-7.0998e-2200 +1.0i
 z =  		-7.09979999999999999999999999998153e-2200  	1.00000000000000000000000000000000e+0
 cxlog(z) = 	0.00000000000000000000000000000000e+0  	1.57079632679489661923132169163975e+0
*string in : 	-6.8432e-17   +0.9999999999999999999864i
 z =  		-6.84319999999999999999999999999986e-17  	9.99999999999999999986399999999974e-1
 cxlog(z) = 	-1.36000000000250070933004954621954e-20  	1.57079632679489668766332169163975e+0
*string in : 	+0.99999999999999989098732i
 z =  		0.00000000000000000000000000000000e+0  	9.99999999999999890987319999999974e-1
 cxlog(z) = 	-1.09012680000000033872455856279863e-16  	1.57079632679489661923132169163975e+0
*string in : 	+4.3329e-16   +0.9999999999999999997877i
 z =  		4.33289999999999999999999999999992e-16  	9.99999999999999999787699999999974e-1
 cxlog(z) = 	-2.12299999999932706941351006147258e-19  	1.57079632679489618594132169163975e+0
*string in : 	-1.0i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
 cxlog(z) = 	0.00000000000000000000000000000000e+0  	-1.57079632679489661923132169163975e+0
*string in : 	+2.7762e-2111 -1.0i
 z =  		2.77619999999999999999999999999307e-2111  	-1.00000000000000000000000000000000e+0
 cxlog(z) = 	0.00000000000000000000000000000000e+0  	-1.57079632679489661923132169163975e+0
*string in : 	+3.3321e-17   -0.9999999999999999996555i
 z =  		3.33209999999999999999999999999993e-17  	-9.99999999999999999655499999999974e-1
 cxlog(z) = 	-3.44500000000026747694715437469608e-19  	-1.57079632679489658591032169163975e+0
*string in : 	-0.999999999999999999996999972013i
 z =  		0.00000000000000000000000000000000e+0  	-9.99999999999999999996999972012965e-1
 cxlog(z) = 	-3.00002798703627244460764839879046e-21  	-1.57079632679489661923132169163975e+0
*string in : 	-7.5541e-17   -0.999999999999999999987231i
 z =  		-7.55409999999999999999999999999985e-17  	-9.99999999999999999987230999999972e-1
 cxlog(z) = 	-1.27690000000268255099985551227302e-20  	-1.57079632679489669477232169163975e+0
*string in : 	-2.1156e-2203 -1.0i
 z =  		-2.11559999999999999999999999999449e-2203  	-1.00000000000000000000000000000000e+0
 cxlog(z) = 	0.00000000000000000000000000000000e+0  	-1.57079632679489661923132169163975e+0
*string in : 	-5.9923e-16   -1.00000000000000000089321i
 z =  		-5.99229999999999999999999999999988e-16  	-1.00000000000000000089321000000000e+0
 cxlog(z) = 	8.93210000000179422314566632539467e-19  	-1.57079632679489721846132169163975e+0
*string in : 	-1.000000000000000000007161i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000716100000000e+0
 cxlog(z) = 	7.16099999999963394852823947703790e-21  	-1.57079632679489661923132169163975e+0
*string in : 	+8.6631e-19   -1.00000000000000000008765i
 z =  		8.66309999999999999999999999999981e-19  	-1.00000000000000000008765000000000e+0
 cxlog(z) = 	8.76500000000003579069827146091337e-20  	-1.57079632679489661836501169163975e+0
*string in : 	+1.5876e1876
 z =  		1.58759999999999999999999999999648e+1876  	0.00000000000000000000000000000000e+0
 cxlog(z) = 	4.32011185789875647647630615384661e+3  	0.00000000000000000000000000000000e+0
*string in : 	-6.4391e1965
 z =  		-6.43909999999999999999999999998503e+1965  	0.00000000000000000000000000000000e+0
 cxlog(z) = 	4.52644209651209707815558800308919e+3  	3.14159265358979323846264338327950e+0
*string in : 	+3.4210e2015i
 z =  		0.00000000000000000000000000000000e+0  	3.42099999999999999999999999999185e+2015
 cxlog(z) = 	4.64093889528899752092245674596832e+3  	1.57079632679489661923132169163975e+0
*string in : 	-4.5522e2011i
 z =  		0.00000000000000000000000000000000e+0  	-4.55219999999999999999999999998917e+2011
 cxlog(z) = 	4.63201423264361581766421370094534e+3  	-1.57079632679489661923132169163975e+0
*string in : 	+2.1907e1854  +9.8765e1872i
 z =  		2.19069999999999999999999999999520e+1854  	9.87649999999999999999999999997812e+1872
 cxlog(z) = 	4.31272945228283943867197319844563e+3  	1.57079632679489661900951234622387e+0
*string in : 	-3.2966e1946  +7.5721e1834i
 z =  		-3.29659999999999999999999999999241e+1946  	7.57209999999999999999999999998357e+1834
 cxlog(z) = 	4.48202348260072801964069710182294e+3  	3.14159265358979323846264338327950e+0
*string in : 	-6.0987e2310  -5.4411e2198i
 z =  		-6.09869999999999999999999999998333e+2310  	-5.44109999999999999999999999998585e+2198
 cxlog(z) = 	5.32077964044995852115456716019544e+3  	-3.14159265358979323846264338327950e+0
*string in : 	+7.8895e1854  -3.8699e1943i
 z =  		7.88949999999999999999999999998271e+1854  	-3.86989999999999999999999999999111e+1943
 cxlog(z) = 	4.47527606435434531962187434558018e+3  	-1.57079632679489661923132169163975e+0
*string in : 	+10.009876543100091234
 z =  		1.00098765431000912340000000000000e+1  	0.00000000000000000000000000000000e+0
 cxlog(z) = 	2.30357225989443850653844561322621e+0  	0.00000000000000000000000000000000e+0
*string in : 	+9.8980000000006542009 +6.11002230065009876i
 z =  		9.89800000000065420089999999999997e+0  	6.11002230065009876000000000000000e+0
 cxlog(z) = 	2.45375752806866017249988303001174e+0  	5.53042107025953173852302384116297e-1
*string in : 	+17.9998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+1
 cxlog(z) = 	2.89036558078486415768110085403983e+0  	1.57079632679489661923132169163975e+0
*string in : 	-12.9822227      +20.87777220192i
 z =  		-1.29822227000000000000000000000000e+1  	2.08777722019200000000000000000000e+1
 cxlog(z) = 	3.20213422872824289909037930574844e+0  	2.12710585235075047160057027232326e+0
*string in : 	-17.9877771265
 z =  		-1.79877771265000000000000000000000e+1  	0.00000000000000000000000000000000e+0
 cxlog(z) = 	2.88969247871051087083248405867506e+0  	3.14159265358979323846264338327950e+0
*string in : 	-19.09872210981  -32.9800000654i
 z =  		-1.90987221098100000000000000000000e+1  	-3.29800000653999999999999999999999e+1
 cxlog(z) = 	3.64050063743259822513944216508604e+0  	-2.09570649869183772096177981258929e+0
*string in : 	-43.987600182345i
 z =  		0.00000000000000000000000000000000e+0  	-4.39876001823449999999999999999999e+1
 cxlog(z) = 	3.78390778016360957399957175968881e+0  	-1.57079632679489661923132169163975e+0
*string in : 	+15.982201192361 -43.2189031229i
 z =  		1.59822011923610000000000000000000e+1  	-4.32189031228999999999999999999999e+1
 cxlog(z) = 	3.83036440308134189291625601605708e+0  	-1.21659537262678903716220661644215e+0
*string in : 	+1000.009876543100091234
 z =  		1.00000987654310009123400000000000e+3  	0.00000000000000000000000000000000e+0
 cxlog(z) = 	6.90776515547646441262104786739289e+0  	0.00000000000000000000000000000000e+0
*string in : 	+989.80000000006542009 +611.002230065009876i
 z =  		9.89800000000065420089999999999996e+2  	6.11002230065009875999999999999998e+2
 cxlog(z) = 	7.05892771405675154053586593938047e+0  	5.53042107025953173852302384116296e-1
*string in : 	+179.998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+2
 cxlog(z) = 	5.19295067377890984169909230872420e+0  	1.57079632679489661923132169163975e+0
*string in : 	-1298.22227  +208.7777220192i
 z =  		-1.29822227000000000000000000000000e+3  	2.08777722019199999999999999999999e+2
 cxlog(z) = 	7.18151797523645458970487076578846e+0  	2.98213975150210743249550554091572e+0
*string in : 	-1798.777712650981
 z =  		-1.79877771265098099999999999999999e+3  	0.00000000000000000000000000000000e+0
 cxlog(z) = 	7.49486266469914760920138848346195e+0  	3.14159265358979323846264338327950e+0
*string in : 	-190.9872210981   -329.800700654i
 z =  		-1.90987221098100000000000000000000e+2  	-3.29800700653999999999999999999999e+2
 cxlog(z) = 	5.94308731988731494818898085301411e+0  	-2.09570557823594370914996234699252e+0
*string in : 	+1598.2201192361  -4321.89031229i
 z =  		1.59822011923609999999999999999999e+3  	-4.32189031228999999999999999999998e+3
 cxlog(z) = 	8.43553458906943326095223892542580e+0  	-1.21659537262678903716220661644215e+0

Test of cxlog10
*string in : 	0
 z =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxlog10(z) = 	*** Out of domain
*string in : 	+2.3106e-1571
 z =  		2.31059999999999999999999999999571e-1571  	0.00000000000000000000000000000000e+0
 cxlog10(z) = 	-1.57063627523099362526463725094820e+3  	0.00000000000000000000000000000000e-3362
*string in : 	+8.7792e-20  +9.7732e-20i
 z =  		8.77919999999999999999999999999979e-20  	9.77319999999999999999999999999977e-20
 cxlog10(z) = 	-1.88814924487357019990541555936582e+1  	3.64340480117170214040550437507364e-1
*string in : 	+2.3298e-1571i
 z =  		0.00000000000000000000000000000000e+0  	2.32979999999999999999999999999567e-1571
 cxlog10(z) = 	-1.57063268135907030876124396391480e+3  	6.82188176920920673742891812715677e-1
*string in : 	-4.5592e-19  +8.2219e-18i
 z =  		-4.55919999999999999999999999999990e-19  	8.22189999999999999999999999999983e-18
 cxlog10(z) = 	-1.70843611251755800378665829858125e+1  	7.06245993955399636207018378079062e-1
*string in : 	-3.8932e-1624i
 z =  		0.00000000000000000000000000000000e+0  	-3.89319999999999999999999999999252e-1624
 cxlog10(z) = 	-1.62340969328529665923440947892991e+3  	-6.82188176920920673742891812715677e-1
*string in : 	-1.2291e-17  -7.3342e-18i
 z =  		-1.22909999999999999999999999999998e-17  	-7.33419999999999999999999999999984e-18
 cxlog10(z) = 	-1.68442722949152169923957011681494e+1  	-1.13072630723961497193840082596939e+0
*string in : 	-4.0987e-1617i
 z =  		0.00000000000000000000000000000000e+0  	-4.09869999999999999999999999999216e-1617
 cxlog10(z) = 	-1.61638735386824427240285451954781e+3  	-6.82188176920920673742891812715677e-1
*string in : 	+3.9917e-19  -8.2215e-19i
 z =  		3.99169999999999999999999999999991e-19  	-8.22149999999999999999999999999982e-19
 cxlog10(z) = 	-1.80390872572956288134096654306675e+1  	-4.85888640197695135842145731049475e-1
*string in : 	+1.0
 z =  		1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxlog10(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000000986
 z =  		1.00000000000000000009860000000000e+0  	0.00000000000000000000000000000000e+0
 cxlog10(z) = 	4.28214359156605880363166200217057e-20  	0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000004567 +2.43e-18i
 z =  		1.00000000000000000045670000000000e+0  	2.42999999999999999999999999999995e-18
 cxlog10(z) = 	1.98342289885215211903078242432515e-19  	1.05533559102490194071027150854625e-18
*string in : 	+1.0  +3.4506e-1298i
 z =  		1.00000000000000000000000000000000e+0  	3.45059999999999999999999999999470e-1298
 cxlog10(z) = 	0.00000000000000000000000000000000e+0  	1.49857653925536075649298544761134e-1298
*string in : 	+0.999999999999999998736  +4.6677e-1297i
 z =  		9.99999999999999998735999999999976e-1  	4.66769999999999999999999999999284e-1297
 cxlog10(z) = 	-5.48948225125721298840833456526696e-19  	2.02715635317980855848950008524325e-1297
*string in : 	+0.999999999999999999754
 z =  		9.99999999999999999753999999999976e-1  	0.00000000000000000000000000000000e+0
 cxlog10(z) = 	-1.06836442548210856811559044539590e-19  	0.00000000000000000000000000000000e+0
*string in : 	+0.999999999999999999338  -5.0783e-1145i
 z =  		9.99999999999999999337999999999976e-1  	-5.07829999999999999999999999999312e-1145
 cxlog10(z) = 	-2.87502947019963801673285712489911e-19  	-2.20547766744928375782075420478269e-1145
*string in : 	+1.0000000000000000000    -6.7002e-1301i
 z =  		1.00000000000000000000000000000000e+0  	-6.70019999999999999999999999998969e-1301
 cxlog10(z) = 	0.00000000000000000000000000000000e+0  	-2.90985988764816789562809398252056e-1301
*string in : 	+1.0000000000000007423    -5.3321e-1344i
 z =  		1.00000000000000074230000000000000e+0  	-5.33209999999999999999999999999153e-1344
 cxlog10(z) = 	3.22376793916783712169498108894532e-16  	-2.31570160695632735127328166486976e-1344
*string in : 	-1.0
 z =  		-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxlog10(z) = 	0.00000000000000000000000000000000e+0  	1.36437635384184134748578362543135e+0
*string in : 	-0.99999999999999999998811
 z =  		-9.99999999999999999988109999999973e-1  	0.00000000000000000000000000000000e+0
 cxlog10(z) = 	-5.16376138984190938842255974047479e-21  	1.36437635384184134748578362543135e+0
*string in : 	-0.99999999999999999976632 +2.3344e-15i
 z =  		-9.99999999999999999766319999999974e-1  	2.33439999999999999999999999999996e-15
 cxlog10(z) = 	-1.01485934529980729062536955225646e-19  	1.36437635384184033366874507048029e+0
*string in : 	-1.0  +4.5601e-997i
 z =  		-1.00000000000000000000000000000000e+0  	4.56009999999999999999999999999462e-997
 cxlog10(z) = 	0.00000000000000000000000000000000e+0  	1.36437635384184134748578362543135e+0
*string in : 	-1.00000000000000000000654 +7.4332e-1532i
 z =  		-1.00000000000000000000654000000000e+0  	7.43319999999999999999999999998655e-1532
 cxlog10(z) = 	2.84028591164716050235698238791343e-21  	1.36437635384184134748578362543135e+0
*string in : 	-1.00000000000000000000743
 z =  		-1.00000000000000000000743000000000e+0  	0.00000000000000000000000000000000e+0
 cxlog10(z) = 	3.22680800054113855631154247011532e-21  	1.36437635384184134748578362543135e+0
*string in : 	-1.00000000000000000085023 -6.9983e-18i
 z =  		-1.00000000000000000085023000000000e+0  	-6.99829999999999999999999999999985e-18
 cxlog10(z) = 	3.69250197348601927338139966986235e-19  	-1.36437635384184134444646055272783e+0
*string in : 	-1.0  -5.00098e-1023i
 z =  		-1.00000000000000000000000000000000e+0  	-5.00097999999999999999999999999395e-1023
 cxlog10(z) = 	0.00000000000000000000000000000000e+0  	-1.36437635384184134748578362543135e+0
*string in : 	-0.9999999999999999997643  -6.7321e-16i
 z =  		-9.99999999999999999764299999999974e-1  	-6.73209999999999999999999999999987e-16
 cxlog10(z) = 	-1.02363209384509813290394511059613e-19  	-1.36437635384184105511439546334319e+0
*string in : 	+1.0i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
 cxlog10(z) = 	0.00000000000000000000000000000000e+0  	6.82188176920920673742891812715677e-1
*string in : 	+5.6098e-2198 +1.0i
 z =  		5.60979999999999999999999999998542e-2198  	1.00000000000000000000000000000000e+0
 cxlog10(z) = 	0.00000000000000000000000000000000e+0  	6.82188176920920673742891812715677e-1
*string in : 	+4.9987e-18   +1.0000000000000000005421i
 z =  		4.99869999999999999999999999999989e-18  	1.00000000000000000054210000000000e+0
 cxlog10(z) = 	2.35431038639752852787452615726237e-19  	6.82188176920920671571983986025892e-1
*string in : 	+1.000000000000000000003201i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000320100000000e+0
 cxlog10(z) = 	1.39017663657205957601804404950989e-21  	6.82188176920920673742891812715677e-1
*string in : 	-6.5541e-20   +1.0000000000000000080981i
 z =  		-6.55409999999999999999999999999984e-20  	1.00000000000000000809810000000000e+0
 cxlog10(z) = 	3.51696014390072365740245917475148e-18  	6.82188176920920673771355907354098e-1
*string in : 	-7.0998e-2200 +1.0i
 z =  		-7.09979999999999999999999999998153e-2200  	1.00000000000000000000000000000000e+0
 cxlog10(z) = 	0.00000000000000000000000000000000e+0  	6.82188176920920673742891812715677e-1
*string in : 	-6.8432e-17   +0.9999999999999999999864i
 z =  		-6.84319999999999999999999999999986e-17  	9.99999999999999999986399999999974e-1
 cxlog10(z) = 	-5.90640495389508529868419825379074e-21  	6.82188176920920703462531798319006e-1
*string in : 	+0.99999999999999989098732i
 z =  		0.00000000000000000000000000000000e+0  	9.99999999999999890987319999999974e-1
 cxlog10(z) = 	-4.73436053814849971577683353704329e-17  	6.82188176920920673742891812715677e-1
*string in : 	+4.3329e-16   +0.9999999999999999997877i
 z =  		4.33289999999999999999999999999992e-16  	9.99999999999999999787699999999974e-1
 cxlog10(z) = 	-9.22007185080311380062930195613336e-20  	6.82188176920920485567435748855693e-1
*string in : 	-1.0i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
 cxlog10(z) = 	0.00000000000000000000000000000000e+0  	-6.82188176920920673742891812715677e-1
*string in : 	+2.7762e-2111 -1.0i
 z =  		2.77619999999999999999999999999307e-2111  	-1.00000000000000000000000000000000e+0
 cxlog10(z) = 	0.00000000000000000000000000000000e+0  	-6.82188176920920673742891812715677e-1
*string in : 	+3.3321e-17   -0.9999999999999999996555i
 z =  		3.33209999999999999999999999999993e-17  	-9.99999999999999999655499999999974e-1
 cxlog10(z) = 	-1.49614449015681871002032459829459e-19  	-6.82188176920920659271765381217423e-1
*string in : 	-0.999999999999999999996999972013i
 z =  		0.00000000000000000000000000000000e+0  	-9.99999999999999999996999972012965e-1
 cxlog10(z) = 	-1.30289560032517343180053029076328e-21  	-6.82188176920920673742891812715677e-1
*string in : 	-7.5541e-17   -0.999999999999999999987231i
 z =  		-7.55409999999999999999999999999985e-17  	-9.99999999999999999987230999999972e-1
 cxlog10(z) = 	-5.54550623943427275824387816435601e-21  	-6.82188176920920706549931270169224e-1
*string in : 	-2.1156e-2203 -1.0i
 z =  		-2.11559999999999999999999999999449e-2203  	-1.00000000000000000000000000000000e+0
 cxlog10(z) = 	0.00000000000000000000000000000000e+0  	-6.82188176920920673742891812715677e-1
*string in : 	-5.9923e-16   -1.00000000000000000089321i
 z =  		-5.99229999999999999999999999999988e-16  	-1.00000000000000000089321000000000e+0
 cxlog10(z) = 	3.87916174180881487097411459617699e-19  	-6.82188176920920933985174203601270e-1
*string in : 	-1.000000000000000000007161i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000716100000000e+0
 cxlog10(z) = 	3.10998278490902736367545602922481e-21  	-6.82188176920920673742891812715677e-1
*string in : 	+8.6631e-19   -1.00000000000000000008765i
 z =  		8.66309999999999999999999999999981e-19  	-1.00000000000000000008765000000000e+0
 cxlog10(z) = 	3.80659113388201781306490773403215e-20  	-6.82188176920920673366658160098071e-1
*string in : 	+1.5876e1876
 z =  		1.58759999999999999999999999999648e+1876  	0.00000000000000000000000000000000e+0
 cxlog10(z) = 	1.87620074109023512580103202191965e+3  	0.00000000000000000000000000000000e+0
*string in : 	-6.4391e1965
 z =  		-6.43909999999999999999999999998503e+1965  	0.00000000000000000000000000000000e+0
 cxlog10(z) = 	1.96580882516979020721075689133737e+3  	1.36437635384184134748578362543135e+0
*string in : 	+3.4210e2015i
 z =  		0.00000000000000000000000000000000e+0  	3.42099999999999999999999999999185e+2015
 cxlog10(z) = 	2.01553415307418506254630322315449e+3  	6.82188176920920673742891812715677e-1
*string in : 	-4.5522e2011i
 z =  		0.00000000000000000000000000000000e+0  	-4.55219999999999999999999999998917e+2011
 cxlog10(z) = 	2.01165822133444771083725776842401e+3  	-6.82188176920920673742891812715677e-1
*string in : 	+2.1907e1854  +9.8765e1872i
 z =  		2.19069999999999999999999999999520e+1854  	9.87649999999999999999999999997812e+1872
 cxlog10(z) = 	1.87299460306807077916370283903770e+3  	6.82188176920920673646561237966986e-1
*string in : 	-3.2966e1946  +7.5721e1834i
 z =  		-3.29659999999999999999999999999241e+1946  	7.57209999999999999999999999998357e+1834
 cxlog10(z) = 	1.94651806625429160774639975436960e+3  	1.36437635384184134748578362543135e+0
*string in : 	-6.0987e2310  -5.4411e2198i
 z =  		-6.09869999999999999999999999998333e+2310  	-5.44109999999999999999999999998585e+2198
 cxlog10(z) = 	2.31078523727058527718169189700347e+3  	-1.36437635384184134748578362543135e+0
*string in : 	+7.8895e1854  -3.8699e1943i
 z =  		7.88949999999999999999999999998271e+1854  	-3.86989999999999999999999999999111e+1943
 cxlog10(z) = 	1.94358769974279428505137468613472e+3  	-6.82188176920920673742891812715677e-1
*string in : 	+10.009876543100091234
 z =  		1.00098765431000912340000000000000e+1  	0.00000000000000000000000000000000e+0
 cxlog10(z) = 	1.00042872113755813985962226754415e+0  	0.00000000000000000000000000000000e+0
*string in : 	+9.8980000000006542009 +6.11002230065009876i
 z =  		9.89800000000065420089999999999997e+0  	6.11002230065009876000000000000000e+0
 cxlog10(z) = 	1.06565335436878267382194074186714e+0  	2.40183135341519081107711884526561e-1
*string in : 	+17.9998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+1
 cxlog10(z) = 	1.25526982241795414543946111917637e+0  	6.82188176920920673742891812715677e-1
*string in : 	-12.9822227      +20.87777220192i
 z =  		-1.29822227000000000000000000000000e+1  	2.08777722019200000000000000000000e+1
 cxlog10(z) = 	1.39066922585020113438541513636400e+0  	9.23790334100044054767502648079311e-1
*string in : 	-17.9877771265
 z =  		-1.79877771265000000000000000000000e+1  	0.00000000000000000000000000000000e+0
 cxlog10(z) = 	1.25497749790130488063804159005008e+0  	1.36437635384184134748578362543135e+0
*string in : 	-19.09872210981  -32.9800000654i
 z =  		-1.90987221098100000000000000000000e+1  	-3.29800000653999999999999999999999e+1
 cxlog10(z) = 	1.58104933820224827299454936829642e+0  	-9.10153768070649567125069622850705e-1
*string in : 	-43.987600182345i
 z =  		0.00000000000000000000000000000000e+0  	-4.39876001823449999999999999999999e+1
 cxlog10(z) = 	1.64333026895583853312288160040418e+0  	-6.82188176920920673742891812715677e-1
*string in : 	+15.982201192361 -43.2189031229i
 z =  		1.59822011923610000000000000000000e+1  	-4.32189031228999999999999999999999e+1
 cxlog10(z) = 	1.66350612393686982582067051017989e+0  	-5.28360657040844945427903652000267e-1
*string in : 	+1000.009876543100091234
 z =  		1.00000987654310009123400000000000e+3  	0.00000000000000000000000000000000e+0
 cxlog10(z) = 	3.00000428930698692146312902726118e+0  	0.00000000000000000000000000000000e+0
*string in : 	+989.80000000006542009 +611.002230065009876i
 z =  		9.89800000000065420089999999999996e+2  	6.11002230065009875999999999999998e+2
 cxlog10(z) = 	3.06565335436878267382194074186714e+0  	2.40183135341519081107711884526560e-1
*string in : 	+179.998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+2
 cxlog10(z) = 	2.25526982241795414543946111917637e+0  	6.82188176920920673742891812715677e-1
*string in : 	-1298.22227  +208.7777220192i
 z =  		-1.29822227000000000000000000000000e+3  	2.08777722019199999999999999999999e+2
 cxlog10(z) = 	3.11889362833420613475770719294951e+0  	1.29512683834169989865724007893026e+0
*string in : 	-1798.777712650981
 z =  		-1.79877771265098099999999999999999e+3  	0.00000000000000000000000000000000e+0
 cxlog10(z) = 	3.25497749790154173196422314355230e+0  	1.36437635384184134748578362543135e+0
*string in : 	-190.9872210981   -329.800700654i
 z =  		-1.90987221098100000000000000000000e+2  	-3.29800700653999999999999999999999e+2
 cxlog10(z) = 	2.58105002849624690748574201218002e+0  	-9.10153368321733962470779948539788e-1
*string in : 	+1598.2201192361  -4321.89031229i
 z =  		1.59822011923609999999999999999999e+3  	-4.32189031228999999999999999999998e+3
 cxlog10(z) = 	3.66350612393686982582067051017988e+0  	-5.28360657040844945427903652000267e-1

Test of cxlog2
*string in : 	0
 z =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxlog2(z) = 	*** Out of domain
*string in : 	+2.3106e-1571
 z =  		2.31059999999999999999999999999571e-1571  	0.00000000000000000000000000000000e+0
 cxlog2(z) = 	-5.21754076953907755614077440609499e+3  	0.00000000000000000000000000000000e-3362
*string in : 	+8.7792e-20  +9.7732e-20i
 z =  		8.77919999999999999999999999999979e-20  	9.77319999999999999999999999999977e-20
 cxlog2(z) = 	-6.27229602388587087220757803384856e+1  	1.21031287700597816966226934850468e+0
*string in : 	+2.3298e-1571i
 z =  		0.00000000000000000000000000000000e+0  	2.32979999999999999999999999999567e-1571
 cxlog2(z) = 	-5.21752883095496606416787058021472e+3  	2.26618007091359690481384147285833e+0
*string in : 	-4.5592e-19  +8.2219e-18i
 z =  		-4.55919999999999999999999999999990e-19  	8.22189999999999999999999999999983e-18
 cxlog2(z) = 	-5.67530192049222288104348007614943e+1  	2.34609840922209233788374479516118e+0
*string in : 	-3.8932e-1624i
 z =  		0.00000000000000000000000000000000e+0  	-3.89319999999999999999999999999252e-1624
 cxlog2(z) = 	-5.39285026963690276642420536177429e+3  	-2.26618007091359690481384147285833e+0
*string in : 	-1.2291e-17  -7.3342e-18i
 z =  		-1.22909999999999999999999999999998e-17  	-7.33419999999999999999999999999984e-18
 cxlog2(z) = 	-5.59554613744116856869284046455652e+1  	-3.75619148764751651581607624872155e+0
*string in : 	-4.0987e-1617i
 z =  		0.00000000000000000000000000000000e+0  	-4.09869999999999999999999999999216e-1617
 cxlog2(z) = 	-5.36952256303556150034604940026194e+3  	-2.26618007091359690481384147285833e+0
*string in : 	+3.9917e-19  -8.2215e-19i
 z =  		3.99169999999999999999999999999991e-19  	-8.22149999999999999999999999999982e-19
 cxlog2(z) = 	-5.99245507661349626407328617122619e+1  	-1.61408712485934047037051387707902e+0
*string in : 	+1.0
 z =  		1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxlog2(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000000986
 z =  		1.00000000000000000009860000000000e+0  	0.00000000000000000000000000000000e+0
 cxlog2(z) = 	1.42249731031652026104791058888287e-19  	0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000004567 +2.43e-18i
 z =  		1.00000000000000000045670000000000e+0  	2.42999999999999999999999999999995e-18
 cxlog2(z) = 	6.58878825173990543135319932118730e-19  	3.50574894936018107828354142966173e-18
*string in : 	+1.0  +3.4506e-1298i
 z =  		1.00000000000000000000000000000000e+0  	3.45059999999999999999999999999470e-1298
 cxlog2(z) = 	0.00000000000000000000000000000000e+0  	4.97816350809145713343615610425748e-1298
*string in : 	+0.999999999999999998736  +4.6677e-1297i
 z =  		9.99999999999999998735999999999976e-1  	4.66769999999999999999999999999284e-1297
 cxlog2(z) = 	-1.82356653168368676435025811276154e-18  	6.73406764235741450504578193344213e-1297
*string in : 	+0.999999999999999999754
 z =  		9.99999999999999999753999999999976e-1  	0.00000000000000000000000000000000e+0
 cxlog2(z) = 	-3.54902980058721842983036115123287e-19  	0.00000000000000000000000000000000e+0
*string in : 	+0.999999999999999999338  -5.0783e-1145i
 z =  		9.99999999999999999337999999999976e-1  	-5.07829999999999999999999999999312e-1145
 cxlog2(z) = 	-9.55064117068531123490255798757499e-19  	-7.32643822614642287644600761323109e-1145
*string in : 	+1.0000000000000000000    -6.7002e-1301i
 z =  		1.00000000000000000000000000000000e+0  	-6.70019999999999999999999999998969e-1301
 cxlog2(z) = 	0.00000000000000000000000000000000e+0  	-9.66634531296423262199296734763400e-1301
*string in : 	+1.0000000000000007423    -5.3321e-1344i
 z =  		1.00000000000000074230000000000000e+0  	-5.33209999999999999999999999999153e-1344
 cxlog2(z) = 	1.07091252885187714084866636489619e-15  	-7.69259422752403607417115930046598e-1344
*string in : 	-1.0
 z =  		-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxlog2(z) = 	0.00000000000000000000000000000000e+0  	4.53236014182719380962768294571666e+0
*string in : 	-0.99999999999999999998811
 z =  		-9.99999999999999999988109999999973e-1  	0.00000000000000000000000000000000e+0
 cxlog2(z) = 	-1.71536440362109891597914075517169e-20  	4.53236014182719380962768294571666e+0
*string in : 	-0.99999999999999999976632 +2.3344e-15i
 z =  		-9.99999999999999999766319999999974e-1  	2.33439999999999999999999999999996e-15
 cxlog2(z) = 	-3.37128977151042951527906614860166e-19  	4.53236014182719044180037949452049e+0
*string in : 	-1.0  +4.5601e-997i
 z =  		-1.00000000000000000000000000000000e+0  	4.56009999999999999999999999999462e-997
 cxlog2(z) = 	0.00000000000000000000000000000000e+0  	4.53236014182719380962768294571666e+0
*string in : 	-1.00000000000000000000654 +7.4332e-1532i
 z =  		-1.00000000000000000000654000000000e+0  	7.43319999999999999999999999998655e-1532
 cxlog2(z) = 	9.43522556741396173175614764205868e-21  	4.53236014182719380962768294571666e+0
*string in : 	-1.00000000000000000000743
 z =  		-1.00000000000000000000743000000000e+0  	0.00000000000000000000000000000000e+0
 cxlog2(z) = 	1.07192241538053182640532078363248e-20  	4.53236014182719380962768294571666e+0
*string in : 	-1.00000000000000000085023 -6.9983e-18i
 z =  		-1.00000000000000000085023000000000e+0  	-6.99829999999999999999999999999985e-18
 cxlog2(z) = 	1.22662260461502417843302864105990e-18  	-4.53236014182719379953127024106343e+0
*string in : 	-1.0  -5.00098e-1023i
 z =  		-1.00000000000000000000000000000000e+0  	-5.00097999999999999999999999999395e-1023
 cxlog2(z) = 	0.00000000000000000000000000000000e+0  	-4.53236014182719380962768294571666e+0
*string in : 	-0.9999999999999999997643  -6.7321e-16i
 z =  		-9.99999999999999999764299999999974e-1  	-6.73209999999999999999999999999987e-16
 cxlog2(z) = 	-3.40043221137241358098269592696566e-19  	-4.53236014182719283839095446885761e+0
*string in : 	+1.0i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
 cxlog2(z) = 	0.00000000000000000000000000000000e+0  	2.26618007091359690481384147285833e+0
*string in : 	+5.6098e-2198 +1.0i
 z =  		5.60979999999999999999999999998542e-2198  	1.00000000000000000000000000000000e+0
 cxlog2(z) = 	0.00000000000000000000000000000000e+0  	2.26618007091359690481384147285833e+0
*string in : 	+4.9987e-18   +1.0000000000000000005421i
 z =  		4.99869999999999999999999999999989e-18  	1.00000000000000000054210000000000e+0
 cxlog2(z) = 	7.82084981665907736114710771245158e-19  	2.26618007091359689760224177196667e+0
*string in : 	+1.000000000000000000003201i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000320100000000e+0
 cxlog2(z) = 	4.61806682588527188387432980993614e-21  	2.26618007091359690481384147285833e+0
*string in : 	-6.5541e-20   +1.0000000000000000080981i
 z =  		-6.55409999999999999999999999999984e-20  	1.00000000000000000809810000000000e+0
 cxlog2(z) = 	1.16830887106229151143395839005810e-17  	2.26618007091359690490839714853324e+0
*string in : 	-7.0998e-2200 +1.0i
 z =  		-7.09979999999999999999999999998153e-2200  	1.00000000000000000000000000000000e+0
 cxlog2(z) = 	0.00000000000000000000000000000000e+0  	2.26618007091359690481384147285833e+0
*string in : 	-6.8432e-17   +0.9999999999999999999864i
 z =  		-6.84319999999999999999999999999986e-17  	9.99999999999999999986399999999974e-1
 cxlog2(z) = 	-1.96206525561265073806737340116353e-20  	2.26618007091359700354034851097187e+0
*string in : 	+0.99999999999999989098732i
 z =  		0.00000000000000000000000000000000e+0  	9.99999999999999890987319999999974e-1
 cxlog2(z) = 	-1.57272052830015532973750450109812e-16  	2.26618007091359690481384147285833e+0
*string in : 	+4.3329e-16   +0.9999999999999999997877i
 z =  		4.33289999999999999999999999999992e-16  	9.99999999999999999787699999999974e-1
 cxlog2(z) = 	-3.06284157180630478227295072099785e-19  	2.26618007091359627970850720607938e+0
*string in : 	-1.0i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
 cxlog2(z) = 	0.00000000000000000000000000000000e+0  	-2.26618007091359690481384147285833e+0
*string in : 	+2.7762e-2111 -1.0i
 z =  		2.77619999999999999999999999999307e-2111  	-1.00000000000000000000000000000000e+0
 cxlog2(z) = 	0.00000000000000000000000000000000e+0  	-2.26618007091359690481384147285833e+0
*string in : 	+3.3321e-17   -0.9999999999999999996555i
 z =  		3.33209999999999999999999999999993e-17  	-9.99999999999999999655499999999974e-1
 cxlog2(z) = 	-4.97008441586286928324691802693492e-19  	-2.26618007091359685674180001539718e+0
*string in : 	-0.999999999999999999996999972013i
 z =  		0.00000000000000000000000000000000e+0  	-9.99999999999999999996999972012965e-1
 cxlog2(z) = 	-4.32812549942578782023093176700363e-21  	-2.26618007091359690481384147285833e+0
*string in : 	-7.5541e-17   -0.999999999999999999987231i
 z =  		-7.55409999999999999999999999999985e-17  	-9.99999999999999999987230999999972e-1
 cxlog2(z) = 	-1.84217729771502457584068189113755e-20  	-2.26618007091359701379646755665152e+0
*string in : 	-2.1156e-2203 -1.0i
 z =  		-2.11559999999999999999999999999449e-2203  	-1.00000000000000000000000000000000e+0
 cxlog2(z) = 	0.00000000000000000000000000000000e+0  	-2.26618007091359690481384147285833e+0
*string in : 	-5.9923e-16   -1.00000000000000000089321i
 z =  		-5.99229999999999999999999999999988e-16  	-1.00000000000000000089321000000000e+0
 cxlog2(z) = 	1.28862963747269036961067525358335e-18  	-2.26618007091359776931999082475187e+0
*string in : 	-1.000000000000000000007161i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000716100000000e+0
 cxlog2(z) = 	1.03311391878057795993927273555693e-20  	-2.26618007091359690481384147285833e+0
*string in : 	+8.6631e-19   -1.00000000000000000008765i
 z =  		8.66309999999999999999999999999981e-19  	-1.00000000000000000008765000000000e+0
 cxlog2(z) = 	1.26452220333918715800127013187607e-19  	-2.26618007091359690356402033198581e+0
*string in : 	+1.5876e1876
 z =  		1.58759999999999999999999999999648e+1876  	0.00000000000000000000000000000000e+0
 cxlog2(z) = 	6.23260395347614214815393686641437e+3  	0.00000000000000000000000000000000e+0
*string in : 	-6.4391e1965
 z =  		-6.43909999999999999999999999998503e+1965  	0.00000000000000000000000000000000e+0
 cxlog2(z) = 	6.53027556550904514398635402331287e+3  	4.53236014182719380962768294571666e+0
*string in : 	+3.4210e2015i
 z =  		0.00000000000000000000000000000000e+0  	3.42099999999999999999999999999185e+2015
 cxlog2(z) = 	6.69545952930214094371243650927299e+3  	2.26618007091359690481384147285833e+0
*string in : 	-4.5522e2011i
 z =  		0.00000000000000000000000000000000e+0  	-4.55219999999999999999999999998917e+2011
 cxlog2(z) = 	6.68258396276204178300024107100424e+3  	-2.26618007091359690481384147285833e+0
*string in : 	+2.1907e1854  +9.8765e1872i
 z =  		2.19069999999999999999999999999520e+1854  	9.87649999999999999999999999997812e+1872
 cxlog2(z) = 	6.22195339350422780421129456780662e+3  	2.26618007091359690449383823020401e+0
*string in : 	-3.2966e1946  +7.5721e1834i
 z =  		-3.29659999999999999999999999999241e+1946  	7.57209999999999999999999999998357e+1834
 cxlog2(z) = 	6.46619305149595148128459556418054e+3  	4.53236014182719380962768294571666e+0
*string in : 	-6.0987e2310  -5.4411e2198i
 z =  		-6.09869999999999999999999999998333e+2310  	-5.44109999999999999999999999998585e+2198
 cxlog2(z) = 	7.67626240094011692566113917010541e+3  	-4.53236014182719380962768294571666e+0
*string in : 	+7.8895e1854  -3.8699e1943i
 z =  		7.88949999999999999999999999998271e+1854  	-3.86989999999999999999999999999111e+1943
 cxlog2(z) = 	6.45645858465309145411044517431549e+3  	-2.26618007091359690481384147285833e+0
*string in : 	+10.009876543100091234
 z =  		1.00098765431000912340000000000000e+1  	0.00000000000000000000000000000000e+0
 cxlog2(z) = 	3.32335227567908880222358508144293e+0  	0.00000000000000000000000000000000e+0
*string in : 	+9.8980000000006542009 +6.11002230065009876i
 z =  		9.89800000000065420089999999999997e+0  	6.11002230065009876000000000000000e+0
 cxlog2(z) = 	3.54002381728861746329810782039492e+0  	7.97871105209125991064277816946892e-1
*string in : 	+17.9998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+1
 cxlog2(z) = 	4.16991608975447206257066258915025e+0  	2.26618007091359690481384147285833e+0
*string in : 	-12.9822227      +20.87777220192i
 z =  		-1.29822227000000000000000000000000e+1  	2.08777722019200000000000000000000e+1
 cxlog2(z) = 	4.61970317204704169322701396162111e+0  	3.06876506463231931197676358665771e+0
*string in : 	-17.9877771265
 z =  		-1.79877771265000000000000000000000e+1  	0.00000000000000000000000000000000e+0
 cxlog2(z) = 	4.16894500872979050131294777636881e+0  	4.53236014182719380962768294571666e+0
*string in : 	-19.09872210981  -32.9800000654i
 z =  		-1.90987221098100000000000000000000e+1  	-3.29800000653999999999999999999999e+1
 cxlog2(z) = 	5.25213221597711964486953680163372e+0  	-3.02346537282148715841989736172724e+0
*string in : 	-43.987600182345i
 z =  		0.00000000000000000000000000000000e+0  	-4.39876001823449999999999999999999e+1
 cxlog2(z) = 	5.45902498962320547429566632486356e+0  	-2.26618007091359690481384147285833e+0
*string in : 	+15.982201192361 -43.2189031229i
 z =  		1.59822011923610000000000000000000e+1  	-4.32189031228999999999999999999999e+1
 cxlog2(z) = 	5.52604772912306645664784736289443e+0  	-1.75517611085712908286891878434331e+0
*string in : 	+1000.009876543100091234
 z =  		1.00000987654310009123400000000000e+3  	0.00000000000000000000000000000000e+0
 cxlog2(z) = 	9.96579853343147449467941684046899e+0  	0.00000000000000000000000000000000e+0
*string in : 	+989.80000000006542009 +611.002230065009876i
 z =  		9.89800000000065420089999999999996e+2  	6.11002230065009875999999999999998e+2
 cxlog2(z) = 	1.01838800070633421590387466793737e+1  	7.97871105209125991064277816946891e-1
*string in : 	+179.998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+2
 cxlog2(z) = 	7.49184418464183441044098201863964e+0  	2.26618007091359690481384147285833e+0
*string in : 	-1298.22227  +208.7777220192i
 z =  		-1.29822227000000000000000000000000e+3  	2.08777722019199999999999999999999e+2
 cxlog2(z) = 	1.03607403689285825530351979866552e+1  	4.30231823072993605712719315580530e+0
*string in : 	-1798.777712650981
 z =  		-1.79877771265098099999999999999999e+3  	0.00000000000000000000000000000000e+0
 cxlog2(z) = 	1.08128011985053020001283404686201e+1  	4.53236014182719380962768294571666e+0
*string in : 	-190.9872210981   -329.800700654i
 z =  		-1.90987221098100000000000000000000e+2  	-3.29800700653999999999999999999999e+2
 cxlog2(z) = 	8.57406260397150978879468981949935e+0  	-3.02346404488433351056167902545949e+0
*string in : 	+1598.2201192361  -4321.89031229i
 z =  		1.59822011923609999999999999999999e+3  	-4.32189031228999999999999999999998e+3
 cxlog2(z) = 	1.21699039188977911523884862218732e+1  	-1.75517611085712908286891878434330e+0

Test of cxtan
*string in : 	0
 z =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxtan(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.3106e-1571
 z =  		2.31059999999999999999999999999571e-1571  	0.00000000000000000000000000000000e+0
 cxtan(z) = 	2.31059999999999999999999999999571e-1571  	0.00000000000000000000000000000000e+0
*string in : 	+8.7792e-20  +9.7732e-20i
 z =  		8.77919999999999999999999999999979e-20  	9.77319999999999999999999999999977e-20
 cxtan(z) = 	8.77919999999999999999999999999979e-20  	9.77320000000000435661642035976737e-20
*string in : 	+2.3298e-1571i
 z =  		0.00000000000000000000000000000000e+0  	2.32979999999999999999999999999567e-1571
 cxtan(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-4.5592e-19  +8.2219e-18i
 z =  		-4.55919999999999999999999999999990e-19  	8.22189999999999999999999999999983e-18
 cxtan(z) = 	-4.55919999999999999999999999999990e-19  	8.22189999999999994626368727154255e-18
*string in : 	-3.8932e-1624i
 z =  		0.00000000000000000000000000000000e+0  	-3.89319999999999999999999999999252e-1624
 cxtan(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-1.2291e-17  -7.3342e-18i
 z =  		-1.22909999999999999999999999999998e-17  	-7.33419999999999999999999999999984e-18
 cxtan(z) = 	-1.22909999999999999999999999999998e-17  	-7.33420000000000023242731814120121e-18
*string in : 	-4.0987e-1617i
 z =  		0.00000000000000000000000000000000e+0  	-4.09869999999999999999999999999216e-1617
 cxtan(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+3.9917e-19  -8.2215e-19i
 z =  		3.99169999999999999999999999999991e-19  	-8.22149999999999999999999999999982e-19
 cxtan(z) = 	3.99169999999999999999999999999991e-19  	-8.22150000000000474780953325262615e-19
*string in : 	+1.0
 z =  		1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxtan(z) = 	1.55740772465490223050697480745836e+0  	0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000000986
 z =  		1.00000000000000000009860000000000e+0  	0.00000000000000000000000000000000e+0
 cxtan(z) = 	1.55740772465490223084473096319069e+0  	0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000004567 +2.43e-18i
 z =  		1.00000000000000000045670000000000e+0  	2.42999999999999999999999999999995e-18
 cxtan(z) = 	1.55740772465490223207140925292446e+0  	8.32401073457986532545627704509548e-18
*string in : 	+1.0  +3.4506e-1298i
 z =  		1.00000000000000000000000000000000e+0  	3.45059999999999999999999999999470e-1298
 cxtan(z) = 	1.55740772465490223050697480745836e+0  	0.00000000000000000000000000000000e+0
*string in : 	+0.999999999999999998736  +4.6677e-1297i
 z =  		9.99999999999999998735999999999976e-1  	4.66769999999999999999999999999284e-1297
 cxtan(z) = 	1.55740772465490222617711901794842e+0  	0.00000000000000000000000000000000e+0
*string in : 	+0.999999999999999999754
 z =  		9.99999999999999999753999999999976e-1  	0.00000000000000000000000000000000e+0
 cxtan(z) = 	1.55740772465490222966429717753785e+0  	0.00000000000000000000000000000000e+0
*string in : 	+0.999999999999999999338  -5.0783e-1145i
 z =  		9.99999999999999999337999999999976e-1  	-5.07829999999999999999999999999312e-1145
 cxtan(z) = 	1.55740772465490222823928134807891e+0  	0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000000    -6.7002e-1301i
 z =  		1.00000000000000000000000000000000e+0  	-6.70019999999999999999999999998969e-1301
 cxtan(z) = 	1.55740772465490223050697480745836e+0  	0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000007423    -5.3321e-1344i
 z =  		1.00000000000000074230000000000000e+0  	-5.33209999999999999999999999999153e-1344
 cxtan(z) = 	1.55740772465490477326959549825747e+0  	0.00000000000000000000000000000000e+0
*string in : 	-1.0
 z =  		-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxtan(z) = 	-1.55740772465490223050697480745836e+0  	0.00000000000000000000000000000000e+0
*string in : 	-0.99999999999999999998811
 z =  		-9.99999999999999999988109999999973e-1  	0.00000000000000000000000000000000e+0
 cxtan(z) = 	-1.55740772465490223046624538867878e+0  	0.00000000000000000000000000000000e+0
*string in : 	-0.99999999999999999976632 +2.3344e-15i
 z =  		-9.99999999999999999766319999999974e-1  	2.33439999999999999999999999999996e-15
 cxtan(z) = 	-1.55740772465490222970649956938120e+0  	7.99653113530997518146738691121322e-15
*string in : 	-1.0  +4.5601e-997i
 z =  		-1.00000000000000000000000000000000e+0  	4.56009999999999999999999999999462e-997
 cxtan(z) = 	-1.55740772465490223050697480745836e+0  	0.00000000000000000000000000000000e+0
*string in : 	-1.00000000000000000000654 +7.4332e-1532i
 z =  		-1.00000000000000000000654000000000e+0  	7.43319999999999999999999999998655e-1532
 cxtan(z) = 	-1.55740772465490223052937770054649e+0  	0.00000000000000000000000000000000e+0
*string in : 	-1.00000000000000000000743
 z =  		-1.00000000000000000000743000000000e+0  	0.00000000000000000000000000000000e+0
 cxtan(z) = 	-1.55740772465490223053242641229701e+0  	0.00000000000000000000000000000000e+0
*string in : 	-1.00000000000000000085023 -6.9983e-18i
 z =  		-1.00000000000000000085023000000000e+0  	-6.99829999999999999999999999999985e-18
 cxtan(z) = 	-1.55740772465490223341945367447969e+0  	-2.39728083637079343927688323143858e-17
*string in : 	-1.0  -5.00098e-1023i
 z =  		-1.00000000000000000000000000000000e+0  	-5.00097999999999999999999999999395e-1023
 cxtan(z) = 	-1.55740772465490223050697480745836e+0  	0.00000000000000000000000000000000e+0
*string in : 	-0.9999999999999999997643  -6.7321e-16i
 z =  		-9.99999999999999999764299999999974e-1  	-6.73209999999999999999999999999987e-16
 cxtan(z) = 	-1.55740772465490222969958002138981e+0  	-2.30609352536070441621006012216946e-15
*string in : 	+1.0i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
 cxtan(z) = 	0.00000000000000000000000000000000e+0  	7.61594155955764888119458282604793e-1
*string in : 	+5.6098e-2198 +1.0i
 z =  		5.60979999999999999999999999998542e-2198  	1.00000000000000000000000000000000e+0
 cxtan(z) = 	2.35597206158636344408924780667001e-2198  	7.61594155955764888119458282604793e-1
*string in : 	+4.9987e-18   +1.0000000000000000005421i
 z =  		4.99869999999999999999999999999989e-18  	1.00000000000000000054210000000000e+0
 cxtan(z) = 	2.09932574142603211134881479233303e-18  	7.61594155955764888347126373193756e-1
*string in : 	+1.000000000000000000003201i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000320100000000e+0
 cxtan(z) = 	0.00000000000000000000000000000000e+0  	7.61594155955764888120802620472300e-1
*string in : 	-6.5541e-20   +1.0000000000000000080981i
 z =  		-6.55409999999999999999999999999984e-20  	1.00000000000000000809810000000000e+0
 cxtan(z) = 	-2.75255383237248822746590874166717e-20  	7.61594155955764891520452498429338e-1
*string in : 	-7.0998e-2200 +1.0i
 z =  		-7.09979999999999999999999999998153e-2200  	1.00000000000000000000000000000000e+0
 cxtan(z) = 	-2.98173383059126228748704794784052e-2200  	7.61594155955764888119458282604793e-1
*string in : 	-6.8432e-17   +0.9999999999999999999864i
 z =  		-6.84319999999999999999999999999986e-17  	9.99999999999999999986399999999974e-1
 cxtan(z) = 	-2.87396841453310319813995537794057e-17  	7.61594155955764888113746631558832e-1
*string in : 	+0.99999999999999989098732i
 z =  		0.00000000000000000000000000000000e+0  	9.99999999999999890987319999999974e-1
 cxtan(z) = 	0.00000000000000000000000000000000e+0  	7.61594155955764842336929772024271e-1
*string in : 	+4.3329e-16   +0.9999999999999999997877i
 z =  		4.33289999999999999999999999999992e-16  	9.99999999999999999787699999999974e-1
 cxtan(z) = 	1.81970682477941355666785875476497e-16  	7.61594155955764888030297729880183e-1
*string in : 	-1.0i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
 cxtan(z) = 	0.00000000000000000000000000000000e+0  	-7.61594155955764888119458282604793e-1
*string in : 	+2.7762e-2111 -1.0i
 z =  		2.77619999999999999999999999999307e-2111  	-1.00000000000000000000000000000000e+0
 cxtan(z) = 	1.16593276718885917385300184692466e-2111  	-7.61594155955764888119458282604793e-1
*string in : 	+3.3321e-17   -0.9999999999999999996555i
 z =  		3.33209999999999999999999999999993e-17  	-9.99999999999999999655499999999974e-1
 cxtan(z) = 	1.39939650369209626656371962932481e-17  	-7.61594155955764887974777121918750e-1
*string in : 	-0.999999999999999999996999972013i
 z =  		0.00000000000000000000000000000000e+0  	-9.99999999999999999996999972012965e-1
 cxtan(z) = 	0.00000000000000000000000000000000e+0  	-7.61594155955764888118198347826114e-1
*string in : 	-7.5541e-17   -0.999999999999999999987231i
 z =  		-7.55409999999999999999999999999985e-17  	-9.99999999999999999987230999999972e-1
 cxtan(z) = 	-3.17252817398651433087467219357511e-17  	-7.61594155955764888114095630236713e-1
*string in : 	-2.1156e-2203 -1.0i
 z =  		-2.11559999999999999999999999999449e-2203  	-1.00000000000000000000000000000000e+0
 cxtan(z) = 	-8.88497717118633552410997301114309e-2204  	-7.61594155955764888119458282604793e-1
*string in : 	-5.9923e-16   -1.00000000000000000089321i
 z =  		-5.99229999999999999999999999999988e-16  	-1.00000000000000000089321000000000e+0
 cxtan(z) = 	-2.51661224725372841220872381770074e-16  	-7.61594155955764888494583564277971e-1
*string in : 	-1.000000000000000000007161i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000716100000000e+0
 cxtan(z) = 	0.00000000000000000000000000000000e+0  	-7.61594155955764888122465718865091e-1
*string in : 	+8.6631e-19   -1.00000000000000000008765i
 z =  		8.66309999999999999999999999999981e-19  	-1.00000000000000000008765000000000e+0
 cxtan(z) = 	3.63827971883646924128572723219092e-19  	-7.61594155955764888156269033647262e-1
*string in : 	+1.5876e1876
 z =  		1.58759999999999999999999999999648e+1876  	0.00000000000000000000000000000000e+0
 cxtan(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-6.4391e1965
 z =  		-6.43909999999999999999999999998503e+1965  	0.00000000000000000000000000000000e+0
 cxtan(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+3.4210e2015i
 z =  		0.00000000000000000000000000000000e+0  	3.42099999999999999999999999999185e+2015
 cxtan(z) = 	*** Floating point overflow
*string in : 	-4.5522e2011i
 z =  		0.00000000000000000000000000000000e+0  	-4.55219999999999999999999999998917e+2011
 cxtan(z) = 	*** Floating point overflow
*string in : 	+2.1907e1854  +9.8765e1872i
 z =  		2.19069999999999999999999999999520e+1854  	9.87649999999999999999999999997812e+1872
 cxtan(z) = 	*** Floating point overflow
*string in : 	-3.2966e1946  +7.5721e1834i
 z =  		-3.29659999999999999999999999999241e+1946  	7.57209999999999999999999999998357e+1834
 cxtan(z) = 	*** Floating point overflow
*string in : 	-6.0987e2310  -5.4411e2198i
 z =  		-6.09869999999999999999999999998333e+2310  	-5.44109999999999999999999999998585e+2198
 cxtan(z) = 	*** Floating point overflow
*string in : 	+7.8895e1854  -3.8699e1943i
 z =  		7.88949999999999999999999999998271e+1854  	-3.86989999999999999999999999999111e+1943
 cxtan(z) = 	*** Floating point overflow
*string in : 	+10.009876543100091234
 z =  		1.00098765431000912340000000000000e+1  	0.00000000000000000000000000000000e+0
 cxtan(z) = 	6.62480062908500512207294794219256e-1  	0.00000000000000000000000000000000e+0
*string in : 	+9.8980000000006542009 +6.11002230065009876i
 z =  		9.89800000000065420089999999999997e+0  	6.11002230065009876000000000000000e+0
 cxtan(z) = 	8.00080126849400010265058480379592e-6  	9.99994235374314218712696436094126e-1
*string in : 	+17.9998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+1
 cxtan(z) = 	0.00000000000000000000000000000000e-4925  	9.99999999999999535992261553889413e-1
*string in : 	-12.9822227      +20.87777220192i
 z =  		-1.29822227000000000000000000000000e+1  	2.08777722019200000000000000000000e+1
 cxtan(z) = 	-1.08522342368904873772196790127693e-18  	9.99999999999999999010898973830819e-1
*string in : 	-17.9877771265
 z =  		-1.79877771265000000000000000000000e+1  	0.00000000000000000000000000000000e+0
 cxtan(z) = 	1.16574328066105538469359570166157e+0  	0.00000000000000000000000000000000e+0
*string in : 	-19.09872210981  -32.9800000654i
 z =  		-1.90987221098100000000000000000000e+1  	-3.29800000653999999999999999999999e+1
 cxtan(z) = 	-2.15951635769224175349387222735134e-29  	-9.99999999999999999999999999960314e-1
*string in : 	-43.987600182345i
 z =  		0.00000000000000000000000000000000e+0  	-4.39876001823449999999999999999999e+1
 cxtan(z) = 	0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
*string in : 	+15.982201192361 -43.2189031229i
 z =  		1.59822011923610000000000000000000e+1  	-4.32189031228999999999999999999999e+1
 cxtan(z) = 	-1.92592994438723585305597794258493e-34  	-9.99999999999999999999999999999999e-1
*string in : 	+1000.009876543100091234
 z =  		1.00000987654310009123400000000000e+3  	0.00000000000000000000000000000000e+0
 cxtan(z) = 	1.50201354924224414601435796353798e+0  	0.00000000000000000000000000000000e+0
*string in : 	+989.80000000006542009 +611.002230065009876i
 z =  		9.89800000000065420089999999999996e+2  	6.11002230065009875999999999999998e+2
 cxtan(z) = 	0.00000000000000000000000000000000e+0  	9.99999999999999999999999999999999e-1
*string in : 	+179.998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+2
 cxtan(z) = 	-0.00000000000000000000000000000000e-4855  	1.00000000000000000000000000000000e+0
*string in : 	-1298.22227  +208.7777220192i
 z =  		-1.29822227000000000000000000000000e+3  	2.08777722019199999999999999999999e+2
 cxtan(z) = 	0.00000000000000000000000000000000e+0  	9.99999999999999999999999999999998e-1
*string in : 	-1798.777712650981
 z =  		-1.79877771265098099999999999999999e+3  	0.00000000000000000000000000000000e+0
 cxtan(z) = 	4.55918022584424715586842992152169e+0  	0.00000000000000000000000000000000e-4931
*string in : 	-190.9872210981   -329.800700654i
 z =  		-1.90987221098100000000000000000000e+2  	-3.29800700653999999999999999999999e+2
 cxtan(z) = 	9.62964972193617926527988971292464e-35  	-9.99999999999999999999999999999999e-1
*string in : 	+1598.2201192361  -4321.89031229i
 z =  		1.59822011923609999999999999999999e+3  	-4.32189031228999999999999999999998e+3
 cxtan(z) = 	0.00000000000000000000000000000000e+0  	-9.99999999999999999999999999999999e-1

Test of cxtanh
*string in : 	0
 z =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxtanh(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.3106e-1571
 z =  		2.31059999999999999999999999999571e-1571  	0.00000000000000000000000000000000e+0
 cxtanh(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+8.7792e-20  +9.7732e-20i
 z =  		8.77919999999999999999999999999979e-20  	9.77319999999999999999999999999977e-20
 cxtanh(z) = 	8.77920000000000795093312100726045e-20  	9.77319999999999999999999999999977e-20
*string in : 	+2.3298e-1571i
 z =  		0.00000000000000000000000000000000e+0  	2.32979999999999999999999999999567e-1571
 cxtanh(z) = 	0.00000000000000000000000000000000e+0  	2.32979999999999999999999999999567e-1571
*string in : 	-4.5592e-19  +8.2219e-18i
 z =  		-4.55919999999999999999999999999990e-19  	8.22189999999999999999999999999983e-18
 cxtanh(z) = 	-4.55919999999999882056122642566678e-19  	8.22189999999999999999999999999982e-18
*string in : 	-3.8932e-1624i
 z =  		0.00000000000000000000000000000000e+0  	-3.89319999999999999999999999999252e-1624
 cxtanh(z) = 	0.00000000000000000000000000000000e+0  	-3.89319999999999999999999999999252e-1624
*string in : 	-1.2291e-17  -7.3342e-18i
 z =  		-1.22909999999999999999999999999998e-17  	-7.33419999999999999999999999999984e-18
 cxtanh(z) = 	-1.22910000000000001189707920112774e-17  	-7.33419999999999999999999999999984e-18
*string in : 	-4.0987e-1617i
 z =  		0.00000000000000000000000000000000e+0  	-4.09869999999999999999999999999216e-1617
 cxtanh(z) = 	0.00000000000000000000000000000000e+0  	-4.09869999999999999999999999999216e-1617
*string in : 	+3.9917e-19  -8.2215e-19i
 z =  		3.99169999999999999999999999999991e-19  	-8.22149999999999999999999999999982e-19
 cxtanh(z) = 	3.99169999999999779581069919279131e-19  	-8.22149999999999999999999999999981e-19
*string in : 	+1.0
 z =  		1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxtanh(z) = 	7.61594155955764888119458282604793e-1  	0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000000986
 z =  		1.00000000000000000009860000000000e+0  	0.00000000000000000000000000000000e+0
 cxtanh(z) = 	7.61594155955764888160867752687936e-1  	0.00000000000000000000000000000000e+0
*string in : 	+1.0000000000000000004567 +2.43e-18i
 z =  		1.00000000000000000045670000000000e+0  	2.42999999999999999999999999999995e-18
 cxtanh(z) = 	7.61594155955764888311260564419918e-1  	1.02053765012208334791870016079453e-18
*string in : 	+1.0  +3.4506e-1298i
 z =  		1.00000000000000000000000000000000e+0  	3.45059999999999999999999999999470e-1298
 cxtanh(z) = 	7.61594155955764888119458282604793e-1  	1.44916346317335835505265044773507e-1298
*string in : 	+0.999999999999999998736  +4.6677e-1297i
 z =  		9.99999999999999998735999999999976e-1  	4.66769999999999999999999999999284e-1297
 cxtanh(z) = 	7.61594155955764887588610714804655e-1  	1.96031423435178948788690507883221e-1297
*string in : 	+0.999999999999999999754
 z =  		9.99999999999999999753999999999976e-1  	0.00000000000000000000000000000000e+0
 cxtanh(z) = 	7.61594155955764888016144594567732e-1  	0.00000000000000000000000000000000e+0
*string in : 	+0.999999999999999999338  -5.0783e-1145i
 z =  		9.99999999999999999337999999999976e-1  	-5.07829999999999999999999999999312e-1145
 cxtanh(z) = 	7.61594155955764887841435268456298e-1  	-2.13275569901850859035663841189756e-1145
*string in : 	+1.0000000000000000000    -6.7002e-1301i
 z =  		1.00000000000000000000000000000000e+0  	-6.70019999999999999999999999998969e-1301
 cxtanh(z) = 	7.61594155955764888119458282604793e-1  	-2.81391208368229747015700705092288e-1301
*string in : 	+1.0000000000000007423    -5.3321e-1344i
 z =  		1.00000000000000074230000000000000e+0  	-5.33209999999999999999999999999153e-1344
 cxtanh(z) = 	7.61594155955765199866412062696169e-1  	-2.23934518692014587267435676906040e-1344
*string in : 	-1.0
 z =  		-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxtanh(z) = 	-7.61594155955764888119458282604793e-1  	0.00000000000000000000000000000000e+0
*string in : 	-0.99999999999999999998811
 z =  		-9.99999999999999999988109999999973e-1  	0.00000000000000000000000000000000e+0
 cxtanh(z) = 	-7.61594155955764888114464787682991e-1  	0.00000000000000000000000000000000e+0
*string in : 	-0.99999999999999999976632 +2.3344e-15i
 z =  		-9.99999999999999999766319999999974e-1  	2.33439999999999999999999999999996e-15
 cxtanh(z) = 	-7.61594155955764888021318678458159e-1  	9.80388103063782456743471200329126e-16
*string in : 	-1.0  +4.5601e-997i
 z =  		-1.00000000000000000000000000000000e+0  	4.56009999999999999999999999999462e-997
 cxtanh(z) = 	-7.61594155955764888119458282604793e-1  	1.91512499519412027904584457970181e-997
*string in : 	-1.00000000000000000000654 +7.4332e-1532i
 z =  		-1.00000000000000000000654000000000e+0  	7.43319999999999999999999999998655e-1532
 cxtanh(z) = 	-7.61594155955764888122204914798949e-1  	3.12175327608537857899207534224679e-1532
*string in : 	-1.00000000000000000000743
 z =  		-1.00000000000000000000743000000000e+0  	0.00000000000000000000000000000000e+0
 cxtanh(z) = 	-7.61594155955764888122578691962985e-1  	0.00000000000000000000000000000000e+0
*string in : 	-1.00000000000000000085023 -6.9983e-18i
 z =  		-1.00000000000000000085023000000000e+0  	-6.99829999999999999999999999999985e-18
 cxtanh(z) = 	-7.61594155955764888476533067075285e-1  	-2.93910643491743863763718617818390e-18
*string in : 	-1.0  -5.00098e-1023i
 z =  		-1.00000000000000000000000000000000e+0  	-5.00097999999999999999999999999395e-1023
 cxtanh(z) = 	-7.61594155955764888119458282604793e-1  	-2.10028328292491209252049030201023e-1023
*string in : 	-0.9999999999999999997643  -6.7321e-16i
 z =  		-9.99999999999999999764299999999974e-1  	-6.73209999999999999999999999999987e-16
 cxtanh(z) = 	-7.61594155955764888020470330286501e-1  	-2.82730926517978490278573930431825e-16
*string in : 	+1.0i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
 cxtanh(z) = 	0.00000000000000000000000000000000e+0  	1.55740772465490223050697480745836e+0
*string in : 	+5.6098e-2198 +1.0i
 z =  		5.60979999999999999999999999998542e-2198  	1.00000000000000000000000000000000e+0
 cxtanh(z) = 	0.00000000000000000000000000000000e+0  	1.55740772465490223050697480745836e+0
*string in : 	+4.9987e-18   +1.0000000000000000005421i
 z =  		4.99869999999999999999999999999989e-18  	1.00000000000000000054210000000000e+0
 cxtanh(z) = 	1.71231409296067403350631572316550e-17  	1.55740772465490223236394856022204e+0
*string in : 	+1.000000000000000000003201i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000320100000000e+0
 cxtanh(z) = 	0.00000000000000000000000000000000e+0  	1.55740772465490223051793989320378e+0
*string in : 	-6.5541e-20   +1.0000000000000000080981i
 z =  		-6.55409999999999999999999999999984e-20  	1.00000000000000000809810000000000e+0
 cxtanh(z) = 	-2.24511929035020393640011448858182e-19  	1.55740772465490225824716877029836e+0
*string in : 	-7.0998e-2200 +1.0i
 z =  		-7.09979999999999999999999999998153e-2200  	1.00000000000000000000000000000000e+0
 cxtanh(z) = 	0.00000000000000000000000000000000e+0  	1.55740772465490223050697480745836e+0
*string in : 	-6.8432e-17   +0.9999999999999999999864i
 z =  		-6.84319999999999999999999999999986e-17  	9.99999999999999999986399999999974e-1
 cxtanh(z) = 	-2.34415103945995642418244363515882e-16  	1.55740772465490223046038775149516e+0
*string in : 	+0.99999999999999989098732i
 z =  		0.00000000000000000000000000000000e+0  	9.99999999999999890987319999999974e-1
 cxtanh(z) = 	0.00000000000000000000000000000000e+0  	1.55740772465490185708198776000159e+0
*string in : 	+4.3329e-16   +0.9999999999999999997877i
 z =  		4.33289999999999999999999999999992e-16  	9.99999999999999999787699999999974e-1
 cxtanh(z) = 	1.48424304987082725494992998301140e-15  	1.55740772465490222977973716179830e+0
*string in : 	-1.0i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
 cxtanh(z) = 	0.00000000000000000000000000000000e+0  	-1.55740772465490223050697480745836e+0
*string in : 	+2.7762e-2111 -1.0i
 z =  		2.77619999999999999999999999999307e-2111  	-1.00000000000000000000000000000000e+0
 cxtanh(z) = 	0.00000000000000000000000000000000e+0  	-1.55740772465490223050697480745836e+0
*string in : 	+3.3321e-17   -0.9999999999999999996555i
 z =  		3.33209999999999999999999999999993e-17  	-9.99999999999999999655499999999974e-1
 cxtanh(z) = 	1.14141712628368607879311515897622e-16  	-1.55740772465490222932688357368758e+0
*string in : 	-0.999999999999999999996999972013i
 z =  		0.00000000000000000000000000000000e+0  	-9.99999999999999999996999972012965e-1
 cxtanh(z) = 	0.00000000000000000000000000000000e+0  	-1.55740772465490223049669815512580e+0
*string in : 	-7.5541e-17   -0.999999999999999999987231i
 z =  		-7.55409999999999999999999999999985e-17  	-9.99999999999999999987230999999972e-1
 cxtanh(z) = 	-2.58767117243167767696995609071600e-16  	-1.55740772465490223046323435763525e+0
*string in : 	-2.1156e-2203 -1.0i
 z =  		-2.11559999999999999999999999999449e-2203  	-1.00000000000000000000000000000000e+0
 cxtanh(z) = 	0.00000000000000000000000000000000e+0  	-1.55740772465490223050697480745836e+0
*string in : 	-5.9923e-16   -1.00000000000000000089321i
 z =  		-5.99229999999999999999999999999988e-16  	-1.00000000000000000089321000000000e+0
 cxtanh(z) = 	-2.05267364299682849615780280572122e-15  	-1.55740772465490223356668247339639e+0
*string in : 	-1.000000000000000000007161i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000716100000000e+0
 cxtanh(z) = 	0.00000000000000000000000000000000e+0  	-1.55740772465490223053150494773421e+0
*string in : 	+8.6631e-19   -1.00000000000000000008765i
 z =  		8.66309999999999999999999999999981e-19  	-1.00000000000000000008765000000000e+0
 cxtanh(z) = 	2.96756120966003366364935279248272e-18  	-1.55740772465490223080722153210277e+0
*string in : 	+1.5876e1876
 z =  		1.58759999999999999999999999999648e+1876  	0.00000000000000000000000000000000e+0
 cxtanh(z) = 	*** Floating point overflow
*string in : 	-6.4391e1965
 z =  		-6.43909999999999999999999999998503e+1965  	0.00000000000000000000000000000000e+0
 cxtanh(z) = 	*** Floating point overflow
*string in : 	+3.4210e2015i
 z =  		0.00000000000000000000000000000000e+0  	3.42099999999999999999999999999185e+2015
 cxtanh(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	-4.5522e2011i
 z =  		0.00000000000000000000000000000000e+0  	-4.55219999999999999999999999998917e+2011
 cxtanh(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.1907e1854  +9.8765e1872i
 z =  		2.19069999999999999999999999999520e+1854  	9.87649999999999999999999999997812e+1872
 cxtanh(z) = 	*** Floating point overflow
*string in : 	-3.2966e1946  +7.5721e1834i
 z =  		-3.29659999999999999999999999999241e+1946  	7.57209999999999999999999999998357e+1834
 cxtanh(z) = 	*** Floating point overflow
*string in : 	-6.0987e2310  -5.4411e2198i
 z =  		-6.09869999999999999999999999998333e+2310  	-5.44109999999999999999999999998585e+2198
 cxtanh(z) = 	*** Floating point overflow
*string in : 	+7.8895e1854  -3.8699e1943i
 z =  		7.88949999999999999999999999998271e+1854  	-3.86989999999999999999999999999111e+1943
 cxtanh(z) = 	*** Floating point overflow
*string in : 	+10.009876543100091234
 z =  		1.00098765431000912340000000000000e+1  	0.00000000000000000000000000000000e+0
 cxtanh(z) = 	9.99999995958322092917699044500508e-1  	0.00000000000000000000000000000000e+0
*string in : 	+9.8980000000006542009 +6.11002230065009876i
 z =  		9.89800000000065420089999999999997e+0  	6.11002230065009876000000000000000e+0
 cxtanh(z) = 	9.99999995244967501549791535897074e-1  	-1.71595105591083885739888014664147e-9
*string in : 	+17.9998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+1
 cxtanh(z) = 	0.00000000000000000000000000000000e+0  	-1.13756875154001643622368335165726e+0
*string in : 	-12.9822227      +20.87777220192i
 z =  		-1.29822227000000000000000000000000e+1  	2.08777722019200000000000000000000e+1
 cxtanh(z) = 	-1.00000000000645781638411184074387e+0  	-8.39063245572365739856033893438974e-12
*string in : 	-17.9877771265
 z =  		-1.79877771265000000000000000000000e+1  	0.00000000000000000000000000000000e+0
 cxtanh(z) = 	-9.99999999999999524615190438524330e-1  	0.00000000000000000000000000000000e-4925
*string in : 	-19.09872210981  -32.9800000654i
 z =  		-1.90987221098100000000000000000000e+1  	-3.29800000653999999999999999999999e+1
 cxtanh(z) = 	-1.00000000000000005152897341730443e+0  	-6.92879440329552929162184599598029e-19
*string in : 	-43.987600182345i
 z =  		0.00000000000000000000000000000000e+0  	-4.39876001823449999999999999999999e+1
 cxtanh(z) = 	0.00000000000000000000000000000000e+0  	-5.30308179934060945203097630433840e-3
*string in : 	+15.982201192361 -43.2189031229i
 z =  		1.59822011923610000000000000000000e+1  	-4.32189031228999999999999999999999e+1
 cxtanh(z) = 	9.99999999999998845322919378059976e-1  	2.62207875470815054697499402034301e-14
*string in : 	+1000.009876543100091234
 z =  		1.00000987654310009123400000000000e+3  	0.00000000000000000000000000000000e+0
 cxtanh(z) = 	1.00000000000000000000000000000000e+0  	-0.00000000000000000000000000000000e-4499
*string in : 	+989.80000000006542009 +611.002230065009876i
 z =  		9.89800000000065420089999999999996e+2  	6.11002230065009875999999999999998e+2
 cxtanh(z) = 	9.99999999999999999999999999999998e-1  	-1.20370621524202240815998621411558e-35
*string in : 	+179.998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+2
 cxtanh(z) = 	0.00000000000000000000000000000000e+0  	1.33559036188809055620693315045638e+0
*string in : 	-1298.22227  +208.7777220192i
 z =  		-1.29822227000000000000000000000000e+3  	2.08777722019199999999999999999999e+2
 cxtanh(z) = 	-9.99999999999999999999999999999998e-1  	-1.44444745829042688979198345693870e-34
*string in : 	-1798.777712650981
 z =  		-1.79877771265098099999999999999999e+3  	0.00000000000000000000000000000000e+0
 cxtanh(z) = 	-9.99999999999999999999999999999998e-1  	-0.00000000000000000000000000000000e-4151
*string in : 	-190.9872210981   -329.800700654i
 z =  		-1.90987221098100000000000000000000e+2  	-3.29800700653999999999999999999999e+2
 cxtanh(z) = 	-1.00000000000000000000000000000000e+0  	4.81482486096808963263994485646232e-35
*string in : 	+1598.2201192361  -4321.89031229i
 z =  		1.59822011923609999999999999999999e+3  	-4.32189031228999999999999999999998e+3
 cxtanh(z) = 	9.99999999999999999999999999999999e-1  	-2.88889491658085377958396691387739e-34

Test of cxatan
*string in : 	0
 z =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxatan(z) = 	0.00000000000000000000000000000000e+0  	-0.00000000000000000000000000000000e+0
*string in : 	+2.3106e-1571
 z =  		2.31059999999999999999999999999571e-1571  	0.00000000000000000000000000000000e+0
 cxatan(z) = 	2.31059999999999999999999999999571e-1571  	-0.00000000000000000000000000000000e+0
*string in : 	+8.7792e-20  +9.7732e-20i
 z =  		8.77919999999999999999999999999979e-20  	9.77319999999999999999999999999977e-20
 cxatan(z) = 	8.77919999999999999999999999999980e-20  	9.77320000000007657898933488111186e-20
*string in : 	+2.3298e-1571i
 z =  		0.00000000000000000000000000000000e+0  	2.32979999999999999999999999999567e-1571
 cxatan(z) = 	0.00000000000000000000000000000000e+0  	-0.00000000000000000000000000000000e+0
*string in : 	-4.5592e-19  +8.2219e-18i
 z =  		-4.55919999999999999999999999999990e-19  	8.22189999999999999999999999999983e-18
 cxatan(z) = 	-4.55919999999999999999999999999990e-19  	8.22190000000000057219091919739420e-18
*string in : 	-3.8932e-1624i
 z =  		0.00000000000000000000000000000000e+0  	-3.89319999999999999999999999999252e-1624
 cxatan(z) = 	0.00000000000000000000000000000000e+0  	-0.00000000000000000000000000000000e+0
*string in : 	-1.2291e-17  -7.3342e-18i
 z =  		-1.22909999999999999999999999999998e-17  	-7.33419999999999999999999999999984e-18
 cxatan(z) = 	-1.22909999999999999999999999999997e-17  	-7.33420000000000013613082092183941e-18
*string in : 	-4.0987e-1617i
 z =  		0.00000000000000000000000000000000e+0  	-4.09869999999999999999999999999216e-1617
 cxatan(z) = 	0.00000000000000000000000000000000e+0  	-0.00000000000000000000000000000000e+0
*string in : 	+3.9917e-19  -8.2215e-19i
 z =  		3.99169999999999999999999999999991e-19  	-8.22149999999999999999999999999982e-19
 cxatan(z) = 	3.99169999999999999999999999999990e-19  	-8.22150000000000234039710276858133e-19
*string in : 	+1.0
 z =  		1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxatan(z) = 	7.85398163397448309615660845819876e-1  	1.39629920968074599346558400837407e-33
*string in : 	+1.0000000000000000000986
 z =  		1.00000000000000000009860000000000e+0  	0.00000000000000000000000000000000e+0
 cxatan(z) = 	7.85398163397448309664960845819875e-1  	1.20370621524202240815998621411558e-33
*string in : 	+1.0000000000000000004567 +2.43e-18i
 z =  		1.00000000000000000045670000000000e+0  	2.42999999999999999999999999999995e-18
 cxatan(z) = 	7.85398163397448309844010845819876e-1  	1.21500000000000104603715753120909e-18
*string in : 	+1.0  +3.4506e-1298i
 z =  		1.00000000000000000000000000000000e+0  	3.45059999999999999999999999999470e-1298
 cxatan(z) = 	7.85398163397448309615660845819876e-1  	1.39629920968074599346558400837407e-33
*string in : 	+0.999999999999999998736  +4.6677e-1297i
 z =  		9.99999999999999998735999999999976e-1  	4.66769999999999999999999999999284e-1297
 cxatan(z) = 	7.85398163397448308983660845819863e-1  	1.20370621524202240815998621411558e-33
*string in : 	+0.999999999999999999754
 z =  		9.99999999999999999753999999999976e-1  	0.00000000000000000000000000000000e+0
 cxatan(z) = 	7.85398163397448309492660845819863e-1  	1.54074395550978868244478235406794e-33
*string in : 	+0.999999999999999999338  -5.0783e-1145i
 z =  		9.99999999999999999337999999999976e-1  	-5.07829999999999999999999999999312e-1145
 cxatan(z) = 	7.85398163397448309284660845819863e-1  	1.20370621524202240815998621411558e-33
*string in : 	+1.0000000000000000000    -6.7002e-1301i
 z =  		1.00000000000000000000000000000000e+0  	-6.70019999999999999999999999998969e-1301
 cxatan(z) = 	7.85398163397448309615660845819876e-1  	1.39629920968074599346558400837407e-33
*string in : 	+1.0000000000000007423    -5.3321e-1344i
 z =  		1.00000000000000074230000000000000e+0  	-5.33209999999999999999999999999153e-1344
 cxatan(z) = 	7.85398163397448680765660845819737e-1  	1.82963344716787406040317904545568e-33
*string in : 	-1.0
 z =  		-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxatan(z) = 	-7.85398163397448309615660845819876e-1  	1.39629920968074599346558400837407e-33
*string in : 	-0.99999999999999999998811
 z =  		-9.99999999999999999988109999999973e-1  	0.00000000000000000000000000000000e+0
 cxatan(z) = 	-7.85398163397448309609715845819862e-1  	-5.77778983316170755916793382775478e-34
*string in : 	-0.99999999999999999976632 +2.3344e-15i
 z =  		-9.99999999999999999766319999999974e-1  	2.33439999999999999999999999999996e-15
 cxatan(z) = 	-7.85398163397448309498820845821225e-1  	1.16720000000000000134414592899523e-15
*string in : 	-1.0  +4.5601e-997i
 z =  		-1.00000000000000000000000000000000e+0  	4.56009999999999999999999999999462e-997
 cxatan(z) = 	-7.85398163397448309615660845819876e-1  	1.39629920968074599346558400837407e-33
*string in : 	-1.00000000000000000000654 +7.4332e-1532i
 z =  		-1.00000000000000000000654000000000e+0  	7.43319999999999999999999999998655e-1532
 cxatan(z) = 	-7.85398163397448309618930845819875e-1  	1.54074395550978868244478235406794e-33
*string in : 	-1.00000000000000000000743
 z =  		-1.00000000000000000000743000000000e+0  	0.00000000000000000000000000000000e+0
 cxatan(z) = 	-7.85398163397448309619375845819875e-1  	1.20370621524202240815998621411558e-33
*string in : 	-1.00000000000000000085023 -6.9983e-18i
 z =  		-1.00000000000000000085023000000000e+0  	-6.99829999999999999999999999999985e-18
 cxatan(z) = 	-7.85398163397448310040775845819876e-1  	-3.49914999999999949995684819857547e-18
*string in : 	-1.0  -5.00098e-1023i
 z =  		-1.00000000000000000000000000000000e+0  	-5.00097999999999999999999999999395e-1023
 cxatan(z) = 	-7.85398163397448309615660845819876e-1  	1.39629920968074599346558400837407e-33
*string in : 	-0.9999999999999999997643  -6.7321e-16i
 z =  		-9.99999999999999999764299999999974e-1  	-6.73209999999999999999999999999987e-16
 cxatan(z) = 	-7.85398163397448309497810845819976e-1  	-3.36604999999999999200448697367313e-16
*string in : 	+1.0i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
 cxatan(z) = 	*** Out of domain
*string in : 	+5.6098e-2198 +1.0i
 z =  		5.60979999999999999999999999998542e-2198  	1.00000000000000000000000000000000e+0
 cxatan(z) = 	7.85398163397448309615660845819876e-1  	2.53002533325659856917350767567481e+3
*string in : 	+4.9987e-18   +1.0000000000000000005421i
 z =  		4.99869999999999999999999999999989e-18  	1.00000000000000000054210000000000e+0
 cxatan(z) = 	8.39411172501296280794100463241560e-1  	2.02623273908710334887802876039677e+1
*string in : 	+1.000000000000000000003201i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000320100000000e+0
 cxatan(z) = 	1.57079632679489661923132169163975e+0  	2.39419854362236489964194376913058e+1
*string in : 	-6.5541e-20   +1.0000000000000000080981i
 z =  		-6.55409999999999999999999999999984e-20  	1.00000000000000000809810000000000e+0
 cxatan(z) = 	-1.56674972518361760880636060456070e+0  	2.00240083189321174872455016153878e+1
*string in : 	-7.0998e-2200 +1.0i
 z =  		-7.09979999999999999999999999998153e-2200  	1.00000000000000000000000000000000e+0
 cxatan(z) = 	-7.85398163397448309615660845819876e-1  	2.53221014257641200952338751931514e+3
*string in : 	-6.8432e-17   +0.9999999999999999999864i
 z =  		-6.84319999999999999999999999999986e-17  	9.99999999999999999986399999999974e-1
 cxatan(z) = 	-7.85298794682366388535149510884673e-1  	1.89569191416099658020352419074263e+1
*string in : 	+0.99999999999999989098732i
 z =  		0.00000000000000000000000000000000e+0  	9.99999999999999890987319999999974e-1
 cxatan(z) = 	0.00000000000000000000000000000000e-4916  	1.87241073243571180124866063155152e+1
*string in : 	+4.3329e-16   +0.9999999999999999997877i
 z =  		4.33289999999999999999999999999992e-16  	9.99999999999999999787699999999974e-1
 cxatan(z) = 	7.85153177379987955127954214575818e-1  	1.80341357423176401310475560202952e+1
*string in : 	-1.0i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
 cxatan(z) = 	*** Out of domain
*string in : 	+2.7762e-2111 -1.0i
 z =  		2.77619999999999999999999999999307e-2111  	-1.00000000000000000000000000000000e+0
 cxatan(z) = 	7.85398163397448309615660845819876e-1  	-2.43021459770241575620785531954429e+3
*string in : 	+3.3321e-17   -0.9999999999999999996555i
 z =  		3.33209999999999999999999999999993e-17  	-9.99999999999999999655499999999974e-1
 cxatan(z) = 	7.80228934891374435202708472341071e-1  	-1.93167187914004707449624410668777e+1
*string in : 	-0.999999999999999999996999972013i
 z =  		0.00000000000000000000000000000000e+0  	-9.99999999999999999996999972012965e-1
 cxatan(z) = 	0.00000000000000000000000000000000e+0  	-2.39744062578992700577598833643782e+1
*string in : 	-7.5541e-17   -0.999999999999999999987231i
 z =  		-7.55409999999999999999999999999985e-17  	-9.99999999999999999987230999999972e-1
 cxatan(z) = 	-7.85313646381004187026881632300993e-1  	-1.89075016424751170781456492393592e+1
*string in : 	-2.1156e-2203 -1.0i
 z =  		-2.11559999999999999999999999999449e-2203  	-1.00000000000000000000000000000000e+0
 cxatan(z) = 	-7.85398163397448309615660845819876e-1  	-2.53626938429311644969977752862181e+3
*string in : 	-5.9923e-16   -1.00000000000000000089321i
 z =  		-5.99229999999999999999999999999988e-16  	-1.00000000000000000089321000000000e+0
 cxatan(z) = 	-7.86143460978065720469488219499284e-1  	-1.78720161229050015525602334077068e+1
*string in : 	-1.000000000000000000007161i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000716100000000e+0
 cxatan(z) = 	1.57079632679489661923132169163975e+0  	-2.35393922487050765283644167892193e+1
*string in : 	+8.6631e-19   -1.00000000000000000008765i
 z =  		8.66309999999999999999999999999981e-19  	-1.00000000000000000008765000000000e+0
 cxatan(z) = 	8.35814725188219191223994606066916e-1  	-2.11390495119474030669716346299366e+1
*string in : 	+1.5876e1876
 z =  		1.58759999999999999999999999999648e+1876  	0.00000000000000000000000000000000e+0
 cxatan(z) = 	1.57079632679489661923132169163975e+0  	-0.00000000000000000000000000000000e+0
*string in : 	-6.4391e1965
 z =  		-6.43909999999999999999999999998503e+1965  	0.00000000000000000000000000000000e+0
 cxatan(z) = 	-1.57079632679489661923132169163975e+0  	-1.15555796663234151183358676555096e-33
*string in : 	+3.4210e2015i
 z =  		0.00000000000000000000000000000000e+0  	3.42099999999999999999999999999185e+2015
 cxatan(z) = 	1.57079632679489661923132169163975e+0  	3.08148791101957736488956470813588e-33
*string in : 	-4.5522e2011i
 z =  		0.00000000000000000000000000000000e+0  	-4.55219999999999999999999999998917e+2011
 cxatan(z) = 	-1.57079632679489661923132169163975e+0  	-1.15555796663234151183358676555096e-33
*string in : 	+2.1907e1854  +9.8765e1872i
 z =  		2.19069999999999999999999999999520e+1854  	9.87649999999999999999999999997812e+1872
 cxatan(z) = 	1.57079632679489661923132169163975e+0  	2.40741243048404481631997242823116e-33
*string in : 	-3.2966e1946  +7.5721e1834i
 z =  		-3.29659999999999999999999999999241e+1946  	7.57209999999999999999999999998357e+1834
 cxatan(z) = 	-1.57079632679489661923132169163975e+0  	2.31111593326468302366717353110191e-33
*string in : 	-6.0987e2310  -5.4411e2198i
 z =  		-6.09869999999999999999999999998333e+2310  	-5.44109999999999999999999999998585e+2198
 cxatan(z) = 	-1.57079632679489661923132169163975e+0  	2.40741243048404481631997242823116e-33
*string in : 	+7.8895e1854  -3.8699e1943i
 z =  		7.88949999999999999999999999998271e+1854  	-3.86989999999999999999999999999111e+1943
 cxatan(z) = 	1.57079632679489661923132169163975e+0  	-1.15555796663234151183358676555096e-33
*string in : 	+10.009876543100091234
 z =  		1.00098765431000912340000000000000e+1  	0.00000000000000000000000000000000e+0
 cxatan(z) = 	1.47122536632822733605179162726757e+0  	-0.00000000000000000000000000000000e+0
*string in : 	+9.8980000000006542009 +6.11002230065009876i
 z =  		9.89800000000065420089999999999997e+0  	6.11002230065009876000000000000000e+0
 cxatan(z) = 	1.49762406437054198849524222230947e+0  	4.49475220009705616649721481681492e-2
*string in : 	+17.9998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+1
 cxatan(z) = 	1.57079632679489661923132169163975e+0  	5.56131617914866389884326606347439e-2
*string in : 	-12.9822227      +20.87777220192i
 z =  		-1.29822227000000000000000000000000e+1  	2.08777722019200000000000000000000e+1
 cxatan(z) = 	-1.54929516559948883331807091579739e+0  	3.45396400293629547012290287487255e-2
*string in : 	-17.9877771265
 z =  		-1.79877771265000000000000000000000e+1  	0.00000000000000000000000000000000e+0
 cxatan(z) = 	-1.51526018723076295836032569247233e+0  	-5.77778983316170755916793382775478e-34
*string in : 	-19.09872210981  -32.9800000654i
 z =  		-1.90987221098100000000000000000000e+1  	-3.29800000653999999999999999999999e+1
 cxatan(z) = 	-1.55764091293653484043117441063911e+0  	-2.27065671541651725463016359820585e-2
*string in : 	-43.987600182345i
 z =  		0.00000000000000000000000000000000e+0  	-4.39876001823449999999999999999999e+1
 cxatan(z) = 	1.57079632679489661923132169163975e+0  	-2.27375970202338822535695098569872e-2
*string in : 	+15.982201192361 -43.2189031229i
 z =  		1.59822011923610000000000000000000e+1  	-4.32189031228999999999999999999999e+1
 cxatan(z) = 	1.56326630738285435890998459498994e+0  	-2.03562070091229318153678813126373e-2
*string in : 	+1000.009876543100091234
 z =  		1.00000987654310009123400000000000e+3  	0.00000000000000000000000000000000e+0
 cxatan(z) = 	1.56979633700466543117769784885095e+0  	-0.00000000000000000000000000000000e+0
*string in : 	+989.80000000006542009 +611.002230065009876i
 z =  		9.89800000000065420089999999999996e+2  	6.11002230065009875999999999999998e+2
 cxatan(z) = 	1.57006478229094941499785489542764e+0  	4.51581231631634552646531570879406e-4
*string in : 	+179.998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+2
 cxatan(z) = 	1.57079632679489661923132169163975e+0  	5.55564703098558733216771418039192e-3
*string in : 	-1298.22227  +208.7777220192i
 z =  		-1.29822227000000000000000000000000e+3  	2.08777722019199999999999999999999e+2
 cxatan(z) = 	-1.57004546203380075617378661593923e+0  	1.20752645785488646349906193674709e-4
*string in : 	-1798.777712650981
 z =  		-1.79877771265098099999999999999999e+3  	0.00000000000000000000000000000000e+0
 cxatan(z) = 	-1.57024039379108719996064988823248e+0  	-0.00000000000000000000000000000000e+0
*string in : 	-190.9872210981   -329.800700654i
 z =  		-1.90987221098100000000000000000000e+2  	-3.29800700653999999999999999999999e+2
 cxatan(z) = 	-1.56948138587169223060749204617111e+0  	-2.27065687839824134714544377535276e-3
*string in : 	+1598.2201192361  -4321.89031229i
 z =  		1.59822011923609999999999999999999e+3  	-4.32189031228999999999999999999998e+3
 cxatan(z) = 	1.57072105637083424439663534844582e+0  	-2.03545495659669108264118083883059e-4

Test of cxatanh
*string in : 	0
 z =  		0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxatanh(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+2.3106e-1571
 z =  		2.31059999999999999999999999999571e-1571  	0.00000000000000000000000000000000e+0
 cxatanh(z) = 	0.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
*string in : 	+8.7792e-20  +9.7732e-20i
 z =  		8.77919999999999999999999999999979e-20  	9.77319999999999999999999999999977e-20
 cxatanh(z) = 	8.77920000000002239540770391152935e-20  	9.77319999999999999999999999999976e-20
*string in : 	+2.3298e-1571i
 z =  		0.00000000000000000000000000000000e+0  	2.32979999999999999999999999999567e-1571
 cxatanh(z) = 	0.00000000000000000000000000000000e+0  	2.32979999999999999999999999999567e-1571
*string in : 	-4.5592e-19  +8.2219e-18i
 z =  		-4.55919999999999999999999999999990e-19  	8.22189999999999999999999999999983e-18
 cxatanh(z) = 	-4.55920000000000700576349007141915e-19  	8.22189999999999999999999999999982e-18
*string in : 	-3.8932e-1624i
 z =  		0.00000000000000000000000000000000e+0  	-3.89319999999999999999999999999252e-1624
 cxatanh(z) = 	0.00000000000000000000000000000000e+0  	-3.89319999999999999999999999999252e-1624
*string in : 	-1.2291e-17  -7.3342e-18i
 z =  		-1.22909999999999999999999999999998e-17  	-7.33419999999999999999999999999984e-18
 cxatanh(z) = 	-1.22910000000000005523050294984054e-17  	-7.33419999999999999999999999999984e-18
*string in : 	-4.0987e-1617i
 z =  		0.00000000000000000000000000000000e+0  	-4.09869999999999999999999999999216e-1617
 cxatanh(z) = 	0.00000000000000000000000000000000e+0  	-4.09869999999999999999999999999216e-1617
*string in : 	+3.9917e-19  -8.2215e-19i
 z =  		3.99169999999999999999999999999991e-19  	-8.22149999999999999999999999999982e-19
 cxatanh(z) = 	3.99170000000000501804799064492576e-19  	-8.22149999999999999999999999999982e-19
*string in : 	+1.0
 z =  		1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxatanh(z) = 	*** Out of domain
*string in : 	+1.0000000000000000000986
 z =  		1.00000000000000000009860000000000e+0  	0.00000000000000000000000000000000e+0
 cxatanh(z) = 	2.22281814359131574745817670846975e+1  	1.57079632679489661923132169163975e+0
*string in : 	+1.0000000000000000004567 +2.43e-18i
 z =  		1.00000000000000000045670000000000e+0  	2.42999999999999999999999999999995e-18
 cxatanh(z) = 	2.06172155939038428023006696346281e+1  	8.78285802447720835331388461168277e-1
*string in : 	+1.0  +3.4506e-1298i
 z =  		1.00000000000000000000000000000000e+0  	3.45059999999999999999999999999470e-1298
 cxatanh(z) = 	1.49410502487893280835655959664095e+3  	7.85398163397448309615660845819876e-1
*string in : 	+0.999999999999999998736  +4.6677e-1297i
 z =  		9.99999999999999998735999999999976e-1  	4.66769999999999999999999999999284e-1297
 cxatanh(z) = 	2.09526987793640415331819968771055e+1  	1.84640031645566135297165130446374e-1279
*string in : 	+0.999999999999999999754
 z =  		9.99999999999999999753999999999976e-1  	0.00000000000000000000000000000000e+0
 cxatanh(z) = 	2.17710512987512230197936171369566e+1  	0.00000000000000000000000000000000e+0
*string in : 	+0.999999999999999999338  -5.0783e-1145i
 z =  		9.99999999999999999337999999999976e-1  	-5.07829999999999999999999999999312e-1145
 cxatanh(z) = 	2.12760842887489301782325007352235e+1  	-3.83557401812674972849900693074085e-1127
*string in : 	+1.0000000000000000000    -6.7002e-1301i
 z =  		1.00000000000000000000000000000000e+0  	-6.70019999999999999999999999998969e-1301
 cxatanh(z) = 	1.49722710789455785792235656962371e+3  	-7.85398163397448309615660845819876e-1
*string in : 	+1.0000000000000007423    -5.3321e-1344i
 z =  		1.00000000000000074230000000000000e+0  	-5.33209999999999999999999999999153e-1344
 cxatanh(z) = 	1.77649626901646333621481928702597e+1  	-1.57079632679489661923132169163975e+0
*string in : 	-1.0
 z =  		-1.00000000000000000000000000000000e+0  	0.00000000000000000000000000000000e+0
 cxatanh(z) = 	*** Out of domain
*string in : 	-0.99999999999999999998811
 z =  		-9.99999999999999999988109999999973e-1  	0.00000000000000000000000000000000e+0
 cxatanh(z) = 	-2.32858682113649983634186242480938e+1  	0.00000000000000000000000000000000e-4912
*string in : 	-0.99999999999999999976632 +2.3344e-15i
 z =  		-9.99999999999999999766319999999974e-1  	2.33439999999999999999999999999996e-15
 cxatanh(z) = 	-1.71920843358369800738620118292559e+1  	7.85348111992544101771280312905573e-1
*string in : 	-1.0  +4.5601e-997i
 z =  		-1.00000000000000000000000000000000e+0  	4.56009999999999999999999999999462e-997
 cxatanh(z) = 	-1.14742657017125654562372704756289e+3  	7.85398163397448309615660845819876e-1
*string in : 	-1.00000000000000000000654 +7.4332e-1532i
 z =  		-1.00000000000000000000654000000000e+0  	7.43319999999999999999999999998655e-1532
 cxatanh(z) = 	-2.35847484839829247718351637404736e+1  	1.57079632679489661923132169163975e+0
*string in : 	-1.00000000000000000000743
 z =  		-1.00000000000000000000743000000000e+0  	0.00000000000000000000000000000000e+0
 cxatanh(z) = 	-2.35209541373526413692370331966807e+1  	1.57079632679489661923132169163975e+0
*string in : 	-1.00000000000000000085023 -6.9983e-18i
 z =  		-1.00000000000000000085023000000000e+0  	-6.99829999999999999999999999999985e-18
 cxatanh(z) = 	-2.00933427516235702884811780714143e+1  	-8.45847380707912908444605951787500e-1
*string in : 	-1.0  -5.00098e-1023i
 z =  		-1.00000000000000000000000000000000e+0  	-5.00097999999999999999999999999395e-1023
 cxatanh(z) = 	-1.17731403171012003510438891005670e+3  	-7.85398163397448309615660845819876e-1
*string in : 	-0.9999999999999999997643  -6.7321e-16i
 z =  		-9.99999999999999999764299999999974e-1  	-6.73209999999999999999999999999987e-16
 cxatanh(z) = 	-1.78138107382650243066336763074388e+1  	-7.85223106587263424860231128405509e-1
*string in : 	+1.0i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000000000000000e+0
 cxatanh(z) = 	-1.39629920968074599346558400837407e-33  	7.85398163397448309615660845819876e-1
*string in : 	+5.6098e-2198 +1.0i
 z =  		5.60979999999999999999999999998542e-2198  	1.00000000000000000000000000000000e+0
 cxatanh(z) = 	-1.39629920968074599346558400837407e-33  	7.85398163397448309615660845819876e-1
*string in : 	+4.9987e-18   +1.0000000000000000005421i
 z =  		4.99869999999999999999999999999989e-18  	1.00000000000000000054210000000000e+0
 cxatanh(z) = 	2.49934999999999972394342329554194e-18  	7.85398163397448309886710845819876e-1
*string in : 	+1.000000000000000000003201i
 z =  		0.00000000000000000000000000000000e+0  	1.00000000000000000000320100000000e+0
 cxatanh(z) = 	-1.68518870133883137142398069976181e-33  	7.85398163397448309617261345819875e-1
*string in : 	-6.5541e-20   +1.0000000000000000080981i
 z =  		-6.55409999999999999999999999999984e-20  	1.00000000000000000809810000000000e+0
 cxatanh(z) = 	-3.27705000000016237899086049512593e-20  	7.85398163397448313664710845819876e-1
*string in : 	-7.0998e-2200 +1.0i
 z =  		-7.09979999999999999999999999998153e-2200  	1.00000000000000000000000000000000e+0
 cxatanh(z) = 	-1.39629920968074599346558400837407e-33  	7.85398163397448309615660845819876e-1
*string in : 	-6.8432e-17   +0.9999999999999999999864i
 z =  		-6.84319999999999999999999999999986e-17  	9.99999999999999999986399999999974e-1
 cxatanh(z) = 	-3.42160000000000009845740097075220e-17  	7.85398163397448309608860845819864e-1
*string in : 	+0.99999999999999989098732i
 z =  		0.00000000000000000000000000000000e+0  	9.99999999999999890987319999999974e-1
 cxatanh(z) = 	-1.54074395550978868244478235406794e-33  	7.85398163397448255109320845819859e-1
*string in : 	+4.3329e-16   +0.9999999999999999997877i
 z =  		4.33289999999999999999999999999992e-16  	9.99999999999999999787699999999974e-1
 cxatanh(z) = 	2.16644999999999999935868578720314e-16  	7.85398163397448309509510845819910e-1
*string in : 	-1.0i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000000000000000e+0
 cxatanh(z) = 	-1.39629920968074599346558400837407e-33  	-7.85398163397448309615660845819876e-1
*string in : 	+2.7762e-2111 -1.0i
 z =  		2.77619999999999999999999999999307e-2111  	-1.00000000000000000000000000000000e+0
 cxatanh(z) = 	-1.39629920968074599346558400837407e-33  	-7.85398163397448309615660845819876e-1
*string in : 	+3.3321e-17   -0.9999999999999999996555i
 z =  		3.33209999999999999999999999999993e-17  	-9.99999999999999999655499999999974e-1
 cxatanh(z) = 	1.66604999999999997678999550821064e-17  	-7.85398163397448309443410845819863e-1
*string in : 	-0.999999999999999999996999972013i
 z =  		0.00000000000000000000000000000000e+0  	-9.99999999999999999996999972012965e-1
 cxatanh(z) = 	-1.20370621524202240815998621411558e-33  	-7.85398163397448309614160831826358e-1
*string in : 	-7.5541e-17   -0.999999999999999999987231i
 z =  		-7.55409999999999999999999999999985e-17  	-9.99999999999999999987230999999972e-1
 cxatanh(z) = 	-3.77705000000000007331223526135875e-17  	-7.85398163397448309609276345819863e-1
*string in : 	-2.1156e-2203 -1.0i
 z =  		-2.11559999999999999999999999999449e-2203  	-1.00000000000000000000000000000000e+0
 cxatanh(z) = 	-1.39629920968074599346558400837407e-33  	-7.85398163397448309615660845819876e-1
*string in : 	-5.9923e-16   -1.00000000000000000089321i
 z =  		-5.99229999999999999999999999999988e-16  	-1.00000000000000000089321000000000e+0
 cxatanh(z) = 	-2.99615000000000000929807265234648e-16  	-7.85398163397448310062265845819965e-1
*string in : 	-1.000000000000000000007161i
 z =  		0.00000000000000000000000000000000e+0  	-1.00000000000000000000716100000000e+0
 cxatanh(z) = 	-1.68518870133883137142398069976181e-33  	-7.85398163397448309619241345819875e-1
*string in : 	+8.6631e-19   -1.00000000000000000008765i
 z =  		8.66309999999999999999999999999981e-19  	-1.00000000000000000008765000000000e+0
 cxatanh(z) = 	4.33154999999999179277255665234899e-19  	-7.85398163397448309659485845819876e-1
*string in : 	+1.5876e1876
 z =  		1.58759999999999999999999999999648e+1876  	0.00000000000000000000000000000000e+0
 cxatanh(z) = 	0.00000000000000000000000000000000e+0  	1.57079632679489661923132169163975e+0
*string in : 	-6.4391e1965
 z =  		-6.43909999999999999999999999998503e+1965  	0.00000000000000000000000000000000e+0
 cxatanh(z) = 	0.00000000000000000000000000000000e+0  	1.57079632679489661923132169163975e+0
*string in : 	+3.4210e2015i
 z =  		0.00000000000000000000000000000000e+0  	3.42099999999999999999999999999185e+2015
 cxatanh(z) = 	-1.54074395550978868244478235406794e-33  	1.57079632679489661923132169163975e+0
*string in : 	-4.5522e2011i
 z =  		0.00000000000000000000000000000000e+0  	-4.55219999999999999999999999998917e+2011
 cxatanh(z) = 	0.00000000000000000000000000000000e+0  	-1.57079632679489661923132169163975e+0
*string in : 	+2.1907e1854  +9.8765e1872i
 z =  		2.19069999999999999999999999999520e+1854  	9.87649999999999999999999999997812e+1872
 cxatanh(z) = 	-2.02222644160659764570877683971417e-33  	1.57079632679489661923132169163975e+0
*string in : 	-3.2966e1946  +7.5721e1834i
 z =  		-3.29659999999999999999999999999241e+1946  	7.57209999999999999999999999998357e+1834
 cxatanh(z) = 	-1.68518870133883137142398069976181e-33  	1.57079632679489661923132169163975e+0
*string in : 	-6.0987e2310  -5.4411e2198i
 z =  		-6.09869999999999999999999999998333e+2310  	-5.44109999999999999999999999998585e+2198
 cxatanh(z) = 	5.77778983316170755916793382775478e-34  	-1.57079632679489661923132169163975e+0
*string in : 	+7.8895e1854  -3.8699e1943i
 z =  		7.88949999999999999999999999998271e+1854  	-3.86989999999999999999999999999111e+1943
 cxatanh(z) = 	5.77778983316170755916793382775478e-34  	-1.57079632679489661923132169163975e+0
*string in : 	+10.009876543100091234
 z =  		1.00098765431000912340000000000000e+1  	0.00000000000000000000000000000000e+0
 cxatanh(z) = 	1.00235684096621237797404332236930e-1  	1.57079632679489661923132169163975e+0
*string in : 	+9.8980000000006542009 +6.11002230065009876i
 z =  		9.89800000000065420089999999999997e+0  	6.11002230065009876000000000000000e+0
 cxatanh(z) = 	7.31348877096854139391168185831522e-2  	1.52542686990082042846684858664691e+0
*string in : 	+17.9998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+1
 cxatanh(z) = 	0.00000000000000000000000000000000e+0  	1.51529747943119606874175664176563e+0
*string in : 	-12.9822227      +20.87777220192i
 z =  		-1.29822227000000000000000000000000e+1  	2.08777722019200000000000000000000e+1
 cxatanh(z) = 	-2.14565127610994940788120198681559e-2  	1.53625229115350613157340499562554e+0
*string in : 	-17.9877771265
 z =  		-1.79877771265000000000000000000000e+1  	0.00000000000000000000000000000000e+0
 cxatanh(z) = 	-5.56506850622370523498021431876020e-2  	1.57079632679489661923132169163975e+0
*string in : 	-19.09872210981  -32.9800000654i
 z =  		-1.90987221098100000000000000000000e+1  	-3.29800000653999999999999999999999e+1
 cxatanh(z) = 	-1.31433702306985834334020287120254e-2  	-1.54808971225646826710989597768030e+0
*string in : 	-43.987600182345i
 z =  		0.00000000000000000000000000000000e+0  	-4.39876001823449999999999999999999e+1
 cxatanh(z) = 	-1.15555796663234151183358676555096e-33  	-1.54806656259172870610183325706365e+0
*string in : 	+15.982201192361 -43.2189031229i
 z =  		1.59822011923610000000000000000000e+1  	-4.32189031228999999999999999999999e+1
 cxatanh(z) = 	7.52406669213765565629571163732678e-3  	-1.55044343538656476199191638232213e+0
*string in : 	+1000.009876543100091234
 z =  		1.00000987654310009123400000000000e+3  	0.00000000000000000000000000000000e+0
 cxatanh(z) = 	9.99990456878102024554013259226976e-4  	1.57079632679489661923132169163975e+0
*string in : 	+989.80000000006542009 +611.002230065009876i
 z =  		9.89800000000065420089999999999996e+2  	6.11002230065009875999999999999998e+2
 cxatanh(z) = 	7.31544466579880719007116353532427e-4  	1.57034474514132348069276899629488e+0
*string in : 	+179.998888123400000076i
 z =  		0.00000000000000000000000000000000e+0  	1.79998888123400000076000000000000e+2
 cxatanh(z) = 	5.77778983316170755916793382775478e-34  	1.56524079407787215602833401434861e+0
*string in : 	-1298.22227  +208.7777220192i
 z =  		-1.29822227000000000000000000000000e+3  	2.08777722019199999999999999999999e+2
 cxatanh(z) = 	-7.50865021422939229149509899485880e-4  	1.57067557401412464799794424281126e+0
*string in : 	-1798.777712650981
 z =  		-1.79877771265098099999999999999999e+3  	0.00000000000000000000000000000000e+0
 cxatanh(z) = 	-5.55933118354448461056500850195030e-4  	1.57079632679489661923132169163975e+0
*string in : 	-190.9872210981   -329.800700654i
 z =  		-1.90987221098100000000000000000000e+2  	-3.29800700653999999999999999999999e+2
 cxatanh(z) = 	-1.31492887963278047985975500210671e-3  	-1.56852566986914970477173882625079e+0
*string in : 	+1598.2201192361  -4321.89031229i
 z =  		1.59822011923609999999999999999999e+3  	-4.32189031228999999999999999999998e+3
 cxatanh(z) = 	7.52704181096553283288917272285539e-5  	-1.57059278130255255142495874029566e+0

*/
