/*
   Copyright (C)  2000    Daniel A. Atkinson
   Copyright (C)  2004    Ivano Primi  <ivano.primi@tin.it>    

   This file is part of the HPA Library.

   The HPA Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   The HPA Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the HPA Library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307 USA.
*/

#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h> /* for calloc() */
#include <string.h> /* for strcat() */
#include "config.h"
#include "xpre.h"

static double ltn2=.3010299956639812;
static int q[5*XDIM+4];

/* Source code added by Ivano Primi - 11/28/2004 */

static void fileputc (int c, FILE* stream)
{
  if ( fputc (c, stream) == EOF )
    {
      fprintf (stderr, "*** In file  \"%s\" of the HPA library\n", __FILE__); 
      fprintf (stderr, "    fileputc(): I/O Error on stream %p\n", stream);
    }        
}

static void printfmt (FILE* stream, const char* fmt, ...)
{
  va_list ap;

  va_start (ap, fmt);
  if ( vfprintf (stream, fmt, ap) < 0 )
    {
      fprintf (stderr, "*** In file  \"%s\" of the HPA library\n", __FILE__); 
      fprintf (stderr, "    printfmt(): I/O Error on stream %p\n", stream);
    }
  va_end (ap);
}

/* Be really careful to the way you employ this function ! */

static void sprintfmt (char* buffer, const char* fmt, ...)
{
  char ibuff[1024];
  va_list ap;

  va_start (ap, fmt);
  vsprintf (ibuff, fmt, ap);
  va_end (ap);
  strcat (buffer, ibuff);
}

/* sc_not != 0 tells using scientific notation                     */
/* sign   != 0 tells putting a '+' sign before non negative values */

void print_xpr (FILE* stream, struct xpr u, 
		int sc_not, int sign, int lim)
{
  int *p=q, k, m;
  unsigned short *pa=(unsigned short *)&u;
  
  if ( lim < 0 )
    lim = 0;
  if ( lim >= 5*XDIM+2 )
    lim = 5*XDIM+2;
  if((*pa&m_sgn))
    { 
      *pa^=m_sgn; 
      printfmt (stream, " -");
    } 
  else
    {
      if ((sign))
	printfmt (stream, " +");
      else
	printfmt (stream, "  ");
    }
  if(*pa==0)
    { 
      printfmt (stream, "0."); 
      for(k=0; k<lim ;++k) 
	printfmt (stream, "0");
      if ( (sc_not) )
	printfmt (stream, "e+0\n"); 
    }
  else
    { 
      *p=0; m=((*pa&m_exp)-bias); 
      m=(int)((double)(m+1)*ltn2);
      if(m) u=xmul(u,xpwr(ten,-m));
      while((*pa&m_exp)<bias)
	{ 
	  --m; 
	  u=xmul(u,ten);
	}
      for(k=0; k<=lim ;++k)
	{
	  u=sfmod(u,++p); 
	  if(*pa==0) break; 
	  u=xmul(ten,u); 
	}
      for(; k<=lim ;++k) 
	*++p=0;
      if(*pa)
	{ 
	  u=sfmod(u,&k); 
	  if(k>=5) ++(*p);
	  while(*p==10)
	    { 
	      *p=0; 
	      ++(*--p);
	    } 
	}
      p=q; 
      if (*p==0)  
	++p;
      else
	++m;
      /* Now has come the moment to print */
      if ( (sc_not) )
	{
	  printfmt (stream, "%d.",*p++);
	  for(k=0; k<lim ;++k) 
	    printfmt (stream, "%d",*p++);
	  if(m>=0) 
	    printfmt (stream, "e+%d\n",m); 
	  else 
	    printfmt (stream, "e%d\n",m);
	}
      else
	{
	  if ( m >= 0 )
	    {
	      for (k = 0; k <= m; k++)
		{
		  if ( k <= lim )
		    printfmt (stream, "%d", p[k]);
		  else
		    printfmt (stream, "0");
		}
	      if ( k <= lim )
		{
		  printfmt (stream, ".");
		  for (; k<= lim; k++)
		    printfmt (stream, "%d", p[k]);
		}
	      printfmt (stream, "\n");
	    }
	  else
	    {
	      printfmt (stream, "0.");
	      for (k=1; k< -m; k++)
		printfmt (stream, "0");
	      for(k=0; k <= lim ;++k) 
		printfmt (stream, "%d",*p++);
	      printfmt (stream, "\n");
	    }
	}
    } /* End of *pa != 0 */
}

#define BUFF_SIZE 5120 /* 5 Kb */

char* asprint_xpr (struct xpr u, int sc_not, int sign, int lim)
{
  int *p=q, k, m;
  unsigned short *pa=(unsigned short *)&u;
  char *buffer, *ptr;

  if ( lim < 0 )
    lim = 0;
  if ( lim > 5*XDIM+2 )
    lim = 5*XDIM+2;
  if ( !(buffer = (char*)calloc (BUFF_SIZE, sizeof(char))) )
    return NULL;
  else
    {
      if((*pa&m_sgn))
	{ 
	  *pa^=m_sgn; 
	  sprintfmt (buffer, "-");
	} 
      else
	{
	  if ((sign))
	    sprintfmt (buffer, "+");
	}
      if(*pa==0)
	{ 
	  sprintfmt (buffer, "0."); 
	  for(k=0; k<lim ;++k) 
	    sprintfmt (buffer, "0");
	  if ( (sc_not) )
	    sprintfmt (buffer, "e+0"); 
	}
      else
	{ 
	  *p=0; m=((*pa&m_exp)-bias); 
	  m=(int)((double)(m+1)*ltn2);
	  if(m) u=xmul(u,xpwr(ten,-m));
	  while((*pa&m_exp)<bias)
	    { 
	      --m; 
	      u=xmul(u,ten);
	    }
	  for(k=0; k<=lim ;++k)
	    {
	      u=sfmod(u,++p); 
	      if(*pa==0) break; 
	      u=xmul(ten,u); 
	    }
	  for(; k<=lim ;++k) 
	    *++p=0;
	  if(*pa)
	    { 
	      u=sfmod(u,&k); 
	      if(k>=5) ++(*p);
	      while(*p==10)
		{ 
		  *p=0; 
		  ++(*--p);
		} 
	    }
	  p=q; 
	  if (*p==0)  
	    ++p;
	  else
	    ++m;
	  /* Now has come the moment to print */
	  if ( (sc_not) )
	    {
	      sprintfmt (buffer, "%d.",*p++);
	      for(k=0; k<lim ;++k) 
		sprintfmt (buffer, "%d",*p++);
	      if(m>=0) 
		sprintfmt (buffer, "e+%d",m); 
	      else 
		sprintfmt (buffer, "e%d",m);
	    }
	  else
	    {
	      if ( m >= 0 )
		{
		  for (k = 0; k <= m; k++)
		    {
		      if ( k <= lim )
			sprintfmt (buffer, "%d", p[k]);
		      else
			sprintfmt (buffer, "0");
		    }
		  if ( k <= lim )
		    {
		      sprintfmt (buffer, ".");
		      for (; k<= lim; k++)
			sprintfmt (buffer, "%d", p[k]);
		    }
		}
	      else
		{
		  sprintfmt (buffer, "0.");
		  for (k=1; k < -m; k++)
		    sprintfmt (buffer, "0");
		  for(k=0; k <= lim ;++k) 
		    sprintfmt (buffer, "%d",*p++);
		}
	    }
	} /* End of *pa != 0 */
      for (k=0; buffer[k] != '\0'; k++);
      /* Now k is the length of the buffer. */
      /* We shrink the buffer so that it has the exact */
      /* size to contain all its non null chars.       */
      ptr = (char*) realloc (buffer, k+1);
      return (ptr != NULL) ? ptr : buffer;
    } /* End of buffer != 0 */
}

char* xtoa (struct xpr u,int lim)
{
  return asprint_xpr (u, 1, 0, lim);
}

void bxprint (FILE* stream, struct xpr u)
{
  int i;
  unsigned short n;

  for (n = 0x8000; n != 0x0; n >>= 1)
    {
      if ( (n & u.nmm[0]) )
	fileputc ('1', stream);
      else
	fileputc ('0', stream);
    }
  fileputc ('.', stream);
  for(i=0; i<XDIM ;++i)
    {
      for (n = 0x8000; n != 0x0; n >>= 1)
	{
	  if ( (n & u.nmm[i+1]) )
	    fileputc ('1', stream);
	  else
	    fileputc ('0', stream);
	}
      fileputc (' ', stream);
    }
}

/* End Additions */

void prxpr(struct xpr u,int lim)
{
  /* Modified by Ivano Primi - 11/29/2004 */ 
  print_xpr (stdout, u, 1, 0, lim);
}

void xprint(struct xpr u)
{ 
  int i;
  
  printf("%04x.",u.nmm[0]);
  for(i=0; i<XDIM ;++i) printf("%04x",u.nmm[i+1]);
  printf("\n");
}
