;;; tts-testing.el -- Regression/examples for tts parser

;; Copyright (C) 2005 Joe Corneli <jcorneli@math.utexas.edu>

;; Time-stamp: <jac -- Sun Jun 26 10:15:17 CDT 2005>

;; This file is not part of GNU Emacs, but it is distributed under
;; the same terms as GNU Emacs.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published
;; by the Free Software Foundation; either version 2, or (at your
;; option) any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary

;;; Code:

(require 'tts)

(defvar tts-testing-verbose nil)

(defvar tts-testing-save-excursion nil)

;; add appropriate case behavior for different cases
;; of `tts-testing-verbose', and set it in whatever
;; functions need verbose output.
(defun test-fun (test obj &optional goal)
  (let* ((fun (cond (t
                     'tts-parse-string)))
         (output (funcall fun obj))
         (buf (current-buffer)))
    (set-buffer (get-buffer-create "*Testing*"))
    (cond ((eq test 'fails)
           (unless (equal output goal)
             (pop-to-buffer (get-buffer-create "*Testing*"))
             (goto-char (point-max))
             (insert
              (concat
               "FAIL: "
               (format "%s" obj)
               "\n ==> "
               (format "%S" output)
               "\n >>> "
               (format "%S" goal)
               "\n\n"))))
          (t
           (pop-to-buffer (get-buffer-create "*Testing*"))
           (goto-char (point-max))
           (insert
            (concat
             (format "%s" obj)
             "\n ==> "
             (format "%S" output)
             "\n\n"))))
    (if tts-testing-save-excursion
        (switch-to-buffer-other-window buf))))

(defun arithmetic-regression (&optional test)
  (let ((test (or test
                  'fails))
        (tts-testing-save-excursion nil))
    (test-fun test "1*2*3" '(* 1 2 3))
    (test-fun test "1*2+3" '(+ (* 1 2) 3))
    (test-fun test "1+2*3" '(+ 1 (* 2 3)))
    (test-fun test "(1+2)*3" '(* (+ 1 2) 3))
    (test-fun test "1+2*3*4" '(+ 1 (* 2 3 4)))
    (test-fun test "1+(2*3)+4" '(+ 1 (* 2 3) 4))
    (test-fun test "(1+2)*3" '(* (+ 1 2) 3))
    (test-fun test "1+2*3" '(+ 1 (* 2 3)))
    (test-fun test "1+(2+3)" '(+ 1 (+ 2 3)))
    (test-fun test "1*2+3" '(+ (* 1 2) 3))
    (test-fun test "1+2*3" '(+ 1 (* 2 3)))
    (test-fun test "1*2+3*4=2+12=14" '(eq (+ (* 1 2) (* 3 4)) (+ 2 12) 14))
    (test-fun test "2+3=1+4=5" '(eq (+ 2 3) (+ 1 4) 5))
    (test-fun test "2+3=5=7-2" '(eq (+ 2 3) 5 (- 7 2)))
    (test-fun test "2+3=7-2" '(eq (+ 2 3) (- 7 2)))
    (test-fun test "2+7-2" '(+ 2 (- 7 2)))
    (test-fun test "2+7-2+3" '(+ 2 (- 7 2) 3))
    (test-fun test "-2" '-2)
    (test-fun test "3" '3)
    (test-fun test "-2+3" '(+ -2 3))
    (test-fun test "3-2" '(- 3 2))))

(defun algebra-regression (&optional test)
  (let ((test (or test
                  'fails))
        (tts-testing-save-excursion nil))
    ;; note that for "normal algebra", typically
    ;; we don't have explicit *'s in there... right?
    (test-fun test "abc*xyz*\\alpha\\beta\\gamma"
              '(* (*~ a b c) (*~ x y z) (*~ alpha beta gamma)))
    (test-fun test "abc*xyz+\\alpha\\beta\\gamma"
              '(+ (* (*~ a b c) (*~ x y z)) (*~ alpha beta gamma)))
    (test-fun test "abc+xyz*\\alpha\\beta\\gamma"
              '(+ (*~ a b c) (* (*~ x y z) (*~ alpha beta gamma))))
    (test-fun test "(abc+xyz)*\\alpha\\beta\\gamma"
              '(* (+ (*~ a b c) (*~ x y z)) (*~ alpha beta gamma)))
    ;; problem here...
    (test-fun test "abc+xyz*\\alpha\\beta\\gamma*123"
              '(+ (*~ a b c) (* (*~ x y z) (*~ alpha beta gamma) 123)))
    (test-fun test "abc+(xyz*\\alpha\\beta\\gamma)+123"
              '(+ (*~ a b c) (* (*~ x y z) (*~ alpha beta gamma)) 123))
    (test-fun test "(abc+xyz)*\\alpha\\beta\\gamma"
              '(* (+ (*~ a b c) (*~ x y z)) (*~ alpha beta gamma)))
    (test-fun test "abc+xyz*\\alpha\\beta\\gamma"
              '(+ (*~ a b c) (* (*~ x y z) (*~ alpha beta gamma))))
    (test-fun test "abc+(xyz+\\alpha\\beta\\gamma)"
              ;; Even though it might be good to get rid of the
              ;; parens, the parens are respected in the arithmetic
              ;; case above, so I guess it is OK to respect them here.
              ;; (?)
              '(+ (*~ a b c) (+ (*~ x y z) (*~ alpha beta gamma))))
    (test-fun test "abc*xyz+\\alpha\\beta\\gamma"
              '(+ (* (*~ a b c) (*~ x y z)) (*~ alpha beta gamma)))
    (test-fun test "abc+xyz*\\alpha\\beta\\gamma"
              '(+ (*~ a b c) (* (*~ x y z) (*~ alpha beta gamma))))
    ;; note that it is a little weird to allow "abc123"... usually
    ;; people would write "123abc".
    (test-fun test "abc*xyz+\\alpha\\beta\\gamma*123=xyz+abc=abc123"
              '(eq (+ (* (*~ a b c) (*~ x y z))
                      (* (*~ alpha beta gamma) 123))
                   (+ (*~ x y z) (*~ a b c))
                   (*~ a b c 123)))
    (test-fun test "xyz+\\alpha\\beta\\gamma=abc+123=\\lambda\\mu\\nu"
              '(eq 
                (+ (*~ x y z) (*~ alpha beta gamma))
                (+ (*~ a b c) 123)
                (*~ lambda mu nu)))
    (test-fun test "xyz+\\alpha\\beta\\gamma=\\lambda\\mu\\nu=f(x)g(x)h(x)-xyz"
              '(eq (+ (*~ x y z) (*~ alpha beta gamma))
                   (*~ lambda mu nu) 
                   (- (*~ (f x) (g x) (h x))
                      (*~ x y z))))
    (test-fun test "xyz+\\alpha\\beta\\gamma=f(x)g(x)h(x)-xyz"
              '(eq 
                (+ (*~ x y z) (*~ alpha beta gamma))
                (- (*~ (f x) (g x) (h x))
                   (*~ x y z))))
    (test-fun test "xyz+f(x)g(x)h(x)-xyz"
              '(+ (*~ x y z) 
                  (- (*~ (f x) (g x) (h x)) 
                     (*~ x y z))))
    (test-fun test "xyz+f(x)g(x)h(x)-xyz+\\alpha\\beta\\gamma"
              '(+ (*~ x y z)
                  (- (*~ (f x) (g x) (h x)) 
                     (*~ x y z)) 
                  (*~ alpha beta gamma)))
    (test-fun test "-xyz"
              '(- (*~ x y z)))
    (test-fun test "\\alpha\\beta\\gamma"
              '(*~ alpha beta gamma))
    (test-fun test "-xyz+\\alpha\\beta\\gamma"
              '(+ (- (*~ x y z))
                  (*~ alpha beta gamma)))
    (test-fun test "\\alpha\\beta\\gamma-xyz"
              '(- (*~ alpha beta gamma) (*~ x y z)))))

(defun parser-regression (&optional test)
  (interactive)
  (let ((test (or test
                  'fails))
        (tts-testing-save-excursion nil))
    (pop-to-buffer (get-buffer-create "*Testing*"))
    (goto-char (point-max))
    (insert-char (string-to-char "=") 70)
    (insert "\n\n")
    (test-fun test "a" 'a)
    (test-fun test "\\wedge" 'and)
    (test-fun test "\\neg" 'not)
    (test-fun test "*" '*)
    (test-fun test "a \\wedge b" '(and a b))
    (test-fun test "\\neg q" '(not q))
    (test-fun test "\\neg a \\wedge b" '(and (not a) b))
    (test-fun test "(p \\wedge (p \\rightarrow q)) \\rightarrow q"
              '(if (and p (if p q)) q))
    (test-fun test "(a, b, c, d, e)"
              '(tuple a b c d e))
    (test-fun test "abcde"
              '(*~ a b c d e))
    (test-fun test "a+b+c+d+e"
              '(+ a b c d e))
    (test-fun test "f(x)"
              '(f x))
    (test-fun test "(R,*,+) \\rightarrow ring"
              '(if (tuple R * +) (*~ r i n g)))
    (test-fun test "(a,b,(c,d,e))"
              '(tuple a b (tuple c d e)))
    (test-fun test "((a,b,c),d,e)"
              '(tuple (tuple a b c) d e))
    (test-fun test "(a \\wedge b) \\vee c"
              '(or (and a b) c))
    (test-fun test "a \\wedge (b \\vee c)"
              '(and a (or b c)))
    (test-fun test "a \\wedge b \\vee c"
              '(and a (or b c)))
    (test-fun test "a \\vee b \\wedge c"
              '(or a (and b c)))
    ;; This parses to (+ (- a) b), which is a little "annoying", but
    ;; it isn't a bad thing.  Something like Kyle's simplifier can
    ;; run on the result of parses to post-process, if that seems
    ;; like a good idea.
    (test-fun test "-a+b"
              '(+ (- a) b))
    (test-fun test "b-a"
              '(- b a))
    (test-fun test "a*b=b*a"
              '(eq (* a b) (* b a)))
    (test-fun test "\\forall x\\in X: P(x)"
              '(forall x :in X :st (P x)))
    (test-fun test "\\forall a,b\\in G: a*b=b*a"
              '(forall a b :in G :st (eq (* a b) (* b a))))
    (test-fun test "\\forall x\\in X: P(x)"
              '(forall x :in X :st (P x)))
    (test-fun test "\\forall y \\in f(X) : P(y)"
              '(forall y :in (f X) :st (P y)))
    (test-fun test "\\exists x: Q(x)"
              '(exists x :st (Q x)))
    (test-fun test "\\forall x: \\exists y: P(x,y)"
              '(forall x :st (exists y :st (P (tuple x y)))))
    (test-fun test "\\forall x\\in X: \\exists y\\in f(X): P(x,y)"
              '(forall x :in X :st (exists y :in (f X) :st (P (tuple x y)))))
    (test-fun test "f: X \\rightarrow X"
              '(map f X X))
    (test-fun test "g(x): X \\rightarrow Y"
              '(map (g x) X Y))
    (test-fun test "x^2"
              '(pow x 2))
    ;; working with curly brackets will require some small changes to
    ;; `parse-parens' versus what we've seen so far, specifically,
    ;; we'll want to know what "type" of paren to look for.
    (test-fun test "x_{23}"
              '(sub x 23))
    (test-fun test "y_z + 23"
              '(+ (sub y z) 23))
    (test-fun test "x^{y_z + 23}"
              '(pow x (+ (sub y z) 23)))
    (test-fun test "Y \\trianglelefteq X"
              '(normal-subgroup Y X))
    (test-fun test "x \\in X"
              '(elt x X))
    ;; this involves a different sort of tokenization from what we've
    ;; been used to.
    (test-fun test "a=Ya"
              '(eq a (*~ Y a)))
    (test-fun test "aY=Ya"
              '(eq (*~ a Y) (*~ Y a)))
    (test-fun test "f(x)=g(x)"
              '(eq (f x) (g x)))
    (test-fun test "(x)(y)"
              '(*~ x y))
    (test-fun test "(ab)H"
              '(*~ (*~ a b) H))
    (test-fun test "aH\\circ bH"
              '(circ (*~ a H) (*~ b H)))
    (test-fun test "f_1 =f_2"
              '(eq (sub f 1) (sub f 2)))
    ;; trying the contents of one relatively simple-looking
    ;; definition
    (test-fun test "(X,M)"
              '(tuple X M))
    (test-fun test "B\\cap A =\\phi"
              '(eq (intersect B A) emptyset))
    (test-fun test "\\mu_1"
              '(sub mu 1))
    (test-fun test "f_1 (E)"
              '((sub f 1) E))
    (test-fun test "\\mu_1 (E)"
              '((sub mu 1) E))
    (test-fun test "\\mu_1 (A\\cap E)"
              '((sub mu 1) (intersect A E)))
    (test-fun test "f (E)=f (A\\cap E)"
              '(eq (f E) (f (intersect A E))))
    (test-fun test "f (E)=f_1 (A\\cap E)"
              '(eq (f E) ((sub f 1) (intersect A E))))
    (test-fun test "\\forall E \\in M: \\mu_1 (E)=\\mu_1 (A\\cap E)"
              '(forall E :in M :st (eq ((sub mu 1) E) ((sub mu 1) (intersect A E)))))
    (test-fun test "a_kx^k"
              '(*~ (sub a k) (pow x k)))
    (test-fun test "q(x)=f(g(h(x)))"
              '(eq (q x) (f (g (h x)))))
    (test-fun test "(x-1)(x+1)" 
              '(*~ (- x 1) (+ x 1)))
    (test-fun test "f(x)=xe^{-x}"
              '(eq (f x) (*~ x (pow e (- x)))))
    (test-fun test "f(x)=e^{-x}x"
              '(eq (f x) (*~ (pow e (- x)) x)))
    (test-fun test "bH=(ab)H"
              '(eq (*~ b H) (*~ (*~ a b) H)))
    ))

;; Note: there was something on the wiki (maybe discussed in email)
;; that didn't come out properly in previous testing.  I'm not
;; sure which item it was?
(defun parser-testing (&optional test)
  (interactive "P")
  (pop-to-buffer (get-buffer-create "*Testing*"))
  (goto-char (point-max))
  (insert "\n")
  (insert-char (string-to-char "=") 70)
  (insert "\n\n")
  (let ((test (or test
                  'print))
        (tts-testing-save-excursion nil))
    ;; selective regressions
    ;; things that "work" but either need post-processing
    ;; or parser adjustments to get into optimal form.
;    (test-fun test "\\forall y \\in f(X) : P(y)")
;    (test-fun test "abc")
;    (test-fun test "abcde")
;    (test-fun test "a+b+c+d+e")
;    (test-fun test "p(x)=x^2+Bx+C")
;    (test-fun test "a_kx^k")
;    (test-fun test "a_kx^ky^k")
    (test-fun test "e^{-x}x")
    ;; legitimately broken cases
;     (test-fun test "f(x)=e^{-x}\\sum_{k=0}^n k^2")
;    (test-fun test "f(x)=e^{-x}x")
;    (test-fun test "f(x)=e^xx")
;     (test-fun test "f(x)=-\\sum_{k=0}^n k^2")
;     (test-fun test "\\sum_{k=0}^na_kx^k")
    ;; needs major adjustments.
;     (test-fun test "f(\\alpha_1 m_1+\\alpha_2m_2,n)=\\alpha_1 f(m_1,n)+\\alpha_2f(m_2,n)")
    ;; these haven't been attempted yet.
;     (test-fun test "\\mu_1, \\mu_2")
;     (test-fun test "A,B \\in M")
    ))

(defun parser-full-regression ()
  (interactive)
  (pop-to-buffer (get-buffer-create "*Testing*"))
  (erase-buffer)
  (parser-regression 'fail)
  (arithmetic-regression 'fail)
  (algebra-regression 'fail)
  (parser-testing 'print))

(defun parser-lite-regression ()
  (interactive)
  (pop-to-buffer (get-buffer-create "*Testing*"))
  (erase-buffer)
  (parser-regression 'fails)
  (parser-testing 'print))

(defun parser-exhibit ()
  (interactive)
  (pop-to-buffer (get-buffer-create "*Testing*"))
  (erase-buffer)
  (parser-regression 'print)
  (arithmetic-regression 'print)
  (parser-testing 'print))

;;; tts-testing.el ends here
