;;; tts-symbols.el -- symbols recognized by the tts parser

;; Copyright (C) 2005 Joe Corneli <jcorneli@math.utexas.edu>

;; Time-stamp: <jac -- Sun Jun 26 10:14:48 CDT 2005>

;; This file is not part of GNU Emacs, but it is distributed under
;; the same terms as GNU Emacs.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published
;; by the Free Software Foundation; either version 2, or (at your
;; option) any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary

;;; Code:

(defun just-a-symbol-p (token)
  (when (and ; (eq (length token) 1)
             (not (tts-matching-paren token))
             (not (tts-special-symbol token))
             (not (operator token))
             (not (greedy-operator token)))
    t))

(defun tts-matching-paren (paren)
  (cond
   ((equal paren "(") ")")
   ((equal paren "{") "}")
   ((equal paren "[") "]")
   ;; I don't know if these are all for real, or what, but
   ;; hey, here they are (and more can be added too).
   ((equal paren "\\(") "\\)")
   ((equal paren "\\{") "}")
   ((equal paren "\\[") "\\]")
   ;; backward?
   ((equal paren ")") "(")
   ((equal paren "}") "{")
   ((equal paren "]") "[")
   ((equal paren "\\)") "\\(")
   ((equal paren "\\}") "\\{")
   ((equal paren "\\]") "\\[")
   (t nil)))

(defun tts-special-symbol (token)
  (member token '("," "." "_" "^" ":" "=")))

;; maybe this should be merged with the special symbol function above?
(defun tts-greek (token)
  (cdr (assoc token '(("\\alpha"   . alpha)
                      ("\\beta"    . beta)
                      ("\\gamma"   . gamma)
                      ("\\delta"   . delta)
                      ("\\epsilon" . epsilon)
                      ("\\eta"     . eta)
                      ("\\theta"   . theta)
                      ("\\zeta"    . zeta)
                      ("\\iota"    . iota)
                      ("\\kappa"   . kappa)
                      ("\\lambda"  . lambda)
                      ("\\mu"      . mu)
                      ("\\nu"      . nu)
                      ("\\xi"      . xi)
                      ("\\chi"     . chi)
                      ("\\pi"      . pi)
                      ("\\varphi"  . varphi)
                      ("\\rho"     . rho)
                      ("\\sigma"   . sigma)
                      ("\\tau"     . tau)
                      ("\\upsilon" . upsilon)
                      ("\\omega"   . omega)
                      ;; "extended greek"
                      ("\\infty"   . infty)
                      ("\\phi"     . emptyset)
                      ))))


;; Note that if you leave off the extra backslash in your string, bad
;; things happen in the parse.  This warrants some extra testing.
;;
;; Note that these operators have no explicit precidence, and if they
;; are not explicitly parenthesized, they end up being treated as if
;; their precedence increases from left to right, e.g.
;; a \vee b \wedge c ==> (or a (and b c))
;; a \wedge b \vee c ==> (and a (or b c))
;;
;; Note that there can be different interpretations of different
;; symbols depending on the branch of mathematics that is being talked
;; about.  Maybe we should call these things `logic-operator', not
;; just `operator', then call the function depending on context,
;; or just at a different place in the order, or whatever.
(defun operator (token)
  (cdr (assoc token '(("\\rightarrow"     . if)
                      ("\\Rightarrow"     . implies)
                      ("\\leftrightarrow" . iff)
                      ("\\Leftrightarrow" . iff)
                      ("\\wedge"          . and)
                      ("\\vee"            . or)
                      ("_"                . sub)
                      ("^"                . pow)
                      ("\\subset"         . subset)
                      ("\\supset"         . supset)
                      ("\\trianglelefteq" . normal-subgroup)
                      ("\\equiv"          . equiv)
                      ("\\cong"           . cong)
                      ;; some of these are a bit dubiuous
                      ("\\triangle"       . laplacian)
                      ("\\nabla"          . nabla)
                      ("\\partial"        . partial)
                      ("\\int"            . int)
                      ("\\sub"            . sub)
                      ("\\in"             . elt)
                      ("\\cdot"           . cdot)
                      ))))

;; these ones want more... like little otiks, hungry.
;;
;; It seems a bit odd to always parse things from the left, why do we
;; do that?  It seems like we might just divide things up into
;; clusters.
;;
;; Maybe this function can be used to determine the relative priority
;; of two given operators?  Or maybe we want a different function to
;; do that.
;;
;; does it really make sense to treat `-' and `/' as greedy operators?
;; well, it seems to work, but then it does make it seem a little odd
;; to also treat `-' as a modifier.
;;
;; More testing of these curious operators is in order!
(defun greedy-operator (token)
  (cdr (assoc token '(("*"      . *)
                      ("+"      . +)
                      ("="      . eq)
                      ("-"      . -)
                      ("/"      . /)
                      ("\\circ" . circ)
                      ("\\cap"  . intersect)
                      ("\\cup"            . union)
                      ;; we'll need some logic to deal with
                      ;; combinations of =, leq, and geq within
                      ;; the same expression.
                      ("\\leq"  . leq)
                      ("\\geq"  . geq)))))

(defun modifier (token)
  (cdr (assoc token '(("\\neg" . not)
                      ("\\pm"  . pm)
                      ("-"     . -)))))

(defun quantifier (token)
  (cdr (assoc token '(("\\forall"  . forall)
                      ("\\exists"  . exists)))))

;; this could be made more fancy if anyone cares.
(defun number (token)
  (with-temp-buffer
    (insert token)
    (goto-char (point-min))
    (when (looking-at "[0-9]+")
      (string-to-int (match-string-no-properties 0)))))

;; higher numbers denote a lower priority -- this is consistent
;; with day-to-day usage "my number one priority is bla".
;;
;; There is no "zeroth" priority.
(defun priority (operator)
  ;; we could also write a function that would really
  ;; give certain operators the *same* priority (if that seemed
  ;; to be a good idea).
  (let ((len (length
              (member operator 
                      '(geq leq eq circ intersect union + - * *~)))))
  (when (> len 0)
    len)))

(defun highest-priority-operator (op1 op2)
  (let ((p1 (priority op1))
        (p2 (priority op2)))
    ;; require that the objects have a non-0 priority
    (when (and (> p1 0)
               (> p2 0))
      (if (< p1 p2)
          op1
        op2))))

(provide 'tts-symbols)
;;; tts-symbols.el ends here
