;;; tts-greedy.el -- help `parser' with greedy operators

;; Copyright (C) 2005 Joe Corneli <jcorneli@math.utexas.edu>

;; Time-stamp: <jac -- Sun Jun 26 10:14:05 CDT 2005>

;; This file is not part of GNU Emacs, but it is distributed under
;; the same terms as GNU Emacs.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published
;; by the Free Software Foundation; either version 2, or (at your
;; option) any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary

;;; Code:

(defun parser-greedy-operator-case ()
  (let* ((continuation (parser-1 (cdr tokens) 
                                (greedy-operator
                                 (car tokens))))
         (new-parens (third continuation))
         extension)
    ;; we do some gymnastics to support the connection between
    ;; "f" and "(x)" in "f(x)".
    ;;
    ;; specifically, this is currently needed for $f(E)=f_1(A\cap E)$
    ;; but I don't like it, and I think this should be handled by
    ;; `parser-1'.  What about more continuations, like $f(g(h(x)))$?
    ;; Remarkably, this seems to work.
    (when (equal (car (second continuation)) "(")
      (setq extension (parser-1 (second continuation))
            continuation (cons (list (first continuation)
                                     (first extension))
                               (second extension))))
    (cond
     ;; treat a parenthesized chunk the same as we treat the case in
     ;; which waiting is not headed by a greedy operator, i.e., just
     ;; as something that the operator should act on. (But compare
     ;; `greedy-default-waiting-case'...)
     (parens (greedy-parens-case))
     ;; we have something waiting and it is headed by the greedy
     ;; operator symbol we're working on.
     ((and waiting
           (listp waiting)
           (eq (car waiting) (greedy-operator (car tokens))))
      (greedy-same-operator-waiting-case))
     ;; something is waiting but it is headed by a *different* greedy
     ;; operator.
     ((and waiting (listp waiting) (priority (car waiting)))
      (greedy-some-other-operator-waiting-case))
     ;; we have something waiting, but it is not a greedy-operator
     ;; headed list.
     (waiting (greedy-default-waiting-case))
     ;; is a silly error?
     (t (error "Trailing operator")))))

;; Designed so that e.g. $f(x)=f(y)$ goes to (eq (f x) (f y)).
(defun greedy-parens-case ()
  (setq waiting (append
                 (list
                  (greedy-operator (car tokens))
                  waiting)
                 (list (first continuation)))
        tokens (second continuation)
        tree (parser tokens tree waiting)))

;; we get the first complete element and add it to whatever is waiting
(defun greedy-same-operator-waiting-case ()
  (setq waiting (append
                 waiting
                 (list (first continuation)))
        tokens (second continuation)
        tree (parser tokens tree waiting)))

(defun greedy-some-other-operator-waiting-case ()
  (let ((high-pri (highest-priority-operator
                   (car waiting)
                   (greedy-operator
                    (car tokens)))))
    ;; build a new waiting structure using the current greedy
    ;; operator, whatever was waiting already, and the next object as
    ;; returned by `parser-1'.
    ;;
    ;; the way we construct this new waiting structure depends on the
    ;; relative priorities of the two operators we've found
    (cond 
     ;; if what was *waiting* has higher priority, we begin a new
     ;; level of parenthetization, headed by the new operator
     ((eq high-pri (car waiting))
      (setq waiting (append
                     (list
                      (greedy-operator (car tokens))
                      waiting)
                     (list (first continuation)))
            tokens (second continuation)
            tree (parser tokens tree waiting)))
     ;; if the *new thing* has higher priority, then begin a new level
     ;; of parenthetization, headed by the *old* operator
     ((eq high-pri (greedy-operator (car tokens)))
      (setq waiting (append
                     (butlast waiting 1)
                     (list
                      (append
                       (list
                        (greedy-operator (car tokens))
                        (car (last waiting)))
                       (list (first continuation)))))
            tokens (second continuation)
            tree (parser tokens tree waiting))))))

;; E.g. for a+b+c+d+e, we have (b c d e) waiting, but for f(x)=g(x),
;; we would have (f x) waiting.  So, we need to be a little bit
;; careful...  to distinguish between these cases.  (Is what we do
;; here inconsistent with the "gymnastics" above?  I think so, and
;; that's another reason to get that other stuff supported elsewhere.)
(defun greedy-default-waiting-case ()
  (setq waiting  (cond 
                  ((or (eq new-parens 'function)
                       (eq new-parens 'implicit))
                   (list 
                    (greedy-operator (car tokens))
                    waiting
                    (first continuation)))
                  (t
                   (cons
                    (greedy-operator (car tokens))
                    (protect-append waiting
                                    (first continuation)))))
        tokens  (second continuation)
        tree (parser tokens tree waiting)))

(provide 'tts-greedy)
;;; tts-greedy.el ends here
