;;; generic-browser.el -- generic history browsing mechanism

;; Copyright (C) 2005 Joe Corneli <jcorneli@math.utexas.edu>

;; Time-stamp: <jac -- Sat Jun 25 10:19:56 CDT 2005>

;; This file is not part of GNU Emacs, but it is distributed under
;; the same terms as GNU Emacs.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published
;; by the Free Software Foundation; either version 2, or (at your
;; option) any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:

;; Genbro is based on nero, but it browses scholia instead of web
;; pages.  See nero.el for documentation.

;; Or, ALTERNATIVELY: Genbro is a general tool for managing histories,
;; that is used by nero.  I kind of like the sound of that better.

;; Aside: I also like the idea of rewriting the history mechanism so
;; that it uses trees instead of lists.  The list-like structure
;; should be what Ariadne's Thread does -- you go into a certain room
;; A, you are unwinding your thread, but you don't loop the thread
;; around anything in the room, so if you came back out of room A and
;; go into the room you were in before, your thread won't show that
;; you visited room A.  (This could be useful when running away from a
;; minotaur, you don't want to go wandering through any dead-end broom
;; closets, you want to high-tail it for the exit.)

;; Note that the approach of making this a general history managing
;; facility means that I'll have to make a separate file that actually
;; uses the facilities here to use with the scholium system.  That's
;; something I'm OK with.  And of course, it would be fine to rewrite
;; nero.el to use the facilities here too.  (Hooks are good.)

;; On the other hand, I don't know if I want to rewrite the history
;; mechanism right now, I don't think, because that would be somewhat
;; time consuming.

;; In addition to providing a history mechanism, it provides a few
;; other minor tricks for working with the history (e.g. marks).  I
;; don't think it will include any graphical/display features, though.

;; Note that different applications that use histories will have
;; different storage requirements.  Nero seems to want to store
;; all the content that it downloads, whereas the scholium
;; just wants to keep track of names.

;; Note also that one might assume that other people had written
;; a generally applicable history mechanism before.  (But I'm
;; not convinced that their api's will be great...?)  Might be
;; worth checking out before going too far with this code.  OTOH,
;; the revisions I'm proposing aren't complicated.

;; The current model of the history is:

;; genbro-multiverse
;;     genbro-tabs
;;         genbro-history
;;         genbro-future

;; In other words, the multiverse holds the tabs, and each
;; tab holds a history and a future.  Each of these things is
;; a list.

;; The model that I would like at some point is this:

;; genbro-multiverse
;;     genbro-tabs
;;         genbro-istree
;;         genbro-thread

;; Here, the istree is a tree, and the thread is ariadne's thread.
;; I'm not quite sure how the "normal" notion of history and future is
;; supposed to be recovered from the thread and/or the tree, but it is
;; something to think about, and probably isn't hard.  (Maybe we'll
;; need some extra structure?)

;; AND, on the topic of matching - which everyone likes, right,
;; matching? - how can you go wrong -- It should be possible to
;; match things by various different predicates.  For example,
;; *keywords* instead of contents of the text field.

;; Genbro will have to deal with the fact that there are
;; "things that have been browsed" and "things that haven't
;; been browsed".  People will probably want to search both
;; categories of things (when they have the opportunity, like
;; with the scholium system -- of course, web searches sort
;; of provide this opportunity even with the web).


;; -------------------------------------------------------------------

;; It would be kind of nice to be able to create many different
;; simultaneous genbro instances; a history of these sorts of events
;; and a history of those sorts of events, bla bla bla.  Thus,
;; the code here should probably be a macro for creating things,
;; sort of like the code from generic-menu.el that I just wrote.
;; 

;;; Code:

(require 'cl)

(defmacro make-generic-browser (browser-name)
  (let ((genbro-add-history-element
         (intern (concat browser-name "-add-history-element")))
        (genbro-add-multiverse-element
         (intern (concat browser-name "-add-multiverse-element")))
        (genbro-after-loading-internal-hook
         (intern (concat browser-name "-after-loading-internal-hook")))
        (genbro-ariadnes-thread
         (intern (concat browser-name "-ariadnes-thread")))
        (genbro-back
         (intern (concat browser-name "-back")))
        (genbro-back-to-beginning
         (intern (concat browser-name "-back-to-beginning")))
        (genbro-before-browsing-hook
         (intern (concat browser-name "-before-browsing-hook")))
        (genbro-browse
         (intern (concat browser-name "-browse")))
        (genbro-collect-matching-pages
         (intern (concat browser-name "-collect-matching-pages")))
        (genbro-collect-matching-pages-1
         (intern (concat browser-name "-collect-matching-pages-1")))
        (genbro-current-page
         (intern (concat browser-name "-current-page")))
        (genbro-current-tab
         (intern (concat browser-name "-current-tab")))
        (genbro-current-url
         (intern (concat browser-name "-current-url")))
        (genbro-default-new-tab-name
         (intern (concat browser-name "-default-new-tab-name")))
        (genbro-default-page
         (intern (concat browser-name "-default-page")))
        (genbro-delete-tab
         (intern (concat browser-name "-delete-tab")))
        (genbro-environment
         (intern (concat browser-name "-environment")))
        (genbro-exchange-point-and-mark
         (intern (concat browser-name "-exchange-point-and-mark")))
        (genbro-forward
         (intern (concat browser-name "-forward")))
        (genbro-forward-to-end
         (intern (concat browser-name "-forward-to-end")))
        (genbro-future
         (intern (concat browser-name "-future")))
        (genbro-get-tab
         (intern (concat browser-name "-get-tab")))
        (genbro-goto-mark
         (intern (concat browser-name "-goto-mark")))
        (genbro-history
         (intern (concat browser-name "-history")))
        (genbro-long-thread
         (intern (concat browser-name "-long-thread")))
        (genbro-mark
         (intern (concat browser-name "-mark")))
        (genbro-mark-page
         (intern (concat browser-name "-mark-page")))
        (genbro-marks
         (intern (concat browser-name "-marks")))
        (genbro-mode
         (intern (concat browser-name "-mode")))
        (genbro-multiverse
         (intern (concat browser-name "-multiverse")))
        (genbro-new-content
         (intern (concat browser-name "-new-content")))
        (genbro-new-tab
         (intern (concat browser-name "-new-tab")))
        (genbro-next-tab
         (intern (concat browser-name "-next-tab")))
        (genbro-old-layout
         (intern (concat browser-name "-old-layout")))
        (genbro-open-tab
         (intern (concat browser-name "-open-tab")))
        (genbro-page-from-url
         (intern (concat browser-name "-page-from-url")))
        (genbro-previous-page
         (intern (concat browser-name "-previous-page")))
        (genbro-previous-tab
         (intern (concat browser-name "-previous-tab")))
        (genbro-processed-before
         (intern (concat browser-name "-processed-before")))
        (genbro-reload
         (intern (concat browser-name "-reload")))
        (genbro-reload-vanilla
         (intern (concat browser-name "-reload-vanilla")))
        (genbro-rename-tab
         (intern (concat browser-name "-rename-tab")))
        (genbro-restore-page
         (intern (concat browser-name "-restore-page")))
        (genbro-restore-environment
         (intern (concat browser-name "-restore-environment")))
        (genbro-revise-current-page
         (intern (concat browser-name "-revise-current-page")))
        (genbro-revise-or-update-timescape
         (intern (concat browser-name "-revise-or-update-timescape")))
        (genbro-scorched-earth
         (intern (concat browser-name "-scorched-earth")))
        (genbro-set-tab
         (intern (concat browser-name "-set-tab")))
        (genbro-state-URL
         (intern (concat browser-name "-state-URL")))
        (genbro-state-action
         (intern (concat browser-name "-state-action")))
        (genbro-state-curpoint
         (intern (concat browser-name "-state-curpoint")))
        (genbro-state-ephem
         (intern (concat browser-name "-state-ephem")))
        (genbro-state-flags
         (intern (concat browser-name "-state-flags")))
        (genbro-state-handler
         (intern (concat browser-name "-state-handler")))
        (genbro-state-mode
         (intern (concat browser-name "-state-mode")))
        (genbro-state-revise
         (intern (concat browser-name "-state-revise")))
        (genbro-state-tab
         (intern (concat browser-name "-state-tab")))
        (genbro-tab
         (intern (concat browser-name "-tab")))
        (genbro-tab-currently-browsing
         (intern (concat browser-name "-tab-currently-browsing")))
        (genbro-tab-names-use-defaults
         (intern (concat browser-name "-tab-names-use-defaults")))
        (genbro-tabs
         (intern (concat browser-name "-tabs")))
        (genbro-tabula-rosa
         (intern (concat browser-name "-tabula-rosa")))
        (genbro-thread
         (intern (concat browser-name "-thread")))
        (genbro-update-timescape
         (intern (concat browser-name "-update-timescape")))
        (genbro-yank-marked-page
         (intern (concat browser-name "-yank-marked-page"))))
    `(progn 
       (defvar ,genbro-history nil)
       (defvar ,genbro-future nil)
       (defvar ,genbro-mark nil)
       (defvar ,genbro-marks nil)
       (defvar ,genbro-tab nil)
       (defvar ,genbro-tabs nil)
       (defvar ,genbro-tab-names-use-defaults t)
       (defvar ,genbro-multiverse nil)
       (defvar ,genbro-ariadnes-thread nil)

       (defvar ,genbro-state-curpoint nil)
       (defvar ,genbro-state-URL nil)
       (defvar ,genbro-state-flags nil)
       (defvar ,genbro-state-handler nil)
       (defvar ,genbro-state-mode nil) 
       (defvar ,genbro-state-action nil)
       (defvar ,genbro-state-ephem nil) 
       (defvar ,genbro-state-revise nil)
       (defvar ,genbro-state-tab nil)


       (defun ,genbro-tabula-rosa ()
         (interactive)
         (setq ,genbro-history (last ,genbro-history)
               ,genbro-future nil))

       (defun ,genbro-scorched-earth ()
         (interactive)
         (setq ,genbro-history nil
               ,genbro-future nil)
         (setq ,genbro-old-layout nil
               ,genbro-ariadnes-thread nil
               ,genbro-tabs nil
               ,genbro-marks nil
               ,genbro-tab nil
               ,genbro-mark nil
               ,genbro-multiverse nil))

       (defun ,genbro-processed-before (url)
         (let ((i 0)
               (len (length ,genbro-multiverse))
               (found nil))
           (while (and (not found)
                       (< i len))
             (let* ((j 0)
                    (tab (cadr (nth i ,genbro-multiverse)))
                    (len2 (length tab)))
               (setq i (1+ i))
               (while (and (not found)
                           (< j len2))
                 (let ((candidate (nth j tab)))
                   (if (equal (car candidate) url)
                       (setq found candidate)
                     (setq j (1+ j)))))))
           found))

       ;; this is the main interface to the rest of the world, and also
       ;; quite popular among the other functions defined in this
       ;; file.
       (defun ,genbro-browse
         (URL &optional flags handler mode action ephem revise)
         (interactive (list (completing-read 
                             "URL: " 
                             (apply
                              #'concatenate
                              'list
                              (map 'list (lambda (tab) 
                                           (mapcar (lambda (page)
                                                     (car page))
                                                   (cadr tab)))
                                   ,genbro-multiverse)))))
         ;; set up environment
         (run-hooks ',genbro-before-browsing-hook)
         ;; save state for use when asynchronous processes finish
         (setq ,genbro-state-curpoint (point)
               ,genbro-state-URL URL
               ,genbro-state-flags flags
               ,genbro-state-handler handler
               ,genbro-state-mode mode
               ,genbro-state-action action
               ,genbro-state-ephem ephem
               ,genbro-state-revise revise)
         ;; process the content if/as needed
         (let ((old-content (,genbro-processed-before URL)))
           (cond
            ((and revise old-content)
             (,genbro-restore-page old-content nil))
            ((functionp handler)
             (funcall handler URL flags action))
            ;; Maybe a hook should be run here
            (t (,genbro-revise-or-update-timescape)))))

       ;; For some reason an error gets triggered here, before anything
       ;; can even happen with the form that is about to get run.
       (defun ,genbro-revise-or-update-timescape ()
         (let ((props (list ,genbro-state-flags 
                            ,genbro-state-handler 
                            ,genbro-state-mode 
                            ,genbro-state-action
                            ,genbro-state-ephem)))
           (if (and ,genbro-state-revise ,genbro-multiverse)
               (,genbro-revise-current-page ,genbro-state-URL 
                                            ,genbro-state-curpoint 
                                            props)
             (,genbro-update-timescape ,genbro-state-URL 
                                       ,genbro-state-curpoint 
                                       props))))

       ;; Some of this stuff may be overkill for the scholium browser
       ;; system - or maybe we need to go with same pattern ?  I'm going
       ;; to try disabling things... It might be good to have the
       ;; shape of the objects we store determined specially for each
       ;; implementation?
       (defun ,genbro-update-timescape (URL curpoint props)
         ;; add to history
         (unless (equal (,genbro-current-url) URL)
           (,genbro-add-history-element URL props))
         ;; save location of point in previous timeframe
;  (when (> (length ,genbro-history) 1)
;    (setcar (nthcdr 3 (,genbro-previous-page)) curpoint))
         ;; update future as needed.  If we've moved forward in time
         ;; one unit, we don't need the last cell in the future
         ;; anymore.
         (if (equal (caar (last ,genbro-future)) URL)
             (setq ,genbro-future (nbutlast ,genbro-future))
           ;; we maintain this "weird" nil-headed structure for
           ;; `,genbro-future' so we can consistenly use tail
           ;; pointers.
           (setq ,genbro-future
                 (if (not ,genbro-future)
                     (list nil)
                   (nbutlast ,genbro-future
                             (1- (length ,genbro-future))))))
         ;; set up a a default tab if there are no tabs yet
         (unless ,genbro-tabs (,genbro-set-tab "Tab 1"))
         ;; update our ariadne's thread (this is where the non-nil
         ;; return value comes from, by the way)
         (setq ,genbro-ariadnes-thread
               (cons (,genbro-current-url) ,genbro-ariadnes-thread)))

       (defun ,genbro-revise-current-page (url curpoint props)
         (setcar (last ,genbro-history) 
                 (,genbro-new-content url props curpoint))
         (cond
          ((not ,genbro-tabs) (,genbro-set-tab "Tab 1"))
          (t t)))

       (defun ,genbro-add-history-element (URL props)
         (setq ,genbro-history
               (nconc ,genbro-history
                      (list (,genbro-new-content URL props 1)))))

       (defun ,genbro-mark-page ()
         (interactive)
         (message "Page marked.")
         (setq ,genbro-mark (cons ,genbro-tab (,genbro-current-page))
               ,genbro-marks (add-to-list ',genbro-marks ,genbro-mark)))

       (defun ,genbro-exchange-point-and-mark (&optional nonmarking)
         (interactive)
         (let ((mark ,genbro-mark))
           (when mark
             (unless nonmarking (,genbro-mark-page))
             (,genbro-get-tab (car mark))
             (,genbro-restore-page (cdr mark) t))))

       (defun ,genbro-goto-mark ()
         (interactive)
         (,genbro-exchange-point-and-mark 'nonmarking))

       ;; rather than just deleting these, it would be nicer to save
       ;; them.  My excuse for the time being is that this is what
       ;; emacs does with its marks.  Both could be improved.
       (defun ,genbro-yank-marked-page ()
         (interactive)
         (if ,genbro-marks
             (progn (,genbro-goto-mark)
                    (setq ,genbro-marks (cdr ,genbro-marks)
                          ,genbro-mark (car ,genbro-marks)))
           (message "No pages marked.")))

       (defun ,genbro-collect-matching-pages (regexp)
         (interactive "MRegexp: ")
         (let ((matches nil))
           (dolist (page ,genbro-history)
             (let ((pt 
                    (with-temp-buffer (insert (second page))
                                      (goto-char (point-min))
                                      (search-forward-regexp
                                       regexp nil t))))
               (when pt (setq matches (cons 
                                       (cons
                                        (first page) pt) matches)))))
           (dolist (page (cdr ,genbro-future))
             (let ((pt
                    (with-temp-buffer (insert (second page))
                                      (goto-char (point-min))
                                      (search-forward-regexp 
                                       regexp nil t))))
               (when pt (setq matches (cons
                                       (cons 
                                        (first page) pt) matches)))))
           matches))

       (defun ,genbro-collect-matching-pages-1 (regexp)
         (interactive "MRegexp: ")
         (let ((matches nil))
           (dolist (tab ,genbro-multiverse)
             (dolist (page (second tab))
               (let ((pt
                      (with-temp-buffer
                        (insert (second page))
                        (goto-char (point-min))
                        (search-forward-regexp regexp nil t))))
                 (when pt (setq matches 
                                (cons 
                                 (cons 
                                  (first page) pt) matches)))))
             (dolist (page (cdr (third tab)))
               (let ((pt
                      (with-temp-buffer
                        (insert (second page))
                        (goto-char (point-min))
                        (search-forward-regexp regexp nil t))))
                 (when pt (setq matches
                                (cons
                                 (cons
                                  (first page) pt) matches))))))
           matches))

       (defun ,genbro-tab-currently-browsing (name)
         (or (caar (last (cadr (assoc name ,genbro-multiverse))))
             ""))

       ;; should complement this with a `,genbro-delete-tab' --
       ;; this business of the `,genbro-default-page' is pretty
       ;; bogus.
       (defun ,genbro-new-tab (&optional name where)
         (interactive (list (if ,genbro-tab-names-use-defaults
                                (,genbro-default-new-tab-name)
                              (read-string
                               (concat "Name (default: "
                                       (,genbro-default-new-tab-name)
                                       "): ")
                               nil
                               nil
                               (,genbro-default-new-tab-name)))
                            (read-string
                             (concat "URL (default: "
                                     (,genbro-current-page)
                                     "): ")
                             nil
                             nil
                             (,genbro-current-page))))
         (,genbro-set-tab (or name
                              (,genbro-default-new-tab-name))
                          (or where
                              (,genbro-current-page))))

       (defun ,genbro-open-tab (arg)
         (interactive "P")
         (if arg (,genbro-new-tab nil (,genbro-current-url))
           (,genbro-new-tab)))

       (defun ,genbro-set-tab (name &optional url)
         (,genbro-add-multiverse-element name url)
         (setq ,genbro-tabs (add-to-list ',genbro-tabs name t))
         ;; identify the tab we just created as active
         (setq ,genbro-tab name))

       (defun ,genbro-rename-tab (newname)
         (interactive (list (read-string
                             "New name: "
                             nil
                             nil
                             ,genbro-tab)))
         (setcar (assoc ,genbro-tab ,genbro-multiverse) newname)
         (setq ,genbro-tabs 
               (substitute newname
                           ,genbro-tab
                           ,genbro-tabs :test #'equal))
         (setq ,genbro-tab newname))

       (defun ,genbro-add-multiverse-element (name url)
         (setq ,genbro-state-tab name)
         (add-hook 
          ',genbro-after-loading-internal-hook
          (lambda ()
            (setq
             ,genbro-multiverse
             (if ,genbro-multiverse
                 (add-to-list 
                  ',genbro-multiverse
                  (let ((newtab (list 'foo)))
                    (setcar newtab ,genbro-state-tab)
                    (setcdr newtab (list ,genbro-history))
                    (setcdr (cdr newtab) (list ,genbro-future))
                    newtab)
                  t)
               (list (let ((newtab (list 'foo)))
                       (setcar newtab ,genbro-state-tab)
                       (setcdr newtab (list ,genbro-history))
                       (setcdr (cdr newtab) (list ,genbro-future))
                       newtab))))))
         (when url
           (setq ,genbro-history nil
                 ,genbro-future nil)
           (,genbro-browse url)))

       (defun ,genbro-default-new-tab-name ()
         (concat "Tab "
                 (int-to-string
                  (1+ (length ,genbro-tabs)))))

       (defun ,genbro-get-tab (name)
         (interactive (list (let ((completion-ignore-case t))
                              (completing-read "Tab: " ,genbro-tabs))))
         (setcar (nthcdr 3 (,genbro-current-page)) (point))
         (unless (equal name "")
           (setq ,genbro-history nil
                 ,genbro-future nil)
           (let ((newtab (assoc name ,genbro-multiverse)))
             (when newtab
               (setq 
                ,genbro-tab name
                ,genbro-history (nconc ,genbro-history (second newtab))
                ,genbro-future (nconc ,genbro-future (third newtab)))
               (,genbro-restore-page (,genbro-current-page) t)))))

       (defun ,genbro-previous-tab ()
         (interactive)
         (if (eq (length ,genbro-multiverse) 1)
             (message "No previous tab.")
           (let ((current-tab
                  (member-if 
                   (lambda (elt) (equal (car elt) ,genbro-tab)) 
                   ,genbro-multiverse)))
             (if (eq (length current-tab) (length ,genbro-multiverse))
                 (,genbro-get-tab (caar (last ,genbro-multiverse)))
               (,genbro-get-tab (caar
                                 (last
                                  (butlast ,genbro-multiverse
                                           (length current-tab)))))))))

       (defun ,genbro-next-tab ()
         (interactive)
         (if (eq (length ,genbro-multiverse) 1)
             (message "No next tab.")
           (let ((current-tab 
                  (member-if 
                   (lambda (elt) (equal (car elt) ,genbro-tab)) 
                   ,genbro-multiverse)))
             (if (eq (length current-tab) 1)
                 (,genbro-get-tab (caar ,genbro-multiverse))
               (,genbro-get-tab (caadr current-tab))))))

       (defun ,genbro-reload ()
         (interactive)
         (when (apply 
                #',genbro-browse 
                (,genbro-current-url)
                ;; make sure the page is revised.
                (append (fifth (,genbro-current-page)) (list t)))))

       (defun ,genbro-reload-vanilla ()
         (interactive)
         (when (apply #',genbro-browse 
                      (,genbro-current-url)
                      ;; make sure the page is revised.
                      (list nil nil nil nil nil t))))

       (defun ,genbro-back ()
         (interactive)
         (if (eq (length ,genbro-history) 1)
             (message "Already at beginning of history.")
           (setq ,genbro-future (nconc ,genbro-future
                                       (last ,genbro-history))
                 ,genbro-history (nbutlast ,genbro-history))
           (,genbro-restore-page (,genbro-current-page) nil)))

       (defun ,genbro-back-to-beginning ()
         (interactive)
         (,genbro-mark-page)
         (while (> (length ,genbro-history) 1)
           (,genbro-back)))

       (defun ,genbro-forward ()
         (interactive)
         (if (not (cdr ,genbro-future))
             (message "Already at end of future.")
           (setq ,genbro-history (nconc ,genbro-history 
                                        (last ,genbro-future))
                 ,genbro-future (nbutlast ,genbro-future))
           (,genbro-restore-page (,genbro-current-page) nil)))

       (defun ,genbro-forward-to-end ()
         (interactive)
         (,genbro-mark-page)
         (while (> (length ,genbro-future) 2)
           (,genbro-forward)))

       (defun ,genbro-page-from-url (url)
         (let ((mult ,genbro-multiverse)
               (ret nil))
           (while mult
             (let ((tab (cadar mult)))
               (while tab
                 (when (equal (caar tab) url)
                   (setq ret (car tab)))
                 (setq tab (cdr tab)))
               (setq mult (cdr mult))))
           ret))

       ;; everywhere *else* we may as well call `,genbro-current-url'
       ;; - makes the code easier to read.
       (defun ,genbro-current-url ()
         (caar (last ,genbro-history)))

       (defun ,genbro-current-tab ()
         ,genbro-tab)

       (defun ,genbro-current-page ()
         (car (last ,genbro-history)))

       (defun ,genbro-previous-page ()
         (car (last ,genbro-history 2))))))

(provide 'generic-browser)

;;; generic-browser.el ends here
