;;; generic-menu.el -- generic menu mechanism

;; Copyright (C) 2005 Joe Corneli <jcorneli@math.utexas.edu>
;; Copyright (C) 1985, 1986, 1987, 1993, 1994, 1995, 2000, 2001, 2002, 
;; 2003, 2004  Free Software Foundation, Inc.

;; Time-stamp: <jac -- Sat Jun 25 10:19:40 CDT 2005>

;; This file not is part of GNU Emacs, but it is licensed under the
;; same terms as GNU Emacs.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:

;; Loosely based on (portions of) buff-menu.el.

;; This code provides a generic way to perform actions on rich objects
;; from a list.  It is assumed that each object has several attributes
;; that a user may be interested in.  "Name", "size" and "mode" are
;; familiar examples from `list-buffers'; here there can be an
;; arbitrary number of arbitrary attributes.  This makes this
;; mechanism useful in some cases where `completing-read' would not
;; provide the user with enough information.

;; The way attributes are obtained from objects is left up to the
;; user, as are the actual actions that can be performed on the
;; objects.

;; Generality comes at an obvious, unavoidable, cost, namely any given
;; deployment of this interface requires some set up -- examples are
;; provided, look through this code for details.

;; A few "generic" functions for sorting and what not are also
;; available.


;; NOTE: The "setup" function could be a "curried" version of this one
;; that takes in the accessors and not the list of objects.  Just
;; an idea.

;; NOTE: I've cut out most of the functions from buff-menu to get at
;; the "basics".  The additional features would probably be to add
;; back again at some point.

;;; Code:

;; Trying to do it all at once... is trickier.  But it shouldn't be
;; impossible to get the mapname, the modename, and the pretty name
;; all from one string (say).  One more string for documentation and
;; we'd be pretty much set.

;; Use this sort of style to make the code better:
; (defmacro testing (some-string)
;   (let ((some-symbol (intern some-string)))
;     `(defun ,some-symbol () t)))

(defmacro make-generic-menu-mode
  (mapname modename pretty-name bindings)
  `(progn 
     (defvar ,mapname)
     (setq ,mapname (make-keymap))
     (suppress-keymap ,mapname t)
     (dolist (binding ,bindings)
       (define-key ,mapname (car binding) (cdr binding)))
     (defun ,modename ()
       (kill-all-local-variables)
       (use-local-map ,mapname)
       (setq major-mode (quote ,modename))
       (setq mode-name ,pretty-name)
       (setq truncate-lines t)
       (setq buffer-read-only t))))

;; Example
; (make-generic-menu-mode bar-menu-mode-map
;                         bar-menu-mode
;                         "Bar Menu"
;                         '(("q" . Generic-menu-revert)
;                           ("\C-m" . Generic-menu-this-window)))

;; The basic idea of this is that we have some objects and some
;; functions to map across the objects to extract the information
;; from the objects.  

;; We also want to have a function that will give the semantics
;; for "select" and other sorts of things that we might be able
;; to do *to* the objects that we've listed.  (Maybe we should
;; pass this in along with the objects, and set it up as a local
;; variable that can be called later.)

;; So, send in the objects, send in an alist of functions together
;; with names.

;; Again: we map each function over each object -- functions produce
;; the columns; objects correspond to rows.  Note that the functions
;; should be set up so that they produce strings.

;; Maybe the thing to do is to make a function that *sets up*
;; the buffer appropriately (by defining all the functions etc.)
;; I guess that might be more of a macro?  Not necessarily, but
;; it seems like a reasonable idea.

(defun generic-menu-noselect (objects &rest accessors)
  (let (cols)
    (dolist (get-this accessors)
      (setq cols (cons 
                  (cons (car get-this)
                        (mapcar (cdr get-this) objects))
                  cols)))
    ;; find the width of the columns.  Note that if you want to limit
    ;; the length of the columns, you should truncate the elements
    ;; produced by the accessors, since the columns are assumed to be
    ;; as wide as their widest item!
    (let ((lens (mapcar (lambda (col) 
                          (let ((len 0))
                            (dolist (str col)
                              (let ((candidate (length str)))
                                (when (> candidate len)
                                  (setq len candidate))))
                            len))
                        cols)))
      ;; Setting things up to make buttons and sorting work seems like
      ;; an additional step...
      (with-current-buffer (get-buffer-create "*Generic List*")
        (setq buffer-read-only nil)
        (erase-buffer)
        (while cols
          (goto-char (point-min))
          (goto-char (line-end-position))
          (dolist (str (car cols))
            (insert str " ")
            ;; fill with spaces to make up for lost space
            (insert-char 32 (- (car lens) (length str)))
            (unless (equal (forward-line) 0)
              (newline))
            (goto-char (line-end-position)))
          (setq cols (cdr cols))
          (setq lens (cdr lens)))
        (goto-char (point-min))
        (current-buffer)))))

;;Example
;(generic-menu-noselect 
; '((1 2 3) (4 5 6))
;  (cons "Small"
;        (lambda (elt)
;          (int-to-string (first elt))))
;  (cons "Big"
;        (lambda (elt)
;          (int-to-string (third elt)))))


;; Discussion of control panels: alittle control panel in the style of
;; `list-buffers' (making this more generic might be silly, but we
;; could demand that users implement their own control panel through a
;; trivial accessor like the one illustrated here; however, trying to
;; make the *mode* totally generic could lead to some trouble) --
;; using this to mark the "current" thing should have some semantics
;; that will depend on the specific state of the collection of objects
;; we're working with, though, so it might make more sense to ask
;; people to implement their own control panel.  Intresting.


;; Sorting could be done using autocompletion on the name of the
;; column to sort on, too, which would be kind of nice.
;;
;; Or, heck, we could make a command to sort on the current
;; column, or reorder columns according to the current line
;; or any sort of magic you might like

;; Actually, *this* simple approach isn't going to work when the
;; strings have any kind of complexity (i.e. with spaces) to them.
;; Something a bit more clever is going to have to be done.

(defun Generic-menu-sort (col)
  (interactive "P")
  (save-excursion
    (sort-fields (or col 1) (progn (goto-line 2)
                                   (point))
                 (point-max))))

;;; generic-menu.el ends here
