;;; austex.el -- general latex utilities

;; Copyright (C) 2004 Joseph Corneli <jcorneli@math.utexas.edu>

;; Time-stamp: <jac -- Wed Jun  8 00:33:03 CDT 2005>

;; This file is not part of GNU Emacs, but it is distributed under
;; the same terms as GNU Emacs.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published
;; by the Free Software Foundation; either version 2, or (at your
;; option) any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:

;; This file collects my code for working with LaTeX under Emacs.  I
;; use these commands together with LaTeX mode.  I tend to have no
;; trouble typing math "in real time", as I take lecture notes for
;; example.

;; Since this file makes available a large number of functions that
;; one needs to be able to select and execute very quickly, the choice
;; of bindings is important (see austex-bindings.el).

;; In order to use these commands and bindings with LaTeX mode, place
;;
;;   (add-hook 'latex-mode-hook 'austex-mode)
;;
;; in your .emacs configuration file.

;; Some people might wonder why I don't just use AUCTeX.  Other than
;; the simple answer that I think my bindings are better, there are
;; two additional reasons.  When I was learning to type math with
;; Emacs, the AUCTeX learning curve seemed too steep.  Also, AUCTeX's
;; advertised functionality seemed like overkill for what I wanted to
;; do.  Eventually, if I continue to develop this code, I will
;; probably cherrypick things I like from AUCTeX's code.  The name of
;; this file is a joke on AUCTeX and AUStin TEXas.

;; Some people are sure to frown on the duplication of effort and the
;; fact that my code is not as nice in many ways.  Perhaps others will
;; appreciate the speed and simplicity of the approach I've been
;; using.  The speediness probably comes down to choice of bindings.
;; The simplicity in part a side-effect of the fact that I haven't
;; made any effort to provide an additional customization layer: if
;; you want to change the way this package works, change the code.  No
;; effort has been made to be fully general, so you may well have to
;; do some editing to get this file to do what you want.

;; Thanks to Klaus Berndl for help with the macro evaluation strategy.
;; An alternative approach was suggested by Daniel Pittman, viz.,
;;
;; (dolist (name '("alpha" "beta"))
;;   (fset (intern (concat "austex-" name))
;;        `(lambda () (interactive) (insert "\\" ,name))))
;;
;; And this is tighter...

;;; History:

;; I decided to revise my LaTeX-under-Emacs code into this
;; "publishable" form following email conversations with Ross Moore
;; about what makes a good LaTeX macro.  Many "convenience" LaTeX
;; macros should actually be handled at the editor-macro level.  My
;; hope is that this code will be a positive contribution to the
;; development of a tightly integrated and correctly divied up editing
;; environment for mathematical writing.

;;; Future:

;; First of all, add navigation features.  Cf. cdlatex.el where
;; `cdlatex-tab' is defined; its docstring says
;;
;;     The cursor stops... 
;;     - before closing brackets if preceding-char is any of -({[]})
;;     - after  closing brackets, but not if following-char is any 
;;       of ({[_^
;;     - just after $, if the cursor was before that $.
;;     - at end of non-empty lines
;;     - at the beginning of empty lines
;;     - before a SPACE at beginning of line
;;     - after first of several SPACE).
;;
;; It would be good to do this sort of thing, but (probably) in more
;; generality.  (I should test out `cdlatex-tab' to see how it
;; behaves.)  In particular, I think it would be nice to be able to
;; navigate both backwards and forwards.  See comments at the
;; beginning of the section "Forms to fill in" below.

;; Another feature that would be useful to emulate would be that
;; provided by `texinfo-insert-@-with-arg', which allows one to put
;; braces around things already in the buffer.

;; (defun texinfo-insert-@-with-arg (string &optional arg)
;;   (if arg
;;       (progn
;;         (setq arg (prefix-numeric-value arg))
;;         (if (< arg 0)
;;             (progn
;;               (skip-chars-backward " \t\n\r\f")
;;               (save-excursion
;;                 (forward-sexp arg)
;;                 (insert "@" string "{"))
;;               (insert "}"))
;;           (skip-chars-forward " \t\n\r\f")
;;           (insert "@" string "{")
;;           (forward-sexp arg)
;;           (insert "}")))
;;     (insert "@" string "{}")
;;     (backward-char)))

;; With an eye to the distant future, let me mention that I am
;; interested in developing software that can parse and "understand"
;; mathematical writing.  Some beginning steps towards linguistic
;; analysis of mathematical writing may (eventually) be included in
;; this package.

;;; Code:

;;; Environment:

(defvar austex-visual-registers t)

;;; Basic navigation:

;; The basic pattern "$<math>$" has to be filled in time and again.
;; We provide some simple commands related to this pattern.  Note that
;; these should probably be made to work with other math environments,
;; as in auctex's texmathp.el.

;; Some commands to skip _over_ a math environment or a block of text
;; between two math environments would be handy.

(defun austex-end-of-math ()
  "Jump forwards past text, positioning cursor just before the next $."
  (interactive)
  (search-forward "$")
  (backward-char 1))

(defun austex-beginning-of-math ()
  "Jump backwards past text, positioning cursor just after the previous $."
  (interactive)
  (search-backward "$")
  (forward-char 1))

(defun austex-out-of-math ()
  "Jump past next $ and insert a space (if there isn't one already)."
  (interactive)
  (search-forward "$")
  (just-one-space))

;; these require that you are inside of a "form" to start with.  This
;; code is not fully general!  and I'm not sure it works anyway!
;; "Forms" are something like sexps and perhaps the code for moving
;; around between sexps can be modified to produce the desired effect
;; here.
(defun austex-next-form ()
  "Jump to next } or $."
  (interactive)
  (up-list)
  (unless (search-forward "}" nil t)
    (search-forward "$"))
  (backward-char 1))

(defun austex-previous-form ()
  "Jump to previous } or $."
  (interactive)
  (backward-up-list)
  (unless (search-backward "}" nil t)
    (search-backward "$")))

;; Even more questionable...

(defun austex-next-bracket ()
  (interactive)
  (search-forward "{}")
  (backward-char))

;;; Environments:

(defmacro austex-define-tex-environment (name)
  "Create a function that inserts the TeX environment NAME."
  `(defun ,(intern (concat "austex-" name)) ()
     (interactive)
     (insert "\\begin{" ,name "}")
     (newline)
     (insert "\\end{" ,name "}")
     (beginning-of-line)
     (newline)
     (backward-char)))

;; Actually, arrays should maybe be handled separately, and read in
;; number of cols.  Also, inserting some arrays can be quick, cf. 
;; section on sequences.
(dolist
    (elt '("defn" "eqnarray*" "proof" "quote" "array" "cases" "nota"))
  (eval `(austex-define-tex-environment ,elt)))

;; this is to be used in combination with `mmm-mode' and `multi-task'.
;; the purpose is to have a nice literate programming environment for
;; elisp/whatever.  We only get fancy if `mmm-mode' is available
;; however.
(defun austex-verbatim ()
  (interactive)
  (let ((beg (point)))
    (insert "\\begin{verbatim}")
    (newline)
    (insert "\\end{verbatim}")
    (let ((end (point)))
      (beginning-of-line)
      (newline)
      (backward-char)
      (when (require 'mmm-mode nil t)
        (save-excursion
          (mmm-parse-region beg (1+ end)))))
    (message nil)))

(defun austex-lisp ()
  (interactive)
  (let ((beg (point)))
    (insert "\\begin{lisp}")
    (newline)
    (insert "\\end{lisp}")
    (let ((end (point)))
      (beginning-of-line)
      (newline)
      (backward-char)
      (when (require 'mmm-mode nil t)
        (save-excursion
          (mmm-parse-region beg (1+ end)))))
    (message nil)))
  
; (austex-define-tex-environment "verbatim")
; (austex-define-tex-environment "lisp")

;;; Lists:

(defmacro austex-define-tex-list (name)
  "Create a function that inserts the TeX list NAME."
  `(defun ,(intern (concat "austex-" name)) ()
     (interactive)
     (insert
      "\\begin{" ,name "}
\\item 
\\end{" ,name "}")
     (search-backward "\\item ")
     (end-of-line)))

(dolist
    (elt '("enumerate" "itemize"))
  (eval `(austex-define-tex-list ,elt)))

;; This decorative line between items should probably be optional/customizable.
(defun austex-item ()
  (interactive)
  (newline)
  (insert
   "%------------------------------------------------------------------*
\\item "))

;;; Greek letters and other simple symbols:

;; Note that some capitalized greek letters need nonstandard latex support.
(defmacro austex-define-tex-symbol (name)
  "Create a function that inserts the TeX symbol NAME."
  `(defun ,(intern (concat "austex-" name)) (&optional capitalize)
     ,(concat "Insert the TeX symbol \\" name 
              ", or \\" (upcase-initials name) 
              " if CAPITALIZE is non-nil.")
     (interactive "P")
     (insert "\\" (if capitalize
                      (upcase-initials ,name)
                    ,name))))

(dolist
    (elt '("alpha" "beta" "gamma" "delta" "epsilon" "zeta" "eta" "theta"
           "iota" "kappa" "lambda" "mu" "nu" "xi" "omicron" "pi" "rho"
           "sigma" "tau" "upsilon" "phi" "chi" "psi" "omega" "varphi"
           "infty"))
  (eval `(austex-define-tex-symbol ,elt)))


(defmacro austex-define-tex-symbol-1 (name)
  "Create a function that inserts the TeX symbol NAME, followed by a space."
  `(defun ,(intern (concat "austex-" name)) ()
     (interactive)
     (insert "\\" ,name " ")))

;; int should maybe be different.
(dolist
    (elt '("Leftarrow" "Leftrightarrow" "Rightarrow" "Subset" "Supset"
           "approx" "cap" "cdot" "cdots" "circ" "cong" "cup" "equiv"
           "exists" "forall" "geq" "in" "int" "ldots" "leq"
           "nabla" "ni" "neq" "openset" "otimes" "partial"
           "rightarrow" "setminus" "sim" "simeq" "square" "subset"
           "supset" "times" "triangle" "vdots" "vee" "wedge"))
  (eval `(austex-define-tex-symbol-1 ,elt)))

(defmacro austex-define-tex-symbol-2 (name)
  "Create a function that inserts NAME.
Simple, maybe even silly, but useful in combination with `austex-fast-math-insert'."
  `(defun ,(intern (concat "austex-" name)) (&optional upcase)
     (interactive "P")
     (insert (if upcase 
                 (upcase ,name)
               ,name))))

(dolist
    (elt '("1" "2" "3" "4" "5" "6" "7" "8" "9" "0"
           "A" "B" "C" "D" "E" "F" "G" "H" "I" "J" "K" "L" "M" 
           "N" "O" "P" "Q" "R" "S" "T" "U" "V" "W" "X" "Y" "Z"
           "a" "b" "c" "d" "e" "f" "g" "h" "i" "j" "k" "l" "m"
           "n" "o" "p" "q" "r" "s" "t" "u" "v" "w" "x" "y" "z"))
  (eval `(austex-define-tex-symbol-2 ,elt)))

;; Th openset symbol is part of Unicode now I think, though not yet
;; part of a standardard LaTeX font as far as I know.

;;; Fontification and unary operators:

(defmacro austex-define-tex-unary-markup (name)
  "Create a function that inserts the TeX unary markup command NAME.
Examples of such commands are emph, mathop, mathrm, and hat."
  `(defun ,(intern (concat "austex-" name)) ()
     (interactive)
     (insert "\\" ,name "{}")
     (backward-char 1)))

(dolist
    (elt '("bar" "check" "dot" "emph" "hat" "mathbf" "mathcal" "mathit"
           "mathop" "mathord" "mathrm" "overdoubledot" "text" "tilde"))
  (eval `(austex-define-tex-unary-markup ,elt)))

;;; Speedy insertion of single letters.

;; Enables extra-speedy insertion of common single symbol phrases,
;; e.g. $\alpha$ or $3$ (cf. cdlatex.el).

;; This should take an optional argument that is passed
;; to symbol producer
(defun austex-fast-math-insert (symbol-producer &rest arguments)
  "A quick way to insert TeX symbols in between dollar signs. 
SYMBOL-PRODUCER is a function that actually inserts the symbols."
  (insert "$") 
  (funcall symbol-producer arguments)
  (insert "$ "))

(defun austex-period ()
  (interactive)
  (when (not (looking-at "[^\n\\.]*\\.[^\n\\.]$"))
      (end-of-line))
  (while (search-backward-regexp "\\s-$" (line-beginning-position) t)
    (replace-match ""))
  (insert ".  "))

(defun austex-comma ()
  (interactive)
  (when (not (looking-at "[^\n\\.]*\\.[^\n\\.]$"))
      (end-of-line))
  (while (search-backward-regexp "\\s-$" (line-beginning-position) t)
    (replace-match ""))
  (insert ", "))

;;; Fancy exponents:

;; These could be built by a combination of other functions from this
;; file.  But I don't know if that would make them "more transparent".

(defun austex-complement ()
  (interactive)
  (insert "^{\\complement}"))

(defun austex-inverse ()
  (interactive)
  (insert "^{-1}"))

(defun austex-to-the-star ()
  (interactive)
  (insert "^{\\star}"))

(defun austex-to-the-ast ()
  (interactive)
  (insert "^*"))

;;; Forms to fill in:

;; By a "form" I mean something that is designed for entering data
;; into.  For example, if a variable has a subscript "form", the data
;; it is looking for is the expression giving the value of the
;; subscript.  The simplest form in LaTeX is probably "$$", which asks
;; you to fill in some math expression.  Most of the forms here take
;; one argument, but some (e.g. `austex-langle-rangle', `austex-frac')
;; take two -- of course, you are free to leave any of the forms blank
;; and LaTeX will still work.  Forms like `austex-partial-derivative'
;; _expect_ to take a variable number of data (1 or 2).

;; If you want to be _literal_, the various environments are also
;; forms, as is the buffer itself.

;; The LaTeX code for forms is associated with a set of regular
;; expressions that one should be able to use to decide e.g. what the
;; "next" or "previous" form insertion point is, and move the cursor
;; to that point.

;; Since forms tend to be nested, navigating forms should be similar
;; to navigating sexps.  It would be nice if "next" could move you
;; "up" or "forward" depending on context, but this would also be a
;; nice feature to have for navigating sexps.

;; Note that the syntax of sexps is not quite right for us in places.
;; For example, <1000 is considered to be one sexp.  I don't know if
;; this is a valid variable name in Lisp: I guess it is.  But it is
;; still wrong for us.  

;; I guess we could call the new sexps "texps" -- that would be funny.

;; But what is not so funny is that in order to hack these things I'll
;; probably have to read and modify the Emacs C code in syntax.c.  :(

;; Luckily the sexps for LaTeX mode aren't *too* far off from how they
;; should be.


;; This isn't really the right way to accomplish what we want.  We
;; don't just want the *first* match (which is found here), nor do we
;; want the *minimal* match.  Instead, we want the match that makes
;; sense syntactically.  Again, this is like with sexps.  We find the
;; left "paren" and then skip over "paren pairs" until we find the
;; correct "right paren".  This gives some clue about how
;; `austex-define-tex-form' should be defined -- it can take a MIDDLE
;; as well as an end, and the stuff in the MIDDLE you should have to
;; skip over, in addition to whatever stuff you skip over by matching
;; paren pairs.
;;
;; But in the mean time, this sort of works -- in cases where the
;; first match is the correct match, which should be the case for
;; example if you are outside of math mode and request the previous
;; piece of text that was in math mode.
(defun austex-get-prev-form (beginning end)
  "Retrieve and insert any previous text between BEGINNING and END.
BEGINNING and END are buffer substrings to match."
  (let ((beg (save-excursion
               (search-backward end nil t)
               (search-backward beginning nil t)))
        (end (save-excursion
               (if (search-backward end nil t)
                   (match-end 0)))))
    (if (and beg end)
        (insert (buffer-substring beg end))
      (message "No match."))))

(defmacro austex-define-tex-form (name beginning end)
  "Create a function that inserts the TeX form named NAME.
The form is based on BEGINNING and ENDING text.  Cursor will
be positioned between the two pieces of text."
  ;; if given a prefix argument, this should insert the previous
  ;; matching form.  We could also set it up to take a numeric
  ;; argument and select the nth previous form.  That could be a
  ;; little challenging for the user without additional visual
  ;; feedback to help in seeing what n is supposed to be.
  `(defun ,(intern (concat "austex-" name)) (&optional copy-previous)
     (interactive "P")
     (if copy-previous
         (austex-get-prev-form ,beginning ,end)
       (insert ,beginning ,end)
       (backward-char (length ,end)))))

;; Note: the tex environments could be treated like forms too.  The
;; only one I have special concern about is arrays, because we would
;; probably be best served by giving `austex-array' a prefix argument.
;; If instead of using a prefix arg to enter previous text selection
;; "mode", but *actually* used a *mode* and some *other* key to enter
;; that mode, this wouldn't be a problem at all.  That's something
;; that can wait until we have the basic selection functions down
;; however.
;;
;; Including of course a way to select something that was earlier than
;; just the immediately previous expression.  So, once we have the
;; immediately previous expression part taken care of, we should be
;; able to give a numeric argument to select something earlier.  (Like
;; `forward-sexp' does, for example.)
;;
;; These should maybe be able to have functions associated with
;; them that they run when they insert too.
(dolist (elt '(("math" "$" "$")
               ("paragraph" "\\paragraph{" "}")
               ("section" "\\section{" "}")
               ("chapter" "\\chapter{" "}")
               ("exponent" "^{" "}")
               ("subscript" "_{" "}")
               ("set" "\\{" "\\}")
               ("norm" "\\|" "\\|")
               ("absolute-value" "|" "|")
               ("sqrt" "\\sqrt{" "}")
               ("prod" "\\prod_{" "}")
               ("sum" "\\sum_{" "}")
               ("quote-marks" "\"" "\"")
               ("parentheses" "(" ")")
               ("braces" "{" "}")
               ("brackets" "[" "]")
               ("lfloor-rfloor" "\\lfloor " "\\rfloor")
               ("verb" "\\verb|" "|")
               ;; Note: unlike the previous forms, the following have
               ;; a "variable" quality to them, i.e. the end or the
               ;; beginning of the form after it is edited won't
               ;; always match the end or the beginning as it is
               ;; indicated here.
               ("langle-rangle" "\\langle " ",\\rangle")
               ("frac" "\\frac{" "}{}")
               ("partial-derivative" "\\frac{\\partial}{\\partial " "}")
               ("derivative" "\\frac{d}{d " "}")))
  (eval `(austex-define-tex-form ,@elt)))

;;; Operations on the current equation:

;; The simplest thing is just selecting the text of the current
;; equation.  Should add this.

;; Also should have a function for converting quickly between
;; displayed and non-displayed equations.

;; Eventually it might be cool to be able to select previous equations
;; by number (like you can do with alternative spellings under
;; ispell).  That would involve some fancy work.

;; this macro lets you operate on just the current equation, nothing
;; else.  It could do a little bit better checking to see whether or
;; not we are inside an equation according to e.g. fontlock or
;; auctex's mathp (texmathp.el/`texmathp-match-environment' etc.).
(defmacro austex-narrowed-to-current-equation (&rest args)
  "Run ARGS on the current equation."
  `(save-excursion
     ;; this should match one _or_ two $'s
     (search-forward-regexp "\\$+")
     (let ((end (point)))
       (goto-char (match-beginning 0))
       (search-backward-regexp "\\$+")
       (save-restriction
         (narrow-to-region (point) end)
         ,@args))))

;; probably should be able to modify the previous macro to include an
;; `if' in the appropriate place & get this behavior as a bonus
(defmacro austex-narrowed-to-current-equation-excursive (&rest args)
  "Run ARGS on the current equation.
Point is allowed to move."
  `(progn
     ;; this should match one _or_ two $'s
     (search-forward-regexp "\\$+")
     (let ((end (point)))
       (goto-char (match-beginning 0))
       (search-backward-regexp "\\$+")
       (save-restriction
         (narrow-to-region (point) end)
         ,@args))))

;; Does the prefix arg really do what the docstring says it does?
;;
;; Anyway, one intelligent thing to do would be to make "real"
;; equation seperators have a space (or ^) before and space (or $)
;; after, and then only process those, skipping any of these symbols
;; if they don't have the spaces.
(defun austex-turn-current-equation-into-equation-array (&optional prefix)
  "Turn a simple equation into an unnumbered equation array.
A non-nil PREFIX argument switches to a full replacement mode for
inequalities.  With no prefix, the equal sign is treated as the
only comparison operator.  This is to one's advantage e.g. when
greater-than or less-than signs are used inside some of the
members of an equation."
  (interactive "p")
  (austex-narrowed-to-current-equation
   (let ((bs (buffer-substring-no-properties (point-min) (point-max))))
     ;; replace dollar signs at the beginning and end of the equation
     ;; with empty string
     (delete-region (point-min) (point-max))
     (insert (replace-regexp-in-string "\\(^\\$+\\)\\|\\(\\$+$\\)" "" bs)))
   (goto-char (point-min))
   (insert "\n\\begin{eqnarray*}\n")
   (let ((past-first nil))
     ;; more different equation separators could be added to this list
     ;; and the regexp could be optimized.
     (while (re-search-forward
             (if prefix
                 "\\s-?[=><]\\|\\(\\\\geq\\)\\|\\(\\\\leq\\)\\s-?"
               "\\s-?=\\s-?")
             nil t)
       (if past-first
           (replace-match "\\\\\\\\\n&\\&& ")
         ;; we could be fancy and record the number of characters by
         ;; which the comparison operator is inserted, then use this
         ;; number of spaces later on.  But it isn't clear that this
         ;; makes for more readable code (esp.  in the case of long
         ;; equations!).  More ideally, we'd have another function for
         ;; fixing up equation arrays to look like that.
         (replace-match " &\\&& ")
         (setq past-first t))))
   (goto-char (point-max))
   (insert "\n\\end{eqnarray*}")))

;;; Sequences:

;; This section still has quite a ways to go.  The idea is to speed up
;; the entry of sequences, which otherwise take a long time to type
;; out.

;; It would be good to have a two-level prefix as used
;; for example with rectangles! that would allow you to
;; input various different kinds of sequences cleanly.

;; It would also be cool to have a "graphical" way
;; to select inputs, maybe using something like ratmen
;; to input things by moving down the branches of some
;; tree.

;; But anyway, the different kinds of sequences we might
;; want are as follows:

;; a_0, a_1, ..., a_n
;; a_1, a_2, ..., a_n
;; a_1, a_2, ..., a_{n+1}

;; TODO: check that the subscript is a number, and
;; iterate the number.  But this provides basic
;; functionality.
(defun austex-insert-sequence-of-variables
  (term substart subend interterm)
  ;; read in three strings; if subend is empty,
  ;; we will insert "..." at the end of the sequence.
  (interactive (list (read-string
                      "Term: ")
                     (read-string
                      "Subscript starting: ")
                     (read-string
                      "Subscript ending: ")
                     (read-string
                      "What goes between terms?: ")))
  (insert term
          "_"
          (if (> (length substart) 1)
              (concat "{" substart "}")
            substart)
          interterm
          ;; would be good to be able to specify
          ;; that we want ldots with some
          ;; prefix argument.
          "\\cdots"
          interterm
          (if (not (equal subend ""))
              (concat term
                      "_"
                      (if (> (length subend) 1)
                          (concat "{" subend "}")
                        subend))
            "\\ldots")))

;;; Registers:

;; Basically copied right out of register.el (I think a code snippet
;; this small constitutes "fair use".), I personally find this
;; behavior to be more natural in general.  For the public, this is
;; only turned on in `austex-mode'.  (What it does: sets point and
;; mark opposite to the way `insert-register' does it.)
;;
;; I'm not sure why this messes with the mark?
(defun austex-insert-register (register)
  "Insert contents of register REGISTER.  (REGISTER is a character.)
Normally puts mark before and point after the inserted text."
  (interactive "*cInsert register: \n")
  (push-mark)
  (let ((val (get-register register)))
    (cond
     ((consp val)
      (insert-rectangle val))
     ((stringp val)
      (insert-for-yank val))
     ((numberp val)
      (princ val (current-buffer)))
     ((and (markerp val) (marker-position val))
      (princ (marker-position val) (current-buffer)))
     (t
      ;; avoid error
      nil))))

;; This should move anything it is replacing further down the list.
;;
;; Kyle Schalm suggested having the registers be populated
;; automatically by something like Huffman coding that detects which
;; phrases are commonly typed into the buffer.  I think this is an
;; excellent idea!  But probably we won't be able to do it that
;; effectively until we have the issues with fields sorted out.
;; However, we could play around with doing it with words only first.
(defun austex-copy-rectangle-to-register (register &optional overwrite)
  "Like `copy-rectangle-to-register' but assumes you mean \"copy current region\".
REGISTER is a character stating which register to use.  Further
differences from `copy-rectangle-to-register' are that visual
feedback on the register bank is provided (if `austex-visual-registers' 
is non-nil), and an attempt is made to preserve the register's
previous contents, if any.  Optional argument OVERWRITE says
don't bother trying to preserve the old contents of REGISTER."
  (interactive "cRegister: \nP:")
  (let ((old-contents (get-register register)))
    (when (and (not overwrite)
               old-contents
               (austex-move-contents-of-register register old-contents)))
    (copy-rectangle-to-register register 
                                (region-beginning) 
                                (region-end))
    (austex-update-register-display)))

(defun austex-update-register-display ()
  (when austex-visual-registers
    (let ((curbuff (current-buffer)))
      (save-excursion
        (set-buffer (get-buffer-create "*Registers*"))
        (delete-region (point-min) (point-max))
        (dolist 
            (elt (string-to-list
                  ;; "etaoinshrdlucmfgypwbvkxjqz"
                  "abcdefghijklmnopqrstuvwxyz"))
          (let ((letter (char-to-string elt)))
            (insert "-" letter "- ")
            (or (austex-insert-register elt)
                "")
            (insert        "\n"))))
      (display-buffer "*Registers*")
      (set-buffer curbuff))))

(defun austex-move-contents-of-register (register old-contents)
  "If move content of this register to next nil register, if any."
  (let* ((remaining-registers 
          (member register 
                  (string-to-list "abcdefghijklmnopqrstuvwxyz")))
         (under-consideration (car remaining-registers)))
    (while (and remaining-registers
                (not (eq under-consideration (string-to-char "z"))))
      (setq remaining-registers (cdr remaining-registers))
      (setq under-consideration (car remaining-registers))
      (unless (get-register under-consideration)
        (set-register under-consideration old-contents)
        (setq remaining-registers nil)))))

(defun austex-delete-matching-registers (regexp)
  "Set to nil any register that contains text matching REGEXP."
  (interactive "MRegexp: ")
  (dolist 
      (elt (string-to-list
            "abcdefghijklmnopqrstuvwxyz"))
    (let ((reg (get-register elt)))
      (when reg
        (with-temp-buffer
          (austex-insert-register elt)
          (when (string-match regexp (buffer-string))
            (set-register elt nil))))))
  (austex-update-register-display))

(defun austex-clear-all-registers ()
  "Set all registers to nil."
  (interactive)
  (austex-delete-matching-registers ".*")
  (austex-update-register-display))

(defun austex-clear-register (char)
  (interactive "cRegister: ")
  (set-register char nil)
  (austex-update-register-display))

;;; Bonus Features:

(defun parenthesize ()
  "Add parentheses around everything on this line (modulo whitespace)."
  (interactive)
  (save-excursion
    (end-of-line)
    (delete-horizontal-space)
    (insert ")")
    (beginning-of-line)
    (while (looking-at " \\|\t")
      (forward-char 1))
    (insert "(")))

(defun put-next-sexp-in-dollar-signs ()
  (interactive)
  (insert "$")
  (forward-sexp)
  (insert "$"))

(defun put-prev-sexp-in-dollar-signs ()
  (interactive)
  (backward-sexp)
  (put-next-sexp-in-dollar-signs))

;; The list of modes here should be customizable.
(when (require 'mmm-mode nil t)
  (defun multi-task (mode)
    "Use `emacs-lisp-mode' inside of `latex-mode'."
    (interactive (list (let ((completion-ignore-case t))
                         (intern (completing-read "Mode: "
                                                  '("emacs-lisp-mode"
                                                    "R-mode"))))))
    (mmm-ify-by-regexp mode
    "\\\\begin{\\(verbatim\\|lisp\\|idea\\)}" 0 
    "\\\\end{\\(verbatim\\|lisp\\|idea\\)}" 0 25)))

(require 'austex-bindings)
(provide 'austex)

;;; austex.el ends here

