;;; austex-bindings.el -- bindings for the austex latex utilities

;; Copyright (C) 2004 Joseph Corneli <jac@sextuplebucky.com>

;; Time-stamp: <jac -- Mon Apr  4 16:35:38 CDT 2005>

;; This file is not part of GNU Emacs, but it is distributed under
;; the same terms as GNU Emacs.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published
;; by the Free Software Foundation; either version 2, or (at your
;; option) any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:

;; This file includes the bindings I use with AusTeX (see austex.el).

;; The long-term goal is to make this into an "optimal" keyboard
;; layout in the sense of Dvorak.  Right now it is just a good one in
;; the sense of the default Emacs map.  The bindings are built on top
;; of a Dvorak layout, and so some of them may appear awkward or
;; confusing to users of other layouts.  If you are already a Dvorak
;; user, adding these bindings to your repertoire should not be very
;; hard.  On the other hand, if you are not a Dvorak user and you
;; still want to use AusTeX, you have at least three options: (1) see
;; if the bindings make sense to you with your layout after all; (2)
;; port the bindings to your layout in such a way that they do make
;; sense; (3) learn Dvorak.

;; Unlike what you may be used to in Emacs, these bindings involve
;; using a lot of modifiers!  It is helpful to have a sense of the
;; physical layout of modifiers on my keyboard, even if you use some
;; other layout.

;; I actually have two Xmodmap setups; I will describe the one I use
;; on my laptop first, because it turns out that I seem to spend the
;; majority of my time typing on the laptop, especially when I am
;; taking lecture notes.

;; I replaced the number row on my laptop by modifiers. The layout is
;;
;;                     A H S C M G A G M C S H A
;;
;; where
;;
;;    A = Alt
;;    H = Hyper
;;    S = Super
;;    C = Control
;;    M = Meta
;;    G = AltGr (Mode_switch)
;;
;; The reason for the choice of the order is not arbitrary.  The basic
;; principle that is I want commonly-used modifiers to be reachable
;; without moving my wrists.  I can reach Mode_switch, Meta, Control,
;; and Super just by extending my fingers.  The middle Alt key is a
;; slightly longer reach.  Hyper is again a longer reach, and if I try
;; to press outside Alt keys without lifting my hand up and moving it,
;; I definitely feel a bit tweaked in the wrist.

;; As for what happens to the numbers -- they have been moved to the
;; AltGr layer in the right hand, in a traditional "embedded keypad"
;; arrangement.  In my view, a better configuration would probably
;; require a better keyboard layout to begin with and/or better logic
;; in the keyboard, together with a more robust system of modifiers
;; inside of X.

;; This setup makes the two-modifier chords Super+Control,
;; Control+Meta, Super+Meta easy to reach as well.  Hyper+Super and
;; Hyper+Control aren't that bad either, but they aren't quite as
;; nice.  Other two-modifier chords aren't particularly comfortable,
;; but at least it is _possible_ to enter Alt+Hyper and Alt+Super
;; (this would be much more dubious if it weren't for the extra Alt
;; keys out on the ends of the row).

;; Though three-modifier chords may seem silly, Hyper+Super+Control
;; and Super+Control+Meta are not very hard to press down using this
;; keyboard layout.  I don't bind to these much yet.

;; In general, the main benefit of using modifiers (and modifier
;; chords) instead of prefix keys is *speed*.  It takes less time, for
;; example, to press C-n than it does to press C-c n.  Sometimes
;; keypresses within the same modifier can be faster than you'd
;; expect, e.g. C-t C-t or G-. G-, (i.e., Up followed by backspace,
;; according to my Xmodmap).

;; Single keypresses tend to be the fastest, then single modifiers
;; plus single keys, then two modifiers plus single keys, then a
;; prefix key followed by a single key.

;; A time-savings is also afforded by finding the keys right where you
;; seek them.  This is not quite the same as knowing the keyboard
;; layout: it also means that the layout should have some internal
;; logic that is easy to understand.  Subjectively, the Dvorak layout
;; seems to be quite good in this regard.  One goal for this file is
;; to map out some of the psychological space behind the act of typing
;; math.

;; The second Xmodmap file I use is the one I have set up for typing
;; with my Kinesis Classic keyboard.  It is similar to the one I use
;; with my laptop's keyboard, but because the Kinesis keyboard comes
;; with an extra row of easy-to-reach keys below the standard four
;; rows, it is not necessary to replace the number row with modifiers.
;; This is the main difference between the two Xmodmap files.  Because
;; I don't have to remap the number row, I don't use an embedded
;; number keypad approach with this keyboard, instead, I have two
;; symmetrical "editing" keypads.  Because the extra fifth row has
;; only four keys for either hand, I do not bind the Alt modifier in
;; the same place with this Xmodmap file (rather it is bound to one of
;; the Kinesis "thumb keys").

;; There seems to be some risk associated with trying to share
;; something as "personal" as a keyboard layout.  However, I think
;; I've put enough work into these bindings to make them worth sharing
;; with anyone who's interested.  I've reduced the risk by not
;; including all of my bindings!  For anyone who is _really_
;; interested, the rest of the bindings (and various nonstandard
;; functions that are bound) are available on my website.  Oh yes, and
;; finally, the xmodmap files described above are also available on my
;; website.

;;; Code:

;; This is a very long variable definition...
(defvar austex-mode-map
  (let ((map (make-sparse-keymap)))

;;; Latin-1 (AltGr layer on standard keys):

    ;; The layout on my keyboards are:
    ;;
    ;; Laptop
    ;;          `  BS   Up   Del   =   + 7    8    9     ^
    ;;            Left Down Right    - 4    5    6     _
    ;;          [  ]    {    }         1    2    3     0
    ;;
    ;; Kinesis
    ;;          `  BS   Up   Del   =   + BS   Up   Del   ^
    ;;            Left Down Right    - Left Down Right _
    ;;          [  ]    {    }         {    }    (     )

    (define-key map (quote [2212]) 'austex-math)                      ; currency
    (define-key map (quote [2213]) 'austex-out-of-math)               ; yen

    ;; This is used sort of like "tab", and when the function
    ;; `austex-next-field' is written, this should probably be bound to
    ;; that instead of `austex-end-of-math'.
    (define-key map (quote [2239]) 'austex-end-of-math)               ; questiondown

    ;; This probably won't be used very much (I might be wrong).  The
    ;; main use would seem to be for selecting text, and there should
    ;; be a seperate function that does that.
    (define-key map (quote [2209]) 'austex-beginning-of-math)         ; exclamdown

;;; Latin-1 (Peripheral keys):

    (define-key map [insert] 'parentheses)
    (define-key map [select] 'parenthesize)

    ;; Everything that doesn't have to take place very quickly should
    ;; probably not be bound in this section. `austex-math' and
    ;; `austex-out-of-math' are two examples of functions that need to
    ;; be able to be called very quickly.

;;; Control Characters:

    ;; These are things like C-w, C-y, etc.  Usually they are bound by
    ;; default in Emacs. Some of these keys have been cannibalized to
    ;; make prefix keymaps (see below).  It is worth knowing the
    ;; default bindings!!

;;; Keymaps:

    ;; Things just got more interesting.  These keymaps let us bind
    ;; lots more editor commands to speedy (though not lightning fast)
    ;; keyboard shortcuts.  The sections that give the bindings with
    ;; the associated "prefix commands" appear in later sections.  Not
    ;; sure whether these should be sparse keymaps or not?

    (define-key map [?\C-.] (make-keymap))
    (define-key map [?\C-,] (make-keymap))
    (define-key map [?\C-\;] (make-keymap))

    ;; Setting up hyper and super for a non-X-enabled Emacs. Though
    ;; these are not strictly speaking keymaps, they are "prefix
    ;; commands" in the sense that you have to press the key, release
    ;; it, then find the next key in the sequence.

    (define-key function-key-map "\e[17~" 'event-apply-hyper-modifier); f6
    (define-key function-key-map "\e[20~" 'event-apply-super-modifier); f9

    ;; Next follow several sections of "single modifier plus
    ;; character" bindings.

;;; Hyper:

    ;; Mostly greek letters.

    (define-key map [(hyper ,)] 'austex-leq)
    (define-key map [(hyper .)] 'austex-geq)
    (define-key map [(hyper ?')] 'austex-neq)
    (define-key map [(hyper ?\040)] 'austex-out-of-math)              ; H-SPC
    (define-key map [(hyper ?\134)] 'austex-pipe)                     ; H-\
    (define-key map [(hyper ?\;)] 'austex-sim)
    (define-key map [(hyper a)] 'austex-alpha)
    (define-key map [(hyper b)] 'austex-beta)
    (define-key map [(hyper c)] 'austex-psi)
    (define-key map [(hyper d)] 'austex-delta)
    (define-key map [(hyper e)] 'austex-epsilon)
    (define-key map [(hyper f)] 'austex-varphi)
    (define-key map [(hyper g)] 'austex-gamma)
    (define-key map [(hyper h)] 'austex-theta)
    (define-key map [(hyper i)] 'austex-iota)
    (define-key map [(hyper j)] 'austex-ldots)
    (define-key map [(hyper k)] 'austex-kappa)
    (define-key map [(hyper l)] 'austex-lambda)
    (define-key map [(hyper m)] 'austex-mu)
    (define-key map [(hyper n)] 'austex-nu)
    (define-key map [(hyper o)] 'austex-omicron)
    (define-key map [(hyper o)] 'austex-phi)
    (define-key map [(hyper p)] 'austex-pi)
    (define-key map [(hyper q)] 'austex-chi)
    (define-key map [(hyper r)] 'austex-rho)
    (define-key map [(hyper s)] 'austex-sigma)
    (define-key map [(hyper t)] 'austex-tau)
    (define-key map [(hyper u)] 'austex-upsilon)
    (define-key map [(hyper v)] 'austex-infty)
    (define-key map [(hyper w)] 'austex-omega)
    (define-key map [(hyper x)] 'austex-xi)
    (define-key map [(hyper y)] 'austex-eta)
    (define-key map [(hyper z)] 'austex-zeta)

;;; Super:

    ;; Math symbols and the simplest math forms.

    ;; I have a sort of weird "graphical" mnemonic for these symbols.
    ;; It tends to work out pretty well, the only difficulty that
    ;; stands out to be being that I seem to forget which finger to
    ;; use for \cap and which to use for \cup.  Some of the other
    ;; commands require a little bit of hunt-and-peck activity as
    ;; well.  Eventually it would be good to optimize this layout in a
    ;; Dvorakian way or find other means of reorganizing the confusing
    ;; symbols to make them easier to find.

    ;; I tend to use this super binding on the laptop and the
    ;; corresponding Latin 1 binding with the Kinesis.  It might
    ;; actually be better to use one binding all the time.
    (define-key map [(super ?\040)] 'austex-math)                     ; s-SPC
    (define-key map [(super ,)] 'austex-subset)
    (define-key map [(super .)] 'austex-supset)
    (define-key map [(super <)] 'austex-Subset)
    (define-key map [(super >)] 'austex-Supset)
    (define-key map [(super ?\134)] 'austex-setminus)                 ; s-\
    (define-key map [(super \')] 'austex-equiv)
    (define-key map [(super \;)] 'austex-cong)
    (define-key map [(super a)] 'austex-simeq)
    (define-key map [(super b)] 'austex-nabla)
    (define-key map [(super c)] 'austex-cap)
    (define-key map [(super d)] 'austex-forall)
    (define-key map [(super e)] 'austex-ni)
    (define-key map [(super f)] 'austex-frac)
    (define-key map [(super g)] 'austex-int)
    (define-key map [(super h)] 'austex-circ)
    (define-key map [(super i)] 'austex-exists)
    (define-key map [(super j)] 'austex-cdots)
    (define-key map [(super k)] 'austex-wedge)
    (define-key map [(super l)] 'austex-Leftrightarrow)
    (define-key map [(super m)] 'austex-vee)
    (define-key map [(super n)] 'austex-rightarrow)
    (define-key map [(super o)] 'austex-in)
    (define-key map [(super p)] 'austex-partial)
    (define-key map [(super q)] 'austex-vdots)
    (define-key map [(super r)] 'austex-cup)
    (define-key map [(super s)] 'austex-Rightarrow)
    (define-key map [(super t)] 'austex-cdot)
    (define-key map [(super u)] 'austex-times)

    ;; we assume that there will be a subscript.  Not always true in
    ;; all math writing, but there _could_ always be one.
    (define-key map [(super v)] 'austex-sum)
    (define-key map [(super w)] 'austex-prod)
    (define-key map [(super x)] 'austex-triangle)
    (define-key map [(super y)] 'austex-sqrt)
    (define-key map [(super z)] 'austex-Leftarrow)

;;; Hyper+Super

    ;; Additional symbols, LaTeX environments and forms, fancy
    ;; exponents.

    ;; Symbols can be moved back and forth with Super until they are
    ;; all settled.

    ;; Some environments probably need to be fast, others can be as slow
    ;; as a prefix map. Forms need to be fast.

    ;; Probably all the fancy exponents could be moved to a prefix map.
    ;; On the other hand, if there aren't enough other things to put here,
    ;; they might as well stay.

    (define-key map [(hyper super ,)] 'austex-openset)
    (define-key map [(hyper super .)] 'austex-alist)
    (define-key map [(hyper super a)] 'austex-absolute-value)
    (define-key map [(hyper super b)] 'austex-next-bracket)
    (define-key map [(hyper super c)] 'austex-complement)
    (define-key map [(hyper super d)] 'austex-defn)
    (define-key map [(hyper super e)] 'austex-emph)
    (define-key map [(hyper super i)] 'austex-inverse)
    (define-key map [(hyper super l)] 'austex-langle-rangle)
    (define-key map [(hyper super o)] 'austex-otimes)
    (define-key map [(hyper super p)] 'austex-proof)
    (define-key map [(hyper super q)] 'austex-quote)
    (define-key map [(hyper super r)] 'austex-eqnarray*)
    (define-key map [(hyper super s)] 'austex-set)
    (define-key map [(hyper super v)] 'austex-verbatim)
    ;; "w" here stands for "wave equation"
    (define-key map [(hyper super w)] 'austex-square)
    (define-key map [(hyper super x)] 'austex-to-the-ast)
    (define-key map [(hyper super y)] 'austex-to-the-star)
    (define-key map [(hyper super z)] 'austex-lisp)

;;; Hyper+Control:

    ;; Nothing here yet.

;;; Hyper+Meta:

    ;; Nothing here yet.

;;; Super+Control:

    ;; "fontification" and other unary-style macros, with a few
    ;; missing Emacs functions thrown in, and some math forms.  A nice
    ;; varied collection.

    ;; Note: There is a somewhat clear division between Hyper and
    ;; Super modified characters, namely, the former are mostly greek
    ;; letters and the latter are mostly math symbols.  It might be
    ;; sensible to have similar clear division between Hyper+Super and
    ;; Super+Control modified characters (shuttling anything that
    ;; doesn't fit into our categories off into another double
    ;; modifier group or prefix mapping).

    ;; Probably it would be best for the more infrequently-used
    ;; decorations to be placed into a prefix map (and perhaps all of
    ;; the decorations should go in the same place)

    (define-key map [(super control b)] 'austex-bar)
    (define-key map [(super control c)] 'austex-check)
    (define-key map [(super control d)] 'austex-derivative)
    (define-key map [(super control f)] 'austex-mathbf)
    (define-key map [(super control h)] 'austex-hat)
    (define-key map (quote [8388617]) 'austex-mathit)                 ; s-C-i = s-TAB
    (define-key map [(super control o)] 'austex-mathop)
    (define-key map [(super control p)] 'austex-partial-derivative)
    (define-key map [(super control r)] 'austex-mathrm)
    (define-key map [(super control s)] 'austex-subscript)
    (define-key map [(super control t)] 'austex-text)
    (define-key map [(super control l)] 'austex-exponent)
    (define-key map [(super control w)] 'austex-tilde)

;;; Alt:

    ;; Speedy insertion of single-character math patterns, like $1$,
    ;; $g$, and $\lambda$.  This is a fairly long and fairly ugly
    ;; section.

    ;; If possible, clean up so that prefix arg does good things for
    ;; the greek letters.

    (define-key map [(alt ?\040)] 'austex-period)                     ; A-SPC
    (define-key map (quote [4194350]) 'austex-period)                 ; A-.
    (define-key map (quote [4194348]) 'austex-comma)                  ; A-,
    (define-key map (quote [A-right]) 'put-next-sexp-in-dollar-signs)
    (define-key map (quote [A-left]) 'put-prev-sexp-in-dollar-signs)

    ;; double check: are the question marks needed?
    (define-key map [(alt ?1)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-1 arg)))
    (define-key map [(alt ?2)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-2 arg)))
    (define-key map [(alt ?3)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-3 arg)))
    (define-key map [(alt ?4)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-4 arg)))
    (define-key map [(alt ?5)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-5 arg)))
    (define-key map [(alt ?6)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-6 arg)))
    (define-key map [(alt ?7)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-7 arg)))
    (define-key map [(alt ?8)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-8 arg)))
    (define-key map [(alt ?9)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-9 arg)))
    (define-key map [(alt ?0)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-0 arg)))

    (define-key map [(alt a)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-a arg)))
    (define-key map [(alt b)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-b arg)))
    (define-key map [(alt c)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-c arg)))
    (define-key map [(alt d)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-d arg)))
    (define-key map [(alt e)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-e arg)))
    (define-key map [(alt f)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-f arg)))
    (define-key map [(alt g)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-g arg)))
    (define-key map [(alt h)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-h arg)))
    (define-key map [(alt i)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-i arg)))
    (define-key map [(alt j)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-j arg)))
    (define-key map [(alt k)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-k arg)))
    (define-key map [(alt l)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-l arg)))
    (define-key map [(alt m)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-m arg)))
    (define-key map [(alt n)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-n arg)))
    (define-key map [(alt o)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-q arg)))
    (define-key map [(alt p)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-p arg)))
    (define-key map [(alt q)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-q arg)))
    (define-key map [(alt r)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-r arg)))
    (define-key map [(alt s)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-s arg)))
    (define-key map [(alt t)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-t arg)))
    (define-key map [(alt u)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-u arg)))
    (define-key map [(alt v)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-v arg)))
    (define-key map [(alt w)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-w arg)))
    (define-key map [(alt x)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-x arg)))
    (define-key map [(alt y)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-y arg)))
    (define-key map [(alt z)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-z arg)))

    ;; Works well enough if you use the "outside" Alt keys.

    (define-key map [(alt hyper a)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-alpha arg)))
    (define-key map [(alt hyper b)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-beta arg)))
    (define-key map [(alt hyper c)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-psi arg)))
    (define-key map [(alt hyper d)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-delta arg)))
    (define-key map [(alt hyper e)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-epsilon arg)))
    (define-key map [(alt hyper f)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-phi arg)))
    (define-key map [(alt hyper g)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-gamma arg)))
    (define-key map [(alt hyper h)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-theta arg)))
    (define-key map [(alt hyper i)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-iota arg)))
    (define-key map [(alt hyper k)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-kappa arg)))
    (define-key map [(alt hyper l)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-lambda arg)))
    (define-key map [(alt hyper m)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-mu arg)))
    (define-key map [(alt hyper n)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-nu arg)))
    (define-key map [(alt hyper o)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-omicron arg)))
    (define-key map [(alt hyper p)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-pi arg)))
    (define-key map [(alt hyper r)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-rho arg)))
    (define-key map [(alt hyper s)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-sigma arg)))
    (define-key map [(alt hyper t)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-tau arg)))
    (define-key map [(alt hyper u)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-upsilon arg)))
    (define-key map [(alt hyper w)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-omega arg)))
    (define-key map [(alt hyper x)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-xi arg)))
    (define-key map [(alt hyper y)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-eta arg)))
    (define-key map [(alt hyper z)] '(lambda (&optional arg) (interactive) (austex-fast-math-insert 'austex-zeta arg)))


;; alt + super

;;     (define-key map [(alt super a)] '(lambda () (interactive) (austex-fast-math-insert '(lambda () (austex-alpha t)))))
;;     (define-key map [(alt super b)] '(lambda () (interactive) (austex-fast-math-insert '(lambda () (austex-beta t)))))
;;     (define-key map [(alt super c)] '(lambda () (interactive) (austex-fast-math-insert '(lambda () (austex-psi t)))))
;;     (define-key map [(alt super d)] '(lambda () (interactive) (austex-fast-math-insert '(lambda () (austex-delta t)))))
;;     (define-key map [(alt super e)] '(lambda () (interactive) (austex-fast-math-insert '(lambda () (austex-epsilon t)))))
;;     (define-key map [(alt super f)] '(lambda () (interactive) (austex-fast-math-insert '(lambda () (austex-phi t)))))
;;     (define-key map [(alt super g)] '(lambda () (interactive) (austex-fast-math-insert '(lambda () (austex-gamma t)))))
;;     (define-key map [(alt super h)] '(lambda () (interactive) (austex-fast-math-insert '(lambda () (austex-theta t)))))
;;     (define-key map [(alt super i)] '(lambda () (interactive) (austex-fast-math-insert '(lambda () (austex-iota t)))))

;;     (define-key map [(alt super k)] '(lambda () (interactive) (austex-fast-math-insert '(lambda () (austex-kappa t)))))
;;     (define-key map [(alt super l)] '(lambda () (interactive) (austex-fast-math-insert '(lambda () (austex-lambda t)))))
;;     (define-key map [(alt super m)] '(lambda () (interactive) (austex-fast-math-insert '(lambda () (austex-mu t)))))
;;     (define-key map [(alt super n)] '(lambda () (interactive) (austex-fast-math-insert '(lambda () (austex-nu t)))))
;;     (define-key map [(alt super o)] '(lambda () (interactive) (austex-fast-math-insert '(lambda () (austex-omicron t)))))
;;     (define-key map [(alt super p)] '(lambda () (interactive) (austex-fast-math-insert '(lambda () (austex-pi t)))))

;;     (define-key map [(alt super r)] '(lambda () (interactive) (austex-fast-math-insert '(lambda () (austex-rho t)))))
;;     (define-key map [(alt super s)] '(lambda () (interactive) (austex-fast-math-insert '(lambda () (austex-sigma t)))))
;;     (define-key map [(alt super t)] '(lambda () (interactive) (austex-fast-math-insert '(lambda () (austex-tau t)))))
;;     (define-key map [(alt super u)] '(lambda () (interactive) (austex-fast-math-insert '(lambda () (austex-upsilon t)))))

;;     (define-key map [(alt super w)] '(lambda () (interactive) (austex-fast-math-insert '(lambda () (austex-omega t)))))
;;     (define-key map [(alt super x)] '(lambda () (interactive) (austex-fast-math-insert '(lambda () (austex-xi t)))))
;;     (define-key map [(alt super y)] '(lambda () (interactive) (austex-fast-math-insert '(lambda () (austex-eta t)))))
;;     (define-key map [(alt super z)] '(lambda () (interactive) (austex-fast-math-insert '(lambda () (austex-zeta t)))))

    (define-key map [(alt super a)] '(lambda () (interactive) (austex-alpha t)))
    (define-key map [(alt super b)] '(lambda () (interactive) (austex-beta t)))
    (define-key map [(alt super c)] '(lambda () (interactive) (austex-psi t)))
    (define-key map [(alt super d)] '(lambda () (interactive) (austex-delta t)))
    (define-key map [(alt super e)] '(lambda () (interactive) (austex-epsilon t)))
    (define-key map [(alt super f)] '(lambda () (interactive) (austex-phi t)))
    (define-key map [(alt super g)] '(lambda () (interactive) (austex-gamma t)))
    (define-key map [(alt super h)] '(lambda () (interactive) (austex-theta t)))
    (define-key map [(alt super i)] '(lambda () (interactive) (austex-iota t)))

    (define-key map [(alt super k)] '(lambda () (interactive) (austex-kappa t)))
    (define-key map [(alt super l)] '(lambda () (interactive) (austex-lambda t)))
    (define-key map [(alt super m)] '(lambda () (interactive) (austex-mu t)))
    (define-key map [(alt super n)] '(lambda () (interactive) (austex-nu t)))
    (define-key map [(alt super o)] '(lambda () (interactive) (austex-omicron t)))
    (define-key map [(alt super p)] '(lambda () (interactive) (austex-pi t)))

    (define-key map [(alt super r)] '(lambda () (interactive) (austex-rho t)))
    (define-key map [(alt super s)] '(lambda () (interactive) (austex-sigma t)))
    (define-key map [(alt super t)] '(lambda () (interactive) (austex-tau t)))
    (define-key map [(alt super u)] '(lambda () (interactive) (austex-upsilon t)))

    (define-key map [(alt super w)] '(lambda () (interactive) (austex-omega t)))
    (define-key map [(alt super x)] '(lambda () (interactive) (austex-xi t)))
    (define-key map [(alt super y)] '(lambda () (interactive) (austex-eta t)))
    (define-key map [(alt super z)] '(lambda () (interactive) (austex-zeta t)))

;;; Super+Meta:

    ;; Nothing here yet.

;;; Control+Meta:

    ;; Plenty here by default.

    ;; Next follow several sections of "three modifiers plus
    ;; character" bindings.

;;; Hyper+Super+Control:

    ;; Nothing here yet.

;;; Super+Control+Meta:

    ;; Nothing here yet.

;;; Control x prefix:

    ;; this overrides the standard behavior for this key sequence
    (define-key map "\C-xri" 'austex-insert-register)

;;; Control . prefix:

    (define-key map [?\C-.?a] '(lambda () (interactive) (austex-insert-register (string-to-char "a"))))
    (define-key map [?\C-.?b] '(lambda () (interactive) (austex-insert-register (string-to-char "b"))))
    (define-key map [?\C-.?c] '(lambda () (interactive) (austex-insert-register (string-to-char "c"))))
    (define-key map [?\C-.?d] '(lambda () (interactive) (austex-insert-register (string-to-char "d"))))
    (define-key map [?\C-.?e] '(lambda () (interactive) (austex-insert-register (string-to-char "e"))))
    (define-key map [?\C-.?f] '(lambda () (interactive) (austex-insert-register (string-to-char "f"))))
    (define-key map [?\C-.?g] '(lambda () (interactive) (austex-insert-register (string-to-char "g"))))
    (define-key map [?\C-.?h] '(lambda () (interactive) (austex-insert-register (string-to-char "h"))))
    (define-key map [?\C-.?i] '(lambda () (interactive) (austex-insert-register (string-to-char "i"))))
    (define-key map [?\C-.?j] '(lambda () (interactive) (austex-insert-register (string-to-char "j"))))
    (define-key map [?\C-.?k] '(lambda () (interactive) (austex-insert-register (string-to-char "k"))))
    (define-key map [?\C-.?l] '(lambda () (interactive) (austex-insert-register (string-to-char "l"))))
    (define-key map [?\C-.?m] '(lambda () (interactive) (austex-insert-register (string-to-char "m"))))
    (define-key map [?\C-.?n] '(lambda () (interactive) (austex-insert-register (string-to-char "n"))))
    (define-key map [?\C-.?o] '(lambda () (interactive) (austex-insert-register (string-to-char "o"))))
    (define-key map [?\C-.?p] '(lambda () (interactive) (austex-insert-register (string-to-char "p"))))
    (define-key map [?\C-.?q] '(lambda () (interactive) (austex-insert-register (string-to-char "q"))))
    (define-key map [?\C-.?r] '(lambda () (interactive) (austex-insert-register (string-to-char "r"))))
    (define-key map [?\C-.?s] '(lambda () (interactive) (austex-insert-register (string-to-char "s"))))
    (define-key map [?\C-.?t] '(lambda () (interactive) (austex-insert-register (string-to-char "t"))))
    (define-key map [?\C-.?u] '(lambda () (interactive) (austex-insert-register (string-to-char "u"))))
    (define-key map [?\C-.?v] '(lambda () (interactive) (austex-insert-register (string-to-char "v"))))
    (define-key map [?\C-.?w] '(lambda () (interactive) (austex-insert-register (string-to-char "w"))))
    (define-key map [?\C-.?x] '(lambda () (interactive) (austex-insert-register (string-to-char "x"))))
    (define-key map [?\C-.?y] '(lambda () (interactive) (austex-insert-register (string-to-char "y"))))
    (define-key map [?\C-.?z] '(lambda () (interactive) (austex-insert-register (string-to-char "z"))))

;;; Control , prefix:

    ;; faster copying to register

    ;; (copy-rectangle-to-register register start end &optional
    ;; delete-flag)

    (define-key map [?\C-,?a] '(lambda (&optional ow) (interactive "P") (austex-copy-rectangle-to-register (string-to-char "a") ow)))
    (define-key map [?\C-,?b] '(lambda (&optional ow) (interactive "P") (austex-copy-rectangle-to-register (string-to-char "b") ow)))
    (define-key map [?\C-,?c] '(lambda (&optional ow) (interactive "P") (austex-copy-rectangle-to-register (string-to-char "c") ow)))
    (define-key map [?\C-,?d] '(lambda (&optional ow) (interactive "P") (austex-copy-rectangle-to-register (string-to-char "d") ow)))
    (define-key map [?\C-,?e] '(lambda (&optional ow) (interactive "P") (austex-copy-rectangle-to-register (string-to-char "e") ow)))
    (define-key map [?\C-,?f] '(lambda (&optional ow) (interactive "P") (austex-copy-rectangle-to-register (string-to-char "f") ow)))
    (define-key map [?\C-,?g] '(lambda (&optional ow) (interactive "P") (austex-copy-rectangle-to-register (string-to-char "g") ow)))
    (define-key map [?\C-,?h] '(lambda (&optional ow) (interactive "P") (austex-copy-rectangle-to-register (string-to-char "h") ow)))
    (define-key map [?\C-,?i] '(lambda (&optional ow) (interactive "P") (austex-copy-rectangle-to-register (string-to-char "i") ow)))
    (define-key map [?\C-,?j] '(lambda (&optional ow) (interactive "P") (austex-copy-rectangle-to-register (string-to-char "j") ow)))
    (define-key map [?\C-,?k] '(lambda (&optional ow) (interactive "P") (austex-copy-rectangle-to-register (string-to-char "k") ow)))
    (define-key map [?\C-,?l] '(lambda (&optional ow) (interactive "P") (austex-copy-rectangle-to-register (string-to-char "l") ow)))
    (define-key map [?\C-,?m] '(lambda (&optional ow) (interactive "P") (austex-copy-rectangle-to-register (string-to-char "m") ow)))
    (define-key map [?\C-,?n] '(lambda (&optional ow) (interactive "P") (austex-copy-rectangle-to-register (string-to-char "n") ow)))
    (define-key map [?\C-,?o] '(lambda (&optional ow) (interactive "P") (austex-copy-rectangle-to-register (string-to-char "o") ow)))
    (define-key map [?\C-,?p] '(lambda (&optional ow) (interactive "P") (austex-copy-rectangle-to-register (string-to-char "p") ow)))
    (define-key map [?\C-,?q] '(lambda (&optional ow) (interactive "P") (austex-copy-rectangle-to-register (string-to-char "q") ow)))
    (define-key map [?\C-,?r] '(lambda (&optional ow) (interactive "P") (austex-copy-rectangle-to-register (string-to-char "r") ow)))
    (define-key map [?\C-,?s] '(lambda (&optional ow) (interactive "P") (austex-copy-rectangle-to-register (string-to-char "s") ow)))
    (define-key map [?\C-,?t] '(lambda (&optional ow) (interactive "P") (austex-copy-rectangle-to-register (string-to-char "t") ow)))
    (define-key map [?\C-,?u] '(lambda (&optional ow) (interactive "P") (austex-copy-rectangle-to-register (string-to-char "u") ow)))
    (define-key map [?\C-,?v] '(lambda (&optional ow) (interactive "P") (austex-copy-rectangle-to-register (string-to-char "v") ow)))
    (define-key map [?\C-,?w] '(lambda (&optional ow) (interactive "P") (austex-copy-rectangle-to-register (string-to-char "w") ow)))
    (define-key map [?\C-,?x] '(lambda (&optional ow) (interactive "P") (austex-copy-rectangle-to-register (string-to-char "x") ow)))
    (define-key map [?\C-,?y] '(lambda (&optional ow) (interactive "P") (austex-copy-rectangle-to-register (string-to-char "y") ow)))
    (define-key map [?\C-,?z] '(lambda (&optional ow) (interactive "P") (austex-copy-rectangle-to-register (string-to-char "z") ow)))

;;; Control ; prefix:

    ;; Miscellaneous stuff (add things here as needed).
    
    ;; Mainly for commands that are supposed to take place in the
    ;; "math mode" parts of a latex buffer.

    ;; These two are also duplicated in the Latin 1 section above,
    ;; where they should be even faster to execute.
    (define-key map [?\C-\;?a] 'austex-beginning-of-math)
    (define-key map [?\C-\;?e] 'austex-end-of-math)

    ;; `austex-next-bracket' is another alternate name for the same
    ;; sort of function, I think.  The question now is whether any of
    ;; these functions work the way they should.
    (define-key map [?\C-\;?n] 'austex-next-field)
    (define-key map [?\C-\;?p] 'austex-previous-field)
    ;; How to make this function behave better when there are equality
    ;; operators inside of the terms that this set of equations is
    ;; talking about (e.g. "P(X=0)")?  Maybe something sort of like
    ;; ispell to select which comparison operators to transform?  That
    ;; would be reasonable.  Some kind of parenthesis counting would
    ;; be sufficient to deal with these examples from probability.
    (define-key map [?\C-\;?r] 'austex-turn-current-equation-into-equation-array)
    (define-key map [?\C-\;?\;] 'austex-brackets)        ; see below

    map)
  "Keymap for AusTeX minor mode.")

;; OK! we're finally done defining `austex-mode-map'!

;;; Minor mode:

(define-minor-mode austex-mode
  "Toggle AusTeX minor mode.
With no argument, this command toggles the mode.
Non-null prefix argument turns on the mode.
Null prefix argument turns off the mode.

AusTeX mode provides many useful LaTeX functions and key bindings."
  :init-value nil
  :lighter " ATX")

;;; Hacks:

;; This is a section for code that is used to maintain this file.

;; I think it is a little bit strange that single-semicolon comments
;; are not sent to the `comment-column' by TAB when there are other
;; things on the line.  This accomplishes the desired end-result.

(defun force-lisp-indent ()
  "Move comments on this line to comment column.
Unlike with `lisp-indent-line', this effect is accomplished even
when there is other text on the line."
  (interactive)
  (goto-char (line-beginning-position))
  (search-forward-regexp " +\\(;[^\n]*\\)" (line-end-position) t)
  (let ((comment (match-string 1)))
    (replace-match "")
    (goto-char (line-beginning-position))
    (let ((count comment-column))
      (while (looking-at "[^\n]")
        (forward-char 1)
        (decf count))
      (insert-char (string-to-char " ") count)
      (insert comment))))

(defun insert-over-alphabet (&rest strings)
  "Insert string with the letters of the alphabet substituted in useful places.
For example

 (insert-over-alphabet  \"\\n(define-key map [?\\\\C-,?\" \"] '(lambda () (interactive) 
                                 (austex-copy-rectangle-to-register 
                                 (string-to-char \" \"))))\")

will insert the bindings for the \\C-, used in this package."
  (dolist (elt (string-to-list "abcdefghijklmnopqrstuvwxyz"))
    (let ((letter (char-to-string elt))
          result)
      (dolist (string (butlast strings 1))
        (setq result (concat result (concat string letter))))
      (insert result (car (last strings))))))

;; Local Variables:
;; mode:Emacs-Lisp
;; comment-column:70
;; End:

(provide 'austex-bindings)

;;; End of austex-bindings.el

