/*
 * happydigger - program for cataloging archaeological finds
 * Copyright (C) 2004, 2005, 2007 Joop Stakenborg <pg4i@amsat.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include <gtk/gtk.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <sqlite3.h>

#include "interface.h"
#include "utils.h"
#include "callbacks.h"
#include "db.h"
#include "preferences.h"
#include "gui_utils.h"
#include "gui_warningdialog.h"

#ifdef WIN32
#include <windows.h>
#include <shlobj.h>
#endif

GtkWidget *window, *startupwindow = NULL;
sqlite3 *db;
statestype state;

extern preferencestype preferences;

static void
dircheck (void)
{
	struct stat statdir;
	gchar *mddir;

	mddir = g_strdup_printf ("%s%s.%s", 
		g_get_home_dir (), G_DIR_SEPARATOR_S, PACKAGE);
	if (stat (mddir, &statdir) == -1)
	{
#ifdef WIN32
		if (mkdir (mddir) == -1)
			g_error (_("Creating ~/.%s directory."), PACKAGE);
#else
		if (mkdir (mddir, S_IRUSR | S_IWUSR | S_IXUSR) == -1)
			g_error (_("Creating ~/.%s directory."), PACKAGE);
#endif
	}
	else if (!S_ISDIR (statdir.st_mode))
		g_error (_("~/.%s is not a directory."), PACKAGE);
}

/* limit length of textviews to 500 characters */
void text_changed (GtkTextBuffer *textbuffer, gpointer user_data)
{
	GtkTextIter first, last, start, stop;
	gchar *text;
	GtkTextMark *mark;

	gtk_text_buffer_get_bounds (textbuffer, &first, &last);
	text = gtk_text_buffer_get_text (textbuffer, &first, &last, FALSE);
	if (strlen(text) == 500)
	{
		mark = gtk_text_buffer_get_insert (textbuffer);
		gtk_text_buffer_get_iter_at_mark (textbuffer, &stop, mark);
		start = stop;
		gtk_text_iter_backward_chars (&start, 1);
		gtk_text_buffer_delete (textbuffer, &start, &stop);
		gdk_beep ();
	}
	gtk_text_buffer_get_bounds (textbuffer, &first, &last);
	gtk_text_buffer_apply_tag_by_name (textbuffer, "tiny", &first, &last);
}

int
main (int argc, char *argv[])
{
	GtkWidget *entry, *notes, *spotnotes, *identry, *tablecombo, *label, *widget;
	gchar *filename, *id, *msg, *title, *dom;
	gint result, rc;
	GtkTextBuffer *buffer;

	dircheck ();
	filename = g_strdup_printf ("%s%s.%s%sdb.happydigger", 
		g_get_home_dir (), G_DIR_SEPARATOR_S, PACKAGE, G_DIR_SEPARATOR_S);

	rc = sqlite3_open(filename, &db);
	if (rc)
	{
		g_print (_("Failed to open database '%s'.\n"), filename);
		return -1;
	}

	setlocale (LC_ALL, "");
#ifdef WIN32
	dom = bindtextdomain (PACKAGE, ".");
#else
	dom = bindtextdomain (PACKAGE, PACKAGE_LOCALE_DIR);
#endif
	bind_textdomain_codeset (PACKAGE, "UTF-8");
	textdomain (PACKAGE);

	gtk_set_locale ();
	gtk_init (&argc, &argv);

#ifdef WIN32
	add_pixmap_directory (".");
#else
	add_pixmap_directory (PACKAGE_DATA_DIR "/pixmaps");
#endif
	window = create_window ();
	gtk_widget_realize (window);
	
	loadpreferences ();
	state.imageloaddir = g_strconcat (g_get_home_dir(), G_DIR_SEPARATOR_S, NULL);
	state.table = g_strdup ("finds");
	state.webmapdir = g_strconcat (g_get_home_dir(), G_DIR_SEPARATOR_S, NULL);
	state.sortby = g_strdup ("find");

	/* fill in combo strings */
	set_combostrings ();

	/*
	 * See if we can read a record. If not, create a database called 'finds'
	 * and add 1 record.
	 */
	clearall ();
	identry = lookup_widget (window, "identry");
	entry = lookup_widget (window, "entry");
	gtk_entry_set_text (GTK_ENTRY (entry), "1");
	result = load_from_db (1);
	if (result == SQLITE_ERROR)
	{
		result = create_db ("finds");
		if (result != SQLITE_OK)
		{
			msg = g_strdup_printf
				(_("Error creating table %s: %s"), "finds", sqlite3_errmsg (db));
			warningdialog (_("happydigger - error"), msg, "gtk-dialog-error");
			g_free (msg);
			sqlite3_close (db);
			g_free (state.imageloaddir);
			g_free (state.table);
			g_free (state.webmapdir);
			gtk_main_quit ();
		}
		else
		{
			msg = g_strdup_printf
				(_("Table %s succesfully created"), "finds");
			warningdialog (_("happydigger - startup"), msg, "gtk-dialog-info");
			result = add_to_db ();
			id = g_strdup_printf ("%d", result);
			gtk_entry_set_text (GTK_ENTRY(identry), id);
			g_free (id);
		}
	}
	else
	{	/* find current tables in the database and add to the combobox */
		startupwindow = create_startupdialog ();
		find_tables ();
		tablecombo = lookup_widget (startupwindow, "tablecombo");
		gtk_combo_box_set_active (GTK_COMBO_BOX(tablecombo), 0);
		gtk_entry_set_activates_default
			(GTK_ENTRY (gtk_bin_get_child (GTK_BIN (tablecombo))), TRUE);
		gtk_dialog_set_default_response
			(GTK_DIALOG (startupwindow), GTK_RESPONSE_OK);
		gtk_widget_show (startupwindow);
	}

	notes = lookup_widget (window, "notes");
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (notes));
	gtk_text_buffer_create_tag (buffer, "tiny", "size", 8 * PANGO_SCALE, NULL);
	g_signal_connect (G_OBJECT (buffer), "changed", G_CALLBACK (text_changed), NULL);
	spotnotes = lookup_widget (window, "spotnotes");
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (spotnotes));
	gtk_text_buffer_create_tag (buffer, "tiny", "size", 8 * PANGO_SCALE, NULL);
	g_signal_connect (G_OBJECT (buffer), "changed", G_CALLBACK (text_changed), NULL);

	label = lookup_widget (window, "label6");
	widget = lookup_widget (window, "typecombo");
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), widget);
	label = lookup_widget (window, "label7");
	widget = lookup_widget (window, "materialcombo");
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), widget);
	label = lookup_widget (window, "label14");
	widget = lookup_widget (window, "periodcombo");
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), widget);
	label = lookup_widget (window, "label13");
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), notes);
	label = lookup_widget (window, "label12");
	gtk_label_set_mnemonic_widget (GTK_LABEL (label), spotnotes);

/* uncomment next for debugging purposes */
	gtk_widget_hide (identry);

	if (!startupwindow)
	{
		title = g_strdup_printf (_("happydigger - table %s"), state.table);
		gtk_window_set_title (GTK_WINDOW(window), title);
		g_free (title);
		gtk_widget_show (window);
	}

	gtk_main ();
	return 0;
}

