/*
 * happydigger - program for cataloging archaeological finds
 * Copyright (C) 2004, 2005, 2007 Joop Stakenborg <pg4i@amsat.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include <gtk/gtk.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sqlite3.h>
#include <math.h>
#include "callbacks.h"
#include "utils.h"
#include "db.h"
#include "interface.h"
#include "preferences.h"
#include "gui_utils.h"

extern GtkWidget *window;
extern preferencestype preferences;
extern sqlite3 *db;
extern statestype state;
GtkWidget *mapchooserdialog;

/* In points */
#define HEADER_HEIGHT (10*72/25.4)
#define HEADER_GAP (3*72/25.4)

typedef struct 
{
	gdouble font_size;
	gint lines_per_page;
	gchar **lines;
} PrintData;

/* TOOLBAR */
void
on_firstbutton_clicked (GtkButton *button,	gpointer user_data)
{
	GtkWidget *entry;
	
	if (max_db () > 0)
	{
		clearall ();
		load_from_db (1);
		entry = lookup_widget (window, "entry");
		gtk_entry_set_text (GTK_ENTRY(entry), "1");
	}
}


void
on_previousbutton_clicked	(GtkButton *button, gpointer user_data)
{
	GtkWidget *entry;
	gchar *line, *newline;
	gint nr;
	
	entry = lookup_widget (window, "entry");
	line = gtk_editable_get_chars (GTK_EDITABLE(entry), 0, -1);
	nr = atoi (line);
	if (nr > 1)
	{
		nr--;
		clearall ();
		load_from_db (nr);
		newline = g_strdup_printf ("%d", nr);
		gtk_entry_set_text (GTK_ENTRY(entry), newline);
		g_free (newline);
	}
	g_free (line);
}


void
on_nextbutton_clicked	(GtkButton	*button, gpointer user_data)
{
	GtkWidget *entry;
	gchar *line, *newline;
	gint nr;

	entry = lookup_widget (window, "entry");
	line = gtk_editable_get_chars (GTK_EDITABLE(entry), 0, -1);
	nr = atoi (line);
	if (nr < max_db ())
	{
		nr++;
		clearall ();
		load_from_db (nr);
		newline = g_strdup_printf ("%d", nr);
		gtk_entry_set_text (GTK_ENTRY(entry), newline);
		g_free (newline);
	}
	g_free (line);
}

void
on_rewindbutton_clicked (GtkButton *button,	gpointer user_data)
{
	GtkWidget *entry;
	gchar *line, *newline;
	gint nr;
	
	entry = lookup_widget (window, "entry");
	line = gtk_editable_get_chars (GTK_EDITABLE(entry), 0, -1);
	nr = atoi (line);
	if (nr > 5)
	{
		nr = nr - 5;
		clearall ();
		load_from_db (nr);
		newline = g_strdup_printf ("%d", nr);
		gtk_entry_set_text (GTK_ENTRY(entry), newline);
		g_free (newline);
	}
	g_free (line);
}


void
on_ffbutton_clicked	 (GtkButton	 *button, gpointer user_data)
{
	GtkWidget *entry;
	gchar *line, *newline;
	gint nr;
	
	entry = lookup_widget (window, "entry");
	line = gtk_editable_get_chars (GTK_EDITABLE(entry), 0, -1);
	nr = atoi (line);
	if (nr <= max_db () - 5)
	{
		nr = nr + 5;
		clearall ();
		load_from_db (nr);
		newline = g_strdup_printf ("%d", nr);
		gtk_entry_set_text (GTK_ENTRY(entry), newline);
		g_free (newline);
	}
	g_free (line);
}

void
on_lastbutton_clicked (GtkButton *button, gpointer user_data)
{
	GtkWidget *entry;
	gchar *str;
	
	clearall ();
	load_from_db (max_db ());
	entry = lookup_widget (window, "entry");
	str = g_strdup_printf ("%d", max_db ());
	gtk_entry_set_text (GTK_ENTRY(entry), str);
	g_free (str);
}

void
on_applybutton_clicked                 (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget *identry;
	gchar *id;
	gint findnr;

	identry = lookup_widget (window, "identry");
	id = gtk_editable_get_chars (GTK_EDITABLE(identry), 0, -1);
	findnr = atoi (id);
	update_db (findnr);
	g_free (id);
}


void
on_removebutton_clicked                (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget *entry, *identry;
	gchar *id;
	gint findnr, nr, max;

	entry = lookup_widget (window, "entry");
	identry = lookup_widget (window, "identry");
	id = gtk_editable_get_chars (GTK_EDITABLE(identry), 0, -1);
	findnr = atoi (id);
	delete_from_db (findnr);
	clearall ();
	max = max_db ();

	if (max == 0)
	{
		findnr = add_to_db ();
		id = g_strdup_printf("%d", findnr);
		gtk_entry_set_text (GTK_ENTRY(identry), id);
		gtk_entry_set_text (GTK_ENTRY(entry), "1");
	}
	else
	{
		id = gtk_editable_get_chars (GTK_EDITABLE(entry), 0, -1);
		nr = atoi (id);
		if (nr > 1)	nr --;
		load_from_db (nr);
		id = g_strdup_printf("%d", nr);
		gtk_entry_set_text (GTK_ENTRY(entry), id);
	}
	g_free (id);
}


void
on_newbutton_clicked                   (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget *entry, *identry;
	gchar *line;
	gint max, id;

	entry = lookup_widget (window, "entry");
	identry = lookup_widget (window, "identry");
	clearall ();
	max = max_db () + 1;
	line = g_strdup_printf ("%d", max);
	gtk_entry_set_text (GTK_ENTRY(entry), line);
	id = add_to_db ();
	line = g_strdup_printf ("%d", id);
	gtk_entry_set_text (GTK_ENTRY(identry), line);
	g_free (line);
}

static int list_callback (void *store, int argc, char **argv, char **columnNames)
{
	GtkTreeIter iter;

	gtk_tree_store_append (store, &iter, NULL);
	/* unselected comboboxentries produce a NULL entry */
	if (argv[3] && strcmp(argv[3], "(NULL)"))
		gtk_tree_store_set (store, &iter, 3, argv[3], -1);
	if (argv[4] && strcmp(argv[4], "(NULL)"))
		gtk_tree_store_set (store, &iter, 4, argv[4], -1);
	if (argv[5] && strcmp(argv[5], "(NULL)"))
		gtk_tree_store_set (store, &iter, 5, argv[5], -1);
	/* rest of the data */
	gtk_tree_store_set (store, &iter, 0, argv[0], 1, argv[1], 2, argv[2],
		6, argv[6], 7, argv[7], 8, argv[8], 9, argv[9], 10, argv[10],
		11, argv[11], 12, argv[12], 13, argv[13], 14, argv[14], 15, argv[15], -1);
	return 0;
}

void
on_listbutton_clicked                  (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget *listwindow, *listscrolledwindow, *treeview, *handlebox, *identry;
	GtkTreeStore *store;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;
	gchar **split, *query, *err, *id;
	gint i, ret, findnr, pathnr;
	GObject *selection;
	GtkTreePath *path;
	
	listwindow = create_listwindow ();
	listscrolledwindow = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW(listscrolledwindow),
		GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_widget_show (listscrolledwindow);
	gtk_container_add (GTK_CONTAINER (listwindow), listscrolledwindow);

	store = gtk_tree_store_new (16, G_TYPE_STRING,
		G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, 
		G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, 
		G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING);
	treeview = gtk_tree_view_new_with_model (GTK_TREE_MODEL(store));
	g_object_unref (G_OBJECT (store));

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes
		(_("Id"), renderer, "text", 0, NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes
		(_("Number"), renderer, "text", 1, NULL);
	gtk_tree_view_column_set_sort_column_id (column, 1);
	gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes
		(_("Description"), renderer, "text", 2, NULL);
	gtk_tree_view_column_set_sort_column_id (column, 2);
	gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes
		(_("Type"), renderer, "text", 3, NULL);
	gtk_tree_view_column_set_sort_column_id (column, 3);
	gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes
		(_("Material"), renderer, "text", 4, NULL);
	gtk_tree_view_column_set_sort_column_id (column, 4);
	gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes
		(_("Period"), renderer, "text", 5, NULL);
	gtk_tree_view_column_set_sort_column_id (column, 5);
	gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes
		(_("Findnotes"), renderer, "text", 6, NULL);
	gtk_tree_view_column_set_sort_column_id (column, 6);
	gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes
		(_("Date"), renderer, "text", 7, NULL);
	gtk_tree_view_column_set_sort_column_id (column, 7);
	gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes
		(_("Value"), renderer, "text", 8, NULL);
	gtk_tree_view_column_set_sort_column_id (column, 8);
	gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes
		(_("Reference"), renderer, "text", 9, NULL);
	gtk_tree_view_column_set_sort_column_id (column, 9);
	gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes
		(_("Find date"), renderer, "text", 10, NULL);
	gtk_tree_view_column_set_sort_column_id (column, 10);
	gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes
		(_("Location"), renderer, "text", 11, NULL);
	gtk_tree_view_column_set_sort_column_id (column, 11);
	gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes
		(_("Coordinates"), renderer, "text", 12, NULL);
	gtk_tree_view_column_set_sort_column_id (column, 12);
	gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes
		(_("Spotnotes"), renderer, "text", 13, NULL);
	gtk_tree_view_column_set_sort_column_id (column, 13);
	gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes
		(_("Image1"), renderer, "text", 14, NULL);
	gtk_tree_view_column_set_sort_column_id (column, 14);
	gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes
		(_("Image2"), renderer, "text", 15, NULL);
	gtk_tree_view_column_set_sort_column_id (column, 15);
	gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);

	/* column with ID is hidden, other depend on preferences */
	column = gtk_tree_view_get_column (GTK_TREE_VIEW(treeview), 0);
	gtk_tree_view_column_set_visible (GTK_TREE_VIEW_COLUMN(column), FALSE);
	split = g_strsplit(preferences.colvisible, ",", 0);
	for (i = 0; i < 15; i++)
	{
		column = gtk_tree_view_get_column (GTK_TREE_VIEW(treeview), i + 1);
		gtk_tree_view_column_set_visible
			(GTK_TREE_VIEW_COLUMN(column), atoi(split[i]) ? TRUE : FALSE);
	}
	g_strfreev(split);
	gtk_container_add (GTK_CONTAINER (listscrolledwindow), treeview);
					
	query = g_strdup_printf ("SELECT * from %s;", state.table);
	ret = sqlite3_exec (db, query, list_callback, store, &err);
	if (ret != SQLITE_OK)
		g_print ("Error on SELECT: %s\n", err);
	g_free (query);

	/* hide handlebox, so we can't update when the list is visible */
	handlebox = lookup_widget (window, "handlebox1");
	gtk_widget_hide (handlebox);

	gtk_widget_show_all (listwindow);

	/* select current record in the list */
 	identry = lookup_widget (window, "identry");
	id = gtk_editable_get_chars (GTK_EDITABLE(identry), 0, -1);
	findnr = atoi (id);
	pathnr = findnr - 1;
	id = g_strdup_printf ("%d", pathnr);
	path = gtk_tree_path_new_from_string (id);
	g_free (id);
	selection = G_OBJECT (gtk_tree_view_get_selection(GTK_TREE_VIEW (treeview)));
	gtk_tree_selection_select_path (GTK_TREE_SELECTION(selection), path);
	gtk_tree_path_free (path);

	g_signal_connect (selection, "changed", G_CALLBACK (on_list_select_row), NULL);
}

void
on_prefsbutton_clicked                 (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget *preferencesdialog, *typestextview, *materialstextview,
		*periodstextview;
	GtkWidget *p1, *p2, *p3, *p4, *p5, *p6, *p7, *p8, *p9, *p10, *p11, *p12, *p13, *p14, *p15;
	gchar **split;
	GtkTextBuffer *buffer;

	preferencesdialog = create_preferencesdialog ();

	typestextview = lookup_widget (preferencesdialog, "typestextview");
	materialstextview = lookup_widget (preferencesdialog, "materialstextview");
	periodstextview = lookup_widget (preferencesdialog, "periodstextview");
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (typestextview));
	gtk_text_buffer_set_text (buffer, preferences.types, -1);
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (materialstextview));
	gtk_text_buffer_set_text (buffer, preferences.materials, -1);
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (periodstextview));
	gtk_text_buffer_set_text (buffer, preferences.periods, -1);

	split = g_strsplit(preferences.colvisible, ",", 0);
	p1 = lookup_widget (preferencesdialog, "p1");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (p1), atoi(split[0]));
	p2 = lookup_widget (preferencesdialog, "p2");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (p2), atoi(split[1]));
	p3 = lookup_widget (preferencesdialog, "p3");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (p3), atoi(split[2]));
	p4 = lookup_widget (preferencesdialog, "p4");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (p4), atoi(split[3]));
	p5 = lookup_widget (preferencesdialog, "p5");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (p5), atoi(split[4]));
	p6 = lookup_widget (preferencesdialog, "p6");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (p6), atoi(split[5]));
	p7 = lookup_widget (preferencesdialog, "p7");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (p7), atoi(split[6]));
	p8 = lookup_widget (preferencesdialog, "p8");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (p8), atoi(split[7]));
	p9 = lookup_widget (preferencesdialog, "p9");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (p9), atoi(split[8]));
	p10 = lookup_widget (preferencesdialog, "p10");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (p10), atoi(split[9]));
	p11 = lookup_widget (preferencesdialog, "p11");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (p11), atoi(split[10]));
	p12 = lookup_widget (preferencesdialog, "p12");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (p12), atoi(split[11]));
	p13 = lookup_widget (preferencesdialog, "p13");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (p13), atoi(split[12]));
	p14 = lookup_widget (preferencesdialog, "p14");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (p14), atoi(split[13]));
	p15 = lookup_widget (preferencesdialog, "p15");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (p15), atoi(split[14]));
	g_strfreev(split);	gtk_widget_show (preferencesdialog);
}

void
on_helpbutton_clicked                  (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget *helpwindow, *helptextview;
	gchar *helpfile, *b;
	FILE *in;
	GtkTextBuffer *buffer;
	GtkTextIter start, end;
	const gchar *encoding;
	gsize utf8_len, read_len;
	GError *err;

	helpwindow = create_helpwindow ();
	helptextview = lookup_widget (helpwindow, "helptextview");
	gtk_text_view_set_editable (GTK_TEXT_VIEW(helptextview), FALSE);
	gtk_text_view_set_cursor_visible (GTK_TEXT_VIEW(helptextview), FALSE);

/* TRANSLATORS:
 * Do not translate README unless you provide a readme in your language,
 * e.g. the dutch readme is called LEESMIJ.
 */
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW(helptextview));
	gtk_text_buffer_get_bounds (buffer, &start, &end);
	g_get_charset (&encoding);
	b = g_new0 (gchar, 100);

#ifdef WIN32
	helpfile = g_strdup_printf ("%s", _("README"));
#else
	helpfile = g_strdup_printf ("%s%s%s", PACKAGE_DATA_DIR, G_DIR_SEPARATOR_S, _("README"));
#endif
	in = fopen (helpfile, "r");
	if (in)
	{
		do 
		{
			if (fgets (b, 80, in) == NULL) 
				break;
			else
			{
				err = NULL;
				g_convert_with_fallback
(b, strlen(b), "UTF-8", encoding, "?", &read_len, &utf8_len, &err);
				if (err)
				{
					g_print ("%s\n", err->message);
					g_error_free (err);
				}
				gtk_text_buffer_insert (buffer, &start, b, -1);
			}
		} 
		while (!feof (in));
		fclose (in);
	}
	g_free (b);
	g_free (helpfile);
	gtk_widget_show (helpwindow);
}

#define GLADE_HOOKUP_OBJECT(component,widget,name) \
  g_object_set_data_full (G_OBJECT (component), name, \
    gtk_widget_ref (widget), (GDestroyNotify) gtk_widget_unref)

static void
on_reverse_toggled (GtkToggleButton *togglebutton, gpointer user_data)
{
  gboolean state = gtk_toggle_button_get_active (togglebutton);
  if (state)
    preferences.reverseorder = 1;
  else
    preferences.reverseorder = 0;
}

static void
on_itemscombo_changed (GtkComboBox *combo, gpointer user_data)
{
	gint item = gtk_combo_box_get_active (combo);
	preferences.recordsperpage = 2 * (item + 1);
}

static void
on_sortcombo_changed (GtkComboBox *combo, gpointer user_data)
{
	gint item = gtk_combo_box_get_active (combo);
	preferences.sortby = item;
	switch (item)
	{
		case 0: state.sortby = g_strdup ("find");	break;
		case 1: state.sortby = g_strdup ("nr");	break;
		case 2: state.sortby = g_strdup ("description");	break;
		case 3: state.sortby = g_strdup ("type");	break;
		case 4: state.sortby = g_strdup ("material");	break;
		case 5: state.sortby = g_strdup ("period");	break;
		case 6: state.sortby = g_strdup ("notes");	break;
		case 7: state.sortby = g_strdup ("date");	break;
		case 8: state.sortby = g_strdup ("value");	break;
		case 9: state.sortby = g_strdup ("reference"); break;
		case 10: state.sortby = g_strdup ("finddate"); break;
		case 11: state.sortby = g_strdup ("lcoation"); break;
		case 12: state.sortby = g_strdup ("coordinates"); break;
		case 13: state.sortby = g_strdup ("spotnotes"); break;
		default: state.sortby = g_strdup ("find"); break;
	}
}

void
on_webbutton_clicked                   (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget *vbox, *hbox1, *hbox2, *label, *entry, *check, *nritems, *nrlabel,
		*sep, *sortlabel, *sortitems, *hbox3, *footlabel, *footentry;

	mapchooserdialog = create_mapchooserdialog ();
	vbox = gtk_vbox_new (FALSE, 0);
	hbox1 = gtk_hbox_new (FALSE, 10);
	hbox2 = gtk_hbox_new (FALSE, 10);
	hbox3 = gtk_hbox_new (FALSE, 10);

	label = gtk_label_new (_("Webpage title:"));
	entry = gtk_entry_new ();
	GLADE_HOOKUP_OBJECT (mapchooserdialog, entry, "webpageentry");
	if (strncasecmp (preferences.webpagetitle, "^", 1))
		gtk_entry_set_text (GTK_ENTRY(entry), preferences.webpagetitle);

	sortlabel = gtk_label_new (_("Sort by"));
	sortitems = gtk_combo_box_new_text ();
	gtk_combo_box_append_text (GTK_COMBO_BOX(sortitems), "Find");
	gtk_combo_box_append_text (GTK_COMBO_BOX(sortitems), "Number");
	gtk_combo_box_append_text (GTK_COMBO_BOX(sortitems), "Description");
	gtk_combo_box_append_text (GTK_COMBO_BOX(sortitems), "Type");
	gtk_combo_box_append_text (GTK_COMBO_BOX(sortitems), "Material");
	gtk_combo_box_append_text (GTK_COMBO_BOX(sortitems), "Period");
	gtk_combo_box_append_text (GTK_COMBO_BOX(sortitems), "Findnotes");
	gtk_combo_box_append_text (GTK_COMBO_BOX(sortitems), "Date");
	gtk_combo_box_append_text (GTK_COMBO_BOX(sortitems), "Value");
	gtk_combo_box_append_text (GTK_COMBO_BOX(sortitems), "Reference");
	gtk_combo_box_append_text (GTK_COMBO_BOX(sortitems), "Find date");
	gtk_combo_box_append_text (GTK_COMBO_BOX(sortitems), "Location");
	gtk_combo_box_append_text (GTK_COMBO_BOX(sortitems), "Coordinates");
	gtk_combo_box_append_text (GTK_COMBO_BOX(sortitems), "Spotnotes");
	gtk_combo_box_set_active (GTK_COMBO_BOX (sortitems), preferences.sortby);
	g_signal_connect ((gpointer) sortitems, "changed",
		G_CALLBACK (on_sortcombo_changed), NULL);

	check = gtk_check_button_new_with_mnemonic (_("_Reverse order"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check),
		preferences.reverseorder);
	g_signal_connect ((gpointer) check, "toggled",
		G_CALLBACK (on_reverse_toggled), NULL);

	nrlabel = gtk_label_new (_("Items per page"));
	nritems = gtk_combo_box_new_text ();
	gtk_combo_box_append_text (GTK_COMBO_BOX(nritems), "2");
	gtk_combo_box_append_text (GTK_COMBO_BOX(nritems), "4");
	gtk_combo_box_append_text (GTK_COMBO_BOX(nritems), "6");
	gtk_combo_box_append_text (GTK_COMBO_BOX(nritems), "8");
	gtk_combo_box_append_text (GTK_COMBO_BOX(nritems), "10");
	gtk_combo_box_append_text (GTK_COMBO_BOX(nritems), "12");
	gtk_combo_box_set_active (GTK_COMBO_BOX (nritems),
		preferences.recordsperpage/2 - 1);
	g_signal_connect ((gpointer) nritems, "changed",
		G_CALLBACK (on_itemscombo_changed), NULL);

	footlabel = gtk_label_new (_("Footnote:"));
	footentry = gtk_entry_new ();
	GLADE_HOOKUP_OBJECT (mapchooserdialog, footentry, "webpagefootentry");
	if (strncasecmp (preferences.webpagefootnote, "^", 1))
		gtk_entry_set_text (GTK_ENTRY(footentry), preferences.webpagefootnote);

	gtk_box_pack_start (GTK_BOX (hbox1), label, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox1), entry, TRUE, TRUE, 0);

	gtk_box_pack_start (GTK_BOX (hbox2), sortlabel, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox2), sortitems, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox2), check, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox2), nrlabel, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox2), nritems, TRUE, TRUE, 0);

	gtk_box_pack_start (GTK_BOX (hbox3), footlabel, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox3), footentry, TRUE, TRUE, 0);

	sep = gtk_hseparator_new ();
	gtk_box_pack_start (GTK_BOX (vbox), sep, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), hbox1, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), hbox2, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), hbox3, FALSE, FALSE, 0);
	sep = gtk_hseparator_new ();
	gtk_box_pack_start (GTK_BOX (vbox), sep, FALSE, FALSE, 0);

	gtk_file_chooser_set_extra_widget
		(GTK_FILE_CHOOSER(mapchooserdialog), vbox);
	gtk_widget_show_all (vbox);

	gtk_widget_set_sensitive (window, 0);
	gtk_file_chooser_set_filename
		(GTK_FILE_CHOOSER(mapchooserdialog), state.webmapdir);
	gtk_widget_show (mapchooserdialog);
}

static void
begin_print (GtkPrintOperation *operation, GtkPrintContext *context,
	gpointer user_data)
{
	PrintData *data = (PrintData *)user_data;
	gdouble height;

	height = gtk_print_context_get_height (context) - HEADER_HEIGHT - HEADER_GAP;
	data->lines_per_page = floor (height / data->font_size);
	gtk_print_operation_set_n_pages (operation, 1);
}

/* get the current date, returned value has to be freed */
static gchar *
happygetdate (void)
{
	time_t current;
	struct tm *timestruct = NULL;
	gchar datenow[20], *date;
	GError *error;

	time (&current);
	timestruct = localtime (&current);
	strftime (datenow, sizeof(datenow), "%d %b %Y", timestruct);

	if (!g_utf8_validate (datenow, -1, NULL ))
	{
		date = g_locale_to_utf8 (datenow, -1, NULL, NULL, &error);
		if (!date)
		{
				g_warning (_("Unable to convert '%s' to UTF-8: %s"), datenow,
					error->message);
				g_error_free (error);
			}
 	}
	else date = g_strdup (datenow);

	return (date);
}

/* get the current time, returned value has to be freed */
static gchar *
happygettime (void)
{
	time_t current;
	struct tm *timestruct = NULL;
	gchar stimenow[20];

	time (&current);
	timestruct = localtime (&current);
	strftime (stimenow, sizeof(stimenow), "%H:%M", timestruct);
	return (g_strdup (stimenow));
}

static void
draw_page (GtkPrintOperation *operation, GtkPrintContext *context,
	gint page_nr, gpointer user_data)
{
	PrintData *data = (PrintData *)user_data;
	cairo_t *cr;
	PangoLayout *layout;
	gdouble width;
	gint imagey, linecount;
	PangoFontDescription *desc;
	gchar *str;
	GtkWidget *w1, *w2, *w3;
	GtkTextBuffer *b;
	GtkTextIter start, end;

	cr = gtk_print_context_get_cairo_context (context);
	width = gtk_print_context_get_width (context);

	cairo_rectangle (cr, 0, 0, width, HEADER_HEIGHT);
	cairo_set_source_rgb (cr, 0.8, 0.8, 0.8);
	cairo_fill_preserve (cr);
	cairo_set_source_rgb (cr, 0, 0, 0);
	cairo_set_line_width (cr, 1);
	cairo_stroke (cr);

	layout = gtk_print_context_create_pango_layout (context);
	desc = pango_font_description_from_string ("mono");
	pango_layout_set_font_description (layout, desc);
	pango_font_description_free (desc);

	/* actual printing here */
	/* db entry in the header */
	w1 = lookup_widget (window, "entry");
	str = g_strdup_printf ("%s: %s, %s: %s. %s %s %s.",
		_("Table"), state.table,
		_("record"), gtk_editable_get_chars (GTK_EDITABLE(w1), 0, -1),
		_("Printed"), happygetdate (), happygettime ());
	cairo_move_to (cr, 0, HEADER_HEIGHT/3);
	pango_layout_set_text (layout, str, -1);
	pango_layout_set_width (layout, PANGO_SCALE * width);
	pango_layout_set_alignment (layout, PANGO_ALIGN_CENTER);
	pango_cairo_show_layout (cr, layout);
	g_object_unref (layout);

	layout = gtk_print_context_create_pango_layout (context);
	desc = pango_font_description_from_string ("sans");
	pango_font_description_set_size (desc, data->font_size * PANGO_SCALE);
	pango_layout_set_font_description (layout, desc);
	pango_font_description_free (desc);
 
	pango_layout_set_wrap (layout, PANGO_WRAP_WORD);
	pango_layout_set_width (layout, PANGO_SCALE * width);

	/* find data */
	cairo_move_to (cr, 0, HEADER_HEIGHT + HEADER_GAP);
	str = g_strdup_printf
		("<b>%s</b>", _("Find description"));
	pango_layout_set_markup (layout, str, -1);
	linecount = pango_layout_get_line_count (layout);
	imagey = linecount;
	pango_cairo_show_layout (cr, layout);
	cairo_rel_move_to (cr, 0, linecount * data->font_size);

	w1 = lookup_widget (window, "nrentry");
	w2 = lookup_widget (window, "descriptionentry");
	str = g_strdup_printf ("%s: %s.",
		_("Number"), gtk_editable_get_chars (GTK_EDITABLE(w1), 0, -1));
	pango_layout_set_markup (layout, str, -1);
	linecount = pango_layout_get_line_count (layout);
	imagey = imagey + linecount;
	pango_cairo_show_layout (cr, layout);
	cairo_rel_move_to (cr, 0, linecount * data->font_size);
	str = g_strdup_printf ("%s: %s.",
		_("Description"), gtk_editable_get_chars (GTK_EDITABLE(w2), 0, -1));
	pango_layout_set_markup (layout, str, -1);
	linecount = pango_layout_get_line_count (layout);
	imagey = imagey + linecount;
	pango_cairo_show_layout (cr, layout);
	cairo_rel_move_to (cr, 0, linecount * data->font_size);

	w1 = lookup_widget (window, "typecombo");
	w2 = lookup_widget (window, "materialcombo");
	w3 = lookup_widget (window, "periodcombo");
	str = g_strdup_printf ("%s: %s.",
		_("Type"), gtk_combo_box_get_active_text (GTK_COMBO_BOX(w1)));
	pango_layout_set_markup (layout, str, -1);
	linecount = pango_layout_get_line_count (layout);
	imagey = imagey + linecount;
	pango_cairo_show_layout (cr, layout);
	cairo_rel_move_to (cr, 0, linecount * data->font_size);
	str = g_strdup_printf ("%s: %s.",
		_("Material"), gtk_combo_box_get_active_text (GTK_COMBO_BOX(w2)));
	pango_layout_set_markup (layout, str, -1);
	linecount = pango_layout_get_line_count (layout);
	imagey = imagey + linecount;
	pango_cairo_show_layout (cr, layout);
	cairo_rel_move_to (cr, 0, linecount * data->font_size);
	str = g_strdup_printf ("%s: %s.",
		_("Period"), gtk_combo_box_get_active_text (GTK_COMBO_BOX(w3)));
	pango_layout_set_markup (layout, str, -1);
	linecount = pango_layout_get_line_count (layout);
	imagey = imagey + linecount;
	pango_cairo_show_layout (cr, layout);
	cairo_rel_move_to (cr, 0, linecount * data->font_size);

	w1 = lookup_widget (window, "dateentry");
	w2 = lookup_widget (window, "valueentry");
	w3 = lookup_widget (window, "referenceentry");
	str = g_strdup_printf ("%s: %s.",
		_("Date"), strlen (gtk_editable_get_chars (GTK_EDITABLE(w1), 0, -1)) > 0 ?
		gtk_editable_get_chars (GTK_EDITABLE(w1), 0, -1): "- ");
	pango_layout_set_markup (layout, str, -1);
	linecount = pango_layout_get_line_count (layout);
	imagey = imagey + linecount;
	pango_cairo_show_layout (cr, layout);
	cairo_rel_move_to (cr, 0, linecount * data->font_size);
	str = g_strdup_printf ("%s: %s.",
		_("Value"), strlen (gtk_editable_get_chars (GTK_EDITABLE(w2), 0, -1)) > 0 ?
		gtk_editable_get_chars (GTK_EDITABLE(w2), 0, -1): "- ");
	pango_layout_set_markup (layout, str, -1);
	linecount = pango_layout_get_line_count (layout);
	imagey = imagey + linecount;
	pango_cairo_show_layout (cr, layout);
	cairo_rel_move_to (cr, 0, linecount * data->font_size);
	str = g_strdup_printf ("%s: %s.",
		_("Reference"), strlen (gtk_editable_get_chars (GTK_EDITABLE(w3), 0, -1)) > 0 ?
		gtk_editable_get_chars (GTK_EDITABLE(w3), 0, -1): "- ");
	pango_layout_set_markup (layout, str, -1);
	linecount = pango_layout_get_line_count (layout);
	imagey = imagey + linecount;
	pango_cairo_show_layout (cr, layout);
	cairo_rel_move_to (cr, 0, linecount * data->font_size);

	w1 = lookup_widget (window, "notes");
	b = gtk_text_view_get_buffer (GTK_TEXT_VIEW (w1));
	gtk_text_buffer_get_bounds (b, &start, &end);
	str = g_strdup_printf ("%s: %s.",
		_("Findnotes"), strlen (gtk_text_buffer_get_text (b, &start, &end, TRUE)) > 0 ?
		gtk_text_buffer_get_text (b, &start, &end, TRUE): "- ");
	pango_layout_set_markup (layout, str, -1);
	linecount = pango_layout_get_line_count (layout);
	imagey = imagey + linecount + 2;
	pango_cairo_show_layout (cr, layout);
	cairo_rel_move_to (cr, 0, (linecount + 2) * data->font_size);

	/* findspot data */
	str = g_strdup_printf
		("<b>%s</b>", _("Findspot details"));
	pango_layout_set_markup (layout, str, -1);
	linecount = pango_layout_get_line_count (layout);
	imagey = imagey + linecount;
	pango_cairo_show_layout (cr, layout);
	cairo_rel_move_to (cr, 0, linecount * data->font_size);

	w1 = lookup_widget (window, "finddateentry");
	w2 = lookup_widget (window, "locationentry");
	w3 = lookup_widget (window, "coordinatesentry");
	str = g_strdup_printf ("%s: %s.",
		_("Find date"), strlen (gtk_editable_get_chars (GTK_EDITABLE(w1), 0, -1)) > 0 ?
		gtk_editable_get_chars (GTK_EDITABLE(w1), 0, -1): "- ");
	pango_layout_set_markup (layout, str, -1);
	linecount = pango_layout_get_line_count (layout);
	imagey = imagey + linecount;
	pango_cairo_show_layout (cr, layout);
	cairo_rel_move_to (cr, 0, linecount * data->font_size);
	str = g_strdup_printf ("%s: %s.",
		_("Location"), strlen (gtk_editable_get_chars (GTK_EDITABLE(w2), 0, -1)) > 0 ?
		gtk_editable_get_chars (GTK_EDITABLE(w2), 0, -1): "- ");
	pango_layout_set_markup (layout, str, -1);
	linecount = pango_layout_get_line_count (layout);
	imagey = imagey + linecount;
	pango_cairo_show_layout (cr, layout);
	cairo_rel_move_to (cr, 0, linecount * data->font_size);
	str = g_strdup_printf ("%s: %s.",
		_("Coordinates"), strlen (gtk_editable_get_chars (GTK_EDITABLE(w3), 0, -1)) > 0 ?
		gtk_editable_get_chars (GTK_EDITABLE(w3), 0, -1): "- ");
	pango_layout_set_markup (layout, str, -1);
	linecount = pango_layout_get_line_count (layout);
	imagey = imagey + linecount;
	pango_cairo_show_layout (cr, layout);
	cairo_rel_move_to (cr, 0, linecount * data->font_size);

	w1 = lookup_widget (window, "spotnotes");
	b = gtk_text_view_get_buffer (GTK_TEXT_VIEW (w1));
	gtk_text_buffer_get_bounds (b, &start, &end);
	str = g_strdup_printf ("%s: %s.",
		_("Spotnotes"), strlen (gtk_text_buffer_get_text (b, &start, &end, TRUE)) > 0 ?
		gtk_text_buffer_get_text (b, &start, &end, TRUE): "- ");
	pango_layout_set_markup (layout, str, -1);
	linecount = pango_layout_get_line_count (layout);
	imagey = imagey + linecount + 2;
	pango_cairo_show_layout (cr, layout);
	cairo_rel_move_to (cr, 0, (linecount + 2) * data->font_size);

	/* images */
	/* findspot data */
	str = g_strdup_printf
		("<b>%s</b>", _("Images"));
	pango_layout_set_markup (layout, str, -1);
	linecount = pango_layout_get_line_count (layout);
	imagey = imagey + linecount;
	pango_cairo_show_layout (cr, layout);
	cairo_rel_move_to (cr, 0, linecount * data->font_size);

	w1 = lookup_widget (window, "image1entry");
	w2 = lookup_widget (window, "image2entry");
	str = g_strdup_printf ("%s 1: %s.",
		_("Image"), strlen (gtk_editable_get_chars (GTK_EDITABLE(w1), 0, -1)) > 0 ?
		gtk_editable_get_chars (GTK_EDITABLE(w1), 0, -1): "- ");
	pango_layout_set_markup (layout, str, -1);
	linecount = pango_layout_get_line_count (layout);
	imagey = imagey + linecount;
	pango_cairo_show_layout (cr, layout);
	cairo_rel_move_to (cr, 0, linecount * data->font_size);
	str = g_strdup_printf ("%s 2: %s.",
		_("Image"), strlen (gtk_editable_get_chars (GTK_EDITABLE(w2), 0, -1)) > 0 ?
		gtk_editable_get_chars (GTK_EDITABLE(w2), 0, -1): "- ");
	pango_layout_set_markup (layout, str, -1);
	linecount = pango_layout_get_line_count (layout);
	imagey = imagey + linecount;
	pango_cairo_show_layout (cr, layout);
	cairo_rel_move_to (cr, 0, linecount * data->font_size);

	str = gtk_editable_get_chars (GTK_EDITABLE(w1), 0, -1);
	GdkPixbuf *pix = gdk_pixbuf_new_from_file_at_size (str, 250, 250, NULL);
	if (pix)
	{
		gdk_cairo_set_source_pixbuf (cr, pix, 0,
			HEADER_HEIGHT + HEADER_GAP + ((imagey + 2) * data->font_size));
		cairo_paint (cr);
		g_object_unref (pix);
	}
	str = gtk_editable_get_chars (GTK_EDITABLE(w2), 0, -1);
	pix = gdk_pixbuf_new_from_file_at_size (str, 250, 250, NULL);
	if (pix)
	{
		gdk_cairo_set_source_pixbuf (cr, pix, 300,
			HEADER_HEIGHT + HEADER_GAP + ((imagey + 2)* data->font_size));
		cairo_paint (cr);
		g_object_unref (pix);
	}
	g_free (str);
	g_object_unref (layout);
}

static void
end_print (GtkPrintOperation *operation, GtkPrintContext *context,
	gpointer user_data)
{
	PrintData *data = (PrintData *)user_data;

	g_free (data);
}

void
on_printbutton_clicked (GtkButton *button, gpointer user_data)
{
	GtkPrintOperation *operation;
	GError *error = NULL;
	PrintData *data;

	operation = gtk_print_operation_new ();
	data = g_new0 (PrintData, 1);
	data->font_size = 12.0;

	g_signal_connect (G_OBJECT (operation), "begin-print", 
				G_CALLBACK (begin_print), data);
	g_signal_connect (G_OBJECT (operation), "draw-page", 
				G_CALLBACK (draw_page), data);
	g_signal_connect (G_OBJECT (operation), "end-print", 
				G_CALLBACK (end_print), data);

	gtk_print_operation_run (operation, GTK_PRINT_OPERATION_ACTION_PRINT_DIALOG,
		GTK_WINDOW (window), &error);

	g_object_unref (operation);
	if (error)
	{
		GtkWidget *dialog;

		dialog = gtk_message_dialog_new (GTK_WINDOW (window),
			GTK_DIALOG_DESTROY_WITH_PARENT, GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
			"%s", error->message);
		g_error_free (error);
		g_signal_connect (dialog, "response", G_CALLBACK (gtk_widget_destroy), NULL);
		gtk_widget_show (dialog); 
	}

}

