/*
 * $Id: model.c,v 1.16 2003/10/13 09:24:14 nicoo Exp $
 *
 *
 * Copyright (C) 2003 Nicolas LAURENT
 * This file is part of `Haplo'
 * 
 *
 * `Haplo'  is free software;  you can  redistribute  it and/or modify it
 * under the terms of the GNU Library General Public License as published
 * by the Free Software Foundation;  either version 2  of the License, or
 * (at your option) any later version.
 *
 * `Haplo' is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the  implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the  GNU General Public License
 * for more details.
 * 
 * You should have  received  a copy of the   GNU General Public  License
 * along with `Haplo'.  If not, write to  the
 *
 *                                        Free Software Foundation,  Inc.
 *                                        675 Mass Ave, Cambridge, MA
 *                                        02139, USA.
 *
 */

#include <stdio.h>
#include <haplo.h>


#include "model.h"


/*-----------------------------------------------------------------------------
                             P R O T O T Y P E S 
-----------------------------------------------------------------------------*/

fem_model_t *fem_model_new(const fem_meshing_t *meshing);
void fem_model_free(fem_model_t *model);
static const char *model_type(fem_model_type_t type);
void fem_model_display(const fem_model_t *model);
fem_model_t *fem_model_copy(const fem_model_t *model);
void fem_model_print(const fem_model_t *model);
void fem_model_set_material(fem_model_t *model,
			    const fem_material_t *material);
void fem_model_set_property(fem_model_t *model,
			    const fem_property_t *property);
fem_meshing_t *fem_model_meshing(const fem_model_t *model);


/*-----------------------------------------------------------------------------
                         I M P L E M E N T A T I O N 
-----------------------------------------------------------------------------*/

/**
 *
 */
fem_model_t *fem_model_new(const fem_meshing_t *meshing)
{
	model_t *model=NULL;

	if (NB_MESHES(MESHING(meshing)))
	{
		HAPLO_ALLOC(model, 1);

		model->meshing=MESHING(meshing);
		model->type=FEM_MODEL_MECANIC;
		model->element=__fem_element_create(MESHING(meshing));

		if (! model->element)
		{
			haplo_error("Creation of element impossible.");
			HAPLO_FREE(model);
			model=NULL;
		}
	} else {
		haplo_error("This meshing do not contain meshes.");
	}

	return((fem_model_t *)model);
}


/**
 *
 */
void fem_model_free(fem_model_t *model)
{
	HAPLO_FREE(MODEL(model)->element);
	HAPLO_FREE(model);
	
	return;
}


/**
 *
 */
static const char *model_type(fem_model_type_t type)
{
	const char *s="[Undefined]";
	
	switch(type)
	{
	case FEM_MODEL_MECANIC:
		s="Mecanic";
		break;
	}
	
	return(s);
}


/**
 *
 */
void fem_model_display(const fem_model_t *model)
{
	printf("Model (%s %hdD)",
	       model_type(MODEL(model)->type),
	       MODEL(model)->meshing->node_list.dim);
	
	return;
}


/**
 *
 */
fem_model_t *fem_model_copy(const fem_model_t *model)
{
	model_t *copy;

	HAPLO_ALLOC(copy, 1);

	copy->meshing=fem_meshing_copy(MODEL(model)->meshing);
	copy->type=MODEL(model)->type;
	copy->element=__fem_element_create(copy->meshing);

	return((fem_model_t *)model);
}


/**
 *
 */
void fem_model_print(const fem_model_t *model)
{
	unsigned long i;
	
	haplo_bordered("MODEL");
	
	haplo_underlined("%lu element%s",
			 NB_MESHES(MODEL(model)->meshing),
			 (NB_MESHES(MODEL(model)->meshing)>1)?"s":"");
	
	for(i=0; i<NB_MESHES(MODEL(model)->meshing); i++)
	{
		const element_t *e=MODEL(model)->element+i;
		unsigned short j;
		
		printf("%4lu. %s: ", i+1, e->type->name);
		for(j=0; j<e->type->nb_node; j++)
			printf("N%-4lu ", e->mesh->nodes[j]->n+1);
		printf("\n");
	}

	return;
}


/**
 *
 */
void fem_model_set_material(fem_model_t *model, const fem_material_t *material)
{
	const unsigned long nb_meshes=NB_MESHES(MODEL(model)->meshing);
	unsigned long i;
	
	for(i=0; i<nb_meshes; i++)
		MODEL(model)->element[i].material = MATERIAL(material);	
	return;
}


/**
 *
 */
void fem_model_set_property(fem_model_t *model, const fem_property_t *property)
{
	const unsigned long nb_meshes=NB_MESHES(MODEL(model)->meshing);
	unsigned long i;
	
	for(i=0; i<nb_meshes; i++)
		MODEL(model)->element[i].property = PROPERTY(property);

	return;
}


/**
 *
 */
fem_meshing_t *fem_model_meshing(const fem_model_t *model)
{
	return(fem_meshing_copy((fem_meshing_t *)MODEL(model)->meshing));
}
