/*
 * $Id: load.c,v 1.4 2003/10/05 20:03:48 nicoo Exp $
 *
 *
 * Copyright (C) 1999, 2000, 2001 Nicolas LAURENT
 * This file is part of `Haplo'
 * 
 *
 * `Haplo'  is free software;  you can  redistribute  it and/or modify it
 * under the terms of the GNU Library General Public License as published
 * by the Free Software Foundation;  either version 2  of the License, or
 * (at your option) any later version.
 *
 * `Haplo' is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the  implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the  GNU General Public License
 * for more details.
 * 
 * You should have  received  a copy of the   GNU General Public  License
 * along with `Haplo'.  If not, write to  the
 *
 *                                        Free Software Foundation,  Inc.
 *                                        675 Mass Ave, Cambridge, MA
 *                                        02139, USA.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <haplo.h>

#include "dof.h"
#include "load.h"
#include "group.h"
#include "matrix.h"
#include "model.h"
#include "numbering.h"


/*-----------------------------------------------------------------------------
                        G L O B A L   V A R I A B L E 
-----------------------------------------------------------------------------*/

/* A warning at the next line means a serious problem! */
static const char *load_name[DOF_MAX+1]={
	"FX",
	"FY",
	"FZ",
	"MX",
	"MY",
	"MZ",
	NULL
};


/*-----------------------------------------------------------------------------
                          L O A D _ V E C T O R _ S
-----------------------------------------------------------------------------*/

struct load_vector_s
{
	const numbering_t	*numbering;
	const meshing_t		*meshing;
	vector_t		*b;
};


/*-----------------------------------------------------------------------------
                             L O A D _ A D D _ S
-----------------------------------------------------------------------------*/

struct load_add_s
{
	load_list_t		*load_list;
	dofset_t		dof;
	double			value;
};


/*-----------------------------------------------------------------------------
                             P R O T O T Y P E S 
-----------------------------------------------------------------------------*/

dofset_t __fem_load_nb(const char *name);
void fem_load_free(fem_load_t *load);
void fem_load_display(const fem_load_t *load);
fem_load_t *fem_load_copy(const fem_load_t *load);
fem_load_t *fem_load_new(void);
static void load_print(load_t *load);
void fem_load_print(const fem_load_t *load);
static void load_add(unsigned long node, struct load_add_s *load_add);
void fem_load_add(fem_load_t *load, const double *node, const char *force, 
		  const double *value);
void fem_load_add_group(fem_load_t *load, const fem_group_t *group,
			const char *force, const double *value);
static void load_vector(load_t *load, struct load_vector_s *vector);
void __fem_load_vector(vector_t *b, const load_list_t *load,
		       const numbering_t *numbering,
		       const meshing_t *meshing);


/*-----------------------------------------------------------------------------
                         I M P L E M E N T A T I O N 
-----------------------------------------------------------------------------*/

/**
 *
 */
dofset_t __fem_load_nb(const char *name)
{
	dofset_t dofset=DOF_MAX;
	dofset_t i;
	
	for(i=0; load_name[i] != NULL; i++)
		if (strcmp(name, load_name[i]) == 0)
		{
			dofset = i;
			break;
		}
	
	return(dofset);
}


/**
 *
 */
void fem_load_free(fem_load_t *load)
{
	__fem_page_list_free(&LOAD_LIST(load)->load);
	
	HAPLO_FREE(load);
	
	return;
}


/**
 *
 */
void fem_load_display(const fem_load_t *load)
{
	printf("Loading (%lu loads)", LOAD_LIST(load)->load.nb);

	return;
}


/**
 *
 */
fem_load_t *fem_load_copy(const fem_load_t *load)
{
	load_list_t *copy;

	HAPLO_ALLOC(copy, 1);
	
	copy->load=__fem_page_list_copy(&LOAD_LIST(load)->load);
	
	return(copy);
}


/**
 *
 */
fem_load_t *fem_load_new(void)
{
	load_list_t *load_list;
	
	HAPLO_ALLOC(load_list, 1);

	load_list->load=__fem_page_list_new(sizeof(load_t));
	
	return(load_list);
}


/**
 *
 */
static void load_print(load_t *load)
{
	printf("   %s%lu = %e\n",
	       load_name[load->dof],
	       load->n+1,
	       load->value);

	return;
}


/**
 *
 */
void fem_load_print(const fem_load_t *load)
{
	haplo_bordered("Loading");
	haplo_underlined("%lu load%s",
			 LOAD_LIST(load)->load.nb,
			 (LOAD_LIST(load)->load.nb>1)?"s":"");
	__fem_page_list_loop(&LOAD_LIST(load)->load, (page_func_t)load_print);
	
	return;
}


/**
 *
 */
static void load_add(unsigned long node, struct load_add_s *load_add)
{
	load_t *l;

	l=__fem_page_list_add(&load_add->load_list->load);

	l->type=LOAD_NODE;
	l->dof=load_add->dof;
	l->value=load_add->value;
	l->n=node;
	
}


/**
 *
 */
void fem_load_add(fem_load_t *load, const double *node, const char *force, 
		  const double *value)
{
	dofset_t d;

	d=__fem_load_nb(force);
	if (d<DOF_MAX)
	{
		struct load_add_s add;

		add.load_list = LOAD_LIST(load);
		add.dof = d;
		add.value=*value;
		
		load_add(haplo_ulong(*node-1), &add);
	} else 
		haplo_error("`%s' is not a name of force.", force);
	

	return;
}


/**
 *
 */
void fem_load_add_group(fem_load_t *load, const fem_group_t *group,
			const char *force, const double *value)
{
	dofset_t d;

	d=__fem_load_nb(force);
	if (d<DOF_MAX)
	{
		struct load_add_s add;

		add.load_list = LOAD_LIST(load);
		add.dof = d;
		add.value=*value;
		
		__fem_group_node_loop_p(GROUP(group),
					(group_func_p_t)load_add,
					&add);
	} else 
		haplo_error("`%s' is not a name of force.", force);
	

	return;
}


/**
 * 
 */
static void load_node_vector(load_t *load, struct load_vector_s *vector)
{
	unsigned long n;
	
	n=__fem_node_internal(vector->meshing, load->n);

	if (vector->numbering->row[load->dof] && 
	    (vector->numbering->row[load->dof][n] >0))
		vector->b->coef[
			vector->numbering->row[load->dof][n]-1] +=
			load->value;
	else
		haplo_warning("Force `%s%lu' is not defined for this model. "
			      "Ignored.",
			      load_name[load->dof], load->n+1);
		
}


/**
 *
 */
static void load_vector(load_t *load, struct load_vector_s *vector)
{
	switch(load->type)
	{
	case LOAD_NODE:
		load_node_vector(load, vector);
		break;
	case LOAD_MESH:
		haplo_fatal("Not implemented yet.");
		break;
	}

	return;
}


/**
 *
 */
void __fem_load_vector(vector_t *b, const load_list_t *load,
		       const numbering_t *numbering,
		       const meshing_t *meshing)
{
	struct load_vector_s vector;
	
	vector.b = b;
	vector.numbering = numbering;
	vector.meshing = meshing;

	__fem_page_list_loop_p(&LOAD_LIST(load)->load,
			       (page_func_p_t)load_vector,
			       &vector);
	

	return;
}
