/*
 * $Id: group.c,v 1.9 2003/10/05 20:04:52 nicoo Exp $
 *
 *
 * Copyright (C) 1999, 2000, 2001 Nicolas LAURENT
 * This file is part of `Haplo'
 * 
 *
 * `Haplo'  is free software;  you can  redistribute  it and/or modify it
 * under the terms of the GNU Library General Public License as published
 * by the Free Software Foundation;  either version 2  of the License, or
 * (at your option) any later version.
 *
 * `Haplo' is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the  implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the  GNU General Public License
 * for more details.
 * 
 * You should have  received  a copy of the   GNU General Public  License
 * along with `Haplo'.  If not, write to  the
 *
 *                                        Free Software Foundation,  Inc.
 *                                        675 Mass Ave, Cambridge, MA
 *                                        02139, USA.
 *
 */

/*
 *
 */

#include <errno.h>
#include <stdio.h>
#include <string.h>

#include <haplo.h>

#include "group.h"
#include "meshing.h"
#include "page.h"


/*-----------------------------------------------------------------------------
                             P R O T O T Y P E S 
-----------------------------------------------------------------------------*/

static fem_group_t *group_new(const fem_meshing_t *meshing, group_type_t type);
fem_group_t *fem_group_new_mesh(const fem_meshing_t *meshing);
fem_group_t *fem_group_new_node(const fem_meshing_t *meshing);
void fem_group_display(const fem_group_t *group);
fem_group_t *fem_group_copy(const fem_group_t *group);
void fem_group_free(fem_group_t *group);
static void group_print_mesh(mesh_t **mesh);
static void group_print_node(node_t **node);
void fem_group_print(const fem_group_t *group);
static void group_add_node(const meshing_t *meshing,
			   unsigned long n, page_list_t *list);
static void group_add_mesh(const meshing_t *meshing, unsigned long n,
			   page_list_t *list);
void fem_group_add(fem_group_t *group, haplo_list_t *list);
group_t *group_load(const meshing_t *meshing, const char *filename, FILE *fp);
fem_group_t *fem_group_load(const fem_meshing_t *meshing,
			    const char *filename);
fem_group_t *fem_group_inline(const fem_meshing_t *meshing);
static void group_loop(const group_t *group, group_func_t f);
static void group_loop_p(const group_t *group, group_func_p_t f, void *p);
void __fem_group_mesh_loop(const group_t *group, group_func_t f);
void __fem_group_mesh_loop_p(const group_t *group, group_func_p_t f, void *p);
void __fem_group_node_loop(const group_t *group, group_func_t f);
void __fem_group_node_loop_p(const group_t *group, group_func_p_t f, void *p);


/*-----------------------------------------------------------------------------
                         I M P L E M E N T A T I O N 
-----------------------------------------------------------------------------*/

/**
 *
 */
static fem_group_t *group_new(const fem_meshing_t *meshing, group_type_t type)
{
	group_t	*group;

	HAPLO_ALLOC(group, 1);
	
	group->meshing=MESHING(meshing);
	group->type=type;
	group->list=__fem_page_list_new(sizeof(void *));
	
	return((fem_group_t *)group);
}


/**
 *
 */
fem_group_t *fem_group_new_mesh(const fem_meshing_t *meshing)
{
	return(group_new(meshing, GROUP_MESH));
}


/**
 *
 */
fem_group_t *fem_group_new_node(const fem_meshing_t *meshing)
{
	return(group_new(meshing, GROUP_NODE));
}


/**
 *
 */
void fem_group_display(const fem_group_t *group)
{
	switch(GROUP(group)->type)
	{
	case GROUP_MESH:
		printf("Group de %lu maille%s",
		       GROUP(group)->list.nb,
		       (GROUP(group)->list.nb > 1)?"s":"");
		break;
	case GROUP_NODE:
		printf("Groupe de %lu noeud%s",
		       GROUP(group)->list.nb,
		       (GROUP(group)->list.nb > 1)?"s":"");
		break;
	}
	
	return;
}


/**
 *
 */
fem_group_t *fem_group_copy(const fem_group_t *group)
{
	group_t *copy;

	HAPLO_ALLOC(copy, 1);
	
	copy->meshing=GROUP(group)->meshing;
	copy->type=GROUP(group)->type;
	copy->list=__fem_page_list_copy(&GROUP(group)->list);
	
	return((fem_group_t *)copy);
}


/**
 *
 */
void fem_group_free(fem_group_t *group)
{
	__fem_page_list_free(&GROUP(group)->list);
	HAPLO_FREE(group);
	
	return;
}


/**
 *
 */
static void group_print_mesh(mesh_t **mesh)
{
	printf("\t%lu\n", (*mesh)->n+1);
	return;
}


/**
 *
 */
static void group_print_node(node_t **node)
{
	printf("\t%lu\n", (*node)->n+1);
	return;
}


/**
 *
 */
void fem_group_print(const fem_group_t *group)
{
	switch(GROUP(group)->type)
	{
	case GROUP_MESH:
		haplo_bordered("Groupe de mailles");
		group_loop(group, (group_func_t)group_print_mesh);
		break;
	case GROUP_NODE:
		haplo_bordered("Groupe de noeuds");
		group_loop(group, (group_func_t)group_print_node);
		break;
	}
	printf("\n");
	return;
}


/**
 *
 */
static void group_add_node(const meshing_t *meshing, unsigned long n,
			   page_list_t *list)
{
	node_t	*node;
	
	node=__fem_node_get(meshing, n);
	if (node)
		__fem_page_list_add_content(list, &node);
	else
		haplo_error("Le noeud %lu n'existe pas.", n+1);
	return;
}


/**
 *
 */
static void group_add_mesh(const meshing_t *meshing, unsigned long n,
			   page_list_t *list)
{
	mesh_t *mesh;

	mesh=__fem_mesh_get(meshing, n);
	if (mesh)
		__fem_page_list_add_content(list, &mesh);
	else
		haplo_error("La maille %lu n'existe pas.", n+1);
	
	
	return;
}


/**
 *
 */
void fem_group_add(fem_group_t *group, haplo_list_t *list)
{
	unsigned long dim;
	unsigned long i;

	haplo_list_start(list);
	if (haplo_list_type_homogeneous(list) != 
	    haplo_object_type_get("float"))
	{
		haplo_error("Vector should contain only numbers");
		return;
	}

	dim=haplo_list_size(list);

	switch(GROUP(group)->type)
	{
	case GROUP_MESH:
		for(i=0; i<dim; i++)
		{
			unsigned long n;

			n=haplo_ulong(*((double *)haplo_list_arg(list))-1);
			group_add_mesh(GROUP(group)->meshing, n,
				       &GROUP(group)->list);
		}
		break;
	case GROUP_NODE:
		for(i=0; i<dim; i++)
		{
			unsigned long n;

			n=haplo_ulong(*((double *)haplo_list_arg(list))-1);
			group_add_node(GROUP(group)->meshing, n,
				       &GROUP(group)->list);
		}
		break;
	}

	return;
}


/**
 *
 */
group_t *group_load(const meshing_t *meshing, const char *filename, FILE *fp)
{
	group_t *group=NULL;
	int loop=1;
	int line=0;
	unsigned long n;
	
	while(loop)
	{
		int byte;
				
		byte = __fem_meshing_load_header(fp, &line);
		switch(byte)
		{
		case EOF:
			loop=0;
			break;
			
		case 'N':
			if (! group)
				group=group_new(meshing, GROUP_NODE);
			else
				if (group->type != GROUP_NODE)
				{
					haplo_warning(
						"[%s:%d] Ce groupe est "
						"un groupe de noeuds",
						filename, line);
					break;
				}
			while(fscanf(fp, "%lu", &n) == 1)
				group_add_node(meshing, n-1, &group->list);
			break;
		case 'M':
			if (! group)
				group=group_new(meshing, GROUP_MESH);
			else
				if (group->type != GROUP_MESH)
				{
					haplo_warning(
						"[%s:%d] Ce groupe est "
						"un groupe de mailles",
						filename, line);
					break;
				}
			while(fscanf(fp, "%lu", &n) == 1)
				group_add_mesh(meshing, n-1, &group->list);
			
			break;
		case 'X':
			loop=0;
			break;
			
		default:
			haplo_warning("[%s:%d] Erreur de syntaxe (%c).",
				      filename, line, byte);

			
		}

	}
	
	
	
	return(group);
}


/**
 *
 */
fem_group_t *fem_group_load(const fem_meshing_t *meshing, const char *filename)
{
	fem_group_t *group=NULL;
	FILE *fp;

	fp=fopen(filename, "rt");
	if (!fp)
	{
		haplo_warning("Impossible d'ouvrir le fichier `%s': %s",
			      filename, strerror(errno));
	} else {
		group=group_load(meshing, filename, fp);
		(void)fclose(fp);
	}

	return((fem_group_t *)group);
}


/**
 *
 */
fem_group_t *fem_group_inline(const fem_meshing_t *meshing)
{
	return((fem_group_t *)group_load(meshing, "<stdin>", stdin));
}


/**
 *
 */
static void group_save_mesh(const mesh_t **mesh, FILE *fp)
{
	fprintf(fp, "#!M %lu\n", (*mesh)->n+1);
}


/**
 *
 */
static void group_save_node(const node_t **node, FILE *fp)
{
	fprintf(fp, "#!N %lu\n", (*node)->n+1);
}


/**
 *
 */
static void group_save(const group_t *group, FILE *file)
{
	switch(group->type)
	{
	case GROUP_MESH:
		__fem_page_list_loop_p(&group->list,
				       (page_func_p_t)group_save_mesh,
				       file);
		break;			
	case GROUP_NODE:
		__fem_page_list_loop_p(&group->list,
				       (page_func_p_t)group_save_node,
				       file);
		break;
	}
	fprintf(file, "#!X\n");
	
	return;
}


/**
 *
 */
void fem_group_save(const char *filename, const fem_group_t *group)
{
	FILE *fp=stdout;

	fp=fopen(filename, "wt");
	if (fp)
	{
		group_save(group, fp);
		fclose(fp);

	} else
		haplo_error("Couldn't write in file `%s'", filename);
	
	return;
}


/**
 *
 */
static void group_loop(const group_t *group, group_func_t f)
{
	__fem_page_list_loop(&group->list, (page_func_t)f);

	return;
}


/**
 *
 */
static void group_loop_p(const group_t *group, group_func_p_t f, void *p)
{
	__fem_page_list_loop_p(&group->list, (page_func_p_t)f, p);

	return;
}


/**
 *
 */
void __fem_group_mesh_loop(const group_t *group, group_func_t f)
{
	if (group->type == GROUP_MESH)
		group_loop(group, f);
	else
		haplo_error("Cette fonction s'applique aux "
			    "groupes de mailles");

	return;
}


/**
 *
 */
void __fem_group_mesh_loop_p(const group_t *group, group_func_p_t f, void *p)
{
	if (group->type == GROUP_MESH)
		group_loop_p(group, f, p);
	else
		haplo_error("Cette fonction s'applique aux "
			    "groupes de mailles");

	return;
}


/**
 *
 */
void __fem_group_node_loop(const group_t *group, group_func_t f)
{
	if (group->type == GROUP_NODE)
		group_loop(group, f);
	else
		haplo_error("Cette fonction s'applique aux "
			    "groupes de noeuds");

	return;
}


/**
 *
 */
void __fem_group_node_loop_p(const group_t *group, group_func_p_t f, void *p)
{
	if (group->type == GROUP_NODE)
		group_loop_p(group, f, p);
	else
		haplo_error("Cette fonction s'applique aux "
			    "groupes de noeuds");

	return;
}
