/*
 * $Id: element.c,v 1.11 2003/10/13 09:24:14 nicoo Exp $
 *
 *
 * Copyright (C) 1999, 2000, 2001 Nicolas LAURENT
 * This file is part of `Haplo'
 * 
 *
 * `Haplo'  is free software;  you can  redistribute  it and/or modify it
 * under the terms of the GNU Library General Public License as published
 * by the Free Software Foundation;  either version 2  of the License, or
 * (at your option) any later version.
 *
 * `Haplo' is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the  implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the  GNU General Public License
 * for more details.
 * 
 * You should have  received  a copy of the   GNU General Public  License
 * along with `Haplo'.  If not, write to  the
 *
 *                                        Free Software Foundation,  Inc.
 *                                        675 Mass Ave, Cambridge, MA
 *                                        02139, USA.
 *
 */

#include <float.h>
#include <math.h>
#include <string.h>

#include <haplo.h>

#include "element.h"
#include "matrix.h"
#include "model.h"


/*-----------------------------------------------------------------------------
                        G L O B A L   V A R I A B L E 
-----------------------------------------------------------------------------*/

/**
 * List of registred finite element types.
 */
static element_type_t *element_type_list=NULL;


/*-----------------------------------------------------------------------------
                     E L E M E N T _ C R E A T I O N _ S
-----------------------------------------------------------------------------*/

/**
 * Private structure to construct element from a meshing.
 */

struct element_creation_s
{
	unsigned short  dim;		/** space dimension */
	unsigned long   current;	/** current element number */
	int		error;		/** if no element */
	element_t       *element;
};


/*-----------------------------------------------------------------------------
                               P R O T O T Y P E S
-----------------------------------------------------------------------------*/

/*
 * element_type_stuff.
 */
static void element_type_setup(element_type_t *type,
			       const fem_element_type_t *et);
static element_type_t *element_type_new(const fem_element_type_t *et);
static void element_type_free(element_type_t *type);
void __fem_element_type_free(void);
static element_type_t *element_type_get(const char *name);
void fem_element_type_register(const fem_element_type_t *et);
static element_type_t *element_type_get_default(unsigned short n,
						unsigned short dim);


/*
 * element_t stuff.
 */

unsigned long __fem_element_dof_g(const numbering_t *numbering,
				  unsigned long n, const char *dof);
dof_t __fem_element_dof_l(const numbering_t *numbering, unsigned long no,
			  unsigned long dof);
static void element_create(mesh_t *mesh, unsigned short n,
			   struct element_creation_s *ec);
element_t *__fem_element_create(const meshing_t *meshing);
static unsigned short element_dof_count(dofset_t dofset);
numbering_t  __fem_element_dof_numbering(const element_t *element,
					 unsigned long el, unsigned long no);
static void element_gsm_profile(matrix_t *gsm, dofset_t dofset,
				unsigned long n1, unsigned long n2, 
				const numbering_t *numbering);
void __fem_element_gsm_profile(matrix_t *gsm, const element_t *element,
			       unsigned long el, const numbering_t *numbering);

static dofset_t element_gsm_compute_dof(dofset_t dofset,
					      register unsigned short n);
void __fem_element_gsm_compute(matrix_t *gsm, const element_t *element,
			       unsigned long el, const numbering_t *numbering);

/*
 * element_type_t stuff.
 */

/*-----------------------------------------------------------------------------
                        I M P L E M E N T A T I O N
-----------------------------------------------------------------------------*/


/**
 * Initialize a new element from a fem_element_t
 *
 * @param type
 * @param et
 *
 * @see element_type_new
 */
static void element_type_setup(element_type_t *type,
			       const fem_element_type_t *et)
{
	type->type	= et->type;
	type->name	= et->name;
	type->nb_node	= et->nb_node;
	type->dofset	= __fem_dof_dofset(et->dofset);
	type->nb_dof	= element_dof_count(type->dofset);
	if (type->nb_dof == 0)
	{
		haplo_error("Registration of element `%s' with 0 dof. (BUG?)",
			    et->name);
		return;
	}

	type->dim	  = et->dim;
	type->integration = __fem_integration(et->integration);
	type->j		  = et->j;
	type->esm	  = et->esm;
	type->hook	  = __fem_hook_setup(et->material, et->property);
	
	/*
	 * Setup buffers
	 */
	HAPLO_ALLOC(type->x, type->nb_node);
	
	return;
}


/**
 * Allocate a new element_type_t and initialize it
 *
 * @param et
 * @return
 *
 * @see element_type_setup
 * @see fem_element_type_register
 */
static element_type_t *element_type_new(const fem_element_type_t *et)
{
	element_type_t	*type;
	
	HAPLO_ALLOC(type, 1);

	element_type_setup(type, et);
	type->next=NULL;
	
	return(type);
}

/**
 * get element_type_t from its name.
 *
 * @param name
 * @return
 *
 * @see fem_element_type_register
 */
static element_type_t *element_type_get(const char *name)
{
	element_type_t	*et=NULL;
	
	for(et=element_type_list; et; et=et->next)
	{
		if (strcmp(et->name, name) == 0)
			break;
	}
	
	return(et);
}

/**
 * Register a new finite element model
 *
 * @param et is a element type
 */
void fem_element_type_register(const fem_element_type_t *et)
{
	element_type_t	*slot;

	slot=element_type_get(et->name);
	if (slot)
	{
		haplo_warning("Re-implementation of element `%s'.", et->name);
		element_type_free(slot);
		element_type_setup(slot, et);
	} else {
		haplo_info("Dfinition de l'lment `%s'", et->name);
		if (element_type_list)
		{
			element_type_t	*s;
			
			for(s=element_type_list; s->next; s=s->next)
				;
			s->next=element_type_new(et);
		} else {
			element_type_list=element_type_new(et);
		}
	}

	return;
}


/**
 * Free allocated memory by a element type
 *
 * @param type is the considered element type
 */
static void element_type_free(element_type_t *type)
{
	HAPLO_FREE(type->x);
	
	return;
}


/**
 * Free all buffers allocated by the element types
 *
 * @see element_type_free
 * @see element_type_list
 */
void __fem_element_type_free(void)
{
	element_type_t *et, *next_et;

	for(et=element_type_list; et; et=next_et)
	{
		next_et=et->next;
		element_type_free(et);
		HAPLO_FREE(et);
	}
	
	return;
}



/**
 * Get the first element type according to the number of nodes and the
 * dimension of space
 *
 * @param n is the number of nodes
 * @param dim is the dimension of space
 * @return a element type if found or NULL
 */
static element_type_t *element_type_get_default(unsigned short n,
						unsigned short dim)
{
	element_type_t	*et=NULL;
	
	for(et=element_type_list; et; et=et->next)
	{
		if ((et->nb_node == n) && (et->dim == dim))
			break;
	}

	return(et);
}


/*
 * element_t stuff.
 */




/**
 * Assign element type to a mesh
 *
 * @param mesh
 * @param n
 * @param ec is an internaly used structure
 * @see __fem_element_create
 */
static void element_create(mesh_t *mesh, unsigned short n,
			   struct element_creation_s *ec)
{
	element_t *const element=ec->element + ec->current;

	element->type=element_type_get_default(n, ec->dim);
	element->mesh=mesh;
	element->material=NULL;
	element->property=NULL;

	if (element->type)
		ec->current ++;
	else
	{
		haplo_warning("Maille #%lu: aucun lment fini n'est "
			      "dfini avec %u "
			      "noeud%s dans l'espace %uD", mesh->n, n,
			      (n>1)?"s":"", ec->dim);
		ec->error=1;
	}
	
	return;
}


/**
 * Create a element list from a meshing
 *
 * @param meshing
 * @return dynamically allocated list of element
 */
element_t *__fem_element_create(const meshing_t *meshing)
{
	element_t *element;
	struct element_creation_s ec;

	HAPLO_ALLOC(element, NB_MESHES(meshing));
	
	ec.dim=meshing->node_list.dim;
	ec.current=0;
	ec.element=element;
	ec.error=0;
	
	__fem_meshing_loop_p(
		meshing,
		(void (*)(mesh_t *, unsigned short, void *))element_create,
		&ec);

	if (ec.error)
	{
		HAPLO_FREE(element);
		element=NULL;
	}
	
	return(element);
}


/**
 * Count the dofs set in a dofset
 *
 * @param dofset
 * @return the number of dofs set
 */
static unsigned short element_dof_count(dofset_t dofset)
{
	unsigned short n=0;
	unsigned short i;
	
	for(i=0; i<DOF_MAX; i++)
	{
		if (dofset & (1 << i))
			n++;
	}
	
	return(n);
}


/**
 *
 */
static void element_gsm_profile(matrix_t *gsm, dofset_t dofset,
				unsigned long n1, unsigned long n2, 
				const numbering_t *numbering)
{
	dofset_t d1;
	
	for(d1=0; d1 < DOF_MAX; d1++)
	{
		if (dofset & (1<<d1))
		{
			dofset_t d2;
			for(d2=0; d2 < DOF_MAX; d2++)
			{
				if (dofset & (1<<d2))
				{
					const unsigned long i=
						numbering->row[d1][n1]-1;
					const unsigned long j=
						numbering->row[d2][n2]-1;
					
					if (i<=j)
					{
						if (gsm->diag[j] < j-i)
							gsm->diag[j] = j-i;
					} else {
						if (gsm->diag[i] < i-j)
							gsm->diag[i] = i-j;
					}
				}
			}
		}
	}
	return;
}

 
/**
 *
 */
void __fem_element_gsm_profile(matrix_t *gsm, const element_t *element,
			       unsigned long el, const numbering_t *numbering)
{
	unsigned long e;
	
	for(e=0; e<el; e++)
	{ /* loop on elements */
		const unsigned short nb_node=element[e].type->nb_node;
		unsigned short n1, n2;

		for(n1=0; n1<nb_node; n1++)
		{ /* loop on node */
			for(n2=0; n2 <= n1; n2++)
			{ /* loop on node */
				element_gsm_profile(
					gsm,
					element[e].type->dofset,
					element[e].mesh->nodes[n1]->n,
					element[e].mesh->nodes[n2]->n,
					numbering);
			}
		}
	}

	return;
}


/**
 * Get the n th dof of dofset.
 *
 * @param dofset
 * @param n
 * @return 
 */
static dofset_t element_gsm_compute_dof(dofset_t dofset,
					register unsigned short n)
{
	dofset_t i;
	
	for(i=0; i<DOF_MAX; i++)
	{
		if (dofset & (1 << i))
		{
			if (n == 0)
				break;
			n--;
		}
	}

	HAPLO_ASSERT(i < DOF_MAX);

	return(i);
}


/**
 * update GSM with value computed for the link beetween dof #i and #j
 */
static void element_gsm_update_do(matrix_t *gsm, unsigned long i,
				  unsigned long j, double value)
{
	if (i<j)
		gsm->coef[gsm->diag[j]+i-j] += value;
	else
		gsm->coef[gsm->diag[i]+j-i] += value;
	
	return;
}



/**
 * update GSM with computed ESM
 */
static void element_gsm_update(matrix_t *gsm, const element_t *element,
			       const double *esm,
			       const double alpha,
			       const numbering_t *numbering)
{
	const dofset_t dofset=element->type->dofset;
	const unsigned short nb_dof=element->type->nb_dof;
	const unsigned short dim=nb_dof*element->type->nb_node;

	unsigned short offset=0;
	unsigned short i;

	for(i=0; i<dim; i++)
	{
		const unsigned long inode=element->mesh->nodes[i/nb_dof]->n;
		const dofset_t idof=
			element_gsm_compute_dof(dofset, i % nb_dof);
		unsigned short j;
		
		for(j=0; j<=i; j++)
		{
			const unsigned long jnode=
				element->mesh->nodes[j/nb_dof]->n;
			const dofset_t jdof=
				element_gsm_compute_dof(dofset, j % nb_dof);
			
			element_gsm_update_do(
				gsm,
				numbering->row[idof][inode]-1,
				numbering->row[jdof][jnode]-1,
				esm[offset+j]*alpha);
		}	
		offset += i+1;
	}
	
	return;
}


/**
 * Update coordinate buffer in element->type for ESM computation.
 */
static void element_esm_coord_update(fem_element_t *e,
				     const element_t *element)
{
	unsigned short i;
	
	e->x=element->type->x;
	for(i=0; i<element->type->nb_node; i++)
		e->x[i]=element->mesh->nodes[i]->coords;
	return;
}


/**
 * Fill esm with zeros
 */
static void element_esm_clear(double *esm, const element_type_t *type)
{
	const unsigned short n = type->nb_dof * type->nb_node;
	const unsigned short size=((n+1)*n)/2;
	unsigned short i;
	
	for(i=0; i<size; i++)
		esm[i]=0.0;

	return;
}


/**
 *
 */
static void element_gsm_compute(matrix_t *gsm, const numbering_t *numbering,
				const element_t *element)
{
#define N (MESH_BIGGEST*DOF_MAX)
	double esm[((N+1)*N)/2];
#undef N
	fem_element_t	e;
	const char *error;
	
	element_esm_coord_update(&e, element);
	element_esm_clear(esm, element->type);

	error=__fem_hook_update(&e,
				element->material,
				element->property,
				&element->type->hook);
	if (error)
	{
		haplo_error("Missing %s's cara on element #%lu",
			    error,
			    element->mesh->n);
		return;
	}


	if (element->type->integration->nb)
	{
		unsigned short i;

		for(i=0; i<element->type->integration->nb; i++)
		{
			double	j;

			/*
			 * Computing of jacobian
			 */
			j=fabs((*element->type->j)(
				       element->type->integration->point[i],
				       element->type->x));

			if (j < DBL_EPSILON)
			{
				haplo_warning("Element #%lu is warped",
					      element->mesh->n);
				continue;
			}

			e.p = element->type->integration->point[i];
			(*element->type->esm)(&e, esm);

			element_gsm_update(
				gsm, element, esm,
				j*element->type->integration->weight[i],
				numbering);
		}
	}
	else {	
		e.p = NULL;
		(*element->type->esm)(&e, esm);
		element_gsm_update(gsm, element, esm,
				   1.0,
				   numbering);
	}
	
	return;
}


/**
 * Compute the Global Stiffness Matrix from elements.
 *
 * @param gsm is the result
 * @param elements is array of elements
 * @param el size of the previous array
 * @param numbering is 
 *
 * @todo coucou
 */
void __fem_element_gsm_compute(matrix_t *gsm, const element_t elements[],
			       unsigned long el, const numbering_t *numbering)
{
	unsigned long e;	

	for(e=0; e < el; e++)
		element_gsm_compute(gsm, numbering, elements+e);

	return;
}
