/*
 * $Id: bc.c,v 1.11 2003/10/13 09:50:27 nicoo Exp $
 *
 *
 * Copyright (C) 1999, 2000, 2001 Nicolas LAURENT
 * This file is part of `Haplo'
 * 
 *
 * `Haplo'  is free software;  you can  redistribute  it and/or modify it
 * under the terms of the GNU Library General Public License as published
 * by the Free Software Foundation;  either version 2  of the License, or
 * (at your option) any later version.
 *
 * `Haplo' is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the  implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the  GNU General Public License
 * for more details.
 * 
 * You should have  received  a copy of the   GNU General Public  License
 * along with `Haplo'.  If not, write to  the
 *
 *                                        Free Software Foundation,  Inc.
 *                                        675 Mass Ave, Cambridge, MA
 *                                        02139, USA.
 *
 */

#include <float.h>
#include <math.h>
#include <string.h>

#include <haplo.h>

#include "bc.h"
#include "group.h"
#include "matrix.h"
#include "meshing.h"
#include "model.h"
#include "page.h"

/* #define FEM_DEBUG_NUMBERING  */

/*-----------------------------------------------------------------------------
                             B C _ C O U N T _ S
-----------------------------------------------------------------------------*/

struct bc_count_s

{
	const numbering_t	*numbering;
	const meshing_t		*meshing;
	unsigned long		node;
	dofset_t		dof;
	dofset_t		counter;
	unsigned long		n;
	unsigned long		current;
};


/*-----------------------------------------------------------------------------
                           B C _ P R O F I L E _ S
-----------------------------------------------------------------------------*/

struct bc_profile_s
{
	const numbering_t	*numbering;
	const meshing_t		*meshing;
	matrix_t		*gsm;
	unsigned long		n;
	unsigned long		*invert;
};


/*-----------------------------------------------------------------------------
                           B C _ C O M P U T E _ S
-----------------------------------------------------------------------------*/

struct bc_compute_s
{
	const numbering_t	*numbering;
	const meshing_t		*meshing;
	matrix_t		*gsm;
	unsigned long		n;
	double			alpha;
	double			beta;
	unsigned long		*invert;
};


/*-----------------------------------------------------------------------------
                              B C _ L O A D _ S
-----------------------------------------------------------------------------*/

struct bc_load_s
{
	const numbering_t	*numbering;
	vector_t		*b;
	unsigned long		n;
	double			beta;
};


/*-----------------------------------------------------------------------------
                               B C _ A D D _ S
-----------------------------------------------------------------------------*/

struct bc_add_s
{
	bc_list_t		*bc_list;
	dofset_t		dof;
	double			value;
};


/*-----------------------------------------------------------------------------
                             P R O T O T Y P E S 
-----------------------------------------------------------------------------*/

static void bc_list_free(bc_t *bc);
void fem_bc_free(fem_bc_t *bc);
void fem_bc_display(const fem_bc_t *bc);
static void bc_list_copy(bc_t *bc);
fem_bc_t *fem_bc_copy(const fem_bc_t *bc);
fem_bc_t *fem_bc_new(void);
static void bc_numbering_lambda1(bc_t *bc, struct bc_count_s *count);
unsigned long __fem_bc_numbering_l1(const bc_list_t *bc_list,
				    const numbering_t *numbering,
				    const meshing_t *meshing,
				    unsigned long node, dofset_t dof,
				    unsigned long current);
static void bc_numbering_lambda2(bc_t *bc, struct bc_count_s *count);
unsigned long __fem_bc_numbering_l2(const bc_list_t *bc_list,
				    const numbering_t *numbering,
				    const meshing_t *meshing,
				    unsigned long node, dofset_t dof,
				    unsigned long current);
static void bc_gsm_profile(bc_t *bc, struct bc_profile_s *bc_profile);
void __fem_bc_gsm_profile(matrix_t *gsm, const bc_list_t *bc_list, 
			  const numbering_t *numbering,
			  const meshing_t *meshing);
static void bc_gsm_compute(bc_t *bc, struct bc_compute_s *bc_compute);
static double bc_gsm_alpha(matrix_t *gsm);
double __fem_bc_gsm_compute(matrix_t *gsm, const bc_list_t *bc_list, 
			    const numbering_t *numbering,
			    const meshing_t *meshing);
static double bc_normalize(bc_part_t *part, unsigned long n);
static void bc_set(unsigned long node, struct bc_add_s *bc_add);
static void bc_set_node(const node_t *node, struct bc_add_s *bc_add);
void fem_bc_block(fem_bc_t *bc, const double *node, const char *dof);
void fem_bc_block_group(fem_bc_t *bc, const fem_group_t *group,
			const char *dof);
void fem_bc_set(fem_bc_t *bc, const double *node, const char *dof,
		const double *value);
void fem_bc_set_group(fem_bc_t *bc, const fem_group_t *group,
		      const char *dof, double *value);
static int bc_relation(bc_part_t *part, haplo_list_t *list);
void fem_bc_relation(fem_bc_t *bc, haplo_list_t *node, const double *value);


/*-----------------------------------------------------------------------------
                         I M P L E M E N T A T I O N 
-----------------------------------------------------------------------------*/


/*
 * Haplo object stuff.
 */

/**
 *
 */
static void bc_list_free(bc_t *bc)
{
	HAPLO_FREE(bc->part);
	
	return;
}


/**
 *
 */
void fem_bc_free(fem_bc_t *bc)
{
	__fem_page_list_loop(&BC_LIST(bc)->bc, (page_func_t)bc_list_free);
	__fem_page_list_free(&BC_LIST(bc)->bc);	
	HAPLO_FREE(bc);

	return;
}


/**
 *
 */
void fem_bc_display(const fem_bc_t *bc)
{
	printf("Boundry conditions set (%lu constraint%s)",
	       BC_LIST(bc)->bc.nb,
	       (BC_LIST(bc)->bc.nb>1)?"s":"");
	
	return;
}


/**
 *
 */
static void bc_list_copy(bc_t *bc)
{
	bc_part_t *part;
	
	HAPLO_ALLOC(part, bc->nb);
	memcpy(part, bc->part, bc->nb*sizeof(bc_part_t));
	bc->part=part;
	
	return;
}


/**
 *
 */
fem_bc_t *fem_bc_copy(const fem_bc_t *bc)
{
	bc_list_t *copy;
	
	HAPLO_ALLOC(copy, 1);

	copy->bc=__fem_page_list_copy(&BC_LIST(bc)->bc);
	__fem_page_list_loop(&copy->bc, (page_func_t)bc_list_copy);
	
	return(copy);
}


/**
 *
 */
fem_bc_t *fem_bc_new(void)
{
	bc_list_t *bc_list;

	HAPLO_ALLOC(bc_list, 1);
	
	bc_list->bc=__fem_page_list_new(sizeof(bc_t));

	return(bc_list);
}


static void bc_print(bc_t *bc)
{
	unsigned long i;

	if (fabs(1.0 - bc->part[0].coef) > DBL_EPSILON)
		printf("   %g * ", bc->part[0].coef);
	else
		fputs("   ", stdout);
	
	printf("%s%lu ", __fem_dof_name(bc->part[0].dof), bc->part[0].node+1);
	
	for(i=1; i<bc->nb; i++)
	{
		if (fabs(1.0 - bc->part[i].coef) > DBL_EPSILON)
			printf("+ %g * ", bc->part[i].coef);
		else
			fputs("+ ", stdout);
		printf("%s%lu ",
		       __fem_dof_name(bc->part[i].dof),
		       bc->part[i].node+1);
	}
	printf("= %g\n", bc->value);

	return;
}


/**
 *
 */
void fem_bc_print(const fem_bc_t *bc)
{
	haplo_bordered("Boundary conditions");
	haplo_underlined("%lu condition%s",
			 BC_LIST(bc)->bc.nb,
			 (BC_LIST(bc)->bc.nb>1)?"s":"");
			 
	__fem_page_list_loop(&BC_LIST(bc)->bc, (page_func_t)bc_print);
	
	return;
}


/*
 * Add relations
 */

/**
 *
 */
static double bc_normalize(bc_part_t *part, unsigned long n)
{
	double norm=0;
	unsigned long i;
	
	for(i=0; i<n; i++)
		norm += part[i].coef * part[i].coef;
	
	norm = 1.0 / sqrt(norm);
	
	for(i=0; i<n; i++)
		part[i].coef *= norm;
	
	return(norm);
}


/**
 *
 */
static void bc_set(unsigned long node, struct bc_add_s *bc_add)
{
	bc_t *bc;
	
	bc=__fem_page_list_add(&bc_add->bc_list->bc);

	bc->nb = 1;
	HAPLO_ALLOC(bc->part, 1);
	
	bc->part[0].node = node;
	bc->part[0].dof  = bc_add->dof;
	bc->part[0].coef = 1.0;
	bc->value        = bc_add->value;

	return;
}

/**
 *
 */
static void bc_set_node(const node_t *node, struct bc_add_s *bc_add)
{
	bc_set(node->n, bc_add);

	return;
}


/**
 *
 */
void fem_bc_block(fem_bc_t *bc, const double *node, const char *dof)
{
	dofset_t d;

	d=__fem_dof_nb(dof);
	if (d<DOF_MAX)
	{
		struct bc_add_s bc_add;
	
		bc_add.bc_list = BC_LIST(bc);
		bc_add.dof     = d;
		bc_add.value   = 0.0;
	
		bc_set(haplo_ulong(*node-1), &bc_add);
	} else
		haplo_error("`%s' is not a name of dof.", dof);
	
	return;
}


/**
 *
 */
void fem_bc_block_group(fem_bc_t *bc, const fem_group_t *group,
			const char *dof)
{
	dofset_t d;

	d=__fem_dof_nb(dof);
	if (d<DOF_MAX)
	{
		struct bc_add_s bc_add;
	
		bc_add.bc_list = BC_LIST(bc);
		bc_add.dof     = d;
		bc_add.value   = 0.0;

		__fem_group_node_loop_p(GROUP(group),
					(group_func_p_t)bc_set_node,
					&bc_add);
	} else
		haplo_error("`%s' is not a name of dof.", dof);
	
	return;
}


/**
 *
 */
void fem_bc_set(fem_bc_t *bc, const double *node, const char *dof,
		const double *value)
{
	dofset_t d;

	d=__fem_dof_nb(dof);
	if (d<DOF_MAX)
	{
		struct bc_add_s bc_add;
	
		bc_add.bc_list = BC_LIST(bc);
		bc_add.dof     = d;
		bc_add.value   = *value;

		bc_set(haplo_ulong(*node-1), &bc_add);
	} else
		haplo_error("`%s' is not a name of dof.", dof);
	
	return;
}


/**
 *
 */
void fem_bc_set_group(fem_bc_t *bc, const fem_group_t *group,
		      const char *dof, double *value)
{
	dofset_t d;

	d=__fem_dof_nb(dof);
	if (d<DOF_MAX)
	{
		struct bc_add_s bc_add;
	
		bc_add.bc_list = BC_LIST(bc);
		bc_add.dof     = d;
		bc_add.value   = *value;

		__fem_group_node_loop_p(GROUP(group),
					(group_func_p_t)bc_set_node,
					&bc_add);
	} else
		haplo_error("`%s' is not a name of dof.", dof);
	
	return;
}


/**
 *
 * @todo Check type in list
 */
static int bc_relation(bc_part_t *part, haplo_list_t *list)
{
	haplo_type_t type_float  = haplo_object_type_get("float");
	haplo_type_t type_string = haplo_object_type_get("string");
	const double *coef;
	const char *dof;
	const double *node;
	int status=-1;

	haplo_list_start(list);
	coef=haplo_list_arg_type(list, type_float);
	dof= haplo_list_arg_type(list, type_string);
	node=haplo_list_arg_type(list, type_float);
	
	if (coef && dof && node)
	{
		part->coef = *coef;
		part->dof  = __fem_dof_nb(dof);
		part->node = haplo_ulong(*node-1);
		status=0;
	}
	
	return(status);
}


/**
 *
 */
void fem_bc_relation(fem_bc_t *bc, haplo_list_t *list, const double *value)
{
	bc_t *eqn;
	unsigned long i;

	haplo_list_start(list);
	if (haplo_list_type_homogeneous(list) != 
	    haplo_object_type_get("list"))
	{
		haplo_error("Vector should contain only vectors");
		return;
	}

	eqn=__fem_page_list_add(&BC_LIST(bc)->bc);
	eqn->nb = haplo_list_size(list);
	HAPLO_ALLOC(eqn->part, eqn->nb);
	
	for(i=0; i<eqn->nb; i++)
	{
		if (bc_relation(eqn->part+i, haplo_list_arg(list)))
		{
			haplo_error("Invalid format in %luth term.", i+1);
			HAPLO_FREE(eqn->part);
			__fem_page_list_remove_last(&BC_LIST(bc)->bc);
			eqn=NULL;
			break;
		}
	}
	
	if (eqn)
		eqn->value = (*value) *	bc_normalize(eqn->part, eqn->nb);
	
	return;
}


/*
 * Numbering stuff.
 */


/**
 * Assign lambda1 before first involved dof.
 */
static void bc_numbering_lambda1(bc_t *bc, struct bc_count_s *count)
{
	if (count->numbering->lambda1[count->n] == 0)
	{
		int doit=0;
		unsigned long i;
		
		for(i=0; i< bc->nb; i++)
		{
			unsigned long n;
			
			n=__fem_node_internal(count->meshing,
					      bc->part[i].node);
			if ((n == count->node) && 
			    (bc->part[i].dof  == count->dof))
			{
				doit=1;
				break;
			}
		}
		if (doit)
		{
#ifdef FEM_DEBUG_NUMBERING			
			haplo_debug("l1 before %s%lu -> %lu",
				    __fem_dof_name(count->dof),
			 	    count->node+1,
				    count->current);
#endif
			count->numbering->lambda1[count->n] = count->current;
			count->current += 1;
			count->counter += 1;
		}
	}

	count->n += 1;
	
	return;
}


/**
 *
 */
unsigned long __fem_bc_numbering_l1(const bc_list_t *bc_list,
				    const numbering_t *numbering,
				    const meshing_t *meshing,
				    unsigned long node, dofset_t dof,
				    unsigned long current)
{
	unsigned long n=0;
	
	if (bc_list)
	{
		struct bc_count_s bc_count;

		bc_count.numbering	= numbering;
		bc_count.meshing	= meshing;
		bc_count.node		= node;
		bc_count.dof		= dof;
		bc_count.counter	= 0;
		bc_count.current	= current;
		bc_count.n		= 0;
		__fem_page_list_loop_p(&bc_list->bc,
				       (page_func_p_t)bc_numbering_lambda1,
				       &bc_count);

		n=bc_count.counter;
	}
#ifdef FEM_DEBUG_NUMBERING
	if (n>0)
		haplo_debug("%lu lambda before dof %s%lu",
			    n, __fem_dof_name(dof), n+1);
#endif	
	return(n);
}


/**
 * Assign lambda2 after last involved dof
 */
static void bc_numbering_lambda2(bc_t *bc, struct bc_count_s *count)
{
	if (count->numbering->lambda2[count->n] == 0)
	{
		const unsigned long N=NB_NODES(count->meshing);
		int doit=1;
		unsigned long i;

		for(i=0; i < bc->nb; i++)
		{
			unsigned long n;
			
			n=__fem_node_internal(count->meshing,
					      bc->part[i].node);

			if ((n<N) &&
			    (count->numbering->row[bc->part[i].dof]) &&
			    (count->numbering->row[bc->part[i].dof][n] == 1))
			{
				/* not assigned yet */
				doit = 0;
				break;
			}
		}

		if (doit)
		{
#ifdef FEM_DEBUG_NUMBERING
			haplo_debug("l2 after %s%lu -> %lu",
				    __fem_dof_name(count->dof),
				    count->node+1,
				    count->current);
#endif			 
			count->numbering->lambda2[count->n] = count->current;
			count->current += 1;
			count->counter += 1;
		}
	}

	count->n += 1;
	
	return;
}




/**
 *
 */
unsigned long __fem_bc_numbering_l2(const bc_list_t *bc_list,
				    const numbering_t *numbering,
				    const meshing_t *meshing,
				    unsigned long node, dofset_t dof,
				    unsigned long current)
{
	unsigned long n=0;
	
	if (bc_list)
	{
		struct bc_count_s bc_count;

		bc_count.numbering	= numbering;
		bc_count.meshing	= meshing;
		bc_count.node		= node;
		bc_count.dof		= dof;
		bc_count.counter	= 0;
		bc_count.current	= current;
		bc_count.n		= 0;
		__fem_page_list_loop_p(&bc_list->bc,
				       (page_func_p_t)bc_numbering_lambda2,
				       &bc_count);
		
		n=bc_count.counter;
	}
#ifdef FEM_DEBUG_NUMBERING
	if (n>0)
		haplo_debug("%lu lambda after dof %s%lu",
			    n, __fem_dof_name(dof), n+1);
#endif
	return(n);
}


/**
 *
 */
unsigned long __fem_bc_numbering_error(const bc_list_t *bc_list,
				       const numbering_t *numbering,
				       unsigned long no,
				       unsigned long n)
{
	unsigned long i;
	const bc_t *bc;
	
	bc=__fem_page_list_get_n(&bc_list->bc, n);
	
	for(i=0; i<bc->nb; i++)
	{
		if ((numbering->row[bc->part[i].dof] == 0) || 
		    (bc->part[i].node > no) ||
		    (numbering->row[bc->part[i].dof][bc->part[i].node] == 0))
		{
			haplo_error("DOF `%s%lu' which is involved in "
				    "boundary conditions is unknown.",
				    __fem_dof_name(bc->part[i].dof),
				    bc->part[i].node+1);
		}
	}
	return(bc->nb);
}


/*
 * GSM stuff.
 */

/**
 *
 */
static void bc_gsm_profile(bc_t *bc, struct bc_profile_s *bc_profile)
{
	const unsigned long l1=bc_profile->numbering->lambda1[bc_profile->n]-1;
	const unsigned long l2=bc_profile->numbering->lambda2[bc_profile->n]-1;
	unsigned long i;
	
	/* Lambda 1 */
 	for(i=0; i<bc->nb; i++)
	{
		unsigned long row;
		unsigned long n;
		
		n=__fem_node_internal(bc_profile->meshing, bc->part[i].node);

		row=bc_profile->numbering->row[bc->part[i].dof][n]-1;

		if (row-l1 > bc_profile->gsm->diag[row])
			bc_profile->gsm->diag[row] = row-l1;
	}
	
	/* Lambda 2 */
	bc_profile->gsm->diag[l2] = l2 - l1;
	

	bc_profile->n += 1;

	return;
}


/**
 *
 */
void __fem_bc_gsm_profile(matrix_t *gsm, const bc_list_t *bc_list, 
			  const numbering_t *numbering,
			  const meshing_t *meshing)
{
	struct bc_profile_s bc_profile;


	bc_profile.n=0;
	bc_profile.gsm=gsm;
	bc_profile.numbering=numbering;
	bc_profile.meshing=meshing;

	__fem_page_list_loop_p(&bc_list->bc,
			       (page_func_p_t)bc_gsm_profile,
			       &bc_profile);

	return;
}


/*
 *
 */
static void bc_gsm_compute(bc_t *bc, struct bc_compute_s *bc_compute)
{
	const unsigned long l1=bc_compute->numbering->lambda1[bc_compute->n]-1;
	const unsigned long l2=bc_compute->numbering->lambda2[bc_compute->n]-1;
	const double alpha = bc_compute->alpha;
	const double beta = bc_compute->beta;
	unsigned long i;

	/*
	 * Lambda 2
	 */
	for(i=0; i < bc->nb; i++)
	{	
		unsigned long row;
		unsigned long n;

		n=__fem_node_internal(bc_compute->meshing, bc->part[i].node);
		row=bc_compute->numbering->row[bc->part[i].dof][n]-1;

		/* Lambda 1 */
		bc_compute->gsm->coef[bc_compute->gsm->diag[row]-row+l1] =
			beta * bc->part[i].coef;
		
		/* Lambda 2 */
		bc_compute->gsm->coef[bc_compute->gsm->diag[l2]-l2+row] = 
			beta * bc->part[i].coef;
	}
	
	/* Lambda 1 */
	bc_compute->gsm->coef[bc_compute->gsm->diag[l1]]       += -alpha;
	
	/* Lambda 2 */
	bc_compute->gsm->coef[bc_compute->gsm->diag[l2]-l2+l1] += +alpha;
	bc_compute->gsm->coef[bc_compute->gsm->diag[l2]]       += -alpha;
	


	bc_compute->n += 1;
}


/**
 *
 */
static double bc_gsm_alpha(matrix_t *gsm)
{
	unsigned long i;
	int set=0;
	double max=1.0;
	double min=1.0;
	double alpha;
	
	
	for(i=0; i<gsm->size; i++)
	{
		const double kii = gsm->coef[gsm->diag[i]];
		if (fabs(kii) > DBL_EPSILON) 
		{
			if (set)
			{
				if (kii > max)
					max = kii;
				if (kii < min)
					min = kii;
			} else {
				max=kii;
				min=kii;
				set=1;
			}
		}
	}

	alpha=(max+min)/2.0;	
	haplo_info("BC coefficient: alpha=%e", alpha);

	return(alpha);
}


/**
 *
 */
double  __fem_bc_gsm_compute(matrix_t *gsm, const bc_list_t *bc_list, 
			     const numbering_t *numbering,
			     const meshing_t *meshing)
{
	struct bc_compute_s bc_compute;
	
	bc_compute.n=0;
	bc_compute.gsm=gsm;
	bc_compute.numbering=numbering;
	bc_compute.meshing=meshing;
	
	bc_compute.alpha=bc_gsm_alpha(gsm);
	/*
	 * In this implementation, beta = alpha
	 */
	bc_compute.beta=bc_compute.alpha;
	
	__fem_page_list_loop_p(&bc_list->bc,
			       (page_func_p_t)bc_gsm_compute,
			       &bc_compute);

	return(bc_compute.beta);
}


/*
 * Loadind stuff.
 */

/**
 *
 */
static void bc_loading_compute(bc_t *bc, struct bc_load_s *bc_load)
{
	/* Lambda 1 */
	bc_load->b->coef[bc_load->numbering->lambda1[bc_load->n]-1] +=
		bc_load->beta * bc->value;
	
	/* Lambda 2 */
	bc_load->b->coef[bc_load->numbering->lambda2[bc_load->n]-1] +=
		bc_load->beta * bc->value;
	
	bc_load->n += 1;
	
	return;
}


/**
 *
 */
void __fem_bc_vector(vector_t *b, const bc_list_t *bc_list, 
		     const numbering_t *numbering, double beta)
{
	struct bc_load_s bc_load;
	
	bc_load.n=0;
	bc_load.b=b;
	bc_load.numbering=numbering;
	bc_load.beta=beta;
	
	__fem_page_list_loop_p(&bc_list->bc,
			       (page_func_p_t)bc_loading_compute,
			       &bc_load);

	return;
}
