/*
 * $Id: utils.c,v 1.20 2003/12/01 09:50:15 nicoo Exp $
 *
 *
 * Copyright (C) 1999, 2000, 2001 Nicolas LAURENT
 * This file is part of `Haplo'
 * 
 *
 * `Haplo'  is free software;  you can  redistribute  it and/or modify it
 * under the terms of the GNU Library General Public License as published
 * by the Free Software Foundation;  either version 2  of the License, or
 * (at your option) any later version.
 *
 * `Haplo' is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the  implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the  GNU General Public License
 * for more details.
 * 
 * You should have  received  a copy of the   GNU General Public  License
 * along with `Haplo'.  If not, write to  the
 *
 *                                        Free Software Foundation,  Inc.
 *                                        675 Mass Ave, Cambridge, MA
 *                                        02139, USA.
 *
 */

#include "extensions.h"
#ifdef HAVE_CONFIG_H
#	include "config.h"
#endif
#include "version.h"


#ifdef HAVE_LIMITS_H
#	include <limits.h>
#endif /* HAVE_LIMITS_H */
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#ifdef HAVE_SYS_IOCTL_H
#	include <sys/ioctl.h>
#endif /* HAVE_SYS_IOCTL_H */
#ifdef HAVE_UNISTD_H
#	include <unistd.h>
#endif
#ifdef ENABLE_BACKTRACE
#include "bt.h"
#endif
#include "utils.h"
#ifdef HAVE_SYS_TIME_H
#	include <sys/time.h>
#endif /* HAVE_SYS_TIME_H */
#ifdef HAVE_SYS_RESOURCE_H
#	include <sys/resource.h>
#endif /* HAVE_SYS_RESOURCE_H */
#include <sys/types.h>


#ifdef HAPLO_DEBUG_MEMORY
#	include "bt.h"
#endif /* HAPLO_DEBUG_MEMORY */

#ifdef ENABLE_BACKTRACE
#	undef ENABLE_BACKTRACE
#endif


/*-----------------------------------------------------------------------------
                       G L O B A L   V A R I A B L E S 
-----------------------------------------------------------------------------*/

static int		colors=HAPLO_COLORS_NONE;
static memory_block_t	*memory_usage=NULL;
static int		memory_number_malloc=0;
static int		memory_number_free=0;
static int		memory_number_realloc=0;
static size_t		memory_size=0;
static size_t		memory_max=0;
static const char	*function=NULL;


/*-----------------------------------------------------------------------------
                             P R O T O T Y P E S 
-----------------------------------------------------------------------------*/

/*
 * Memory stuff.
 */
static const char *xbasename(const char *path);
static void memfill(void *ptr, size_t size);
static char *utils_strdup(const char *s);
void *haplo_malloc(size_t size);
void *haplo_malloc_debug(size_t size, const char *filename,
			 unsigned long line);
void haplo_free_debug(void *ptr, const char *filename, unsigned long line);
void *haplo_realloc(void *ptr, size_t size);
void *haplo_realloc_debug(void *ptr, size_t size, const char *filename,
			  unsigned long line);
void haplo_memory_watch(void *ptr);
static void memory_usage_print(size_t l, char *buffer, size_t len);
int haplo_memory_usage(void);

/*
 * Display Stuff.
 */
void __haplo_colors_policy(int colors);
int __haplo_colors_set(const char *color);
void __haplo_colors_reset(void);
void __haplo_prefix(const char *s);
static void prefix(void);
void haplo_fatal(const char *s, ...);
void haplo_warning(const char *s, ...);
void haplo_error(const char *s, ...);
void haplo_info(const char *s, ...);
void haplo_debug(const char *s, ...);
void __haplo_result_display(const object_t *object);
static int get_width(void);
void haplo_centered(const char *s, ...);
void haplo__bordered(const char *s, ...);
void haplo_underlined(const char *s, ...);
double haplo_clamp(double in, double min, double max);
int haplo_int(double in);
unsigned int haplo_uint(double in);
unsigned short haplo_ushort(double in);
unsigned long haplo_ulong(double in);
char *haplo_strdup(const char *s);

/*
 * Timer stuff.
 */
void haplo_timer_start(haplo_timer_t *timer);
void haplo_timer_stop(haplo_timer_t *timer);
void haplo_timer_print(const haplo_timer_t *timer, const char *s, ...);
void __haplo_timer_print(const haplo_timer_t *timer);


/*-----------------------------------------------------------------------------
                         I M P L E M E N T A T I O N 
-----------------------------------------------------------------------------*/

/*
 * Memory stuff.
 */

/**
 *
 */
static const char *xbasename(const char *path)
{
	const char *p;
	
	for(p=path+strlen(path)-1; p!=path; --p)
	{
		if (*p == '/')
		{
			++p;
			break;
		}
	}
	return(p);
}


/**
 *
 */
static void memfill(void *ptr, size_t size)
{
	size_t		i;
	unsigned char	*p=ptr;
	
	for(i=0; i<size; i++)
		p[i] = 0xFF;
	
	return;
}


/**
 *
 */
static char *utils_strdup(const char *s)
{
	char	*c;
	
	c=malloc(strlen(s)+1);
	strcpy(c, s);

	return(c);
}


/**
 *
 */
void *haplo_malloc(size_t size)
{
	void	*ptr;
	
	ptr=malloc(size);
	if (!ptr)
	{
		char s[20];
		
		memory_usage_print(size, s, 20);
		
		haplo_fatal("Fail to allocate %s", s);
	}
	
	return(ptr);
}


/**
 *
 */
void *haplo_malloc_debug(size_t size, const char *file, unsigned long line)
{
	void		*ptr;
	memory_block_t	*block;

	if (!size)
	{
		haplo_fatal("Allocating 0 byte. (%s:%lu)", file, line);
		return(NULL);
	}

	ptr=malloc(size);
	memory_number_malloc++;
	memory_size += size;
	if (memory_max < memory_size)
		memory_max=memory_size;

	if (!ptr)
	{
		char s[20];
		
		memory_usage_print(size, s, 20);		
		haplo_fatal(_("Fail to allocate %s (%s:%lu)"),
			    s, xbasename(file), line);
	}
	
	if (! memory_usage)
	{
		memory_usage=malloc(sizeof(*memory_usage));
		block=memory_usage;
	}
	else
	{
		for(block=memory_usage; block->next; block=block->next)
			;
		block->next=malloc(sizeof(*block->next));
		block=block->next;
	}
	if (!block)
		haplo_fatal(_("Memory debugger unavailable "
			    "(virtual memory exhausted)"));

	block->addr=ptr;
	block->size=size;
	block->flags=0;
	block->file=utils_strdup(file);
	block->line=line;
	block->next=NULL;
	
	memfill(ptr, size);
	
	return(ptr);
}


/**
 *
 */
void haplo_free_debug(void *ptr, const char *file, unsigned long line)
{

	if (! ptr)
	{
		haplo_fatal(_("Freeing nul-pointer. (%s:%lu)"),
			    xbasename(file), line);
	}
	else
	{
		memory_block_t	*block, *prev=NULL;
		for(block=memory_usage; block; block=block->next)
		{
			if (block->addr == ptr)
			{
				memfill(ptr, block->size);
				if (block->flags & BLOCK_LOOKED)
				{
					haplo_debug(
						_("Freeing memory at %p "
						  "(%s:%lu)"), ptr,
						xbasename(file), line);
				}
				free(ptr);
				memory_size -= block->size;
				if (block == memory_usage)
					memory_usage=memory_usage->next;
				else
					prev->next=block->next;
				free(block->file);
				free(block);
				memory_number_free++;
				return;
			}
			prev=block;
		}
		haplo_fatal(_("Freeing non-allocated memory at %p. (%s:%lu)"),
			    ptr, xbasename(file), line);

	}
}


/**
 *
 */
void *haplo_realloc(void *ptr, size_t size)
{
	void	*newptr;

	newptr=realloc(ptr, size);
	if (!newptr)
	{
		char s[20];
		
		memory_usage_print(size, s, 20);
		
		haplo_fatal(_("Fail to re-allocate %s"), s);
	}
	
	return(newptr);
}


/**
 *
 */
void *haplo_realloc_debug(void *ptr, size_t size, const char *file,
			  unsigned long line)
{
	void		*newptr;
	memory_block_t	*block;
	size_t		prev=0;

	if (!ptr)
	{
		haplo_warning(_("Re-allocating nul-pointer (%s:%lu)"),
			      xbasename(file), line);

		if (memory_usage == NULL)
		{
			memory_usage=malloc(sizeof(*memory_usage));
			block=memory_usage;
		}
		else
		{
			for(block=memory_usage; block->next; block=block->next)
				;
			block->next=malloc(sizeof(*block->next));
			block=block->next;
		}
		if (!block)
			haplo_fatal(_("Memory debugger unavailable "
				      "(virtual memory exhausted)"));
		block->next=NULL;
		prev=0;
	}
	else
	{
		int ok=0;
		for(block=memory_usage; block; block=block->next)
		{
			if (block->addr == ptr)
			{
				prev=block->size;
				ok=1;
				break;
			}
		}
		if (!ok)
		{
			haplo_fatal(_("Reallocation unallocated memory at "
				      "%p (%s:%lu)"), ptr,
				    xbasename(file), line);
		}
	}
	newptr=realloc(ptr, size);
	memory_number_realloc++;
	memory_size += (size - prev);
	if (!newptr)
		haplo_fatal("Impossible d'agrandir le tampon  %.2f ko",
			    (double)size/1024);

	block->addr=newptr;
	block->size=size;
	block->file=utils_strdup(file);
	block->line=line;
	
	memfill((char *)newptr+(size-prev), size-prev);
	
	return(newptr);
}


/**
 *
 */
void haplo_memory_watch(void *ptr)
{
	memory_block_t	*block;
	for(block=memory_usage; block; block=block->next)
	{
		if (ptr == block->addr)
		{
			block->flags |= BLOCK_LOOKED;
			return;
		}
	}
	haplo_debug("Impossible de surveiller la mmoire pointe  %p", ptr);
	return;
}


/**
 *
 */
static void memory_usage_print(size_t l, char *buffer, size_t len)
{
	if (l<1024)
		snprintf(buffer, len, "%d b", l);
	else if (l < 1024*1024)
		snprintf(buffer, len, "%.2f kb", (double)l/1024.0);
	else if (l < 1024*1024*1024)
  		snprintf(buffer, len, "%.2f Mb", (double)l/(1024.0*1024));
	else
		snprintf(buffer, len, "%.2f Gb",
			 (double)l/(1024.0*1024*1024));

	return;
}


/**
 *
 */
int haplo_memory_usage(void)
{	
	const int leaks=memory_number_malloc+memory_number_realloc
		-memory_number_free;
	char rss[20];
	
	haplo_debug(_("Number of malloc()....: %d"), memory_number_malloc);
	haplo_debug(_("Number of free()......: %d"), memory_number_free);
	haplo_debug(_("Number of realloc()...: %d"), memory_number_realloc);
	memory_usage_print(memory_max, rss, 20);
	haplo_debug(_("Maximal RSS...........: %s\n"), rss);
	if (memory_usage)
	{
		memory_block_t	*block;

		haplo_debug("Number of leak%c.......: %d", (leaks>1)?'s':'.',
			    leaks);
		
		haplo_debug(",-----------------------------------------------"
			    "-----------------------------.");
		haplo_debug(_("|                                Memory leaks "
			    "                               |"));
		haplo_debug("|-----------------------------------------------"
			    "-----------------------------|");
		haplo_debug(_("|                         Where               "
			    "     |    Size    |   Pointer  |"));
		haplo_debug("|-----------------------------------------------"
			    "---+------------+------------|");

		for(block=memory_usage; block; block=block->next)
		{
			char	info[49];

			snprintf(info, 48, "%s:%lu",
				 xbasename(block->file),
				 block->line);
			info[48]='\0';
			haplo_debug("| %-48s | %10u | 0x%-8x |", info,
				    block->size, (unsigned int)block->addr);
			free(block->addr);
			free(block->file);
			free(block);
		}
		haplo_debug("`-----------------------------------------------"
			    "-----------------------------'");
	}
	else
		haplo_debug(_("No memory leaks."));
		
	return(leaks);
}


/*
 * Display Stuff.
 */

/**
 *
 */
void __haplo_colors_policy(int c)
{
	colors = c;
	return;
}


/**
 *
 */
int __haplo_colors_set(const char *color)
{
	int ret=1;
	
	if (colors & HAPLO_COLORS_ANSI)
	{
		if (colors & HAPLO_COLORS_FLASH)
			fputs(COLOR_FLASH, stdout);
		fputs(color, stdout);
		ret=0;
	}
	return(ret);
}


/**
 *
 */
void __haplo_colors_reset(void)
{
	if (colors)
		puts(COLOR_RESET);
	else
		fputc('\n', stdout);

	return;
}


/**
 *
 */
void __haplo_prefix(const char *s)
{
	function = s;
	
	return;
}

/**
 *
 */
static void prefix(void)
{
	if (function)
	{
		__haplo_colors_set(COLOR_CYAN);
		fputs(function, stdout);
		if (colors)
			fputs(COLOR_RESET ": ", stdout);
		else
			fputs(": ", stdout);
	}
	
	return;
}


/**
 *
 */
void haplo_fatal(const char *s, ...)
{
	va_list	ap;

	prefix();
	
	if (__haplo_colors_set(COLOR_FATAL))
		fputs(_("Fatal error: "), stdout);

	va_start(ap, s);
	vprintf(s, ap);
	va_end(ap);
	__haplo_colors_reset();

#if defined HAPLO_DEBUG_MEMORY && defined ENABLE_BACKTRACE
	__haplo_bt();
#endif /* HAPLO_DEBUG_MEMORY */

	puts(_("End."));
	exit(EXIT_FAILURE);

	/* never reached */
	return;
}


/**
 *
 */
void haplo_error(const char *s, ...)
{
	va_list	ap;

	prefix();
	
	if (__haplo_colors_set(COLOR_ERROR))
		fputs(_("Error: "), stdout);

	va_start(ap, s);
     	vprintf(s, ap);
	va_end(ap);

	__haplo_colors_reset();
	
	return;
}


/**
 *
 */
void haplo_warning(const char *s, ...)
{
	va_list	ap;

	prefix();
	
	if (__haplo_colors_set(COLOR_WARNING))
		fputs(_("Warning: "), stdout);


	va_start(ap, s);
	vprintf(s, ap);     
	va_end(ap);

	__haplo_colors_reset();
	
	return;
}


/**
 *
 */
void haplo_info(const char *s, ...)
{
	va_list	ap;

	prefix();
	
	if (__haplo_colors_set(COLOR_INFO))
		fputs(_("Info: "), stdout);

	va_start(ap, s);
	vprintf(s, ap);     
	va_end(ap);

	__haplo_colors_reset();

	return;
}


/**
 *
 */
void haplo_debug(const char *s, ...)
{
	va_list	ap;

	prefix();
	
	if (__haplo_colors_set(COLOR_DEBUG))
		fputs(_("Debug: "), stdout);

	va_start(ap, s);
	vprintf(s, ap);
	va_end(ap);

	__haplo_colors_reset();
	
	return;
}


/**
 *
 */
void __haplo_result_display(const object_t *object)
{
	if (object)
	{
		__haplo_colors_set(COLOR_RESULT);

		fputs("   ", stdout);
		__haplo_object_display(object);
	
		__haplo_colors_reset();
	}
	
	return;
}


/**
 *
 */
static int get_width(void)
{
	int width=DEFAULT_SCREEN_WIDTH;
#ifdef TIOCGWINSZ
	struct winsize win;

	if (ioctl(STDOUT_FILENO, TIOCGWINSZ, (char *)&win) == 0)
		width = win.ws_col;
	else
	{
#endif /* TIOCGWINSZ*/
		char *txt;
		
		txt=getenv("COLUMNS");
 		if (txt)
			width=atoi(txt);
#ifdef  TIOCGWINSZ
	}
#endif /* TIOCGWINSZ*/
	return(width);
}


/**
 *
 */
void haplo_centered(const char *s, ...)
{
	va_list	ap;
	char	*title;
	const int width=get_width();
	int len;
	int i;
	int mid;	
	
	title=HAPLO_MALLOC(width+1);

	va_start(ap, s);
	len=vsnprintf(title, width+1, s, ap);
	va_end(ap);

	mid=(width-len)/2;
	
	for(i=0; i<mid; i++)
		putchar(' ');
	puts(title);

	putchar('\n');

	HAPLO_FREE(title);

	return;
}


/**
 *
 */
void haplo_underlined(const char *s, ...)
{
	va_list	ap;
	char	*title;
	const int width=get_width();
	int len;
	int i;
	int mid;

	title=HAPLO_MALLOC(width+1);

	va_start(ap, s);
	len=vsnprintf(title, width+1, s, ap);
	va_end(ap);

	mid=(width-len)/2;
	
	for(i=0; i<mid; i++)
		putchar(' ');
	puts(title);
	for(i=0; i<mid; i++)
		putchar(' ');
	for(i=0; i<len; i++)
		putchar('-');

	putchar('\n');
	putchar('\n');

	HAPLO_FREE(title);
     
	return;
}


/**
 *
 */
void haplo_bordered(const char *s, ...)
{
	va_list	ap;
	char	*title;
	const int width=get_width();
	int len;
	int i;
	int mid;	

	title=HAPLO_MALLOC(width+1);

	va_start(ap, s);
	len=vsnprintf(title, width+1, s, ap);
	va_end(ap);

	mid=(width-len)/2;

	for(i=0; i<mid-2; i++)
		putchar(' ');
	
	putchar('+');
	for(i=0; i<len+2; i++)
		putchar('-');
	putchar('+');
	putchar('\n');
	for(i=0; i<mid-2; i++)
		putchar(' ');
	putchar('|');
	putchar(' ');
	fputs(title, stdout);
	puts(" |");
	for(i=0; i<mid-2; i++)
		putchar(' ');
	putchar('+');
	for(i=0; i<len+2; i++)
		putchar('-');
	puts("+");

	__haplo_colors_reset();
	putchar('\n');

	HAPLO_FREE(title);

	return;
}


/**
 *
 */
double haplo_clamp(double in, double min, double max)
{
	double out;

	if (in > max)
	{
		out=max;
		haplo_info(_("Overflow: `%.8g' is rounded to `%.8g'"),
			     in, out);
	} else if (in < min)
	{
		out=min;
		haplo_info(_("Underflow: `%.8g' is rounded to `%.8g'"),
			     in, out);
	} else {
		out=in;
	}
	
	return(out);
}


/**
 *
 */
int haplo_int(double in)
{
	int	out;
	
	if (in < (double)INT_MIN)
	{
		out=INT_MIN;
		haplo_info(_("Underflow: `%.8g' is rounded to `%u'"), in, out);
	}
	else if (in > (double)INT_MAX)
	{
		out=INT_MAX;
		haplo_info(_("Overflow: `%.8g' is rounded to `%u'"), in, out);
	}
	else
	{
		out=(int)in;
		if (in != out)
			haplo_info(_("`%.8g' is rounded to `%u'"), in, out);
	}
	return(out);
}


/**
 *
 */
unsigned int haplo_uint(double in)
{
	unsigned int	out;

	if (in < 0)
	{
		out=0;
		haplo_info(_("Underflow: `%e' is rounded to `%u'"), in, out);
	}
	else if (in > (double)UINT_MAX)
	{
		out=UINT_MAX;
		haplo_info(_("Overflow: `%e' is rounded to `%u'"), in, out);
	}
	else
	{
		out=(unsigned int)in;
		if (in != out)
			haplo_info(_("`%e' is rounded to `%u'"), in, out);
	}
	return(out);
}


/**
 *
 */
unsigned short haplo_ushort(double in)
{
	unsigned short	out;

	if (in < 0)
	{
		out=0;
		haplo_info(_("Underflow: `%e' is rounded to `%u'"), in, out);
	}
	else if (in > (double)USHRT_MAX)
	{
		out=USHRT_MAX;
		haplo_info(_("Overflow: `%e' is rounded to `%u'"), in, out);
	}
	else
	{
		out=(unsigned short)in;
		if (in != out)
			haplo_info(_("`%e' is rounded to `%u'"), in, out);
	}
	return(out);
}


/**
 *
 */
unsigned long haplo_ulong(double in)
{
	unsigned long	out;

	if (in < 0)
	{
		out=0;
		haplo_info(_("Underflow: `%e' is rounded to `%lu'"), in, out);
	}
	else if (in > (double)ULONG_MAX)
	{
		out=ULONG_MAX;
		haplo_info(_("Overflow: `%e' is rounded to `%lu'"), in, out);
	}
	else
	{
		out=(unsigned long)in;
		if (in != out)
			haplo_info(_("`%e' is rounded to `%lu'"), in, out);
	}
	return(out);
}


/**
 *
 */
char *haplo_strdup(const char *s)
{
	char	*copy;
	copy=HAPLO_MALLOC(strlen(s)+1);
	strcpy(copy, s);
	
	return(copy);
}


/*
 * Timer stuff.
 */

/**
 *
 */
void haplo_timer_start(haplo_timer_t *timer)
{
#ifdef WIN32
	__FIXME__
#else
	struct rusage usage;
	
	if (getrusage(RUSAGE_SELF, &usage)<0)
	{
		haplo_warning(_("Execution statistics unavailable."));
	}

	timer->sec =usage.ru_utime.tv_sec + usage.ru_stime.tv_sec;
	timer->usec=usage.ru_utime.tv_usec + usage.ru_stime.tv_usec;
	
	if (timer->usec > 1000000)
	{
		timer->usec -= 1000000;
		timer->sec  += 1;
	}
#endif
	return;
}


/**
 *
 */
void haplo_timer_stop(haplo_timer_t *timer)
{
#ifdef WIN32
	__FIXME__
#else
	struct rusage usage;
	unsigned long sec;
	unsigned long usec;
	
	if (getrusage(RUSAGE_SELF, &usage)<0)
	{
		haplo_warning(_("Execution statistics unavailable."));
	}

	sec =usage.ru_utime.tv_sec + usage.ru_stime.tv_sec;
	usec=usage.ru_utime.tv_usec + usage.ru_stime.tv_usec;

	
	if (usec > 1000000)
	{
		usec -= 1000000;
		sec  += 1;
	}
	
	if (usec < timer->usec)
	{
		timer->sec += 1;
		usec       += 1000000;
	}

	timer->sec  = sec  - timer->sec;
	timer->usec = usec - timer->usec;
#endif
	return;
}


/**
 *
 */
void haplo_timer_print(const haplo_timer_t *timer, const char *s, ...)
{
	va_list	ap;

	prefix();
	
	if (colors & HAPLO_COLORS_ANSI)
	{
		if (colors & HAPLO_COLORS_FLASH)
			fputs(COLOR_FLASH, stdout);
		fputs(COLOR_GREEN, stdout);
	}
	else
		fputs("Info  : ", stdout);

	va_start(ap, s);
	vprintf(s, ap);     
	va_end(ap);

	fputs(": ", stdout);
	if (colors & HAPLO_COLORS_ANSI)
	{
		if (colors & HAPLO_COLORS_FLASH)
			fputs(COLOR_FLASH, stdout);
		fputs(COLOR_WHITE, stdout);
	}
	
	__haplo_timer_print(timer);
	
	if (colors)
		puts(COLOR_RESET);
	else
		fputc('\n', stdout);

	return;
}


/**
 *
 */
void __haplo_timer_print(const haplo_timer_t *timer)
{
	unsigned long	t;
	unsigned long	d;
	unsigned long	h;
	unsigned long	m;
	float		s;
	
	t=timer->sec;
	
	d  = t / (24*60*60);
	t -= d * (24*60*60);
	
	h  = t / (60*60);
	t -= h * (60*60);
	
	m  = t / (60);
	t -= m * (60);
	
	s  = (float)(t + timer->usec/1.0E6);
	
	if (d)
		printf("%lu day%s %luh %lum %.2fs", d, (d>1)?_("s"):"",
		       h, m, s);
	else
		if (h)
			printf("%luh %lum %.2fs", h, m, s);
		else
			if (m)
				printf("%lum %.2fs", m, s);
			else
				if (s >= 0.01)
					printf("%.2fs", s);
				else
					fputs("<0.01s", stdout);
	return;
}
