/*
 * $Id: plugins.c,v 1.18 2003/12/01 09:50:15 nicoo Exp $
 *
 *
 * Copyright (C) 1999, 2000, 2001 Nicolas LAURENT
 * This file is part of `Haplo'
 * 
 *
 * `Haplo'  is free software;  you can  redistribute  it and/or modify it
 * under the terms of the GNU Library General Public License as published
 * by the Free Software Foundation;  either version 2  of the License, or
 * (at your option) any later version.
 *
 * `Haplo' is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the  implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the  GNU General Public License
 * for more details.
 * 
 * You should have  received  a copy of the   GNU General Public  License
 * along with `Haplo'.  If not, write to  the
 *
 *                                        Free Software Foundation,  Inc.
 *                                        675 Mass Ave, Cambridge, MA
 *                                        02139, USA.
 *
 */

/* 
 *              C h a r g e m e n t   d e s   P L U G ' I N S
 *              ---------------------------------------------
 *
 * Il s'agit de charger une bibliotques dynamique.
 * Le nom du fichier correspondant est de la forme libXXX.so ou libXXX.sl selon
 * l'implementation.
 * Une fois le dlopen(3) ou shl_load() effectu on value le symbole XXX_init.
 *
 */

#ifdef HAVE_CONFIG_H
#	include "config.h"
#endif
#include "version.h"

#include <stdio.h>
#ifdef HAVE_STRING_H
#	include <string.h>
#endif /* HAVE_STRING_H */

#include "func.h"
#include "plugins.h"
#include "slink.h"
#include "utils.h"


#if HAPLO_PLUGINS_IMPL != HAPLO_PLUGINS_IMPL_NONE
#	ifdef HAVE_DL_H
#		include <dl.h>
#	endif /* HAVE_DL_H */
#	ifdef HAVE_DLFCN_H
#		include <dlfcn.h>
#	endif /* HAVE_DLFCN_H */
#	ifdef HAVE_LIMITS_H
#		include <limits.h>
#	endif /* HAVE_LIMITS_H */
#	include <sys/stat.h>
#	include <sys/types.h>
#	ifdef HAVE_UNISTD_H
#		include <unistd.h>
#	endif
#	ifndef PATH_MAX
#		define PATH_MAX	1024
#	endif /* PATH_MAX */
#	ifndef HAVE_DLERROR
#		ifdef __NetBSD__
#			include <errno.h>
#			define dlerror()	strerror(errno)
#		else /* ! __NetBSD__ */
#			define dlerror()	"Unknown dl-error"
#		endif /* __NetBSD__ */
#	endif /* HAVE_DLERROR */
#	ifndef RTLD_NOW
#		define RTLD_NOW	0
#	endif /* RTLD_NOW */
#endif /* HAPLO_PLUGINS_IMPL != HAPLO_PLUGINS_IMPL_NONE */


/*-----------------------------------------------------------------------------
                       G L O B A L   V A R I A B L E S 
-----------------------------------------------------------------------------*/

#if HAPLO_PLUGINS_IMPL != HAPLO_PLUGINS_IMPL_NONE
static slink_t		*plugins_list=NULL;
static slink_t		*plugins_path=NULL;
#endif /* */


/*-----------------------------------------------------------------------------
                             P R O T O T Y P E S 
-----------------------------------------------------------------------------*/

int haplo_plugins_path_add(const char *path);
#if HAPLO_PLUGINS_IMPL != HAPLO_PLUGINS_IMPL_NONE
static plugins_handle_t plugins_library_load(const char *path);
static int plugins_library_unload(plugins_handle_t handle);
plugins_handle_t *  __haplo_plugins_library_load(const char *path);
void *__haplo_plugins_symbol_get(plugins_handle_t handle, const char *symbol);
static plugins_t * plugins_already_loaded(const char *name);
static char *plugins_full_path(const char *dir, const char *name);
static int plugins_load_magic(const char *magic);
static int plugins_load_check(int major, int minor);
static int plugins_load_depend(const char *const* depend);
static int plugins_load_init(plugins_t *plugin, const char *name);
static plugins_t * plugins_load(const char *dir, const char *name);
int __haplo_plugins_load(const char *name);
static void plugins_status_loaded(plugins_t *plugin);
static void plugins_status_path(const char *path);
void __haplo_plugins_status(void);
static void plugins_unload(plugins_t *plugin);
void __haplo_plugins_unload_all(void);
void __haplo_plugins_reload(const char *name);
#endif /* HAPLO_PLUGINS_IMPL != HAPLO_PLUGINS_IMPL_NONE */


/*-----------------------------------------------------------------------------
                         I M P L E M E N T A T I O N 
-----------------------------------------------------------------------------*/


/**
 * Add a path in plugins searchpath
 *
 * @param path
 */
int haplo_plugins_path_add(const char *path)
{
#if HAPLO_PLUGINS_IMPL != HAPLO_PLUGINS_IMPL_NONE
	char *full_path=NULL;
	int len;
	
	if (path[0] != '\0')
	{
		if (path[0] == '/')
		{
			full_path=haplo_strdup(path);
			len=strlen(full_path);
		} else {
			char here[PATH_MAX];

			getcwd(here, PATH_MAX);
			here[PATH_MAX-1]=0;
		
			len=strlen(here)+strlen(path)+1;
			HAPLO_ALLOC(full_path, len+1);
			strcpy(full_path, here);
			strcat(full_path, "/");
			strcat(full_path, path);
		}
		if (full_path[len-1] == '/')
			full_path[len-1]=0;
	} else
		full_path=haplo_strdup(path);
	
	plugins_path=__haplo_slink_prepend(plugins_path, full_path);
#else	/* HAPLO_PLUGINS_IMPL == HAPLO_PLUGINS_IMPL_NONE */
	(void)path;	/* avoid gcc to complain... */
#endif /* HAPLO_PLUGINS_IMPL != HAPLO_PLUGINS_IMPL_NONE */	

	return(0);
}


#if HAPLO_PLUGINS_IMPL != HAPLO_PLUGINS_IMPL_NONE
/*
 * the rest of the module support is not needed if the arch does not support
 * loadable library mecanism.
 */


/**
 * dlopen() wrapper
 *
 * @param path is the fully qualified path of the object we're trying to load
 */

static plugins_handle_t plugins_library_load(const char *path)
{
	plugins_handle_t handle=NULL;
	
#	if HAPLO_PLUGINS_IMPL == HAPLO_PLUGINS_IMPL_DL
	handle=dlopen(path, RTLD_NOW);
#	endif
#	if HAPLO_PLUGINS_IMPL == HAPLO_PLUGINS_IMPL_DLD
	handle=shl_load(path,
			BIND_IMMEDIATE |
			BIND_NONFATAL |
			BIND_VERBOSE,
			0);
#	endif
	if (!handle)
	{
		struct stat unused;
		/*
		 * Display diagnostic only if a object was found or searched
		 * in LD_LIBRAY_PATH
		 */
		if ((path[0] != '/') || (access(path, F_OK) == 0))
			haplo_error(_("Failed in loading: %s"), dlerror());
	}
	
	return(handle);
}


/**
 * dlclose() wrapper
 * 
 * @param handle
 */
static int plugins_library_unload(plugins_handle_t handle)
{
	int status;
	
#	if HAPLO_PLUGINS_IMPL == HAPLO_PLUGINS_IMPL_DL
	status=dlclose(handle);
#	endif
#	if HAPLO_PLUGINS_IMPL == HAPLO_PLUGINS_IMPL_DLD
	status=shl_unload(plugin->handle);
#	endif

	return(status);
}


/**
 * Haplo interface for loading plugins
 *
 * @param path
 */
plugins_handle_t * __haplo_plugins_library_load(const char *path)
{
	plugins_handle_t	*handle;
	
	HAPLO_ALLOC(handle, 1);
	*handle=plugins_library_load(path);
	
	if (! *handle)
	{	
		HAPLO_FREE(handle);
		handle=NULL;
	}

	return(handle);
}


/**
 * Resolve a symbol in a plugin
 *
 * @param handle
 * @param symbol
 */
void *__haplo_plugins_symbol_get(plugins_handle_t handle, const char *symbol)
{
	void *addr=NULL;

	if (symbol && symbol[0])
	{
#	if HAPLO_PLUGINS_IMPL == HAPLO_PLUGINS_IMPL_DL
		addr=dlsym(handle, symbol);
#	endif
#	if HAPLO_PLUGINS_IMPL == HAPLO_PLUGINS_IMPL_DLD
		shl_findsym(&handle, symbol, TYPE_UNDEFINED, (void *)&func);
#	endif 
	}
	
	return(addr);
}


/**
 * check if a plugin is already loaded
 *
 * @param name
 */
static plugins_t * plugins_already_loaded(const char *name)
{
	slink_t	*s;
	plugins_t *plugin=NULL;
	
	for(s=plugins_list; s; s=s->next)
	{
		if (strcmp(PLUGINS(s->data)->name, name)==0)
		{
			plugin=PLUGINS(s->data);
			break;
		}
	}
	return(plugin);
}


/**
 * try to resolve to fully qualified path from a module name
 *
 * @param dir
 * @param name
 */
static char * plugins_full_path(const char *dir, const char *name)
{
	char *path;
	
	if (dir[0] != '\0')
	{
		HAPLO_ALLOC(path,
			    strlen(dir)+
			    strlen(name)+
			    strlen(HAPLO_PLUGINS_PREFIX)+
			    strlen(HAPLO_PLUGINS_SUFFIX)+2);
		strcpy(path, dir);
		strcat(path, "/" HAPLO_PLUGINS_PREFIX);
		strcat(path, name);
		strcat(path, HAPLO_PLUGINS_SUFFIX);
	}
	else
	{
		HAPLO_ALLOC(path,
			    strlen(name)+
			    strlen(HAPLO_PLUGINS_PREFIX)+
			    strlen(HAPLO_PLUGINS_SUFFIX)+1);
		strcpy(path, HAPLO_PLUGINS_PREFIX);
		strcat(path, name);
		strcat(path, HAPLO_PLUGINS_SUFFIX);
	}
	
	return(path);
}


/**
 *
 */
static int plugins_load_magic(const char *magic)
{
	int status=-1;
	
	if ((magic[0]==HAPLO_MAG0) &&
	    (magic[1]==HAPLO_MAG1) &&
	    (magic[2]==HAPLO_MAG2) &&
	    (magic[3]==HAPLO_MAG3) &&
	    (magic[4]==HAPLO_MAG4))
		status=0;

	return(status);
}


/** 
 *
 */
static int plugins_load_check(int major, int minor)
{
	int status=0;
	
	if (major == HAPLO_MAJOR)
	{
		if (minor > HAPLO_MINOR)
		{
			haplo_error(
				"Haplo-kernel is outdated (required minor %d)",
				minor);
			status=-1;
		}
	} else if (major < HAPLO_MAJOR)
	{
		haplo_warning("This module is rather old.");
	} else {
		haplo_error("Haplo-kernel is outdated (required major %d)",
			    major);
		status=-1;
	}
}


/**
 *
 */
static int plugins_load_depend(const char *const* depend)
{
	int	i;

	for(i=0; depend[i]!=NULL; i++)
	{
		if (!plugins_already_loaded(depend[i]))
		{
			if (__haplo_plugins_load(depend[i])<0)
			{
				haplo_error("Une dpendance de module n'a pas"
					    " t satisfaite.");
				return(-1);
			}
		}
	}

	return(0);
}


/**
 *
 */
static int plugins_load_init(plugins_t *plugin, const char *name)
{
	haplo_module_t *module;
	int status=-1;
	char *symbol;

	HAPLO_ALLOC(symbol, strlen(name)+sizeof(HAPLO_PLUGINS_SYMBOL)+1);
	strcpy(symbol, name);
	strcat(symbol, HAPLO_PLUGINS_SYMBOL);
	module=__haplo_plugins_symbol_get(plugin->handle, symbol);
	HAPLO_FREE(symbol);
	
	if (module && (plugins_load_magic(module->magic) == 0))
	{
		if (plugins_load_check(module->required_major,
				       module->required_minor) == 0)
		{
			plugins_load_depend(module->depend);
			if (module->version)
				haplo_info("<%s> version %s (c) %s",
					   (module->title)?module->title:name, 
					   module->version,
					   module->author);
			else
				haplo_info("<%s> version (c) %s",
					   (module->title)?module->title:name,
					   module->author);

			__haplo_prefix(name);
			if (module->init)
				(*module->init)();

			plugin->types=__haplo_object_type_register(
				module->types, plugin);

			plugin->functions=__haplo_func_register(
				module->functions, plugin);

			plugin->module=module;
			status=0;
		}
	} else {
		plugins_library_unload(plugin->handle);
		haplo_error(_("Library `%s' (%s) is not a haplo-module."),
			    name, plugin->path);
	}
	
	return(status);
}


/**
 *
 */
static plugins_t * plugins_load(const char *dir, const char *name)
{
	plugins_handle_t handle;
	plugins_t *plugin=NULL;
	char *path;

	path=plugins_full_path(dir, name);

	/* Loading */
	handle=plugins_library_load(path);
	
	if (handle)
	{
		HAPLO_ALLOC(plugin, 1);
		plugin->handle=handle;
		plugin->path=path;
		
		/* Init */
		if (plugins_load_init(plugin, name) == 0)
		{
			plugin->name = haplo_strdup(name);

		} else {
			HAPLO_FREE(path);
			HAPLO_FREE(plugin);
			plugin=NULL;
		}
	} else
		HAPLO_FREE(path);		

	return(plugin);
}


/**
 * haplo interface: use()
 *
 * @param name
 */
int __haplo_plugins_load(const char *name)
{
	slink_t		*p;
	plugins_t	*plugin;
	char		here[PATH_MAX];
	
	if (plugins_already_loaded(name))
	{
		haplo_info("Le module `%s' est dj charg.", name);
		return(1);
	}

	/*
	 * First: try current working directory
	 */
	getcwd(here, PATH_MAX);
	plugin=plugins_load(here, name);
	if (plugin)
	{
		plugins_list=__haplo_slink_prepend(plugins_list, plugin);
		return(0);
	}

	/*
	 * Second: try `plugins_path'...
	 * Note: "" is the last path in this list, so LD_LIBRARY_PATH will
	 * be the last try.
	 */
	for(p=plugins_path; p; p=p->next)
	{
		plugin=plugins_load((char *)p->data, name);
		if (plugin)
		{
			plugins_list=__haplo_slink_prepend(plugins_list,
							   plugin);
			return(0);
		}
	}
	haplo_error(_("Failed to load module `%s'."), name);

	return(-1);
}


/**
 *
 */
static void plugins_status_loaded(plugins_t *plugin)
{
	printf("  - %s (%s)\n", plugin->name, plugin->path);
	printf("    | %u types and %u functions\n\n", plugin->types,
	       plugin->functions);
	
	return;
}


/**
 *
 */
static void plugins_status_path(const char *path)
{
	if (path[0] == '\0')
		printf("  - Default system pathes\n");
	else
		printf("  - %s\n", path);
	
	return;
}


/**
 *
 */
void __haplo_plugins_status(void)
{
	haplo_bordered("Plugins status");

	haplo_underlined("loaded");
	__haplo_slink_loop(plugins_list, SLINK_FUNC(plugins_status_loaded));
	printf("\n");

	haplo_underlined("Search pathes");
	__haplo_slink_loop(plugins_path, SLINK_FUNC(plugins_status_path));
	printf("\n");

	return;
}


/**
 * 
 */
static void plugins_unload(plugins_t *plugin)
{
	if (plugin->module->fini)	
		(*plugin->module->fini)();

	if (plugins_library_unload(plugin->handle))
	{
		haplo_error("Dchargement de `%s': %s", plugin->name,
			    dlerror());
	}
	HAPLO_FREE(plugin->path);
	HAPLO_FREE(plugin->name);
	HAPLO_FREE(plugin);
	
	return;
}


/**
 *
 */
void __haplo_plugins_unload_all(void)
{
	__haplo_slink_free_f(plugins_list, SLINK_FUNC(plugins_unload));
	__haplo_slink_free_f(plugins_path, SLINK_FUNC(HAPLO_FREE_FUNC));

	return;
}

#endif /* HAPLO_PLUGINS_IMPL != HAPLO_PLUGINS_IMPL_NONE */
