/*
 * $Id: func.c,v 1.20 2003/12/01 09:50:15 nicoo Exp $
 *
 *
 * Copyright (C) 1999, 2000, 2001 Nicolas LAURENT
 * This file is part of `Haplo'
 * 
 *
 * `Haplo'  is free software;  you can  redistribute  it and/or modify it
 * under the terms of the GNU Library General Public License as published
 * by the Free Software Foundation;  either version 2  of the License, or
 * (at your option) any later version.
 *
 * `Haplo' is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the  implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the  GNU General Public License
 * for more details.
 * 
 * You should have  received  a copy of the   GNU General Public  License
 * along with `Haplo'.  If not, write to  the
 *
 *                                        Free Software Foundation,  Inc.
 *                                        675 Mass Ave, Cambridge, MA
 *                                        02139, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#	include "config.h"
#endif
#include "version.h"

#include <ctype.h>
#include <math.h>
#ifdef HAVE_STDLLIB_H
#	include <stdlib.h>
#endif
#include <stdarg.h>
#include <stdio.h>
#include <string.h>

#include "code.h"
#include "func.h"
#include "help.h"
#include "object.h"
#include "utils.h"
#include "plugins.h"


/*-----------------------------------------------------------------------------
                       G L O B A L   V A R I A B L E S 
-----------------------------------------------------------------------------*/

static func_t *func_list[FUNC_HASH_TABLE_SIZE];


/*-----------------------------------------------------------------------------
                             P R O T O T Y P E S 
-----------------------------------------------------------------------------*/

static int func_hash(const char *key);
static int func_name_cmp(const void *f1, const void *f2);
void __haplo_func_list_display(void);
void __haplo_func_init(void);
static void func_free(func_t *l);
void __haplo_func_fini(void);
static int func_cmp(const func_t *f1, const func_t *f2);
func_t * __haplo_func_get(const char *funcname);
static int func_check(func_t *func);
static func_t * func_register(func_t *new);
static int func_signature(signature_t *signature, const char *sig);
static func_t *func_register_sig(const char *name, const char *symbol,
				 haplo_func_t function, const char *sig);
unsigned int __haplo_func_register(const haplo_func_descr_t *functions,
				   const plugins_t *plugin);
#if HAPLO_PLUGINS_IMPL != HAPLO_PLUGINS_IMPL_NONE
void __haplo_func_bind(const plugins_handle_t *lib, const char *symbol,
		       const char *name, const char *sig);
#endif
#ifdef HAVE_READLINE
static char *func_strdup(const char *s);
char *__haplo_func_completion(const char *beginning, int seq);
#endif


/*-----------------------------------------------------------------------------
                         I M P L E M E N T A T I O N 
-----------------------------------------------------------------------------*/

/**
 * hash-function for function names
 *
 * @param key is the function name
 */
static int func_hash(const char *key)
{
	unsigned long int val=0;

	while(*key != '\0')
	{
		unsigned long int tmp;
		val = (val << 4) + (unsigned long)(*key);
		tmp = val & 0xf0000000;
		if (tmp)
		{
			val ^= tmp >> 24;
			val ^= tmp;
		}
		
		key++;
	}
	
	return((int)(val % FUNC_HASH_TABLE_SIZE));
}


/*
 *
 */
static int func_name_cmp(const void *f1, const void *f2)
{
	return(strcmp((*(func_t **)f1)->name.constant,
		      (*(func_t **)f2)->name.constant));
}


/**
 *
 */
void __haplo_func_loop(func_loop_t l, void *data)
{
	func_t **tab;
	size_t i;
	size_t nbfunc;

	nbfunc=0;
	for(i=0; i<FUNC_HASH_TABLE_SIZE; i++)
	{
		const char *s="";
		func_t *f;

		for(f=func_list[i]; f; f=f->next)
			if (strcmp(f->name.constant, s) != 0)
			{
				s=f->name.constant;
				nbfunc++;
			}
	}
	HAPLO_ALLOC(tab, nbfunc);
	nbfunc=0;
	for(i=0; i<FUNC_HASH_TABLE_SIZE; i++)
	{
		const char *s="";
		func_t *f;

		for(f=func_list[i]; f; f=f->next)
			if (strcmp(f->name.constant, s) != 0)
			{
				s=f->name.constant;
				tab[nbfunc]=f;
				nbfunc++;
			}
	}
	qsort(tab, nbfunc, sizeof(func_t *), func_name_cmp);
	
	for(i=0; i<nbfunc; i++)
		l(tab[i], data);

	HAPLO_FREE(tab);

	return;
}


/**
 * Display functions list
 */
void __haplo_func_list_display(void)
{
	func_t **tab;
	int i;
	int size;
	size_t nbfunc;

	haplo_bordered(_("List of functions"));

	nbfunc=0;
	for(i=0; i<FUNC_HASH_TABLE_SIZE; i++)
	{
		const char *s="";
		func_t *f;

		for(f=func_list[i]; f; f=f->next)
			if (strcmp(f->name.constant, s) != 0)
			{
				s=f->name.constant;
				nbfunc++;
			}
	}
	HAPLO_ALLOC(tab, nbfunc);
	nbfunc=0;
	for(i=0; i<FUNC_HASH_TABLE_SIZE; i++)
	{
		const char *s="";
		func_t *f;

		for(f=func_list[i]; f; f=f->next)
			if (strcmp(f->name.constant, s) != 0)
			{
				s=f->name.constant;
				tab[nbfunc]=f;
				nbfunc++;
			}
	}
	qsort(tab, nbfunc, sizeof(func_t *), func_name_cmp);
	
#define COLS 4
	size = nbfunc/COLS + 1;
	for(i=0; i<size; i++)
	{
		size_t j;
		
		for(j=0; j < COLS; j++)
			if (j*size+i < nbfunc)
				printf("%-18s ", tab[j*size+i]->name.constant);
		putchar('\n');
	}
#undef COLS
	HAPLO_FREE(tab);
	
	return;
}


/**
 * Initialize functions global variables
 */
void __haplo_func_init(void)
{
	int i;

	for(i=0; i<FUNC_HASH_TABLE_SIZE; i++)
		func_list[i]=NULL;

	return;
}


/**
 * Free all function buffers
 *
 * @param l is the first function in hash table row
 */
static void func_free(func_t *l)
{
	if (l)
	{
		func_free(l->next);

		if (l->flags & FUNC_FLAG_DYNAMIC)
			HAPLO_FREE(l->name.dynamic);
		
		HAPLO_FREE(l->args);
		if (l->args_flags)
			HAPLO_FREE(l->args_flags);
		HAPLO_FREE(l);
	}
	return;
}


/**
 * Free all registred functions
 */
void __haplo_func_fini(void)
{
	int	i;

	for(i=0; i<FUNC_HASH_TABLE_SIZE; i++)
		func_free(func_list[i]);

	return;
}


/**
 * Compare to functions
 *
 * @param f1
 * @param f2
 * @return 0 if f1 = f2. non-null value otherwise.
 */
static int func_cmp(const func_t *f1, const func_t *f2)
{
	int status;
	
	status=strcmp(f1->name.constant, f2->name.constant);
	
	if (status == 0)
	{
		
		if (f1->n == f2->n)
		{
			unsigned int j;
			
			for(j=1; j <= f1->n; j++)
			{
				if (f1->args[j] != f2->args[j])
				{
					status=1;
					break;
				}
			}
		} else
			status=1;
	}
	
	return(status);
}


/**
 * get a function by name
 *
 * @param funcname is the function name
 * @return the function
 */
func_t * __haplo_func_get(const char *funcname)
{
	func_t	*l=NULL;

	for(l=func_list[func_hash(funcname)]; l; l=l->next)
	{
		if (strcmp(funcname, l->name.constant) == 0)
			break;
		
	}

	return(l);
}


/**
 * Check if function could be registrable
 *
 * @param func
 */
static int func_check(func_t *func)
{
	unsigned int	i;
	int		error=0;

	if (!func->name.constant || (func->name.constant[0]=='\0'))
	{
		haplo_error(_("Failed to register function with nul name."));
		return(1);
	}
	
	if (!func->func.f1)
	{
		haplo_error(_("Failed to register function `%s': "
			      "pointer-to-function is nil."),
			    func->name);
		error=1;
	}
	
	if (func->n > 5)
	{
		haplo_error(_("Failed to register function `%s': "
			      "too many parameters."),
			    func->name);
		error=1;
	}

	for(i=0; i<func->n; i++)
	{
		if (!func->args[i+1])
		{
			haplo_error(_("Failed to register function `%s': "
				      "argument #%d cannot have void type."),
				    func->name, i+1);
			error=1;
			continue;
		}
		if (!func->args[0] &&
		    (func->args_flags[i] & FUNC_ARG_PROTECT))
		{
			haplo_error(_("Failed to register function `%s': "
				      "cannot protect void result."),
				    func->name);
			error=1;
			continue;
		}
		
		if ((func->args_flags[i] & FUNC_ARG_OUT) &&
		    ! func->args[i+1]->copy)
		{
			haplo_error(_("Failed to register function `%s': "
				      "type `%s' lacks of copy operator."),
				    func->name, func->args[i+1]->name);
			error=1;
		}
	}
	return(error);
}


/**
 * Register a function
 *
 * @param new is the new function to regsiter
 */
static func_t * func_register(func_t *new)
{
	int		index=func_hash(new->name.constant);
	func_t		*func, *pos=NULL;
	
	
	/*
	 * Find a place where insert the new function
	 */
	if (func_check(new))
	{
		haplo_error(_("Registration of function `%s' failed."),
			    new->name);
		HAPLO_FREE(new->args);
		if (new->n)
			HAPLO_FREE(new->args_flags);	
		HAPLO_FREE(new);

		return(NULL);
	}
	
	for(func=func_list[index]; func; func=func->next)
	{
		if (strcmp(func->name.constant, new->name.constant) == 0)
		{
			func_t	*fu;
			for(fu=func; fu; fu=fu->next)
			{
				if (func_cmp(fu, new)==0)
				{
					haplo_warning(
						"Masquage de la fonction "
						"`%s' par le nouvel "
						"enregistrement.",
						new->name.constant);
					break;
				}
			}
			break;
		}
		pos=func;
	}

	/*
	 * Then add entry point!
	 */
	if (pos)
	{
		new->next=pos->next;
		pos->next=new;
	}
	else
	{
		new->next=func_list[index];
		func_list[index]=new;
	}

	new->plugin=NULL;
	
	return(new);
}


/**
 * Computed flags of registration from a signature
 *
 * @param signature contains the computed flags
 * @param sig is the given signature
 */
static int func_signature(signature_t *signature, const char *sig)
{
	size_t len;
	size_t i;
	size_t last=0;

	len=strlen(sig);

	signature->n=0;
	for(i=0; i<5; i++)
	{
		signature->flag[i]=0;
		signature->type[i]=NULL;
	}	
	signature->type[i]=NULL;

	for(i=0; i<len; i++)
	{
		switch(sig[i])
		{
		case ':': signature->flag[signature->n]=FUNC_ARG_IN; break;
		case '!': signature->flag[signature->n]=FUNC_ARG_OUT; break;
		case '/': signature->flag[signature->n]=
				  (FUNC_ARG_IN|FUNC_ARG_PROTECT); break;
		case '*': signature->flag[signature->n]=
				  (FUNC_ARG_OUT|FUNC_ARG_PROTECT); break;
		}
		if (signature->flag[signature->n])
		{
			signature->type[signature->n]=
				__haplo_object_type_nget(sig+last, i-last);
			last=i+1;
			if (++signature->n > 5)
				break;
		}
	}
	signature->type[signature->n]=
		haplo_object_type_get(sig+last);

	return(0);
}


/**
 * Register a function by signature
 *
 * @param name is the haplo name of the function
 * @param symbol is the C symbol of the function
 * @param function is a pointer to function
 * @sig is the signature
 */
static func_t *func_register_sig(const char *name, const char *symbol,
				 haplo_func_t function, const char *sig)
{
	signature_t signature;
	func_t *new;
	unsigned int i;

	
	func_signature(&signature, sig);
	/*
	 * Creation of the new entry point
	 */

	HAPLO_ALLOC(new, 1);
	
	new->name.constant=name;
	new->symbol=symbol;
	new->func.f1=function;
	new->n=signature.n;
	new->flags=0;
	
	HAPLO_ALLOC(new->args, signature.n+1);

	if (signature.n)
		HAPLO_ALLOC(new->args_flags, signature.n);
	else
		new->args_flags=NULL;
		
	new->args[0]=signature.type[0];
	
	for(i=0; i<signature.n; i++)
	{
		new->args[i+1]=signature.type[i+1];
		new->args_flags[i]=signature.flag[i];
	}

	return (func_register(new));
}


/**
 *
 */
unsigned int __haplo_func_register(const haplo_func_descr_t *functions,
			   const plugins_t *plugin)
{
	unsigned int i;
	
	for(i=0; functions[i].name != NULL; i++)
	{
		func_t *func;
		
		func=func_register_sig(functions[i].name,
				       functions[i].symbol,
				       functions[i].func, 
				       functions[i].signature);
		func->plugin=plugin;
	}
	
	return(i);
}


/**
 * Haplo interface to bind function
 *
 * @param lib is the library handler
 * @param symbol is the function symbol
 * @param name is the haplo name
 * @param sig is the signature of function
 */
#if HAPLO_PLUGINS_IMPL != HAPLO_PLUGINS_IMPL_NONE
void __haplo_func_bind(const plugins_handle_t *lib, const char *symbol,
		       const char *name, const char *sig)
{
	void *func;
	
	func=__haplo_plugins_symbol_get(*lib, symbol);

	if (func)
	{
		func_t	*f;
		char	*n=haplo_strdup(name);
		
		f=func_register_sig(n, symbol, func, sig);
		if (f)
			f->flags=FUNC_FLAG_DYNAMIC;
		else
			HAPLO_FREE(n);
	}
       
	else
		haplo_error(("Failed to register function `%s': "
			     "symbol `%s' not found."), name, symbol);

	return;
}
#endif


/**
 * Local implementation of strdup(2)
 *
 * @param s is a null terminated string
 * @return a copy of s in a buffer allocated by malloc()
 */
#ifdef HAVE_READLINE
static char *func_strdup(const char *s)
{
	char	*copy;
	
	copy=malloc(strlen(s)+1);
	strcpy(copy, s);

	return(copy);
}
#endif


/**
 * Handle completion
 *
 * @param beginning is the beginning of completing word
 * @param seq is the choice number
 */
#ifdef HAVE_READLINE
char *__haplo_func_completion(const char *beginning, int seq)
{
	static int	i;
	static func_t	*pos;
	static size_t	len;

	if (seq == 0)
	{
		i=0;
		while(!(pos=func_list[i]))
			i++;
		len=strlen(beginning);
	}

	while(1)
	{
		if (pos->next)
			pos=pos->next;
		else
		{
			i++;
			while(!(pos=func_list[i]))
			{
				i++;
				if (i==FUNC_HASH_TABLE_SIZE)
					return(NULL);
			}
		}
		if (isalpha(pos->name.constant[0]))
		{
			if (len)
			{
				if (strncmp(pos->name.constant,
					    beginning, len)==0)
					break;
			}
			else
				break;
		}
	}
	
	return(func_strdup(pos->name.constant));
}
#endif /* HAVE_READLINE */
