/*
 * $Id: code.h,v 1.12 2003/07/04 17:40:14 nicoo Exp $
 *
 *
 * Copyright (C) 1999, 2000, 2001 Nicolas LAURENT
 * This file is part of `Haplo'
 * 
 *
 * `Haplo'  is free software;  you can  redistribute  it and/or modify it
 * under the terms of the GNU Library General Public License as published
 * by the Free Software Foundation;  either version 2  of the License, or
 * (at your option) any later version.
 *
 * `Haplo' is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the  implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the  GNU General Public License
 * for more details.
 * 
 * You should have  received  a copy of the   GNU General Public  License
 * along with `Haplo'.  If not, write to  the
 *
 *                                        Free Software Foundation,  Inc.
 *                                        675 Mass Ave, Cambridge, MA
 *                                        02139, USA.
 *
 *
 *
 *                 ^         /\
 *                 |        /  \
 *              parents        /\
 *                 |          /\
 *                 V           /\
 *          
 *                  <- brothers ->
 */
 
#ifndef __HAPLO_CODE_H__
#define __HAPLO_CODE_H__

#include "func.h"
#include "object.h"
#include "utils.h"

/*
 * parse_param_t is not yet known.
 * we using struct parse_param_t instead.
 */

/*-----------------------------------------------------------------------------
                            E N U M E R A T I O N S
-----------------------------------------------------------------------------*/

enum leaf_action
{
	LEAF_ACTION_ASSIGN,	/* assign child, arg is a ref */
	LEAF_ACTION_BLOCK,	/* this is a block there's a code */
	LEAF_ACTION_BREAK,
	LEAF_ACTION_CONTINUE,
	LEAF_ACTION_EXECUTE,	/* execute children */
	LEAF_ACTION_EXTRACT,	/* get a member of a list */
	LEAF_ACTION_FREE,	/* free an object */
	LEAF_ACTION_FUNCTION,	/* there's a function to execute */
	LEAF_ACTION_IF,		/* conditionnal branchment */
	LEAF_ACTION_LOAD,	/* load a file */
	LEAF_ACTION_LOCAL_ASSIGN,
	LEAF_ACTION_LOCAL_REF,	/* there's local reference to update */
	LEAF_ACTION_OBJECT,	/* the result is a given object */
	LEAF_ACTION_MULTI,	/* just execute children */
	LEAF_ACTION_NONE,
	LEAF_ACTION_REF,	/* there's a reference to update*/
	LEAF_ACTION_LIST,	/* construct a list */
	LEAF_ACTION_WHILE	/* WHILE statement */
};

enum leaf_status
{
	STATUS_UNDEF=0,		/* not evaluated yet */
	STATUS_OK,		/* computation achieved with no problem */
	STATUS_CONTINUE,	/* `continuer' keyword encounted */
	STATUS_BREAK,		/* `break' keyword encounted */
	STATUS_ERROR,		/*  error during computation */
	STATUS_INTERRUPTED	/* [CTRL][C] */
};


/*-----------------------------------------------------------------------------
                            L E A F _ D E B U G _ T
-----------------------------------------------------------------------------*/

typedef struct leaf_debug_t
{
	char	*buf;
	int	begin;
	int	end;
} leaf_debug_t;


/*-----------------------------------------------------------------------------
                                  L E A F _ T
-----------------------------------------------------------------------------*/

typedef struct leaf_t
{
	object_t		*result;
	struct leaf_t		*brother;
	struct leaf_t		*child;
	unsigned long		n;
	enum leaf_action	action;
	union
	{
		struct code_t			*code;	/* BLOCK */
		reference_t			*ref;  /* REF */
		const func_t			*function; /* FUNCTION */
		struct leaf_t			*execute; /* EXECUTE */
		const struct parse_param_t	*parse_param; /* LOAD */
	} arg;
	leaf_debug_t		*debug;
} leaf_t;


/*-----------------------------------------------------------------------------
                                B R A N C H _ T
-----------------------------------------------------------------------------*/

typedef struct branch_t
{
	leaf_t			*first;
	leaf_t			*last;
	unsigned int		n;
} branch_t;

/*-----------------------------------------------------------------------------
                             P R E C O D E _ T
-----------------------------------------------------------------------------*/

typedef struct precode_t
{
	reference_t	*db[OBJECT_HASH_TABLE_SIZE];	/* local var. */
	unsigned int	n;				/* nb of args. */
	reference_t	**args;				/* ref. of args */
	leaf_t		*first;
	leaf_t		*last;
} precode_t;


/*-----------------------------------------------------------------------------
                                  C O D E _ T
-----------------------------------------------------------------------------*/

typedef struct code_t
{
	unsigned long	n;	/* size of local context       */
	unsigned int	args;	/* numbers of args */
	unsigned int	ops;	/* total of leaf */
	leaf_t		*code;
	unsigned int	flags;
} code_t;

#define	CODE_HAVE_INTERNAL	1
#define CODE_HAVE_EXTERNAL	2
#define CODE_BAD		4


/*-----------------------------------------------------------------------------
                               C O N T E X T _ T
-----------------------------------------------------------------------------*/

typedef struct context_t {
	object_t		**db;
	const code_t		*code;
	const struct context_t	*father;
} context_t;


/*-----------------------------------------------------------------------------
                                 M A C R O S 
-----------------------------------------------------------------------------*/

#define CODE_CONTEXT_POOL_SIZE	64

#define CODE(x)		((code_t *)x)
#define PRECODE(x)	((precode_t *)x)


/*-----------------------------------------------------------------------------
                              P R O T O T Y P E S
-----------------------------------------------------------------------------*/

extern void __haplo_code_init(void);
extern void __haplo_code_fini(void);

/*
 * Constructors and destructor
 */

extern void __haplo_code_leaf_free(leaf_t *leaf);
extern leaf_t *__haplo_code_leaf_from_ref(reference_t *ref);
extern leaf_t *__haplo_code_leaf_from_object(object_t *object);
extern leaf_t *__haplo_code_leaf_from_function(const func_t *f,
					       const branch_t *argl);
extern leaf_t *__haplo_code_leaf_from_execute(leaf_t *code,
					      const branch_t *argl);
extern leaf_t *__haplo_code_leaf_from_block(code_t *code);
extern leaf_t *__haplo_code_leaf_from_unary(const char *function, leaf_t *a);
extern leaf_t *__haplo_code_leaf_from_binary(const char *function, leaf_t *a,
					     leaf_t *b);
extern leaf_t *__haplo_code_leaf_from_assign(leaf_t *result, reference_t *ref);
extern leaf_t *__haplo_code_leaf_from_free(reference_t *ref);
extern leaf_t *__haplo_code_leaf_from_list(leaf_t *first);
extern leaf_t *__haplo_code_leaf_from_extract(leaf_t *list, leaf_t *indice);
extern leaf_t *__haplo_code_leaf_from_load(leaf_t *filename,
					   const struct parse_param_t *param);
extern leaf_t *__haplo_code_leaf_from_if(leaf_t *cond, leaf_t *t, leaf_t *f);
extern leaf_t *__haplo_code_leaf_from_while(leaf_t *cond, leaf_t *body);
extern leaf_t *__haplo_code_leaf_from_for(leaf_t *pre, leaf_t *cond,
					  leaf_t *post, leaf_t *body);
extern void __haplo_code_add_child(leaf_t *father, leaf_t *child);
extern void __haplo_code_add_leaf(precode_t *code, leaf_t *leaf);


/*
 * Execution
 */
extern object_t * __haplo_code_leaf_execute(leaf_t *leaf);


/*
 * Object related stuff.
 */
extern precode_t * __haplo_precode_new(void);
extern void __haplo_precode_free(precode_t *precode);
extern code_t *__haplo_code_from_precode(const precode_t *precode);
extern unsigned int __haplo_code_count_object(const object_t *object,
					      const leaf_t *leaf);
extern void __haplo_code_count_reference(leaf_t *leaf);
extern void __haplo_code_replace_object(const object_t *object, leaf_t *leaf);
extern void __haplo_code_replace_reference(const reference_t *reference,
					   leaf_t *leaf);
extern void __haplo_code_free(code_t *code);
extern void __haplo_code_display(const code_t *code);
extern code_t * __haplo_code_copy(const code_t *code);
extern void __haplo_code_print(const code_t *code);

#endif /* __HAPLO_CODE_H__ */

