/*
 * $Id: code.c,v 1.31 2003/12/01 09:50:15 nicoo Exp $
 *
 *
 * Copyright (C) 1999, 2000, 2001 Nicolas LAURENT
 * This file is part of `Haplo'
 * 
 *
 * `Haplo'  is free software;  you can  redistribute  it and/or modify it
 * under the terms of the GNU Library General Public License as published
 * by the Free Software Foundation;  either version 2  of the License, or
 * (at your option) any later version.
 *
 * `Haplo' is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the  implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the  GNU General Public License
 * for more details.
 * 
 * You should have  received  a copy of the   GNU General Public  License
 * along with `Haplo'.  If not, write to  the
 *
 *                                        Free Software Foundation,  Inc.
 *                                        675 Mass Ave, Cambridge, MA
 *                                        02139, USA.
 *
 */

/*
 * [continued on the next page]
 * UML Model of how code_t is organized. Thanks to Yves Connetable :-)
 * This diagram is not very up-to-date... but this may help!
 *
 *                     +-------------------------------------------+
 *                     | Une r?f?rence peut ?tre associ?e ? un ou  |
 *                     | plusieurs code_t ou leaf_t. La supression |
 *                     | de la derni?re occurence d'association    |
 *                     | entraine la suppression de la r?f?rence   |
 *                     +-------------------------------------------+
 *                             |  |
 *                             |  |
 * +-----------------+         |  |                         +--------+
 * |      ref_t      |+db      |  |                         | code_t |
 * |-----------------|<--------+----------------------------|--------+
 * | name : char *   |* +ref      |                         +--------+
 * | instances : int |<-----------+-------------+       +code^  |
 * +-----------------+  0..1                    |            |  |
 *          |                                   +-+          |  |
 *          |                                     |          |  |
 *          |                                     +-+        |  |
 *          |  +---------------------------------+  |        |  |
 *          |  | Un object peut ?tre partag?     |  +-+      |  |
 *          |  | entre plusieurs ref_t ou leaf_t |    |      |  |
 *          |  | La supression de la derni?re    |    |      |  |
 *          +--| ref_t ou leaf_t pointant sur    |    |      |  |
 *          |  | un object_t entraine la         |    |      |  |
 *          |  | supression de celui-ci          |    |      |  |
 *          |  +---------------------------------+    |      |  |
 *          |                    |                    |      |  |0..1
 *          |                    |                    |      |  |+code
 *          |                    |                    |      |  |
 *          |                    |                    |      |  |
 *          |                    |                    |      |  | +----------+
 *          |                    |                    |      |  | |          |
 *      0..1v+object             |                    |      |  v |          |
 * +------------------+1         |                  +---------------+0..1    |
 * |     object_t     |<---------+------------------|     leaf_t    |<-------+
 * |------------------|+result                      |---------------|+child
 * | references : int |                     +brother| n : int       |+execute
 * +------------------+                    +------->| action : enum |<-------+
 *                                         |    0..1+---+-----------+0..1    |
 *                                         |          | |     |   |          |
 *                                         |          | |     |   |          |
 *                                         +----------+ |     |   +----------+
 *                                                      |     |
 *                    +----------------------------+    |     |
 *                    | Selon la valeur de action, |    | 0..1v+function
 *                    | au plus une seule des      ?    |    +---------------+
 *                    | relations suivantes existe |----+    |     func_t    |
 *                    | ref, code, function ou     |         |---------------|
 *                    | execute                    |         | name : char * |
 *                    +----------------------------+         +---------------+
 *                                       
 */

#include "extensions.h"
#ifdef HAVE_CONFIG_H
#	include "config.h"
#endif
#include "version.h"

#include <stdio.h>
#ifdef HAVE_STDLIB_H
#	include <stdlib.h>
#endif
#ifndef WIN32
#	define HAPLO_INTERRUPTIBLE
#endif
#include <signal.h>
#ifdef SIGINT
#	include <setjmp.h>
#endif
#include <string.h>

#include "builtin.h"
#include "code.h"
#include "func.h"
#include "object.h"
#include "parser.h"
#include "pool.h"
#include "utils.h"


#define CODE_OPTIMIZE
/*#define HAPLO_DEBUG_CODE*/


/*-----------------------------------------------------------------------------
                       G L O B A L   V A R I A B L E S 
-----------------------------------------------------------------------------*/

static pool_t 			*code_context_pool=NULL;
#ifdef SIGINT
static volatile sig_atomic_t	code_interrupted=0; /* set when interrupted */
static volatile sig_atomic_t	code_function=0; /* set when in function */
static jmp_buf			code_jmp;
#endif /* SIGINT */


/*-----------------------------------------------------------------------------
                             P R O T O T Y P E S 
-----------------------------------------------------------------------------*/

void __haplo_code_init(void);
void __haplo_code_fini(void);

/*
 * Interruptions stuff. [optional]
 */
#ifdef SIGINT
static void code_interruption_handler(int sig);
#endif /* SIGINT */

/*
 * Accessing result methods
 */
static void code_result_free(leaf_t *leaf);
static enum leaf_status code_result_set(leaf_t *leaf, object_t *result);

/*
 * Constructors and destructor
 */
void __haplo_code_leaf_free(leaf_t *leaf);
static leaf_t *code_leaf_copy(const code_t *old, code_t *new, leaf_t *leaf);
leaf_t *__haplo_code_leaf_from_assign(leaf_t *result, reference_t *ref);
leaf_t *__haplo_code_leaf_from_block(code_t *code);
leaf_t *__haplo_code_leaf_from_execute(leaf_t *code, const branch_t *argl);
leaf_t *__haplo_code_leaf_from_extract(leaf_t *list, leaf_t *indice);
leaf_t *__haplo_code_leaf_from_free(reference_t *ref);
leaf_t *__haplo_code_leaf_from_function(const func_t *f, const branch_t *argl);
leaf_t *__haplo_code_leaf_from_unary(const char *function, leaf_t *a);
leaf_t *__haplo_code_leaf_from_binary(const char *function, leaf_t *a,
				      leaf_t *b);
leaf_t *__haplo_code_leaf_from_if(leaf_t *cond, leaf_t *t, leaf_t *f);
leaf_t *__haplo_code_leaf_from_load(leaf_t *filename,
				    const struct parse_param_t *param);
leaf_t *__haplo_code_leaf_from_object(object_t *object);
leaf_t *__haplo_code_leaf_from_ref(reference_t *ref);
leaf_t *__haplo_code_leaf_from_list(leaf_t *first);
leaf_t *__haplo_code_leaf_from_while(leaf_t *cond, leaf_t *body);
leaf_t *__haplo_code_leaf_from_for(leaf_t *pre, leaf_t *cond, leaf_t *post, leaf_t *body);
void __haplo_code_add_child(leaf_t *father, leaf_t *child);
void __haplo_code_add_leaf(precode_t *code, leaf_t *leaf);

/*
 * Execution
 */
static enum leaf_status code_leaf_execute_assign(leaf_t *leaf);
static enum leaf_status code_leaf_execute_block(const context_t *context,
						leaf_t *leaf);
static unsigned int code_nb_args(leaf_t *values);
static void code_local_remove(const context_t *context, object_t *old,
			      object_t *new, leaf_t *leaf);
static context_t *code_context_new(const context_t *father, code_t *code,
				   leaf_t *values);
static void code_context_free(context_t *context, code_t *code);
static object_t *code_execute(const context_t *context, code_t *code,
			      leaf_t *values);
static enum leaf_status code_leaf_execute_execute(const context_t *context,
						  leaf_t *leaf);
static enum leaf_status code_leaf_execute_extract(const context_t *context,
						  leaf_t *leaf);
static int code_func_args_compare(const func_t *func, const leaf_t *first);
static void code_func_args_check(const context_t *context, const func_t *func,
				 leaf_t *leaf);
static void code_func_args_protect(object_t *object, const func_t *func,
				   const leaf_t *leaf);
static const func_t *code_func_choose(const leaf_t *leaf);
static void code_leaf_execute_func_do(void **result, union f f, leaf_t *leaf);
static enum leaf_status code_leaf_execute_func(const context_t *context,
					       leaf_t *leaf);
static enum leaf_status code_leaf_execute_if(const context_t *context,
					     leaf_t *leaf);
static enum leaf_status code_leaf_execute_load(leaf_t *leaf);
static enum leaf_status code_leaf_execute_local_assign(
	const context_t *context, leaf_t *leaf);
static enum leaf_status code_leaf_execute_local_ref(const context_t *context,
						    leaf_t *leaf);
static void code_context_update(const context_t *context, leaf_t *leaf);
static enum leaf_status code_leaf_execute_ref(leaf_t *leaf);
static enum leaf_status code_leaf_execute_list(const context_t *context,
					       leaf_t *leaf);
static enum leaf_status code_leaf_execute_while(const context_t *context,
						leaf_t *leaf);
static enum leaf_status code_leaf_execute_action(const context_t *context,
						 leaf_t *leaf);
static enum leaf_status code_leaf_execute_children(const context_t *context,
						   leaf_t *leaf);
static enum leaf_status code_leaf_execute(const context_t *context,
					  leaf_t *leaf);
object_t *__haplo_code_leaf_execute(leaf_t *leaf);

/*
 * Object related stuff.
 */
precode_t *__haplo_precode_new(void);
void __haplo_precode_free(precode_t *precode);
static void code_from_precode_replace_ref(code_t *code,
					  reference_t *const *trans,
					  leaf_t *l);
static void code_from_precode_replace_assign(code_t *code,
					     reference_t *const *trans,
					     leaf_t *l);
static void code_from_precode_replace(code_t *code,
				      reference_t *const *trans, leaf_t *l);
static unsigned int code_from_precode_pos(const precode_t *precode,
					  const reference_t *ref,
					  unsigned int *proposed);
static void code_stat(code_t *code, leaf_t *leaf);
code_t *__haplo_code_from_precode(const precode_t *precode);
unsigned int __haplo_code_count_object(const object_t *object,
				       const leaf_t *leaf);
void __haplo_code_count_reference(leaf_t *leaf);
void __haplo_code_replace_object(const object_t *object, leaf_t *leaf);
void __haplo_code_replace_reference(const reference_t *reference,
				    leaf_t *leaf);
void __haplo_code_free(code_t *code);
void __haplo_code_display(const code_t *code);
code_t *__haplo_code_copy(const code_t *code);
void __haplo_code_print(const code_t *code);
static void code_leaf_display(const code_t *code, int level,
			      const leaf_t *leaf);

/*
 * debug stuff 
 */

#ifdef HAPLO_DEBUG_CODE
static char *code_debug_offset(int offset);
static const char *code_debug_action(enum leaf_action action);
static void code_debug_leaf(leaf_t *leaf, const char *domain, int offset);
#endif /* HAPLO_DEBUG_CODE */


/*-----------------------------------------------------------------------------
                         I M P L E M E N T A T I O N 
-----------------------------------------------------------------------------*/


/**
 *
 */
void __haplo_code_init(void)
{
#ifdef SIGINT
	struct sigaction handler;
	sigset_t mask;

	code_interrupted=0;
	code_function=0;

	sigemptyset(&mask);
	handler.sa_handler=(void (*)(int))code_interruption_handler;
	handler.sa_mask=mask;
	handler.sa_flags=SA_RESTART;
	sigaction(SIGINT, &handler, NULL);
#endif /* SIGINT */

	code_context_pool=__haplo_pool_new(CODE_CONTEXT_POOL_SIZE,
					   sizeof(context_t));

	return;
}


/**
 *
 */
void __haplo_code_fini(void)
{
	__haplo_pool_free(code_context_pool);
	
	return;
}


/*
 * Interruptions stuff.
 */

/**
 * Handler called when [Ctr][C] is pressed.
 *
 * @arg sig is the signal number. Normally it should be SIGINT.
 */
#ifdef SIGINT
static void code_interruption_handler(int sig)
{
#ifdef HAPLO_DEBUG_SIGNAL
	haplo_debug("INT: sig=%d interrupted=%d function=%d",
		    sig,
		    code_interrupted,
		    code_function);
#endif /* HAPLO_DEBUG_SIGNAL */

	if (!code_interrupted)
	{
		/* do not stack interruptions.... */
		code_interrupted=1;
		if (code_function)
		{
			code_function=0;
			longjmp(code_jmp, sig);
		}
	}
	
	return;
}
#endif


/*
 * Accessing result methods
 */

/**
 * Clear the result of a leaf
 * 
 * @param leaf
 */
static  void code_result_free(leaf_t *leaf)
{
	__haplo_object_free(leaf->result);
	leaf->result=NULL;

	return;
}


/**
 * Set a result to a leaf
 *
 * @param leaf
 * @param result
 * 
 * @return the status of action
 */
static  enum leaf_status code_result_set(leaf_t *leaf, object_t *result)
{
	enum leaf_status status=STATUS_ERROR;
	
	if (result)
	{
		status=STATUS_OK;
		if (leaf->result != result)
		{
			code_result_free(leaf);
			leaf->result=result;
			OBJECT_REF(result);
#ifdef HAPLO_DEBUG_CODE
			if (result->type == OBJECT_CODE)
			{
				haplo_debug("protect %p %s %u->%u",
					    result,
					    result->type->name,
					    result->references,
					    result->references+1);
			}
#endif /* HAPLO_DEBUG_CODE */			
		}	
	}

	return(status);
}


/*
 * Constructors and destructor
 */

/**
 * free tree of leaves
 *
 * @param leaf is the first leaf
 */
void __haplo_code_leaf_free(leaf_t *leaf)
{
#ifdef HAPLO_DEBUG_CODE
	static int	tab=0;

	if (tab == 0)
	{
		haplo_debug("--- FREE BEGIN ---");
	}
#endif /* HAPLO_DEBUG_CODE */
	
	if (leaf)
	{
#ifdef HAPLO_DEBUG_CODE
		code_debug_leaf(leaf, "FREE", tab++);
#endif /* HAPLO_DEBUG_CODE */



		switch(leaf->action)
		{
		case LEAF_ACTION_ASSIGN:
			__haplo_object_ref_free(leaf->arg.ref);
			break;

		case LEAF_ACTION_BLOCK:
			__haplo_code_free(leaf->arg.code);
			break;
			
		case LEAF_ACTION_BREAK:
		case LEAF_ACTION_CONTINUE:
			break;
			
		case LEAF_ACTION_EXECUTE:
			__haplo_code_leaf_free(leaf->arg.execute);
			break;

		case LEAF_ACTION_EXTRACT:
			break;

		case LEAF_ACTION_FREE:
			__haplo_object_ref_free(leaf->arg.ref);
			break;

		case LEAF_ACTION_FUNCTION:
		case LEAF_ACTION_IF:
		case LEAF_ACTION_LOAD:
		case LEAF_ACTION_LOCAL_ASSIGN:
		case LEAF_ACTION_LOCAL_REF:
		case LEAF_ACTION_MULTI:
		case LEAF_ACTION_NONE:
		case LEAF_ACTION_OBJECT:
			break;

		case LEAF_ACTION_REF:
			__haplo_object_ref_free(leaf->arg.ref);
			break;

		case LEAF_ACTION_LIST:
		case LEAF_ACTION_WHILE:
			break;
		}
		code_result_free(leaf);
		__haplo_code_leaf_free(leaf->child);
		__haplo_code_leaf_free(leaf->brother);

		HAPLO_FREE(leaf);
#ifdef HAPLO_DEBUG_CODE
		if (--tab == 0)
		{
			haplo_debug("--- FREE END ---");
		}
#endif /* HAPLO_DEBUG_CODE */
	}

	return;
}


/**
 * Copy a tree of leaves
 *
 * @param old
 * @param new
 * @param leaf is the first leaf
 * 
 * @return the copy of the first leaf of the new tree
 */
static leaf_t *code_leaf_copy(const code_t *old, code_t *new, leaf_t *leaf)
{
	leaf_t	*copy=NULL;

	if (new && leaf)
	{
		HAPLO_ALLOC(copy, 1);
		
		copy->action=leaf->action;
		copy->result=leaf->result;
		OBJECT_REF(copy->result);
		copy->n=leaf->n;

		copy->brother=code_leaf_copy(old, new, leaf->brother);
		copy->child  =code_leaf_copy(old, new, leaf->child);

	
		switch(copy->action)
		{
		case LEAF_ACTION_ASSIGN:
			copy->arg.ref=leaf->arg.ref;
			copy->arg.ref->instances += 1;
			break;
			
		case LEAF_ACTION_BLOCK:
			copy->arg.code=__haplo_code_copy(leaf->arg.code);
			break;
			
		case LEAF_ACTION_BREAK:
		case LEAF_ACTION_CONTINUE:
			break;
			
		case LEAF_ACTION_EXECUTE:
			copy->arg.execute=code_leaf_copy(old, new,
							 leaf->arg.execute);
			break;

		case LEAF_ACTION_EXTRACT:
			break;

		case LEAF_ACTION_FREE:
			copy->arg.ref=leaf->arg.ref;
			copy->arg.ref->instances += 1;
			break;
			
		case LEAF_ACTION_FUNCTION:
			copy->arg.function=leaf->arg.function;
			break;
			
		case LEAF_ACTION_IF:
			break;
			
		case LEAF_ACTION_LOAD:
			copy->arg.parse_param=leaf->arg.parse_param;
			break;
			
		case LEAF_ACTION_LOCAL_ASSIGN:
			if (leaf->arg.code == old)
				copy->arg.code=new; /* change context */
			else
			{
				haplo_error(_("This code makes unallowed "
					      "reference to another one. %p"),
					      copy->result);
				
				new->flags=CODE_BAD;
				OBJECT_UNREF(copy->result);
				copy->action=LEAF_ACTION_NONE;
				copy->result=NULL;
				copy->arg.code=new;
			}
			break;

		case LEAF_ACTION_LOCAL_REF:
			if (leaf->arg.code == old)
				copy->arg.code=new; /* change context */
			else
				copy->arg.code=leaf->arg.code;
			break;

		case LEAF_ACTION_MULTI:
		case LEAF_ACTION_NONE:
			break;
			
		case LEAF_ACTION_OBJECT:
			break;			
			
		case LEAF_ACTION_REF:
			copy->arg.ref=leaf->arg.ref;
			copy->arg.ref->instances += 1;
			break;
			
		case LEAF_ACTION_LIST:
			__FIXME__
			break;

		case LEAF_ACTION_WHILE:
			break;			
		}
	}
	
	return(copy);
}


/**
 * Create a new leaf
 * 
 * @param result
 * @param ref
 *
 * @return a lewly allocated leaf
 */
leaf_t *__haplo_code_leaf_from_assign(leaf_t *result, reference_t *ref)
{
	leaf_t	*leaf;
	
	HAPLO_ALLOC(leaf, 1);
	leaf->result=NULL;
	leaf->brother=NULL;
	leaf->child=result;
	leaf->n=1;
	leaf->action=LEAF_ACTION_ASSIGN;
	leaf->arg.ref=ref;

	return(leaf);
}


/**
 * Create a new leaf
 * 
 * @param code
 *
 * @return a lewly allocated leaf
 */
leaf_t * __haplo_code_leaf_from_block(code_t *code)
{
	leaf_t	*leaf;
	
	HAPLO_ALLOC(leaf, 1);
	leaf->result=NULL;
	leaf->brother=NULL;
	leaf->child=NULL;
	leaf->n=0;
	leaf->action=LEAF_ACTION_BLOCK;
  	leaf->arg.code=code;
	
	return(leaf);
}


/**
 * Create a new leaf
 * 
 * @param code
 * @param argl
 *
 * @return a lewly allocated leaf
 */
leaf_t * __haplo_code_leaf_from_execute(leaf_t *code, const branch_t *argl)
{
	leaf_t	*leaf;
	
	HAPLO_ALLOC(leaf, 1);
	leaf->result=NULL;
	leaf->brother=NULL;
	leaf->child=argl->first;
	leaf->n=argl->n;
	leaf->action=LEAF_ACTION_EXECUTE;
	leaf->arg.execute=code;
	
	return(leaf);
}


/**
 * Create a new leaf
 * 
 * @param list
 * @param indice
 *
 * @return a lewly allocated leaf
 */
leaf_t *__haplo_code_leaf_from_extract(leaf_t *list, leaf_t *indice)
{
	leaf_t	*leaf;
	
	HAPLO_ALLOC(leaf, 1);
	leaf->result=NULL;
	leaf->brother=NULL;
	leaf->child=list;
	list->brother=indice;
	leaf->n=2;
	leaf->action=LEAF_ACTION_EXTRACT;
	
	return(leaf);
}


/**
 * Create a new leaf
 * 
 * @param ref
 *
 * @return a lewly allocated leaf
 */
leaf_t *__haplo_code_leaf_from_free(reference_t *ref)
{
	leaf_t	*leaf;
	
	HAPLO_ALLOC(leaf, 1);
	leaf->result=NULL;
	leaf->brother=NULL;
	leaf->child=NULL;
	leaf->n=0;
	leaf->action=LEAF_ACTION_FREE;
	leaf->arg.ref=ref;
	
	ref->instances++;
	return(leaf);
}


/**
 * Create a new leaf
 * 
 * @param f
 * @param argl
 *
 * @return a lewly allocated leaf
 */
leaf_t *__haplo_code_leaf_from_function(const func_t *f, const branch_t *argl)
{
	leaf_t	*leaf;
	
	HAPLO_ALLOC(leaf, 1);
	leaf->result=NULL;
	leaf->brother=NULL;
	leaf->child=argl->first;
	leaf->n=argl->n;
	leaf->action=LEAF_ACTION_FUNCTION;
	leaf->arg.function=f;
	
	return(leaf);
}


/**
 * Create a new leaf
 * 
 * @param function
 * @param a
 *
 * @return a lewly allocated leaf
 */
leaf_t *__haplo_code_leaf_from_unary(const char *function, leaf_t *a)
{
	leaf_t	*leaf;
	
	HAPLO_ALLOC(leaf, 1);
	leaf->result=NULL;
	leaf->brother=NULL;
	leaf->child=a;
	leaf->n=1;
	leaf->action=LEAF_ACTION_FUNCTION;
	leaf->arg.function=__haplo_func_get(function);
	
	return(leaf);
}


/**
 * Create a new leaf
 * 
 * @param function
 * @param a
 * @param b
 *
 * @return a lewly allocated leaf
 */
leaf_t *__haplo_code_leaf_from_binary(const char *function, leaf_t *a,
				      leaf_t *b)
{
	leaf_t	*leaf;
	
	HAPLO_ALLOC(leaf, 1);
	leaf->result=NULL;
	leaf->brother=NULL;
	leaf->child=a;
	a->brother=b;
	leaf->n=2;
	leaf->action=LEAF_ACTION_FUNCTION;
	leaf->arg.function=__haplo_func_get(function);
	
	return(leaf);
}


/**
 * Create a new leaf
 * 
 * @param cond is the test
 * @param t is the "true" branch
 * @param f is the "false" branch
 *
 * @return a lewly allocated leaf
 */
leaf_t *__haplo_code_leaf_from_if(leaf_t *cond, leaf_t *t, leaf_t *f)
{
	leaf_t	*leaf;
	
	HAPLO_ALLOC(leaf, 1);
	leaf->result=NULL;
	leaf->brother=NULL;
	leaf->child=cond;
	leaf->child->brother=t;
	if (t)
	{
		leaf->child->brother->brother=f;
		if (f)
			leaf->n=3;
		else
			leaf->n=2;
		leaf->action=LEAF_ACTION_IF;
	}
	
	return(leaf);

}


/**
 * Create a new leaf
 * 
 * @param filename
 * @param param
 *
 * @return a lewly allocated leaf
 */
leaf_t *__haplo_code_leaf_from_load(leaf_t *filename,
				    const struct parse_param_t *param)
{
	leaf_t	*leaf;
	
	HAPLO_ALLOC(leaf, 1);
	leaf->result=NULL;
	leaf->brother=NULL;
	leaf->child=filename;
	leaf->n=1;
	leaf->arg.parse_param=param;
	leaf->action=LEAF_ACTION_LOAD;
	
	return(leaf);
}


/**
 * Create a new leaf
 * 
 * @param object
 *
 * @return a lewly allocated leaf
 */
leaf_t *__haplo_code_leaf_from_object(object_t *object)
{
	leaf_t	*leaf;

	HAPLO_ALLOC(leaf, 1);

	leaf->result=NULL;
	leaf->brother=NULL;
	leaf->child=NULL;
	leaf->n=0;
	leaf->action=LEAF_ACTION_OBJECT;

	(void)code_result_set(leaf, object);
	

	return(leaf);	
}


/**
 * Create a new leaf
 * 
 * @param ref
 *
 * @return a lewly allocated leaf
 */
leaf_t *__haplo_code_leaf_from_ref(reference_t *ref)
{
	leaf_t	*leaf;

	HAPLO_ALLOC(leaf, 1);

	leaf->result=NULL;
	leaf->brother=NULL;
	leaf->child=NULL;
	leaf->n=0;

	leaf->action=LEAF_ACTION_REF;
	leaf->arg.ref=ref;

	return(leaf);
}


/**
 * Create a new leaf
 * 
 * @param first
 *
 * @return a lewly allocated leaf
 */
leaf_t *__haplo_code_leaf_from_list(leaf_t *first)
{
	leaf_t	*leaf;
	
	HAPLO_ALLOC(leaf, 1);
	leaf->result=NULL;
	leaf->brother=NULL;
	leaf->child=first;
	leaf->n=1;
	leaf->action=LEAF_ACTION_LIST;
	
	return(leaf);
}


/**
 * Create a new leaf
 * 
 * @param cond
 * @param body
 *
 * @return a lewly allocated leaf
 */
leaf_t *__haplo_code_leaf_from_while(leaf_t *cond, leaf_t *body)
{
	leaf_t	*leaf;
	
	HAPLO_ALLOC(leaf, 1);
	leaf->result=NULL;
	leaf->brother=NULL;
	leaf->child=cond;
	leaf->child->brother=body;
	leaf->n=2;
	leaf->action=LEAF_ACTION_WHILE;
	
	return(leaf);
}


/**
 * Create a new leaf
 * 
 * @param pre
 * @param cond
 * @param post
 * @param body
 *
 * @return a lewly allocated leaf
 */
leaf_t *__haplo_code_leaf_from_for(leaf_t *pre, leaf_t *cond, leaf_t *post,
				   leaf_t *body)
{
	leaf_t	*leaf;
	leaf_t	*real_body;
	
	if (body)
	{
		HAPLO_ALLOC(real_body, 1);
		real_body->result=NULL;
		real_body->brother=NULL;
		real_body->child=body;
		real_body->child->brother=post;
		real_body->n=2;
		real_body->action=LEAF_ACTION_MULTI;
	} else
		real_body=post;
	
	HAPLO_ALLOC(leaf, 1);
	leaf->result=NULL;
	leaf->brother=NULL;
	leaf->child=pre;
	leaf->child->brother=__haplo_code_leaf_from_while(cond, real_body);
	leaf->n=2;
	leaf->action=LEAF_ACTION_MULTI;	

	return(leaf);
}


/**
 * add a child to a leaf
 * 
 * @param father
 * @param child
 */
void __haplo_code_add_child(leaf_t *father, leaf_t *child)
{
	father->n += 1;
	
	if (father->child)
	{
		leaf_t	*i;
		
		for(i=father->child; i->brother; i=i->brother)
			/*nothing*/ ;
	
		i->brother=child;
	}
	else
		father->child=child;
	
	return;
}


/**
 * Add a leaf to a precode
 *
 * @param code
 * @param leaf
 */
void __haplo_code_add_leaf(precode_t *code, leaf_t *leaf)
{
	if (code->first)
	{
		code->last->brother=leaf;
		code->last=leaf;
	}
	else
	{
		code->first=leaf;
		code->last=leaf;
	}
	
	return;
}


/*
 * Execution
 */

/**
 * Execute leaf
 * 
 * @param leaf
 *
 * @return the status of action
 */
static  enum leaf_status code_leaf_execute_assign(leaf_t *leaf)
{
	enum leaf_status	status;

	status=code_result_set(leaf, leaf->child->result);
	if (status == STATUS_OK)
	{
		__haplo_object_register(leaf->child->result,
					leaf->arg.ref);
	}	
	else
	{
		haplo_error(_("Can't assign nul object to `%s'"),
			    leaf->arg.ref->name);
	}		
	return(status);
}


/**
 * Execute leaf
 * 
 * @param context
 * @param leaf
 *
 * @return the status of action
 */
static  enum leaf_status code_leaf_execute_block(const context_t *context,
						 leaf_t *leaf)
{
	enum leaf_status status;
	
	status=code_leaf_execute_children(context, leaf);
	if (status == STATUS_OK)
		code_result_set(leaf, code_execute(context,
						   leaf->arg.code,
						   NULL));
	return(status);
}


/**
 * Count number of brothers :-)
 * 
 * @param values
 *
 * @return the number of brothers
 */
static unsigned int code_nb_args(leaf_t *values)
{
	unsigned int	nb;
	
	for(nb=0; values; values=values->brother)
		nb++;

	return(nb);
}


/**
 * ?? :-))
 *
 * @param context
 * @param old
 * @param new
 * @param leaf
 *
 * @see code_copy()
 */
static void code_local_remove(const context_t *context, object_t *old, 
			      object_t *new, leaf_t *leaf)
{
	if (leaf)
	{
		code_local_remove(context, old, new, leaf->child);
		code_local_remove(context, old, new, leaf->brother);
		if ((leaf->action == LEAF_ACTION_OBJECT) &&
		    (leaf->result->type==OBJECT_CODE))
		{

			code_local_remove(
				context, old, new,
				CODE(leaf->result->content)->code);
		}
		if (leaf->action == LEAF_ACTION_BLOCK)
		{
			code_local_remove(context, old, new,
					  leaf->arg.code->code);
		}
		if (leaf->action == LEAF_ACTION_EXECUTE)
		{
			code_local_remove(context, old, new,
					  leaf->arg.execute);
		}
		
		if ((leaf->action == LEAF_ACTION_LOCAL_REF) &&
		    (leaf->arg.code==context->code))
		{
			leaf->action=LEAF_ACTION_OBJECT;
			if (leaf->result == old)
			{
				leaf->action=LEAF_ACTION_NONE;
				(void)code_result_set(leaf, new);
			}
			else
			{
				(void)code_result_set(
					leaf,
					__haplo_object_copy(leaf->result));
			}
			
		}
	}
	return;
	
}


/**
 * Create a new context
 *
 * @paran father is the previous context
 * @param code is the current frame 
 * @param values is the list of arguments
 *
 * @return a new context
 */
static  context_t *code_context_new(const context_t *father,
				    code_t *code, leaf_t *values)
{
	leaf_t		*stmt;
	unsigned long	i;
	context_t	*context;

	context=__haplo_pool_get(code_context_pool);
	context->father=father;
	context->code=code;

	if (code->n)
	{
		HAPLO_ALLOC(context->db, code->n);
		context->father=father;
#ifdef HAPLO_DEBUG_CODE
		haplo_debug("CONTEXT: initialize");
#endif /* HAPLO_DEBUG_CODE */
		for(i=0, stmt=values; stmt; i++, stmt=stmt->brother)
		{
#ifdef HAPLO_DEBUG_CODE
			haplo_debug(
				"CONTEXT: #%ld initialized with <%s> "
				"(object=%p)",
				i,
				stmt->result->type->name,
				stmt->result);
#endif /* HAPLO_DEBUG_CODE */
		
			context->db[i]=stmt->result;
			OBJECT_REF(context->db[i]);
		}
		for(; i<code->n; i++)
		{
#ifdef HAPLO_DEBUG_CODE
			haplo_debug("CONTEXT: #%ld initialized", i);
#endif /* HAPLO_DEBUG_CODE */
			context->db[i]=NULL;
		}
	}
	else
		context->db=NULL;

#ifdef HAPLO_DEBUG_CODE
	haplo_debug("CONTEXT: context=%p size=%lu code=%p father=%p",
		    context, code->n, context->code, context->father);
#endif /* HAPLO_DEBUG_CODE*/

	return(context);
}


/**
 * pop a context
 *
 * @param context is the current context
 * @param code is the frame
 */
static  void code_context_free(context_t *context, code_t *code)
{
	if (code->n)
	{
		unsigned long i;

		for(i=0; i<code->n; i++)
			__haplo_object_free(context->db[i]);

		HAPLO_FREE(context->db);
	}
	__haplo_pool_release(code_context_pool, context);
	
	return;
}


/**
 * Execute a code
 *
 * @param context 
 * @param code
 * @param values
 *
 * @return the result
 */
static  object_t *code_execute(const context_t *context,
			       code_t *code, leaf_t *values)
{
	leaf_t		*stmt;
	object_t	*result=NULL;
	context_t	*new_context;
	
	/*
	 * Check number of arguments
	 */
	if (code->args != code_nb_args(values))
	{
		haplo_error(_("Execution of this code needs %u parameter%s"),
			    code->args, (code->args>1)?"s":"");
		return(NULL);
	}

	new_context=code_context_new(context, code, values);
	/* for(c=new_context; c; c=c->father)*/
	code_context_update(new_context, code->code);
	
	for(stmt=code->code; stmt; stmt=stmt->brother)
	{
		enum leaf_status	status;

		status=code_leaf_execute(new_context, stmt);
		if (status != STATUS_OK)
			break;
		else
			result=stmt->result;
	}
	
	if (result &&
	    (result->type == OBJECT_CODE)
#ifdef CODE_OPTIMIZE
	    && (CODE(result->content)->flags & CODE_HAVE_EXTERNAL)
#endif /* CODE_OPTIMIZE */
		)
	{
		object_t	*copy;
		
		copy=__haplo_object_copy(result);
		if (copy)
		{
			code_local_remove(new_context, result, copy,
					  CODE(copy->content)->code);
			CODE(copy->content)->flags &=
				~CODE_HAVE_EXTERNAL;
		}
		result=copy;	
	}

	code_context_free(new_context, code);
	/*
	 * Handling right-recursion (recursive_func_2 & recursive_func_3)
	 */
	if (context)
		code_context_update(context, context->code->code);
	
	return(result);
}


/**
 * Execute a leaf
 *
 * @param context
 * @param leaf
 *
 * @return the status of action
 */
static  enum leaf_status code_leaf_execute_execute(const context_t *context,
						   leaf_t *leaf)
{
	enum leaf_status	status;

	status=code_leaf_execute_children(context, leaf);
	if (status != STATUS_OK)
		return(status);

	status=code_leaf_execute(context, leaf->arg.execute);
	if (status != STATUS_OK)
		return(status);

	if (leaf->arg.execute->result->type == OBJECT_CODE)
	{
#define CODE_TO_EXECUTE	CODE(leaf->arg.execute->result->content)
		status=code_result_set(
			leaf,
			code_execute(context, 
				     CODE_TO_EXECUTE, leaf->child));
#undef CODE_TO_EXECUTE
	}
	else
	{
		haplo_error(_("Try to execute object of type <%s> "
			      "instead of <code>"),
			    (leaf->arg.execute->result)?
			    leaf->arg.execute->result->type->name:"nil");
		status=STATUS_ERROR;
	}

	return(status);		
}


/**
 * Execute an extraction
 * 
 * @param context
 * @param leaf
 *
 * @return the status of action
 */
static  enum leaf_status code_leaf_execute_extract(const context_t *context,
						   leaf_t *leaf)
{
	enum leaf_status status;
	
	status=code_leaf_execute_children(context, leaf);
	
	if (status != STATUS_OK)
		return(status);

	if (leaf->child->result->type == OBJECT_LIST)
	{
		if (leaf->child->brother->result->type == OBJECT_FLOAT)
		{

			status=code_result_set(
				leaf,
				__haplo_builtin_list_extract(
					leaf->child->result->content,
					leaf->child->brother->result->content)
				);
		}
		else
		{
			haplo_error(
				"Utilisation d'un objet de type `%s' comme "
				"indice de vecteur",
				leaf->child->brother->result->type->name);
			status=STATUS_ERROR;
		}
	}
	else
	{	
		haplo_error("Application de l'op?rateur `->' sur un objet "
			    "de type `%s'", leaf->child->result->type->name);
		status=STATUS_ERROR;
	}
	
	return(status);
}


/**
 * Compare argument types with signatures
 * 
 * @param func
 * @param first
 *
 * @return comparaison
 */
static  int code_func_args_compare(const func_t *func,
				   const leaf_t *first)
{
	const leaf_t	*l;
	unsigned int	i;
	
	for(i=1, l=first; i <= func->n; i++, l=l->brother)
	{
		if (!l->result || (func->args[i] != l->result->type))
			return(-1);
	}
	
	return(0);
}


/**
 * Check arguments passed to a function
 *
 * @param context
 * @param func
 * @param leaf
 * 
 * @return <0 if an object is not defined, 0 if ok, >0 if an object is not
 * registred.
 */
static  void code_func_args_check(const context_t *context, 
				  const func_t *func, leaf_t *leaf)
{
	unsigned int	i=0;
	
	while(leaf)
	{
		if ((func->args_flags[i] & FUNC_ARG_OUT))
		{
			switch(leaf->action)
			{
			case LEAF_ACTION_REF:
			case LEAF_ACTION_ASSIGN:
				if (leaf->arg.ref->object->names > 1)
				{
					/*
					 * Avoid unneeded copy
					 */
				
					leaf->arg.ref->object=
						__haplo_object_copy(
							leaf->result);
					/*
					 * keep back one entry in the db...
					 */
					OBJECT_UNREF(leaf->result);
					(void)code_result_set(
						leaf,
						leaf->arg.ref->object);
					/*
					 * To keep a place in the db we must
					 * register a second time. Currently
					 * we are registred only one time
					 * (for the leaf * structure...)
					 */
					OBJECT_REF(leaf->result);
				}
				break;
			case LEAF_ACTION_LOCAL_REF:
			case LEAF_ACTION_LOCAL_ASSIGN:
				if (leaf->result->names > 1)
				{
					object_t *copy;
					
					copy=__haplo_object_copy(leaf->result);
					code_result_set(leaf, copy);

					OBJECT_UNREF(context->db[leaf->n]);
					context->db[leaf->n]=copy;
					OBJECT_REF(copy);
				}
				break;
			default:
#ifdef DEBUG_CODE
				haplo_debug(
					_("Function `%s' expected "
					  "named variable as argument "
					  "#%d."),
					func->name, i+1);
#endif /* DEBUG_CODE */
				break;
			}
		}

		leaf=leaf->brother;
		i++;
	}
	return;
}


/**
 *
 * @param object
 * @param func
 * @param leaf
 */
static void code_func_args_protect(object_t *object, const func_t *func,
				   const leaf_t *leaf)
{
	unsigned int	i=0;

	while(leaf)
	{
		if (func->args_flags[i] & FUNC_ARG_PROTECT)
			__haplo_object_protect(object, leaf->result);
		
		leaf=leaf->brother;
		i++;
	}
	
	return;
}


/**
 * Choose a function to execute
 *
 * @param leaf
 *
 *
 */
static const func_t *code_func_choose(const leaf_t *leaf)
{
	const func_t *func;

	for(func=leaf->arg.function; func; func=func->next)
	{
		if (strcmp(func->name.constant,
			   leaf->arg.function->name.constant) != 0)
		{
			func=NULL;
			break;
		}
		if ((leaf->n == func->n) &&
		    (code_func_args_compare(func, leaf->child) == 0))
			break;
	}
	return(func);
}


/**
 * Execute a registred function
 *
 * @param result
 * @param f
 * @param leaf
 */
static void code_leaf_execute_func_do(void **result, union f f, leaf_t *leaf)
{
#define ARG1		(leaf->child)
#define ARG2		(ARG1->brother)
#define ARG3		(ARG2->brother)
#define ARG4		(ARG3->brother)
#define ARG5		(ARG4->brother)
#define DATA1		(ARG1->result->content)
#define DATA2		(ARG2->result->content)
#define DATA3		(ARG3->result->content)
#define DATA4		(ARG4->result->content)
#define DATA5		(ARG5->result->content)

	switch(leaf->n)
	{
	case 0:
		if (result)
			*result=(*f.f0)();
		else
			(*f.f0v)();
		break;

	case 1:
		if (result)
			*result=(*f.f1)(DATA1);
		else
			(*f.f1v)(DATA1);
		break;

	case 2:
		if (result)
			*result=(*f.f2)(DATA1, DATA2);
		else
			(*f.f2v)(DATA1,	DATA2);
		break;

	case 3:
		if (result)
			*result=(*f.f3)(DATA1, DATA2, DATA3);
		else
			(*f.f3v)(DATA1,	DATA2, DATA3);
		break;

	case 4:
		if (result)
			*result=(*f.f4)(DATA1, DATA2, DATA3, DATA4);
		else
			(*f.f4v)(DATA1, DATA2, DATA3, DATA4);
		break;

	case 5:
		if (result)
			*result=(*f.f5)(DATA1, DATA2, DATA3, DATA4, DATA5);
		else
			(*f.f5v)(DATA1, DATA2, DATA3, DATA4, DATA5);
		break;
	}
	
	return;
#undef ARG1
#undef ARG2
#undef ARG3
#undef ARG4
#undef ARG5
#undef DATA1
#undef DATA2
#undef DATA3
#undef DATA4
#undef DATA5
}


/**
 * Execute a leaf
 *
 * @param context
 * @param leaf
 *
 * @return status of action
 */ 
static enum leaf_status code_leaf_execute_func(const context_t *context,
						leaf_t *leaf)
{
	const func_t		*func;
	enum leaf_status	status;
	void			*result=NULL;
#ifdef SIGINT
	int flow;

	if (code_interrupted)
		return STATUS_INTERRUPTED;
#endif /* SIGINT */
	
	status=code_leaf_execute_children(context, leaf);
	if (status != STATUS_OK)
		return(status);

	func=code_func_choose(leaf);
	if (!func)
	{
		haplo_error("La fonction `%s' est incompatible avec les"
		" arguments fournis.", leaf->arg.function->name.constant);
		return(STATUS_ERROR);
	}

	code_func_args_check(context, func, leaf->child);

#ifdef SIGINT
	flow=setjmp(code_jmp);
	status=STATUS_OK;
	code_function=1;
	
	if (flow || code_interrupted)
	{
		status=STATUS_INTERRUPTED;
	}
	else
	{
#endif /* SIGINT */
		__haplo_prefix(func->name.constant);
		if (func->args[0])
			code_leaf_execute_func_do(&result, func->func, leaf);
		else
			code_leaf_execute_func_do(NULL, func->func, leaf);
		__haplo_prefix(NULL);
#ifdef SIGINT
	}
	code_function=0;
#endif /* SIGINT */
	
	if (status != STATUS_INTERRUPTED)
	{
		if (func->args[0])
		{
			if (! result)
			{
				__haplo_prefix(func->name.constant);
				haplo_error(_("Failure."));
				__haplo_prefix(NULL);
				status=STATUS_ERROR;
			}
			else
			{
				object_t	*object;
				
				object=__haplo_object_from_type(
					func->args[0], result);
				code_func_args_protect(object, func,
						      leaf->child);
				status=code_result_set(leaf, object);
			}
		}
		else
			status=STATUS_OK;
	}
	
	return(status);

#undef FUNC
}


/**
 * Execute a leaf
 *
 * @param context
 * @param leaf
 *
 * @return status of action
 */
static  enum leaf_status code_leaf_execute_if(const context_t *context,
						    leaf_t *leaf)
{
	enum leaf_status 	status;
	
	status=code_result_set(leaf, leaf->child->result);

	if (status == STATUS_OK)
	{
		if (leaf->child->result->type == OBJECT_BOOLEAN)
		{
			if (*((int *)leaf->child->result->content))
			{
				/*
				 * TRUE
				 */
				status=code_leaf_execute(context,
							 leaf->child->brother);
				(void)code_result_set(
					leaf,
					leaf->child->brother->result);
			}
			else
			{
				/*
				 * FALSE
				 */
				if (leaf->child->brother->brother)
				{
					status=code_leaf_execute(
						context,
						leaf->child->brother->brother);
					(void)code_result_set(
						leaf,
						leaf->child->brother->
						brother->result);
				}
			}
		}
		else
		{
			haplo_error(_("Object of type <%s> used as <boolean> "
				    "in if statement."),
				leaf->result->type->name);
			status=STATUS_ERROR;
		}	
	}
	return(status);
}


/**
 * Execute a leaf
 *
 * @param leaf
 *
 * @return the status of action
 */ 
static  enum leaf_status code_leaf_execute_load(leaf_t *leaf)
{
	enum leaf_status status;

	if (leaf->child->result->type == OBJECT_STRING){
		__haplo_parser_load(leaf->child->result->content,
				    leaf->arg.parse_param);
		status=STATUS_OK;
	} else {
		haplo_error(_("Object of type <%s> used as <string> "
			      "in `load()'."),
			    leaf->child->result->type->name);
		status=STATUS_ERROR;
	}

	return(status);
}


/**
 * Execute a leaf
 *
 * @param context
 * @param leaf
 *
 * @return the status of action
 */
static  enum leaf_status code_leaf_execute_local_assign(
	const context_t *context, leaf_t *leaf)
{
	const context_t	*c;
#ifdef HAPLO_DEBUG_CODE
	int		ok=0;
	
	HAPLO_ASSERT(leaf->child->result != NULL);
#endif

	for(c=context; c; c=c->father)
	{
		if (leaf->arg.code == c->code)
		{
			if (c->db[leaf->n] != leaf->child->result)
			{
#ifdef HAPLO_DEBUG_CODE
				haplo_debug(
					"CONTEXT: #%ld set to  <%s> "
					"object=%p context=%p",
					leaf->n,
					leaf->child->result->type->name,
					leaf->child->result,
					c);
#endif /* HAPLO_DEBUG_CODE */
				if (c->db[leaf->n])
					c->db[leaf->n]->names -= 1;
				__haplo_object_free(c->db[leaf->n]);

				c->db[leaf->n]=leaf->child->result;
				OBJECT_REF(c->db[leaf->n]);
				if (c->db[leaf->n])
					c->db[leaf->n]->names += 1;
			}
			(void)code_result_set(leaf, leaf->child->result);
			code_context_update(c, c->code->code);

#ifdef HAPLO_DEBUG_CODE
			ok=1;
#endif
			break;
		}
	}
#ifdef HAPLO_DEBUG_CODE
	if (!ok)
		haplo_fatal("code_leaf_execute_local_assign() failed");
#endif
	return(STATUS_OK);
}


/**
 * Execute a leaf
 *
 * @param context
 * @param leaf
 *
 * @return the status of action
 */
static  enum leaf_status code_leaf_execute_local_ref(
	const context_t *context, leaf_t *leaf)
{
	enum leaf_status	status;

	if (leaf->arg.code == context->code)
	{
		status=code_result_set(leaf, context->db[leaf->n]);
#ifdef HAPLO_DEBUG_CODE
		haplo_debug("Updating LOCAL_REF #%ld "
			    "with context %p: <%s>",
			    leaf->n, context,
			    leaf->result->type->name);
#endif /* HAPLO_DEBUG_CODE*/
	}
	else
		status=STATUS_OK;

	HAPLO_ASSERT(leaf->result != NULL);

	return(status);
}


/**
 * Update the  context
 *
 * @param context
 * @param leaf
 */
static void code_context_update(const context_t *context, leaf_t *leaf)
{
	if (leaf)
	{
		switch(leaf->action)
		{
		case LEAF_ACTION_OBJECT:
			if (leaf->result->type == OBJECT_CODE)
				code_context_update(
					context,
					CODE(leaf->result->content)->code);
			break;

		case LEAF_ACTION_EXECUTE:
			code_context_update(context, leaf->arg.execute);
			break;

		case LEAF_ACTION_BLOCK:
			code_context_update(context, leaf->arg.code->code);
			break;
		
		case LEAF_ACTION_LOCAL_REF:
			if (context->code == leaf->arg.code)
			{
#ifdef HAPLO_DEBUG_CODE
				haplo_debug("CONTEXT: #%ld is updated with "
					    "<%s> leaf=%p",
					    leaf->n,
					    (context->db[leaf->n])?context->db[leaf->n]->type->name:"nil",
					    leaf);
#endif /* HAPLO_DEBUG_CODE */

				/*if (leaf->result != context->db[leaf->n])*/
				(void)code_result_set(leaf,
						      context->db[leaf->n]);
			}
			break;
		default:
			/* do nothing */;
		}
		code_context_update(context, leaf->brother);
		code_context_update(context, leaf->child);
	}

	return;
}


/**
 * Execute a leave
 * 
 * @param leaf
 *
 * @return status of action
 */
static  enum leaf_status code_leaf_execute_ref(leaf_t *leaf)
{
	enum leaf_status	status;

	status=code_result_set(leaf, leaf->arg.ref->object);
#ifdef HAPLO_DEBUG_CODE
	haplo_debug("Updating GLOBAL_REF `%s'", leaf->arg.ref->name);
#endif /* HAPLO_DEBUG_CODE */
	if ( status == STATUS_ERROR)
	{
		haplo_error(_("Variable `%s' is undefined."),
			    leaf->arg.ref->name);
	}
	else
		status=STATUS_OK;
	
	return(status);
}



/**
 * Execute a leaf
 *
 * @param context
 * @param leaf
 *
 * @return the status of action
 */
static  enum leaf_status code_leaf_execute_list(const context_t *context,
						leaf_t *leaf)
{
	enum leaf_status status;
	leaf_t	*i;
	
	status=code_leaf_execute_children(context, leaf);
	
	if (status != STATUS_OK)
		return(status);

	(void)code_result_set(
		leaf,
		__haplo_object_from_type(
			OBJECT_LIST,
			__haplo_builtin_list_new(leaf->child->result)));
	__haplo_object_protect(leaf->result, leaf->child->result);
	
	for(i=leaf->child->brother; i; i=i->brother)
	{	
		__haplo_builtin_list_grow(leaf->result->content, i->result);
		__haplo_object_protect(leaf->result, i->result);
	}
	
	return(status);
}


/**
 * Execute a leaf
 * 
 * @param context
 * @param leaf
 * 
 * @return the status of action
 */
static  enum leaf_status code_leaf_execute_while(const context_t *context,
						 leaf_t *leaf)
{
	enum leaf_status status=STATUS_OK;
	

#ifdef SIGINT
	while (code_interrupted == 0)
#else /* ! SIGINT */
	while(1)
#endif /* SIGINT */
	{
		if (code_result_set(leaf, leaf->child->result)==STATUS_OK)
		{
			if (leaf->child->result->type == OBJECT_BOOLEAN)
			{
				if (*((int *)leaf->child->result->content))
				{
					/*
					 * TRUE: execute body once again
					 */
					code_leaf_execute(
						context,
						leaf->child->brother);
					if (leaf->child->brother)
						(void)code_result_set(
							leaf,
							leaf->child->brother
							->result);
				}
				else
				{
					(void)code_result_set(
						leaf,
						leaf->child->brother->result);
					break;
				}
				
			}
			else
			{
				haplo_error(_("bad condition"));
				break;
			}
		} else
			break;
		code_leaf_execute(context, leaf->child);
	}
#ifdef SIGINT
	if (code_interrupted)
		status=STATUS_INTERRUPTED;
#endif /* SIGINT */
	return(status);
}


/**
 *
 */
static void code_reset_children(leaf_t *leaf)
{
	leaf_t *l;
	
	for(l=leaf->child; l; l=l->brother)
		if (l->action != LEAF_ACTION_OBJECT)
			code_result_free(l);
	return;
}


/**
 * Execute a leaf. The big switch.
 *
 * @param context
 * @param leaf
 *
 * @return the status of action
 */
static  enum leaf_status code_leaf_execute_action(const context_t *context,
						  leaf_t *leaf)
{
	enum leaf_status	status=STATUS_ERROR;

	switch(leaf->action)
	{
	case LEAF_ACTION_ASSIGN:
		status=code_leaf_execute_assign(leaf);
		break;

	case LEAF_ACTION_BLOCK:
		status=code_leaf_execute_block(context, leaf);
		break;

	case LEAF_ACTION_BREAK:
		status=STATUS_BREAK;
		break;
		
	case LEAF_ACTION_CONTINUE:
		status=STATUS_BREAK;
		break;
		
	case LEAF_ACTION_EXECUTE:
		status=code_leaf_execute_execute(context, leaf);
		break;

	case LEAF_ACTION_EXTRACT:
		status=code_leaf_execute_extract(context, leaf);
		break;

	case LEAF_ACTION_FREE:
		__haplo_object_ref_free(leaf->arg.ref);
		status=STATUS_OK;
		break;

	case LEAF_ACTION_FUNCTION:
		status=code_leaf_execute_func(context, leaf);
		break;

	case LEAF_ACTION_IF:
		status=code_leaf_execute_if(context, leaf);
		break;

	case LEAF_ACTION_LOAD:
		status=code_leaf_execute_load(leaf);
		break;

	case LEAF_ACTION_LOCAL_ASSIGN:
		status=code_leaf_execute_local_assign(context, leaf);
		break;

	case LEAF_ACTION_LOCAL_REF:
		status=code_leaf_execute_local_ref(context, leaf);
		break;

	case LEAF_ACTION_OBJECT:
		status=STATUS_OK;
		break;

	case LEAF_ACTION_MULTI:
		status=code_leaf_execute_children(context, leaf);
		break;

	case LEAF_ACTION_NONE:
		status=STATUS_OK;
		break;
		
	case LEAF_ACTION_REF:
		status=code_leaf_execute_ref(leaf);
		break;

	case LEAF_ACTION_LIST:
		status=code_leaf_execute_list(context, leaf);
		break;

	case LEAF_ACTION_WHILE:
		status=code_leaf_execute_while(context, leaf);
		break;
	}
	code_reset_children(leaf);
	
	return(status);
}


/**
 * Execute children of a leaf
 *
 * @param context
 * @param leaf
 *
 * @return the status of action
 */
static  enum leaf_status code_leaf_execute_children(const context_t *context,
						    leaf_t *leaf)
{
	leaf_t	*child;
	enum leaf_status status=STATUS_OK;

	if (leaf->child)
	{
		for(child=leaf->child->brother; child; child=child->brother)
		{
			status=code_leaf_execute(context, child);
			if (status != STATUS_OK)
				break;
			
		}
	}
	return(status);
}


/**
 * Execute a leaf. Internal main entry point.
 *
 * @param context
 * @param leaf
 * 
 * @return the status of action
 */
static enum leaf_status code_leaf_execute(const context_t *context,
					  leaf_t *leaf)
{
 	enum leaf_status	status=STATUS_OK;

	if (leaf)
	{
#ifdef HAPLO_DEBUG_CODE
		static int		tab=0;
		code_debug_leaf(leaf, "EXECUTE", tab++);
#endif /* HAPLO_DEBUG_CODE */
		status=code_leaf_execute(context, leaf->child);
		if (status == STATUS_OK)
			status=code_leaf_execute_action(context, leaf);

#ifdef HAPLO_DEBUG_CODE
		tab--;
#endif /* HAPLO_DEBUG_CODE */
	}	
	return(status);
}


/**
 * Execute a leaf. Public entry point.
 *
 * @pram leaf
 * 
 * @return the result
 */
object_t *__haplo_code_leaf_execute(leaf_t *leaf)
{
#ifdef SIGINT
	if (code_interrupted)
	{
		haplo_info("Interrupted");
		code_interrupted=0;
		return(NULL);
	}
	code_interrupted=0;
	code_function=0;
#endif /* SIGINT */

	code_leaf_execute(NULL, leaf);

#ifdef SIGINT
	if (code_interrupted)
	{
		code_interrupted=0;
		haplo_info("Interrupted");
	}
#endif /* SIGINT */

	return(leaf->result);
}


/*
 * Object related stuff.
 */

/**
 * Create a new precode_t
 *
 * @return a new allocated structure
 */
precode_t * __haplo_precode_new(void)
{
	precode_t	*precode;
	
	HAPLO_ALLOC(precode, 1);
	__haplo_object_db_init(precode->db); 
	precode->n=0;
	precode->args=NULL;
	precode->first=NULL;
	precode->last=NULL;
	
	return(precode);
}


/**
 * Free a precode_t
 * 
 * @param precode
 */
void __haplo_precode_free(precode_t *precode)
{	
  	__haplo_object_db_free(precode->db);
	if (precode->args)
		HAPLO_FREE(precode->args);
	HAPLO_FREE(precode);
	return;
}


static void code_from_precode_replace_ref(code_t *code,
					  reference_t * const *trans,
					  leaf_t *l)
{
	unsigned long i;

	for(i=0; i<code->n; i++)
	{
		if (l->arg.ref == trans[i])
		{
#ifdef HAPLO_DEBUG_CODE
			haplo_debug(
				"REF(`%s', object=%p) -> "
				"LOCAL #%lu (code=%p)",
				l->arg.ref->name,
				l->arg.ref->object,
				i, code);
#endif /* HAPLO_DEBUG_CODE */
			l->arg.ref->instances--;
			l->action=LEAF_ACTION_LOCAL_REF;
			l->n=i;
			l->arg.code=code;
			code->flags |= CODE_HAVE_INTERNAL;
			break;
		}
	}
}


/**
 *
 */
static void code_from_precode_replace_assign(code_t *code,
					     reference_t *const *trans,
					     leaf_t *l)
{
	unsigned long i;
	
	for(i=0; i<code->n; i++)
	{
		if (l->arg.ref == trans[i])
		{
#ifdef HAPLO_DEBUG_CODE
			haplo_debug(
				"ASSIGN(`%s') -> LOCAL #%ld "
				"(code=%p)",
				l->arg.ref->name,
				i, code);
#endif /* HAPLO_DEBUG_CODE */	
			l->arg.ref->instances--;
			l->action=LEAF_ACTION_LOCAL_ASSIGN;
			l->n=i;
			l->arg.code=code;
			code->flags |= CODE_HAVE_INTERNAL;
			break;
		}
	}
}

/**
 * Kill references in a precode
 *
 * @param code
 * @param trans
 * @param l
 */
static void code_from_precode_replace(code_t *code, reference_t * const *trans,
				     leaf_t *l)
{
	if (l)
	{
		switch(l->action)
		{
		case LEAF_ACTION_OBJECT:
			/*
			 * nested function should be treated here!
			 */
			if (l->result->type == OBJECT_CODE)
				code_from_precode_replace(code,
				trans,
				CODE(l->result->content)->code);
			break;
		
		case LEAF_ACTION_EXECUTE:
			code_from_precode_replace(code, trans,
						  l->arg.execute);
			break;
			
		case LEAF_ACTION_BLOCK:
			code_from_precode_replace(code, trans,
						  l->arg.code->code);
			break;
			
		case LEAF_ACTION_REF:
			code_from_precode_replace_ref(code, trans, l);
			break;

		case LEAF_ACTION_ASSIGN:
			code_from_precode_replace_assign(code, trans, l);
			break;
			
		default:
			/* do nothing */
			break;
		}
		code_from_precode_replace(code, trans, l->brother);
		code_from_precode_replace(code, trans, l->child);
	}
	return;
}


/**
 * ??
 * 
 * @param precode
 * @param ref
 * @pram proposed
 * 
 * @return
 */
static unsigned int code_from_precode_pos(const precode_t *precode,
					  const reference_t *ref,
					  unsigned int *proposed)
{
	unsigned int	i;

	for(i=0; i<precode->n; i++)
	{
		if (ref == precode->args[i])
			return(i);
	}
	*proposed += 1;
	
	return((*proposed)-1);
}


/**
 * Compute statistics on a code
 *
 * @param code
 * @param leaf
 */
static void code_stat(code_t *code, leaf_t *leaf)
{
	if (leaf)
	{
		code->ops += 1;

		switch(leaf->action)
		{
		case LEAF_ACTION_EXECUTE:
			code_stat(code, leaf->arg.execute);
			break;
			
		case LEAF_ACTION_BLOCK:
			code_stat(code, leaf->arg.code->code);
			break;
			
		case LEAF_ACTION_REF:
		case LEAF_ACTION_ASSIGN:
			code->flags |= CODE_HAVE_EXTERNAL;
			break;
			
		case LEAF_ACTION_LOCAL_REF:
		case LEAF_ACTION_LOCAL_ASSIGN:
			code->flags |= CODE_HAVE_INTERNAL;
			break;
			
		default:
			/* do nothing */
			break;
		}
		
		code_stat(code, leaf->brother);
		code_stat(code, leaf->child);
	}
	
	return;
}


/**
 * Create a code from precode
 *
 * @param precode
 *
 * @return a code
 */
code_t *__haplo_code_from_precode(const precode_t *precode)
{
	code_t		*code;
	unsigned int	i;
	reference_t	**trans;
	
	HAPLO_ALLOC(code, 1);
	
	code->n=__haplo_object_db_size(precode->db);
	code->args=precode->n;
	code->code=precode->first;
	code->ops=0;
	code->flags=0;
	
	if (code->n)
	{
		unsigned int j=precode->n;
		HAPLO_ALLOC(trans, code->n);
		for(i=0, j=precode->n; i<OBJECT_HASH_TABLE_SIZE; i++)
		{
			reference_t	*ref;
			for(ref=precode->db[i]; ref; ref=ref->next)
			{
				trans[code_from_precode_pos(
					precode, ref, &j)]=ref;
			}
		}
		code_from_precode_replace(code, trans, code->code);
		HAPLO_FREE(trans);
	}

	code_stat(code, code->code);
	
	return(code);
}


/**
 * Count occurences of object in a tree of leaves
 *
 * @param object
 * @parm leaf 
 */
unsigned int __haplo_code_count_object(const object_t *object,
				       const leaf_t *leaf)
{
	unsigned int	count=0;
	
	if (leaf)
	{
		if (leaf->action==LEAF_ACTION_EXECUTE)
		{
			count += __haplo_code_count_object(object,
							   leaf->arg.execute);
		}
		if (leaf->action==LEAF_ACTION_BLOCK)
		{
			count += __haplo_code_count_object(object,
						   leaf->arg.code->code);
		}
		
		if (leaf->result == object)
		{
			count++;
		}
		else
		{
			if (leaf->result && 
			    (leaf->result->type == OBJECT_LIST))
			{
				count += __haplo_builtin_list_count_object(
					leaf->result,
					object);
			}
		}

		count += __haplo_code_count_object(object, leaf->brother);
		count += __haplo_code_count_object(object, leaf->child);
	}
	return(count);
}


/**
 * Compute level of recurivity
 *
 * @param leaf
 */
void __haplo_code_count_reference(leaf_t *leaf)
{
	if (leaf)
	{
		if (leaf->action==LEAF_ACTION_EXECUTE)
		{
			__haplo_code_count_reference(leaf->arg.execute);
		}
		
		if (leaf->action==LEAF_ACTION_BLOCK)
		{
			__haplo_code_count_reference(leaf->arg.code->code);
		}
		
		if (leaf->action == LEAF_ACTION_REF)
		{
			leaf->arg.ref->recursive += 1;
		}
		__haplo_code_count_reference(leaf->brother);
		__haplo_code_count_reference(leaf->child);
	}
	return;
}


/**
 * Remove occurence of a particular object
 *
 * @param object
 * @param leaf
 */
void __haplo_code_replace_object(const object_t *object, leaf_t *leaf)
{
	if (leaf)
	{
		if (leaf->action==LEAF_ACTION_EXECUTE)
		{
			__haplo_code_replace_object(object,
						    leaf->arg.execute);
		}
		
		if (leaf->result == object)
		{
			leaf->result=NULL;
		}
		__haplo_code_replace_object(object, leaf->brother);
		__haplo_code_replace_object(object, leaf->child);
	}
	return;
}


/**
 * Remove particular reference 
 *
 * @param reference
 * @param leaf
 */
void __haplo_code_replace_reference(const reference_t *reference, leaf_t *leaf)
{
	if (leaf)
	{
		if (leaf->action==LEAF_ACTION_EXECUTE)
		{
			__haplo_code_replace_reference(reference,
						       leaf->arg.execute);
		}

		if (leaf->action == LEAF_ACTION_REF)
		{
			if (leaf->arg.ref == reference)
			{
#ifdef HAPLO_DEBUG_CODE
				haplo_debug("Removing %s %p",
					    reference->name,
					    reference->object);
#endif /* HAPLO_DEBUG_CODE */
				leaf->action=LEAF_ACTION_NONE;
				if (leaf->result == reference->object)
				{
#ifdef HAPLO_DEBUG_CODE
					haplo_error("Freeing local ref");
#endif /* HAPLO_DEBUG_CODE */					
					leaf->result->references -= 1;
					leaf->result=NULL;
				}
			}
		}
		
		__haplo_code_replace_reference(reference, leaf->brother);
		__haplo_code_replace_reference(reference, leaf->child);
	}
	return;
}


/**
 * Free code object
 *
 * @param code
 */
void  __haplo_code_free(code_t *code)
{
	__haplo_code_leaf_free(code->code);
	HAPLO_FREE(code);
	
	return;
}


/**
 * Display function
 *
 * @param code
 */
void __haplo_code_display(const code_t *code)
{
	/*
	 * Avoid gcc to complain
	 */
	
	printf(_("Code (%d Op%s"), code->ops, (code->ops>1)?_("s"):_(""));
	if (code->flags & CODE_HAVE_INTERNAL)
		if (code->flags & CODE_HAVE_EXTERNAL)
			fputs(_(", Internal/External references"), stdout);
		else
			fputs(_(", Internal references"), stdout);
	else
		if (code->flags & CODE_HAVE_EXTERNAL)
			fputs(", External references", stdout);
	fputs(")", stdout);
	return;
}


/**
 * copy function
 *
 * @param code
 *
 * @return copy of code
 */
code_t * __haplo_code_copy(const code_t *code)
{
	code_t	*copy;

	HAPLO_ALLOC(copy, 1);
	copy->n=code->n;
	copy->ops=code->ops;
	copy->args=code->args;
	copy->flags=code->flags;
	copy->code=code_leaf_copy(code, copy, code->code);

	if (copy->flags == CODE_BAD)
	{
		haplo_debug("impossible de faire la copie");
		__haplo_code_free(copy);
		haplo_debug("code freed");
		copy=NULL;
	}
	return(copy);

}


/**
 * print function
 *
 * @param code
 */
void __haplo_code_print(const code_t *code)
{
	leaf_t	*l;
	unsigned int i;
	
	fputs("function(", stdout);
	for(i=0; i<code->args; i++)
	{
		if (i > 0)
		{
			fputs(", ", stdout);
		}
		printf("<local#%u>", i);		
	}
	fputs(")\n{\n", stdout);
	for(l=code->code; l; l=l->brother)
	{
		code_leaf_display(code, 1, l);
		fputs(";\n", stdout);
		
	}
	fputs("}\n", stdout);

	return;	
}


/**
 * Print subfunction
 * 
 * @param code
 * @param level
 * @param leaf
 */
static void code_leaf_display(const code_t *code, int level,
			      const leaf_t *leaf)
{
	leaf_t		*i;
	int		l;

	for(l=0; l<level; l++)
		fputc('\t', stdout);
	
	switch(leaf->action)
	{
	case LEAF_ACTION_ASSIGN:
		fputs(leaf->arg.ref->name, stdout);
		fputc('=', stdout);
		code_leaf_display(code, 0, leaf->child);
		break;
	case LEAF_ACTION_BLOCK:
		fputs("{\n", stdout);
		for(i=leaf->arg.code->code; i; i=i->brother)
		{
			code_leaf_display(code, level+1, i);
			fputs(";\n", stdout);
		}
		for(l=0; l<level; l++)
			fputc('\t', stdout);
		fputs("}", stdout);
		break;
	case LEAF_ACTION_BREAK:
		fputs("break;", stdout);
		break;
		
	case LEAF_ACTION_CONTINUE:
		fputs("continue;", stdout);
		break;
		
	case LEAF_ACTION_EXECUTE:
		code_leaf_display(code, 0, leaf->arg.execute);
		fputc('(', stdout);
		for(i=leaf->child; i; i=i->brother)
		{
			if (i != leaf->child)
			{
				fputs(", ", stdout);
			}
			code_leaf_display(code, 0, i);
		}
		fputc(')', stdout);
		break;
	case LEAF_ACTION_EXTRACT:
		code_leaf_display(code,0, leaf->child);
		fputs("->", stdout);
		code_leaf_display(code, 0, leaf->child->brother);
		break;
	case LEAF_ACTION_FREE:
		fputs("free(", stdout);
		fputs(leaf->arg.ref->name, stdout);
		fputc(')', stdout);
		break;
	case LEAF_ACTION_FUNCTION:
		if (strlen(leaf->arg.function->name.constant) < 3)
		{
			/*
			 * Binary or unary operators
			 */
			if (leaf->child)
			{
				if (leaf->child->brother)
				{
					code_leaf_display(code, 0,
							  leaf->child);
					fputc(' ', stdout);
					fputs(leaf->
					      arg.function->name.constant,
					      stdout);
					fputc(' ', stdout);
					code_leaf_display(
						code,
						0,
						leaf->child->brother);
				}
				else
				{
					fputs(leaf->
					      arg.function->name.constant,
					      stdout);
					code_leaf_display(
						code,
						0, leaf->child);
				}
			}
			else
			{
				fputs(leaf->arg.function->name.constant,
				      stdout);
			}
		}
		else
		{
			fputs(leaf->arg.function->name.constant, stdout);
			putc('(', stdout);

			for(i=leaf->child; i; i=i->brother)
			{
				if (i != leaf->child)
				{
					fputs(", ", stdout);
				}
				code_leaf_display(code,0, i);
			}
			putc(')', stdout);
		}
		break;
	case LEAF_ACTION_IF:
		fputs("if (", stdout);
		code_leaf_display(code, 0, leaf->child);
		fputs(")\n", stdout);

		if (leaf->child->brother->action==LEAF_ACTION_BLOCK)
			l=0;
		else
			l=1;
		
		code_leaf_display(code, level+l, leaf->child->brother);
		if (leaf->child->brother->brother)
		{
			putc('\n', stdout);
			for(l=0; l<level; l++)
				fputc('\t', stdout);
			fputs("else\n", stdout);
			if (leaf->child->brother->brother->action
			    ==LEAF_ACTION_BLOCK)
				l=0;
			else
				l=1;
			code_leaf_display(code, level+l,
					  leaf->child->brother->brother);
		}      
		break;
	case LEAF_ACTION_LOAD:
		fputs("load(", stdout);
		code_leaf_display(code, 0, leaf->child);
		fputc(')', stdout);
		break;
	case LEAF_ACTION_LOCAL_ASSIGN:
		if (leaf->arg.code == code)
			printf("<local#%lu> = ", leaf->n);
		else
			printf("<sublocal#%lu> = ", leaf->n);
		code_leaf_display(code, 0, leaf->child);
		break;
	case LEAF_ACTION_LOCAL_REF:
		if (leaf->arg.code == code)
			printf("<local#%lu>", leaf->n);
		else
			printf("<sublocal#%lu>", leaf->n);
		break;
	case LEAF_ACTION_OBJECT:
		if (leaf->result->type == OBJECT_CODE)
			__haplo_code_print(leaf->result->content);
		else
			__haplo_object_display(leaf->result);
		break;
	case LEAF_ACTION_MULTI:
		fputs("{\n", stdout);
		for (i=leaf->child; i; i=i->brother)
		{
			code_leaf_display(code, level+1, i);
			fputs(";\n", stdout);
		}
		for(l=0; l<level; l++)
			fputc('\t', stdout);
		fputs("}", stdout);
		
		break;
	case LEAF_ACTION_NONE:
		/* recursive function */
		fputs("&myself", stdout);
		break;
	case LEAF_ACTION_REF:
		fputs(leaf->arg.ref->name, stdout);
		break;
	case LEAF_ACTION_LIST:
		fputs("[\n", stdout);
		for(i=leaf->child; i; i=i->brother)
		{
			if (i != leaf->child)
			{
				fputs(", ", stdout);
			}
			code_leaf_display(code, level+1, i);
		}
		fputc(']', stdout);
		break;
	case LEAF_ACTION_WHILE:
		printf("while(");
		code_leaf_display(code, 0, leaf->child);
		printf(")\n");
		code_leaf_display(code, level+1, leaf->child->brother);
		break;
	}
	
	return;
}


/*
 * Debug stuff
 */

/**
 * Print spaces..
 *
 * @param offset
 *
 * @return strings full of spaces
 */
#ifdef HAPLO_DEBUG_CODE
static char *code_debug_offset(int offset)
{
	char	*string;
	int	i;
	
	HAPLO_ALLOC(string, 2*offset+1);
	for(i=0; i<offset*2; i++)
	{
		string[i]=' ';
		string[i+1]=' ';
	}
	string[i]='\0';
	
	return(string);
}
#endif /* HAPLO_DEBUG_CODE */


/**
 * Symbolic name
 *
 * @param action
 *
 * @return symbolic name
 */ 
#ifdef HAPLO_DEBUG_CODE
static const char *code_debug_action(enum leaf_action action)
{
#define CASE(x) case LEAF_ACTION_ ## x : return(#x)

	switch(action)
	{
		CASE(ASSIGN);
		CASE(BREAK);
		CASE(BLOCK);
		CASE(CONTINUE);
		CASE(EXECUTE);
		CASE(EXTRACT);
		CASE(FREE);
		CASE(FUNCTION);
		CASE(IF);
		CASE(LOAD);
		CASE(LOCAL_ASSIGN);
		CASE(LOCAL_REF);
		CASE(OBJECT);
		CASE(MULTI);
		CASE(NONE);
		CASE(REF);
		CASE(LIST);
		CASE(WHILE);
	}	
	/*normaly never reached */
	return("ERROR");
}
#endif /* HAPLO_DEBUG_CODE */


/**
 * Print leaf informations
 *
 * @param leaf
 * @param domain
 * @param offset
 */
#ifdef HAPLO_DEBUG_CODE
static void code_debug_leaf(leaf_t *leaf, const char *domain, int offset)
{
	char	*s;

	s=code_debug_offset(offset);
	haplo_debug("%s:%sleaf=%p n=%ld action=[%s]",
		    domain, s, leaf, leaf->n, code_debug_action(leaf->action));
	switch(leaf->action)
	{
	case LEAF_ACTION_ASSIGN:
	case LEAF_ACTION_BREAK:
	case LEAF_ACTION_BLOCK:
	case LEAF_ACTION_CONTINUE:
	case LEAF_ACTION_EXECUTE:
	case LEAF_ACTION_EXTRACT:
		break;
		
	case LEAF_ACTION_FREE:
		haplo_debug("%s:%s`-> ref=%p (name=<%s>, instances=%u)",
			    domain, s, leaf->arg.ref,
			    (leaf->arg.ref)?leaf->arg.ref->name:"",
			    (leaf->arg.ref)?leaf->arg.ref->instances:0);
		break;

	case LEAF_ACTION_FUNCTION:
		haplo_debug("%s:%s`-> function=<%s>",
			    domain, s, leaf->arg.function->name);
		break;
		
	case LEAF_ACTION_IF:
	case LEAF_ACTION_LOAD:
	case LEAF_ACTION_LOCAL_ASSIGN:
	case LEAF_ACTION_LOCAL_REF:
	case LEAF_ACTION_OBJECT:
	case LEAF_ACTION_MULTI:
	case LEAF_ACTION_NONE:
	case LEAF_ACTION_REF:
	case LEAF_ACTION_LIST:
	case LEAF_ACTION_WHILE:
		;
	}
	
	if (leaf->result)	
		haplo_debug("%s:%s`-> result=%p (type=[%s] ref=%u)",
			    domain, s, leaf->result,
			    leaf->result->type->name,
			    leaf->result->references);
	HAPLO_FREE(s);
	return;
}
#endif /* HAPLO_DEBUG_CODE */
