/*
 * $Id: builtin.c,v 1.18 2003/12/01 09:50:14 nicoo Exp $
 *
 *
 * Copyright (C) 1999, 2000, 2001 Nicolas LAURENT
 * This file is part of `Haplo'
 * 
 *
 * `Haplo'  is free software;  you can  redistribute  it and/or modify it
 * under the terms of the GNU Library General Public License as published
 * by the Free Software Foundation;  either version 2  of the License, or
 * (at your option) any later version.
 *
 * `Haplo' is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the  implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the  GNU General Public License
 * for more details.
 * 
 * You should have  received  a copy of the   GNU General Public  License
 * along with `Haplo'.  If not, write to  the
 *
 *                                        Free Software Foundation,  Inc.
 *                                        675 Mass Ave, Cambridge, MA
 *                                        02139, USA.
 *
 */

/*
 * Here are defined all registred functions and types.
 */


#include "extensions.h"
/*
 * We should disable strict ansi, otherwise some (math-)prototypes won't be
 * defined. We should do this before first #include. (BUG of GNU Libc ?)
 */
#ifdef __STRICT_ANSI__
#	undef __STRICT_ANSI__
#endif

#ifdef HAVE_CONFIG_H
#	include "config.h"
#endif

#include "version.h"

#include <float.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef HAVE_UNISTD_H
#	include <unistd.h>
#endif
#ifndef PATH_MAX
#	define PATH_MAX	1024
#endif /* PATH_MAX */

#include "haplo/builtin.h"

#include "code.h"
#include "builtin.h"
#include "func.h"
#include "help.h"
#include "pool.h"
#include "utils.h"


/*-----------------------------------------------------------------------------
                       G L O B A L   V A R I A B L E S 
-----------------------------------------------------------------------------*/

const object_type_t		*__haplo_object_type_float;
const object_type_t		*__haplo_object_type_string;
const object_type_t		*__haplo_object_type_list;
const object_type_t		*__haplo_object_type_boolean;
const object_type_t		*__haplo_object_type_code;

static pool_t			*builtin_float_pool=NULL;
static pool_t			*builtin_boolean_pool=NULL;
static pool_t			*builtin_list_pool=NULL;


/*-----------------------------------------------------------------------------
                             P R O T O T Y P E S 
-----------------------------------------------------------------------------*/

static int * boolean_and(const int *a, const int *b);
static int * boolean_or(const int *a, const int *b);
static int * boolean_xor(const int *a, const int *b);
static int * boolean_not(const int *a);
static int * boolean_copy(const int *a);
static void boolean_display(const int *a);
static void boolean_free(int *a);

double *haplo_float(double value);
static double * float_add(const double *a, const double *b);
static double * float_sub(const double *a, const double *b);
static double * float_mul(const double *a, const double *b);
static double * float_div(const double *a, const double *b);
static double * float_pow(const double *a, const double *b);
static double * float_unary_minus(const double *a);
static void float_print(const double *a);
static int * float_equal(const double *a, const double *b);
static int * float_greater(const double *a, const double *b);
static int * float_greater_e(const double *a, const double *b);
static int * float_lower(const double *a, const double *b);
static int * float_lower_e(const double *a, const double *b);
static void float_print(const double *a);
static double *float_copy(const double *a);
static void float_display(const double *a);
static void float_free(double *a);

static char *string_add_string(const char *s1, const char *s2);
static char *string_add_float(const char *s, const double *f);
static char *float_add_string(const double *f, const char *s);
static void string_print(const char *s);
static void string_display(const char *s);

static char *file_pwd(void);

static void list_display(const list_t *list);
static void list_free(list_t *list);
static list_t * list_copy(const list_t *list);
list_t * __haplo_builtin_list_new(object_t *object);
haplo_list_t * haplo_list_new(void *data, haplo_type_t type);
int __haplo_builtin_list_grow(list_t *list, object_t *object);
int haplo_list_grow(haplo_list_t *list, void *data, haplo_type_t type);
unsigned long haplo_list_size(const haplo_list_t *list);
void haplo_list_start(haplo_list_t *list);
haplo_type_t haplo_list_type(const haplo_list_t *list);
haplo_type_t haplo_list_type_homogeneous(const haplo_list_t *list);
void *haplo_list_arg(haplo_list_t *list);
void *haplo_list_arg_type(haplo_list_t *list, const haplo_type_t type);
object_t *__haplo_list_extract(list_t *list, double *indice);
unsigned int __haplo_builtin_list_count_object(const object_t *list,
					       const object_t *object);

HAPLO_MATH1_PROTO(acos);
HAPLO_MATH1_PROTO(asin);
HAPLO_MATH1_PROTO(atan);
HAPLO_MATH1_PROTO(cos);
HAPLO_MATH1_PROTO(cosh);
HAPLO_MATH1_PROTO(exp);
HAPLO_MATH1_PROTO(log);
HAPLO_MATH1_PROTO(log10);
HAPLO_MATH1_PROTO(sin);
HAPLO_MATH1_PROTO(sinh);
HAPLO_MATH1_PROTO(sqrt);
HAPLO_MATH1_PROTO(tan);
HAPLO_MATH1_PROTO(tanh);
HAPLO_MATH1_PROTO(fabs);

HAPLO_MATH2_PROTO(atan2);
HAPLO_MATH2_PROTO(pow);

#ifdef HAVE_ACOSH
HAPLO_MATH1_PROTO(acosh);
#endif
#ifdef  HAVE_ASINH
HAPLO_MATH1_PROTO(asinh);
#endif
#ifdef HAVE_ATANH
HAPLO_MATH1_PROTO(atanh);
#endif
#ifdef HAVE_CBRT
HAPLO_MATH1_PROTO(cbrt);
#endif
#ifdef HAVE_ERF
HAPLO_MATH1_PROTO(erf);
#endif
#ifdef HAVE_ERFC
HAPLO_MATH1_PROTO(erfc);
#endif
#ifdef HAVE_EXPM1
HAPLO_MATH1_PROTO(expm1);
#endif
#ifdef HAVE_J0
HAPLO_MATH1_PROTO(j0);
#endif
#ifdef HAVE_J1
HAPLO_MATH1_PROTO(j1);
#endif
#ifdef HAVE_LGAMMA
HAPLO_MATH1_PROTO(lgamma);
#endif
#ifdef HAVE_LOG1P
HAPLO_MATH1_PROTO(log1p);
#endif
#ifdef HAVE_LOGB
HAPLO_MATH1_PROTO(logb);
#endif
#ifdef HAVE_Y0
HAPLO_MATH1_PROTO(y0);
#endif
#ifdef HAVE_Y1
HAPLO_MATH1_PROTO(y1);
#endif
#ifdef HAVE_HYPOT
HAPLO_MATH2_PROTO(hypot);
#endif


void __haplo_builtin_init(void);


/*-----------------------------------------------------------------------------
                         I M P L E M E N T A T I O N 
-----------------------------------------------------------------------------*/


/**
 * booleand op
 *
 * @param a is a boolean
 * @param b is a boolean
 * @return a && b
 */
static int * boolean_and(const int *a, const int *b)
{
	int *result=__haplo_pool_get(builtin_boolean_pool);

	*result = (*a) & (*b);

	return(result);
}


/**
 * booleand op
 *
 * @param a is a boolean
 * @param b is a boolean
 * @return a || b
 */
static int * boolean_or(const int *a, const int *b)
{
	int *result=__haplo_pool_get(builtin_boolean_pool);

	*result = (*a) | (*b);

	return(result);
}


/**
 * booleand op
 *
 * @param a is a boolean
 * @param b is a boolean
 * @return a ^ b
 */
static int * boolean_xor(const int *a, const int *b)
{
	int *result=__haplo_pool_get(builtin_boolean_pool);

	*result = (*a) ^ (*b);

	return(result);
}


/**
 * booleand op
 *
 * @param a is a boolean
 * @return ! a
 */
static int * boolean_not(const int *a)
{
	int *result=__haplo_pool_get(builtin_boolean_pool);

	*result = (*a)?0:1;

	return(result);
}


/**
 * Copy constructor
 *
 * @param a is a boolean
 * @return a copy of a
 */
static int * boolean_copy(const int *a)
{
	int *result=__haplo_pool_get(builtin_boolean_pool);

	*result=*a;

	return(result);
}


/**
 * Display function 
 *
 * @param a is a boolean
 */
static void boolean_display(const int *a)
{
	if (*a)
		fputs("TRUE", stdout);
	else
		fputs("FALSE", stdout);

	return;
}


/**
 *
 */
static void boolean_free(int *a)
{
	__haplo_pool_release(builtin_boolean_pool, a);

	return;
}


/**
 *
 */
double *haplo_float(double value)
{
	double *f=__haplo_pool_get(builtin_float_pool);

	*f = value;

	return(f);
	
}


/**
 * float op
 *
 * @param a is a float
 * @param b is a float
 * @return a + b
 */
static double * float_add(const double *a, const double *b)
{
	double *result=__haplo_pool_get(builtin_float_pool);

	*result = *a + *b;

	return(result);
}


/**
 * float op
 *
 * @param a is a float
 * @param b is a float
 * @return a - b
 */
static double * float_sub(const double *a, const double *b)
{
	double *result=__haplo_pool_get(builtin_float_pool);

	*result = *a - *b;

	return(result);
}


/**
 * float op
 *
 * @param a is a float
 * @param b is a float
 * @return a * b
 */
static double * float_mul(const double *a, const double *b)
{
	double *result=__haplo_pool_get(builtin_float_pool);

	*result = (*a) * (*b);

	return(result);
}


/**
 * float op
 *
 * @param a is a float
 * @param b is a float
 * @return a / b
 */
static double * float_div(const double *a, const double *b)
{
	double *result=__haplo_pool_get(builtin_float_pool);

	*result = (*a) / (*b);

	return(result);
}


/**
 * float op
 *
 * @param a is a float
 * @param b is a float
 * @return a ^ b
 */
static double * float_pow(const double *a, const double *b)
{
	double *result=__haplo_pool_get(builtin_float_pool);

	*result = pow(*a, *b);

	return(result);
}


/**
 * float op
 *
 * @param a is a float
 * @return -a
 */
static double * float_unary_minus(const double *a)
{
	double *result=__haplo_pool_get(builtin_float_pool);

	*result = - (*a);

	return(result);
}


/**
 * float comparison
 *
 * @param a is a float
 * @param b is a float
 * @return a == b
 */
static int * float_equal(const double *a, const double *b)
{
	int *result=__haplo_pool_get(builtin_boolean_pool);

	*result=(fabs(*a - *b) < DBL_EPSILON)?1:0;

	return(result);
}


/**
 * float comparison
 *
 * @param a is a float
 * @param b is a float
 * @return a > b
 */
static int * float_greater(const double *a, const double *b)
{
	int *result=__haplo_pool_get(builtin_boolean_pool);

	*result =(*a > *b)?1:0;

	return(result);
}


/**
 * float comparison
 *
 * @param a is a float
 * @param b is a float
 * @return a >= b
 */
static int * float_greater_e(const double *a, const double *b)
{
	int *result=__haplo_pool_get(builtin_boolean_pool);

	*result = (*a >= *b)?1:0;

	return(result);
}


/**
 * float comparison
 *
 * @param a is a float
 * @param b is a float
 * @return a < b
 */
static int * float_lower(const double *a, const double *b)
{
	int *result=__haplo_pool_get(builtin_boolean_pool);

	*result = (*a < *b)?1:0;

	return(result);
}


/**
 * float comparison
 *
 * @param a is a float
 * @param b is a float
 * @return a <= b
 */
static int * float_lower_e(const double *a, const double *b)
{
	int *result=__haplo_pool_get(builtin_boolean_pool);

	*result = (*a <= *b)?1:0;

	return(result);
}


/**
 * Print function
 *
 * @param a is a float
 */
static void float_print(const double *a)
{
	printf("%.8g\n", *a);
	return;
}


/**
 * copy constructor
 *
 * @param a is a float
 * @return a copy of a
 */
static double *float_copy(const double *a)
{
	double *result=__haplo_pool_get(builtin_float_pool);

	*result = *a;
	
	return(result);
}


/**
 *
 */
static void float_free(double *a)
{
	__haplo_pool_release(builtin_float_pool, a);

	return;
}


/**
 * display function
 *
 * @param a is a float
 */
static void float_display(const double *a)
{
	printf("%.8g", *a);
	return;
}


/**
 * string op
 *
 * @param s1 is a string
 * @param s2 is a string
 * @return concatenation of s1 and s2
 */
static char *string_add_string(const char *s1, const char *s2)
{
	char	*result;
	
	HAPLO_ALLOC(result, strlen(s1)+strlen(s2)+1);
	strcpy(result, s1);
	strcat(result, s2);
	
	return(result);
}


/**
 * string/float op
 *
 * @param s is a string
 * @param f is a float
 * @return a concatenation of s and string converted f
 */
static char *string_add_float(const char *s, const double *f)
{
	char	*result;
	char	s2[20];
	
	snprintf(s2, 20, "%.8g", *f);
	HAPLO_ALLOC(result, strlen(s)+strlen(s2)+1);
	strcpy(result, s);
	strcat(result, s2);

	return(result);
}


/**
 * string/float op
 *
 * @param f is a float
 * @param s is a string
 * @return a concatenation of string converted f and s
 */
static char *float_add_string(const double *f, const char *s)
{
	char	*result;
	char	s2[20];
	
	snprintf(s2, 20, "%.8g", *f);
	HAPLO_ALLOC(result, strlen(s)+strlen(s2)+1);
	strcpy(result, s2);
	strcat(result, s);

	return(result);
}


/**
 * print function
 *
 * @param s is a string
 */
static void string_print(const char *s)
{
	puts(s);
	return;
}


/**
 * display function
 *
 * @param s is a string
 */
static void string_display(const char *s)
{
	putc('"', stdout);
	fputs(s, stdout);
	putc('"', stdout);
	
	return;
}


/**
 * print working directory name
 */
static char * file_pwd(void)
{
	char pwd[PATH_MAX];

	(void)getcwd(pwd, PATH_MAX);

	return(haplo_strdup(pwd));
}


/**
 * display function
 *
 * @param list
 */
static void list_display(const list_t *list)
{
	printf(_("List (%lu element%s)"), list->size, (list->size>1)?"s":"");
	return;
}


/**
 * free function
 *
 * @param list
 */
static void list_free(list_t *list)
{
	/*
	 * La lib?ration des object_t contenus dans le vecteur
	 * est ? la charge de l'allocateur d'objets.
	 * En effet, le champ protected sert ? cela!
	 */
	__haplo_slink_free(list->first);
	__haplo_pool_release(builtin_list_pool, list);

	return;
}


/**
 * copy constructor
 * 
 * @param list
 */
static list_t * list_copy(const list_t *list)
{
	list_t	*copy;
	slink_t *s;

	copy=__haplo_pool_get(builtin_list_pool);
	copy->size=list->size;
	copy->first=NULL;
	
	copy->last=NULL;
	
	for(s=list->first; s; s=s->next)
	{
		slink_t	*c;

		c=__haplo_slink_new(s->data);
		/*
		 * l'appel ? __haplo_object_protect() est automatiquement
		 * fait par __haplo_object_copy() qui a appeler cette fontion!
		 * Heureusement, car dans ce niveau d'appel, il n'y a pas
		 * moyen d'acc?der aux structures de donn?es d'objet!
		 */
		if (copy->last)
			copy->last->next=c;
		else
			copy->first=c;

		copy->last=c;
		
		if (list->current == s)
			copy->current=c;
	}
	
	return(copy);
}


/**
 * Constructor
 *
 * @param object if the first element of the list
 */
list_t *__haplo_builtin_list_new(object_t *object)
{
	list_t *list=NULL;

	if (object)
	{
		list=__haplo_pool_get(builtin_list_pool);
		list->size=1;
		list->first=__haplo_slink_new(object);
		list->current=list->first;
		list->last=list->first;
		list->flags=LIST_HOMOGENEOUS;
	}

	return(list);
}


/**
 * public constructor
 *
 * @param data is the content of the object which is the first element
 * @param type  is the type of the object which is the first element
 *
 * @see __haplo_builtin_list_new
 */
haplo_list_t * haplo_list_new(void *data, haplo_type_t type)
{
	object_t *object;

	object=__haplo_object_from_type((object_type_t *)type, data);

	return(__haplo_builtin_list_new(object));
}


/**
 * push a element in a list
 *
 * @param list
 * @param object
 * @return 0 if object != NULL. 1 overwise
 */
int __haplo_builtin_list_grow(list_t *list, object_t *object)
{
	int status = 1;
	
	if (object)
	{
		list->size++;
		if (object->type != OBJECT(list->last->data)->type)
			list->flags &= ~LIST_HOMOGENEOUS;
		list->last=__haplo_slink_append(list->last, object);
		status =0;
	}
	
	return(status);
}


/**
 * public interface of __haplo_builtin_list_grow
 *
 * @param list
 * @param data is the content of the new object
 * @param type is the type of the new object
 *
 * @see __haplo_builtin_list_grow
 */
int haplo_list_grow(haplo_list_t *list, void *data, haplo_type_t type)
{
	object_t	*object;
	
	object=__haplo_object_from_type((object_type_t *)type, data);

	return(__haplo_builtin_list_grow(list, object));
}


/**
 * compute length of a list
 *
 * @param list
 * @return the length
 */
unsigned long haplo_list_size(const haplo_list_t *list)
{
	unsigned long size=0;
	
	if (list)
		size=LIST(list)->size;

	return(size);
}


/**
 *
 * @param list
 */
void haplo_list_start(haplo_list_t *list)
{
	LIST(list)->current=LIST(list)->first;

	return;
}


/**
 * @param list
 * @return the type of the first element of a list
 *
 * @see haplo_list_homogeneous_type
 */
haplo_type_t haplo_list_type(const haplo_list_t *list)
{
	const object_type_t *type=NULL;
	
	if (list)
		type=OBJECT(LIST(list)->current->data)->type;

	return(type);
}


/**
 * Check that a list is homogeneous
 *
 * @param list
 * @return the type of the first element of a list
 * 
 * @see haplo_list_type
 */
haplo_type_t haplo_list_type_homogeneous(const haplo_list_t *list)
{
	const object_type_t *type=NULL;
	
	if (LIST(list)->flags & LIST_HOMOGENEOUS)
		type=OBJECT(LIST(list)->first->data)->type;
	
	return(type);
}


/**
 * look at the content of the next object in the list.
 * 
 * @param list
 * @return the content of the next object
 */
void *haplo_list_arg(haplo_list_t *list)
{
	void *content=NULL;

	if (LIST(list)->current)
	{
		content=OBJECT(LIST(list)->current->data)->content;
		LIST(list)->current=LIST(list)->current->next;
	}
	return(content);
}


/**
 *
 *
 */
void *haplo_list_arg_type(haplo_list_t *list, const haplo_type_t type)
{
	void *content=NULL;

	if (LIST(list)->current)
	{
		if (OBJECT(LIST(list)->current->data)->type == type)
			content = OBJECT(LIST(list)->current->data)->content;
		else
			haplo_error(
				"Object of type <%s> was expected in list"
				" instead of <%s",
				OBJECT_TYPE(type)->name,
				OBJECT(LIST(list)->current->data)->type->name);
		LIST(list)->current=LIST(list)->current->next;	
	} else
		haplo_error("The list is too short");

	return(content);
}


/**
 * get the i th element of a list
 *
 * @param list
 * @param indice
 * @return the indice th element of the list
 */
object_t *__haplo_builtin_list_extract(const list_t *list,
				       const double *indice)
{
	unsigned int i;
	object_t *object=NULL;
	
	i=haplo_uint(*indice - 1);
	if (i < list->size)
	{
		slink_t	*s;

		s=list->first;
		while(i--)
			s=s->next;
		object=s->data;
	}
	else
		haplo_warning(_("Indice `%u' exceed size of the list."), i);
	
	return(object);
}


/**
 * Count occurence of an object in a list and sublists
 *
 * @param list
 * @param object
 * @return the count
 */
unsigned int __haplo_builtin_list_count_object(const object_t *list,
					       const object_t *object)
{
	unsigned int	count=0;
	slink_t		*s;
	
	for(s=LIST(list->content)->first; s; s=s->next)
	{
		if (OBJECT(s->data) == object)
			count++;
		else
		{
			if (OBJECT(s->data)->type == OBJECT_LIST)
			{
				count += __haplo_builtin_list_count_object(
					OBJECT(s->data), object);
			}
		}
	}
	return(count);
}


/*
 * Mathematical functions
 */
HAPLO_MATH1_FUNC(acos)
HAPLO_MATH1_FUNC(asin)
HAPLO_MATH1_FUNC(atan)
HAPLO_MATH1_FUNC(cos)
HAPLO_MATH1_FUNC(cosh)
HAPLO_MATH1_FUNC(exp)
HAPLO_MATH1_FUNC(log)
HAPLO_MATH1_FUNC(log10)
HAPLO_MATH1_FUNC(sin)
HAPLO_MATH1_FUNC(sinh)
HAPLO_MATH1_FUNC(sqrt)
HAPLO_MATH1_FUNC(tan)
HAPLO_MATH1_FUNC(tanh)
HAPLO_MATH1_FUNC(fabs)

HAPLO_MATH2_FUNC(atan2)
HAPLO_MATH2_FUNC(pow)

#ifdef HAVE_ACOSH
HAPLO_MATH1_FUNC(acosh)
#endif
#ifdef HAVE_ASINH
HAPLO_MATH1_FUNC(asinh)
#endif
#ifdef HAVE_ATANH
HAPLO_MATH1_FUNC(atanh)
#endif
#ifdef HAVE_CBRT
HAPLO_MATH1_FUNC(cbrt)
#endif
#ifdef HAVE_ERF
HAPLO_MATH1_FUNC(erf)
#endif
#ifdef HAVE_ERFC
HAPLO_MATH1_FUNC(erfc)
#endif
#ifdef HAVE_ERFC
HAPLO_MATH1_FUNC(expm1)
#endif
#ifdef HAVE_EXPM1
HAPLO_MATH1_FUNC(j0)
#endif
#ifdef HAVE_GETCWD
HAPLO_MATH1_FUNC(j1)
#endif
#ifdef HAVE_LGAMMA
HAPLO_MATH1_FUNC(lgamma)
#endif
#ifdef HAVE_LOG1P
HAPLO_MATH1_FUNC(log1p)
#endif
#ifdef HAVE_LOGB
HAPLO_MATH1_FUNC(logb)
#endif
#ifdef HAVE_Y0
HAPLO_MATH1_FUNC(y0)
#endif
#ifdef HAVE_Y1
HAPLO_MATH1_FUNC(y1)
#endif
#ifdef HAVE_HYPOT
HAPLO_MATH2_FUNC(hypot)
#endif


static haplo_type_descr_t builtin_types[]={
	{ "boolean",
	  HAPLO_TYPE_DISPLAY(boolean_display),
	  HAPLO_TYPE_FREE(boolean_free),
	  HAPLO_TYPE_COPY(boolean_copy) },
	{ "code",
	  HAPLO_TYPE_DISPLAY(__haplo_code_display),
	  HAPLO_TYPE_FREE(__haplo_code_free),
	  HAPLO_TYPE_COPY(__haplo_code_copy) },
	{ "float",
	  HAPLO_TYPE_DISPLAY(float_display),
	  HAPLO_TYPE_FREE(float_free),
	  HAPLO_TYPE_COPY(float_copy) },
#if HAPLO_PLUGINS_IMPL != HAPLO_PLUGINS_IMPL_NONE
	{ "library",
	  NULL, NULL,
	  HAPLO_TYPE_FREE(HAPLO_FREE_FUNC),
	  NULL, NULL },
#endif /* HAPLO_PLUGINS_IMPL */
	{ "list",
	  HAPLO_TYPE_DISPLAY(list_display),
	  HAPLO_TYPE_FREE(list_free),
	  HAPLO_TYPE_COPY(list_copy) },
	{ "string",
	  HAPLO_TYPE_DISPLAY(string_display),
	  HAPLO_TYPE_FREE(HAPLO_FREE_FUNC),
	  HAPLO_TYPE_COPY(haplo_strdup) },
	{ NULL, NULL, NULL, NULL, NULL, NULL, NULL }
};


static haplo_func_descr_t builtin_functions[]={
	{ "print", HAPLO_FUNC(__haplo_code_print), ":code" },
	{ "&", HAPLO_FUNC(boolean_and), "boolean:boolean:boolean" },
	{ "|", HAPLO_FUNC(boolean_or), "boolean:boolean:boolean" },
	{ "~", HAPLO_FUNC(boolean_xor), "boolean:boolean:boolean" },
	{ "!", HAPLO_FUNC(boolean_not), "boolean:boolean" },
	{ "+", HAPLO_FUNC(float_add), "float:float:float" },
	{ "-", HAPLO_FUNC(float_sub), "float:float:float" },
	{ "*", HAPLO_FUNC(float_mul), "float:float:float" },
	{ "/", HAPLO_FUNC(float_div), "float:float:float" },
	{ "--", HAPLO_FUNC(float_unary_minus), "float:float" },
	{ "++", HAPLO_FUNC(float_copy), "float:float" },
	{ "^", HAPLO_FUNC(float_pow), "float:float:float" },
	{ "=", HAPLO_FUNC(float_equal), "boolean:float:float" },
	{ ">", HAPLO_FUNC(float_greater), "boolean:float:float" },
	{ ">=", HAPLO_FUNC(float_greater_e), "boolean:float:float" },
	{ "<", HAPLO_FUNC(float_lower), "boolean:float:float" },
	{ "<=", HAPLO_FUNC(float_lower_e), "boolean:float:float" },
	{ "print", HAPLO_FUNC(float_print), ":float" },
	{ "print", HAPLO_FUNC(string_print), ":string" },
	{ "+", HAPLO_FUNC(string_add_string), "string:string:string" },
	{ "+", HAPLO_FUNC(string_add_float), "string:string:float" },
	{ "+", HAPLO_FUNC(float_add_string), "string:float:string" },
	{ "bordered", HAPLO_FUNC(haplo_bordered), ":string" },
	{ "centered", HAPLO_FUNC(haplo_centered), ":string" },
	{ "underlined", HAPLO_FUNC(haplo_underlined), ":string" },
	{ "cd", HAPLO_FUNC(chdir), ":string" },
	{ "pwd", HAPLO_FUNC(file_pwd), "string" },
	{ "type", HAPLO_FUNC(__haplo_object_type_list_display), "" },
	{ "bind", HAPLO_FUNC(__haplo_func_list_display), "" },
#if HAPLO_PLUGINS_IMPL != HAPLO_PLUGINS_IMPL_NONE
	{ "use", HAPLO_FUNC(__haplo_plugins_load), ":string" },
	{ "use", HAPLO_FUNC(__haplo_plugins_status), "" },
	{ "load_library", HAPLO_FUNC(__haplo_plugins_library_load), "library:string" } ,
	{ "bind", HAPLO_FUNC(__haplo_func_bind), ":library:string:string" },
	{ "type", HAPLO_FUNC(__haplo_object_type_register_0), ":string" } ,
	{ "type", HAPLO_FUNC(__haplo_object_type_register_1), ":string:string" } ,
	{ "type", HAPLO_FUNC(__haplo_object_type_register_2), ":string:string:string" } ,
	{ "type", HAPLO_FUNC(__haplo_object_type_register_3), ":string:string:string:string" } ,
#endif /* HAPLO_PLUGINS_IMPL */
	{ "html", HAPLO_FUNC(__haplo_help_html), ":string" },
	{ "abort", HAPLO_FUNC(abort), "" },
	HAPLO_MATH1_REGISTER(acos),
	HAPLO_MATH1_REGISTER(asin),
	HAPLO_MATH1_REGISTER(atan),
	HAPLO_MATH1_REGISTER(cos),
	HAPLO_MATH1_REGISTER(cosh),
	HAPLO_MATH1_REGISTER(exp),
	HAPLO_MATH1_REGISTER(log),
	HAPLO_MATH1_REGISTER(log10),
	HAPLO_MATH1_REGISTER(sin),
	HAPLO_MATH1_REGISTER(sinh),
	HAPLO_MATH1_REGISTER(sqrt),
	HAPLO_MATH1_REGISTER(tan),
	HAPLO_MATH1_REGISTER(tanh),
	HAPLO_MATH1_REGISTER(fabs),	
	HAPLO_MATH2_REGISTER(atan2),
	HAPLO_MATH2_REGISTER(pow),
#ifdef HAVE_ACOSH
	HAPLO_MATH1_REGISTER(acosh),
#endif
#ifdef HAVE_ASINH
	HAPLO_MATH1_REGISTER(asinh),
#endif
#ifdef HAVE_ATANH
	HAPLO_MATH1_REGISTER(atanh),
#endif
#ifdef HAVE_CBRT
	HAPLO_MATH1_REGISTER(cbrt),
#endif
#ifdef HAVE_ERF
	HAPLO_MATH1_REGISTER(erf),
#endif
#ifdef HAVE_ERFC
	HAPLO_MATH1_REGISTER(erfc),
#endif
#ifdef HAVE_EXPM1
	HAPLO_MATH1_REGISTER(expm1),
#endif
#ifdef HAVE_J0
	HAPLO_MATH1_REGISTER(j0),
#endif
#ifdef HAVE_J1
	HAPLO_MATH1_REGISTER(j1),
#endif
#ifdef HAVE_LGAMMA
	HAPLO_MATH1_REGISTER(lgamma),
#endif
#ifdef HAVE_LOG1P
	HAPLO_MATH1_REGISTER(log1p),
#endif
#ifdef HAVE_LOGB
	HAPLO_MATH1_REGISTER(logb),
#endif
#ifdef HAVE_Y0
	HAPLO_MATH1_REGISTER(y0),
#endif
#ifdef HAVE_Y1
	HAPLO_MATH1_REGISTER(y1),
#endif
#ifdef HAVE_HYPOT
	HAPLO_MATH2_REGISTER(hypot),
#endif
	{ NULL, NULL, NULL, NULL }
};


/**
 * Initialize builtin functions
 */
void __haplo_builtin_init(void)
{
	/*
	 * pools initializations
	 */
	builtin_float_pool=__haplo_pool_new(BUILTIN_POOL_SIZE, sizeof(double));
	builtin_boolean_pool=__haplo_pool_new(BUILTIN_POOL_SIZE, sizeof(int));
	builtin_list_pool=__haplo_pool_new(BUILTIN_POOL_SIZE, sizeof(list_t));
	
	/*
	 * Types and functions registrations
	 */
	__haplo_object_type_register(builtin_types, NULL);
	__haplo_func_register(builtin_functions, NULL);

	/*
	 * Initialize cache
	 */
	__haplo_object_type_float=haplo_object_type_get("float");
	__haplo_object_type_string=haplo_object_type_get("string");
	__haplo_object_type_list=haplo_object_type_get("list");
	__haplo_object_type_boolean=haplo_object_type_get("boolean");
	__haplo_object_type_code=haplo_object_type_get("code");

	return;
}


/**
 *
 */
void __haplo_builtin_fini(void)
{
	__haplo_pool_free(builtin_float_pool);
	__haplo_pool_free(builtin_boolean_pool);
	__haplo_pool_free(builtin_list_pool);

	return;
}
