/*
 * $Id: identity.c,v 1.3 2003/10/14 11:19:50 nicoo Exp $
 *
 *
 * Copyright (C) 1999, 2000, 2001 Nicolas LAURENT
 * This file is part of `Haplo'
 * 
 *
 * `Haplo'  is free software;  you can  redistribute  it and/or modify it
 * under the terms of the GNU Library General Public License as published
 * by the Free Software Foundation;  either version 2  of the License, or
 * (at your option) any later version.
 *
 * `Haplo' is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the  implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the  GNU General Public License
 * for more details.
 * 
 * You should have  received  a copy of the   GNU General Public  License
 * along with `Haplo'.  If not, write to  the
 *
 *                                        Free Software Foundation,  Inc.
 *                                        675 Mass Ave, Cambridge, MA
 *                                        02139, USA.
 *
 */

/*
 * About MD5 code :
 *
 * This code implements the MD5 message-digest algorithm.
 * The algorithm is due to Ron Rivest.  This code was
 * written by Colin Plumb in 1993, no copyright is claimed.
 * This code is in the public domain; do with it what you wish.
 *
 * Equivalent code is available from RSA Data Security, Inc.
 * This code has been tested against that, and is equivalent,
 * except that you don't need to include two pages of legalese
 * with every copy.
 *
 * To compute the message digest of a chunk of bytes, declare an
 * MD5Context structure, pass it to MD5Init, call MD5Update as
 * needed on buffers full of bytes, and then call MD5, which
 * will fill a supplied 16-byte array with the digest.
 */


#ifdef HAVE_CONFIG_H
#	include "config.h"
#endif
#include "version.h"

#include <unistd.h>
#include <termios.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <netinet/in.h>

#include "identity.h"


/*-----------------------------------------------------------------------------
                                 U I N T 3 2 
-----------------------------------------------------------------------------*/

typedef unsigned long uint32;


/*-----------------------------------------------------------------------------
                             M D 5 C O N T E X T 
-----------------------------------------------------------------------------*/

struct MD5Context  
{  
	uint32 buf[4];  
	uint32 bits[2];  
	unsigned char in[64];  
};

/*-----------------------------------------------------------------------------
                             P R O T O T Y P E S
-----------------------------------------------------------------------------*/

static void byte_reverse(unsigned char *buf, unsigned longs);
static void MD5Init(struct MD5Context *ctx);
static void MD5Update(struct MD5Context *ctx, unsigned char const *buf,
		      unsigned len);
static void MD5Final(unsigned char digest[16], struct MD5Context *ctx);
static void MD5Transform(uint32 buf[4], uint32 const in[16]);
#ifdef HAPLO_DEBUG_MD5
static void MD5Rewrite(unsigned char digest[32]);
#endif
static int term_setup(struct termios *term);
static void term_restore(const struct termios *term);
static void get_string(unsigned char *buffer, size_t len, int echo, int erase);
void identity_get(identity_t *id);
int identity_cmp(const identity_t *id1, const identity_t *id2);


/*-----------------------------------------------------------------------------
                         I M P L E M E N T A T I O N
-----------------------------------------------------------------------------*/

/**
 *
 */
static void byte_reverse (unsigned char *buf, unsigned longs)
{
	const unsigned long u=(65535U << 1);
	
	if (htonl(u) == u)
	{
		/* MSB First / Little Endian */
		uint32 t;

		do
		{
			t = (uint32) ((unsigned) buf[3] << 8 | buf[2]) << 16 |
				((unsigned) buf[1] << 8 | buf[0]);
			*(uint32 *) buf = t;
			buf += 4;
		}
		while (--longs);
	}
}


/**
 * Start MD5 accumulation.  Set bit count to 0 and buffer to mysterious
 * initialization constants.
 */
static void MD5Init (struct MD5Context *ctx)
{
	ctx->buf[0] = 0x67452301;
	ctx->buf[1] = 0xefcdab89;
	ctx->buf[2] = 0x98badcfe;
	ctx->buf[3] = 0x10325476;

	ctx->bits[0] = 0;
	ctx->bits[1] = 0;
}


/**
 * Update context to reflect the concatenation of another buffer full
 * of bytes.
 */
static void MD5Update (struct MD5Context *ctx, unsigned char const *buf,
		       unsigned len)
{
	uint32 t;

	/* Update bitcount */

	t = ctx->bits[0];
	if ((ctx->bits[0] = t + ((uint32) len << 3)) < t)
		ctx->bits[1]++;	 /* Carry from low to high */
	ctx->bits[1] += len >> 29;

	t = (t >> 3) & 0x3f; /* Bytes already in shsInfo->data */

	/* Handle any leading odd-sized chunks */

	if (t)
	{
		unsigned char *p = (unsigned char *) ctx->in + t;

		t = 64 - t;
		if (len < t)
		{
			memcpy (p, buf, len);
			return;
		}
		memcpy (p, buf, t);
		byte_reverse (ctx->in, 16);
		MD5Transform (ctx->buf, (uint32 *) ctx->in);
		buf += t;
		len -= t;
	}
	/* Process data in 64-byte chunks */

	while (len >= 64)
	{
		memcpy (ctx->in, buf, 64);
		byte_reverse (ctx->in, 16);
		MD5Transform (ctx->buf, (uint32 *) ctx->in);
		buf += 64;
		len -= 64;
	}

	/* Handle any remaining bytes of data. */

	memcpy (ctx->in, buf, len);
}


/**
 * Final wrapup - pad to 64-byte boundary with the bit pattern 
 * 1 0* (64-bit count of bits processed, MSB-first)
 */
static void MD5Final (unsigned char digest[16], struct MD5Context *ctx)
{
	unsigned count;
	unsigned char *p;

	/* Compute number of bytes mod 64 */
	count = (ctx->bits[0] >> 3) & 0x3F;

	/* Set the first char of padding to 0x80.  This is safe since there is
	   always at least one byte free */
	p = ctx->in + count;
	*p++ = 0x80;

	/* Bytes of padding needed to make 64 bytes */
	count = 64 - 1 - count;

	/* Pad out to 56 mod 64 */
	if (count < 8)
	{
		/* Two lots of padding:  Pad the first block to 64 bytes */
		memset (p, 0, count);
		byte_reverse (ctx->in, 16);
		MD5Transform (ctx->buf, (uint32 *) ctx->in);

		/* Now fill the next block with 56 bytes */
		memset (ctx->in, 0, 56);
	}
	else
	{
		/* Pad block to 56 bytes */
		memset (p, 0, count - 8);
	}
	byte_reverse (ctx->in, 14);

	/* Append length in bits and transform */
	((uint32 *) ctx->in)[14] = ctx->bits[0];
	((uint32 *) ctx->in)[15] = ctx->bits[1];

	MD5Transform (ctx->buf, (uint32 *) ctx->in);
	byte_reverse ((unsigned char *) ctx->buf, 4);
	memcpy (digest, ctx->buf, 16);
	memset (ctx, 0, sizeof (ctx));	 /* In case it's sensitive */
}



/**
 * The core of the MD5 algorithm, this alters an existing MD5 hash to
 * reflect the addition of 16 longwords of new data.  MD5Update blocks
 * the data and converts bytes into longwords for this routine.
 */
static void MD5Transform (uint32 buf[4], uint32 const in[16])
{
	register uint32 a, b, c, d;

/* The four core functions - F1 is optimized somewhat */
/* #define F1(x, y, z) (x & y | ~x & z) */
#define F1(x, y, z)	(z ^ (x & (y ^ z)))
#define F2(x, y, z)	F1(z, x, y)
#define F3(x, y, z)	(x ^ y ^ z)
#define F4(x, y, z)	(y ^ (x | ~z))

/* This is the central step in the MD5 algorithm. */
#define MD5STEP(f, w, x, y, z, data, s) \
( w += f(x, y, z) + data,  w = w<<s | w>>(32-s),  w += x )


	a = buf[0];
	b = buf[1];
	c = buf[2];
	d = buf[3];

	MD5STEP (F1, a, b, c, d, in[0] + 0xd76aa478, 7);
	MD5STEP (F1, d, a, b, c, in[1] + 0xe8c7b756, 12);
	MD5STEP (F1, c, d, a, b, in[2] + 0x242070db, 17);
	MD5STEP (F1, b, c, d, a, in[3] + 0xc1bdceee, 22);
	MD5STEP (F1, a, b, c, d, in[4] + 0xf57c0faf, 7);
	MD5STEP (F1, d, a, b, c, in[5] + 0x4787c62a, 12);
	MD5STEP (F1, c, d, a, b, in[6] + 0xa8304613, 17);
	MD5STEP (F1, b, c, d, a, in[7] + 0xfd469501, 22);
	MD5STEP (F1, a, b, c, d, in[8] + 0x698098d8, 7);
	MD5STEP (F1, d, a, b, c, in[9] + 0x8b44f7af, 12);
	MD5STEP (F1, c, d, a, b, in[10] + 0xffff5bb1, 17);
	MD5STEP (F1, b, c, d, a, in[11] + 0x895cd7be, 22);
	MD5STEP (F1, a, b, c, d, in[12] + 0x6b901122, 7);
	MD5STEP (F1, d, a, b, c, in[13] + 0xfd987193, 12);
	MD5STEP (F1, c, d, a, b, in[14] + 0xa679438e, 17);
	MD5STEP (F1, b, c, d, a, in[15] + 0x49b40821, 22);

	MD5STEP (F2, a, b, c, d, in[1] + 0xf61e2562, 5);
	MD5STEP (F2, d, a, b, c, in[6] + 0xc040b340, 9);
	MD5STEP (F2, c, d, a, b, in[11] + 0x265e5a51, 14);
	MD5STEP (F2, b, c, d, a, in[0] + 0xe9b6c7aa, 20);
	MD5STEP (F2, a, b, c, d, in[5] + 0xd62f105d, 5);
	MD5STEP (F2, d, a, b, c, in[10] + 0x02441453, 9);
	MD5STEP (F2, c, d, a, b, in[15] + 0xd8a1e681, 14);
	MD5STEP (F2, b, c, d, a, in[4] + 0xe7d3fbc8, 20);
	MD5STEP (F2, a, b, c, d, in[9] + 0x21e1cde6, 5);
	MD5STEP (F2, d, a, b, c, in[14] + 0xc33707d6, 9);
	MD5STEP (F2, c, d, a, b, in[3] + 0xf4d50d87, 14);
	MD5STEP (F2, b, c, d, a, in[8] + 0x455a14ed, 20);
	MD5STEP (F2, a, b, c, d, in[13] + 0xa9e3e905, 5);
	MD5STEP (F2, d, a, b, c, in[2] + 0xfcefa3f8, 9);
	MD5STEP (F2, c, d, a, b, in[7] + 0x676f02d9, 14);
	MD5STEP (F2, b, c, d, a, in[12] + 0x8d2a4c8a, 20);

	MD5STEP (F3, a, b, c, d, in[5] + 0xfffa3942, 4);
	MD5STEP (F3, d, a, b, c, in[8] + 0x8771f681, 11);
	MD5STEP (F3, c, d, a, b, in[11] + 0x6d9d6122, 16);
	MD5STEP (F3, b, c, d, a, in[14] + 0xfde5380c, 23);
	MD5STEP (F3, a, b, c, d, in[1] + 0xa4beea44, 4);
	MD5STEP (F3, d, a, b, c, in[4] + 0x4bdecfa9, 11);
	MD5STEP (F3, c, d, a, b, in[7] + 0xf6bb4b60, 16);
	MD5STEP (F3, b, c, d, a, in[10] + 0xbebfbc70, 23);
	MD5STEP (F3, a, b, c, d, in[13] + 0x289b7ec6, 4);
	MD5STEP (F3, d, a, b, c, in[0] + 0xeaa127fa, 11);
	MD5STEP (F3, c, d, a, b, in[3] + 0xd4ef3085, 16);
	MD5STEP (F3, b, c, d, a, in[6] + 0x04881d05, 23);
	MD5STEP (F3, a, b, c, d, in[9] + 0xd9d4d039, 4);
	MD5STEP (F3, d, a, b, c, in[12] + 0xe6db99e5, 11);
	MD5STEP (F3, c, d, a, b, in[15] + 0x1fa27cf8, 16);
	MD5STEP (F3, b, c, d, a, in[2] + 0xc4ac5665, 23);

	MD5STEP (F4, a, b, c, d, in[0] + 0xf4292244, 6);
	MD5STEP (F4, d, a, b, c, in[7] + 0x432aff97, 10);
	MD5STEP (F4, c, d, a, b, in[14] + 0xab9423a7, 15);
	MD5STEP (F4, b, c, d, a, in[5] + 0xfc93a039, 21);
	MD5STEP (F4, a, b, c, d, in[12] + 0x655b59c3, 6);
	MD5STEP (F4, d, a, b, c, in[3] + 0x8f0ccc92, 10);
	MD5STEP (F4, c, d, a, b, in[10] + 0xffeff47d, 15);
	MD5STEP (F4, b, c, d, a, in[1] + 0x85845dd1, 21);
	MD5STEP (F4, a, b, c, d, in[8] + 0x6fa87e4f, 6);
	MD5STEP (F4, d, a, b, c, in[15] + 0xfe2ce6e0, 10);
	MD5STEP (F4, c, d, a, b, in[6] + 0xa3014314, 15);
	MD5STEP (F4, b, c, d, a, in[13] + 0x4e0811a1, 21);
	MD5STEP (F4, a, b, c, d, in[4] + 0xf7537e82, 6);
	MD5STEP (F4, d, a, b, c, in[11] + 0xbd3af235, 10);
	MD5STEP (F4, c, d, a, b, in[2] + 0x2ad7d2bb, 15);
	MD5STEP (F4, b, c, d, a, in[9] + 0xeb86d391, 21);

	buf[0] += a;
	buf[1] += b;
	buf[2] += c;
	buf[3] += d;
#undef F1
#undef F2
#undef F3
#undef F4
#undef MD5STEP
}


/**
 *
 */
#ifdef HAPLO_DEBUG_MD5
static void MD5Rewrite(unsigned char digest[32])
{
	int i;
	const char hex[16]={
		'0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
		'A', 'B', 'C', 'D', 'E', 'F' };
	
	i=16;
	do
	{
		i -= 1;
		digest[2*i]   = hex[digest[i] / 16];
		digest[2*i+1] = hex[digest[i] % 16];
	} while(i>0);

	return;
}
#endif /* HAPLO_DEBUG_MD5 */


/**
 *
 */
static int term_setup(struct termios *term)
{
	int erase=0;
	
	if (tcgetattr(STDIN_FILENO, term) == 0)
	{
		struct termios new;
	
		
		new=*term;
	
		new.c_lflag &= ~(ICANON | ECHO);
		new.c_cc[VMIN] 	= 1;
		new.c_cc[VTIME] = 0;
		erase = new.c_cc[VERASE];
		tcsetattr(STDIN_FILENO, TCSAFLUSH, &new);
	}

	return(erase);
}


/**
 *
 */
static void term_restore(const struct termios *term)
{
	tcsetattr(STDIN_FILENO, TCSAFLUSH, term);
	
	return;
}


/**
 *
 */
static void get_string(unsigned char *buffer, size_t len, int echo, int erase)
{
	while(1)
	{
		size_t pos=0;
		int c;
		
		c=fgetc(stdin);
		if (c == erase)
		{
			if (pos > 0)
			{
				fputs("\b \b", stdout);
				buffer[pos]='\0';
				pos--;
			}	
			else
				fputc('\a', stdout);
			continue;
		}

		if (c == '\n')
		{
			fputc('\n', stdout);
			buffer[pos]='\0';
			
			break;
		}
		
		if (pos < len-1)
		{
			if (echo)
				fputc(c, stdout);
			else
				fputc('*', stdout);
			fflush(stdout);
			buffer[pos]=(unsigned char)c;
			pos++;
		}
		else
			fputc('\a', stdout);		
	}

	return;
}


/**
 *
 */
void identity_get(identity_t *id)
{
	struct termios term;
	int erase;
	struct MD5Context MD5;

	memset(id, '\0', sizeof(*id));
	MD5Init(&MD5);
	erase=term_setup(&term);

	fputs("Login: ", stdout); fflush(stdout);
	get_string(id->login, LOGIN_SZ, 1, erase);

	fputs("Password: ", stdout); fflush(stdout);	
	get_string(id->passwd, PASSWD_SZ, 0, erase);

	MD5Update(&MD5, id->passwd, PASSWD_SZ);
	MD5Final(id->passwd, &MD5);

#ifdef HAPLO_DEBUG_MD5
	MD5Rewrite(id->passwd);
	id->passwd[PASSWD_SZ-1] = '\0';
#endif /* HAPLO_DEBUG_MD5 */

	term_restore(&term);
}


/**
 *
 */
int identity_cmp(const identity_t *id1, const identity_t *id2)
{
	int status;
	
	status=strncmp((char *)id1->login, (char *)id2->login, LOGIN_SZ);
	if (status == 0)
		status=memcmp(id1->passwd, id2->passwd, PASSWD_SZ);

	return(status);
}




