/*
 * $Id: haplo.c,v 1.20 2003/12/01 09:50:14 nicoo Exp $
 *
 *
 * Copyright (C) 1999, 2000, 2001 Nicolas LAURENT
 * This file is part of `Haplo'
 * 
 *
 * `Haplo'  is free software;  you can  redistribute  it and/or modify it
 * under the terms of the GNU Library General Public License as published
 * by the Free Software Foundation;  either version 2  of the License, or
 * (at your option) any later version.
 *
 * `Haplo' is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the  implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the  GNU General Public License
 * for more details.
 * 
 * You should have  received  a copy of the   GNU General Public  License
 * along with `Haplo'.  If not, write to  the
 *
 *                                        Free Software Foundation,  Inc.
 *                                        675 Mass Ave, Cambridge, MA
 *                                        02139, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#	include "config.h"
#endif
#include "version.h"

#include <errno.h>
#ifdef HAVE_LOCALE_H
#	include <locale.h>
#endif
#include <signal.h>
#include <stdio.h>
#include <string.h>
#ifdef HAVE_SYS_RESOURCE_H
#	include <sys/resource.h>
#endif /* HAVE_SYS_RESOURCE_H */
#include <sys/stat.h>
#ifdef HAVE_SYS_TIME_H
#	include <sys/time.h>
#endif /* HAVE_SYS_TIME_H */
#include <sys/types.h>
#include <time.h>
#ifdef HAVE_UNISTD_H
#	include <unistd.h>
#endif /* HAVE_UNISTD_H */
#ifdef HAVE_LIBREADLINE
#	include <readline/readline.h>
#	include <readline/history.h>
#endif /* HAVE_LIBREADLINE */

#include "client.h"
#include "haplo.h"
#include "hgetopt.h"
#include "server.h"


/*-----------------------------------------------------------------------------
                             P R O T O T Y P E S
-----------------------------------------------------------------------------*/

static void welcome(void);
static int print_resources(void);
static void quit(int code);
void do_work(haplo_param_t *param);
int main(int argc, char *const argv[]);
static int version(void);
static int config(void);
static int cflags(void);
static int libs(void);
static int setcolors(haplo_param_t *param, const char *optarg);
static int xhaplo(void);
static int limit_cpu(const char *value);
static int limit_mem(const char *value);


/*-----------------------------------------------------------------------------
                         I M P L E M E N T A T I O N
-----------------------------------------------------------------------------*/

/**
 * Display a greeting message
 */
static void welcome(void)
{
	putchar('\n');
	haplo_centered("<  H A P L O  >");
#ifdef ENABLE_NLS
	haplo_centered("Version %d.%d.%d (%s)", HAPLO_MAJOR, HAPLO_MINOR,
			   HAPLO_PL, _("En"));
#else /* ! ENABLE_NLS */
	haplo_centered("Version %d.%d.%d", HAPLO_MAJOR, HAPLO_MINOR, HAPLO_PL);
#endif /* ENABLE_NLS */
	putchar('\n');
	haplo_centered("Copyright (C) 1998-2003 Nicolas LAURENT");
	putchar('\n');
	putchar('\n');

	return;
}


/**
 * Display resources usage
 */
static int print_resources(void)
{
#ifdef WIN32
	__FIXME__;
#else
	struct rusage	usage;

	if (getrusage(RUSAGE_SELF, &usage)<0)
		haplo_warning(_("Execution statistics unavailable."));
	else
	{
		double	usr,
			sys;
		usr=(double)(usage.ru_utime.tv_sec+usage.ru_utime.tv_usec/1E6);
		sys=(double)(usage.ru_stime.tv_sec+usage.ru_stime.tv_usec/1E6);
		haplo_info("CPU   total = %.2fs", usr+sys);
		haplo_info("      user  = %.2fs (%.1f%%)",
			   usr, usr/(usr+sys)*100.0);
		haplo_info("      sys   = %.2fs (%.1f%%)",
			   sys, sys/(usr+sys)*100.0);
	}
#endif
	return(HAPLO_MEMORY_USAGE());
}


/**
 * simply quit the main process
 *
 * @param code is the exit(2) status
 */
static void quit(int code)
{
	int	status=code;
	
	haplo_fini();

	switch(code)
	{
	case 0:
		status=(print_resources())?(EXIT_FAILURE):(EXIT_SUCCESS);
		break;
#ifdef SIGXCPU
	case SIGXCPU:
		haplo_info(_("CPU time limit exceeded"));
		/* do not break here */
#endif /* SIGXCPU */
	default:
		(void)print_resources();
	}

	haplo_info(_("End of processing."));

	exit(status);
}


/**
 *
 */
void do_work(haplo_param_t *param)
{
	welcome();
	haplo_main(param);
	quit(0);
}


/**
 * Entry point
 *
 * @param argc
 * @param argv
 */
int main(int argc, char *const argv[])
{
#ifdef ENABLE_NLS
	char *path;
#endif
	int		ind;
	haplo_param_t	param;
	option_t	options[]={
		{ "-I" , N_("Add <optarg> to scripts search-path."),
		  OPTARG, OPT_F(haplo_parser_path_add) },
		{ "-L" , N_("Add <optarg> to plugins search-path."),
		  OPTARG, OPT_F(haplo_plugins_path_add) },
		{ "-cflags", N_("Print compilation flags and exit."),
		  NONE, OPT_F(cflags) },
		{ "-client", N_("Launch haplo as client."),
		  PARAM, OPT_F(client) },
		{ "-colors", N_("Set <arg> as colors scheme."),
		  PARAM, OPT_F(setcolors) },
		{ "-config", N_("Print configuration informations and exit."),
		  NONE, OPT_F(config) },
		{ "-cpu", N_("Limit CPU usage to <optarg> seconds."),
		  OPTARG, OPT_F(limit_cpu) },
		{ "-help", N_("Print this message."),
		  OPTIONS, OPT_F(usage) },
		{ "-libs", N_("Print linkage flags and exit."),
		  NONE, OPT_F(libs) },
		{ "-mem", N_("Limit memory usage to <optarg> Mb."),
		  OPTARG, OPT_F(limit_mem) },
		{ "-server", N_("Launch haplo as server."),
		  PARAM, OPT_F(server) },
		{ "-version", N_("Print version informations and exit."),
		  NONE, OPT_F(version) },
		{ "-xhaplo", N_("For use with XHaplo."),
		  NONE, OPT_F(xhaplo) },
		{ NULL, NULL, NONE,  NULL }
		
	};
	
#ifdef ENABLE_NLS
	path=haplo_param_path("share/locale");
	bindtextdomain(PACKAGE, path);
	HAPLO_FREE(path);
	if (setlocale(LC_ALL, "") == NULL)
		haplo_error("Locale are not supported in this environement");
	setlocale(LC_NUMERIC, "C");
#endif
	haplo_param_default(&param);
	
	hgetopt(&param, options, argc, argv, 1);
	
	haplo_init(&param);

	ind=hgetopt(&param, options, argc, argv, 2);

	if (ind > 0)
		param.filename=argv[ind];

	do_work(&param);
	/* never reached */
	return(0);	/* avoid warning on non-gcc compiler */
}


/**
 * Display version message
 *
 * @return 1
 */
static int version(void)
{
	printf("Haplo %d.%d.%d\n", HAPLO_MAJOR, HAPLO_MINOR, HAPLO_PL);
	puts("\nCopyright (C) 1998-2003 Nicolas LAURENT");
	puts("This is free software; see the sources for copying conditions.");
	puts("There is NO warranty; not even for MERCHANTABILITY or "
	     "FITNESS FOR A PARTICULAR");
	fputs("PURPOSE.", stdout);
	
	
	return(1);
}


/**
 * Display config message
 *
 * @return 1
 */
static int config(void)
{
	char *path;
	
	printf(_("Version..............: %d.%d.%d ($Name:  $)\n"),
	       HAPLO_MAJOR, HAPLO_MINOR, HAPLO_PL);
	printf(_("Build date...........: %s (%s)\n"), __DATE__, __TIME__);
	printf(_("Architecture.........: %s\n"), HAPLO_HOST_OS);
	printf(_("Modules support......: "));
	switch(HAPLO_PLUGINS_IMPL)
	{
	case HAPLO_PLUGINS_IMPL_NONE:
		puts(_("None"));
		break;
	case HAPLO_PLUGINS_IMPL_DL:
		puts(_("Solaris style (DL)"));
		break;
	case HAPLO_PLUGINS_IMPL_DLD:
		puts(_("HP/UX style (DLD)"));
		break;
	default:
		puts(_("Unknown"));
	}
	printf(_("i18n.................: "));
#ifdef ENABLE_NLS
	printf(_("yes\n"));
	path=haplo_param_path("share/locale");
	printf(_("i18n data directory..: %s\n"), path);
	HAPLO_FREE(path);
#else
	printf("no\n");
#endif /* ENABLE_NLS */
	printf(_("Options..............: %s\n"), HAPLO_OPTIONS);
	path=haplo_param_path(HAPLO_PLUGINS_DIR);
	printf(_("Plugins directory....: %s\n"), path);
	HAPLO_FREE(path);
	path=haplo_param_path(HAPLO_SCRIPTS_DIR);
	printf(_("Scripts directory....: %s"),	 path);
	HAPLO_FREE(path);

	return(1);
}


/**
 * Display cflags message
 *
 * @return 1
 */
static int cflags(void)
{
	char	*path;
	
	path=haplo_param_path(HAPLO_INCLUDE_DIR);
	printf("-I%s ", path);
	HAPLO_FREE(path);
	
	return(1);
}


/**
 * display libs message
 *
 * @return 1
 */
static int libs(void)
{
	char *path;
	
	path=haplo_param_path(HAPLO_PLUGINS_DIR);
	printf("-L%s -lhaplo -lm ", path);
	HAPLO_FREE(path);
	return(1);
}


/**
 * set colors schemes
 *
 * @param param
 * @param optarg
 * @return 0 or -1 in case of error
 */
static int setcolors(haplo_param_t *param, const char *optarg)
{
	if (strcmp(optarg, "ansi")==0)
	{
		param->colors = HAPLO_COLORS_ANSI;
	} else if (strcmp(optarg, "flash")==0)
	{
		param->colors = HAPLO_COLORS_ANSI | HAPLO_COLORS_FLASH;
	} else if (strcmp(optarg, "none")==0)
	{
		param->colors = HAPLO_COLORS_NONE;
	} else {
		fputs(_("Option `-colors' expected one of the "
			"following values:\n"
			"  `none'\tno colors at all\n"
			"  `ansi'\tuse simple ansi colors\n"
			"  `flash'\tuse all ansi colors\n"), stderr);
		return(-1);
	}
	
	
	return(0);
}


/**
 * Set option for use with xhaplo
 *
 * @return 0
 */
static int xhaplo(void)
{
	/*
	 * Cook I/O
	 */
	setvbuf(stdout, (char *)NULL, _IOLBF, 0);
	setvbuf(stdin, (char *)NULL, _IOLBF, 0);

	return(0);
}


/**
 *
 */
#ifdef SIGXCPU
static int limit_cpu(const char *value)
{
	struct rlimit limit;

	if (getrlimit(RLIMIT_CPU, &limit) == 0)
	{
		rlim_t sec;
		
		sec=atoi(value);
		if (sec > limit.rlim_max)
			sec = limit.rlim_max;
		limit.rlim_cur=sec;
		signal(SIGXCPU, quit);
		
		setrlimit(RLIMIT_CPU, &limit);
			
	} else
		perror("getrlimit()");
	return(0);
}
#endif /* SIGXCPU */


/**
 *
 */
static int limit_mem(const char *value)
{
	struct rlimit limit;
	
	if (getrlimit(RLIMIT_DATA, &limit))
	{
		rlim_t bytes;
		
		bytes=atoi(value)*1024*1024;

		if (bytes > limit.rlim_max)
			bytes = limit.rlim_max;
		limit.rlim_cur=bytes;

		setrlimit(RLIMIT_DATA, &limit);
	} else
		perror("getrlimit()");
	
	return(0);
}
