/* meta-protocol.h: Opcode definitions for the meta protocol.
 * Copyright (C) 2018 Julian Graham
 *
 * gzochi is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <glib.h>
#include <sys/time.h>

#include "oids.h"

#ifndef GZOCHID_META_PROTOCOL_H
#define GZOCHID_META_PROTOCOL_H

/* The following byte single-constants are "opcodes" for the meta protocol, and
   correspond to the different messages that can be exchanged between the client
   (gzochi application server) and the server (gzochi meta server) participating
   in the protocol. The format of the message payload that follows each opcode 
   is opcode-specific and is described below. */

/* The following opcodes are for messages sent from the client to the server. */

/*
  Login to the meta server and pass some configuration data from the client.
  Format:
  
  1 byte: Meta protocol version. (0x02)
  `NULL'-terminated string: gzochid admin server base URL
    or 1 `NULL' byte if the client is not running an admin web console
*/

#define GZOCHID_META_PROTOCOL_LOGIN 0x10

/*
  Reserve a block of object ids from the data server. Format:
  
  `NULL'-terminated string: Name of the requesting game application
*/

#define GZOCHID_DATA_PROTOCOL_REQUEST_OIDS 0x20

/*
  Request from the data server the bytes associated with a specified key, 
  establishing a point lock on that key. Format:
  
  `NULL'-terminated string: Name of the requesting game application
  `NULL'-terminated string: Name of the target store
  1 byte (0x00 or 0x01) indicating whether the object should be locked for write
  2 bytes: The big-endian encoding of the length of the key; the key bytes 
    follow
*/

#define GZOCHID_DATA_PROTOCOL_REQUEST_VALUE 0x21

/*
  Request from the data server the key that falls directly after the specified 
  key establishing a range lock on the interval between the two keys. Format:
  
  `NULL'-terminated string: Name of the requesting game application
  `NULL'-terminated string: Name of the target store
  2 bytes: The big-endian encoding of the length of the key
    Two zeros indicates a request for the first key, else the key bytes follow
*/

#define GZOCHID_DATA_PROTOCOL_REQUEST_NEXT_KEY 0x22

/* Transmit a series of object and binding modifications to the data server for
   persistence as a single, transactional unit. See 
   `gzochid_data_protocol_changeset_write' below for format details. */

#define GZOCHID_DATA_PROTOCOL_SUBMIT_CHANGESET 0x30

/*
  Begin the submission of a changeset spanning one or more 
  `GZOCHID_DATA_PROTOCOL_SUBMIT_CHANGESET' messages.
  
  `NULL'-terminated string: Name of the submitting game application
  8 bytes: A connection-unique identifier for the changeset
 */
#define GZOCHID_DATA_PROTOCOL_BEGIN_CHANGESET 0x32

/*
  Complete the submission of a changeset previously initiated via a 
  `GZOCHID_DATA_PROTOCOL_BEGIN_CHANGESET' message.
  
  `NULL'-terminatedstring: Name of the submitting game application
  8 bytes: The connection-unique identifier for the changeset
*/

#define GZOCHID_DATA_PROTOCOL_END_CHANGESET 0x34

/*
  Release the point lock on the specified object key. Format:
  
  `NULL'-terminated string: Name of the requesting game application
  `NULL'-terminated string: Name of the target store
  2 bytes: The big-endian encoding of the length of the key; the key bytes 
    follow
*/  

#define GZOCHID_DATA_PROTOCOL_RELEASE_KEY 0x40

/*
  Release the range lock on the key interval between between the specified start
  and end keys. specified binding name. Format:
  
  `NULL'-terminated string: Name of the requesting game application
  `NULL'-terminated string: Name of the target store
  2 bytes: The big-endian encoding of the length of the lower key
    Two zeros indicates the beginning of the key space, else the key bytes 
    follow
  2 bytes: The big-endian encoding of the length of the upper key
    Two zeros indicates the end of the key space, else the key bytes follow
*/  

#define GZOCHID_DATA_PROTOCOL_RELEASE_KEY_RANGE 0x42

/*
  Notify the meta server that a session has connected to an application server
  node.

  `NULL'-terminated string: Name of the game application that owns the session
  8 bytes: The big-endian encoding of the target session oid
*/

#define GZOCHID_SESSION_PROTOCOL_SESSION_CONNECTED 0x60

/*
  Notify the meta server that a session has disconnected from an application 
  server node.

  `NULL'-terminated string: Name of the game application that owns the session
  8 bytes: The big-endian encoding of the target session oid
*/

#define GZOCHID_SESSION_PROTOCOL_SESSION_DISCONNECTED 0x61

/*
  Relay a disconnect signal via the meta server for the target session.

  `NULL'-terminated string: Name of the game application that owns the session
  8 bytes: The big-endian encoding of the target session oid
*/

#define GZOCHID_SESSION_PROTOCOL_RELAY_DISCONNECT_FROM 0x62

/*
  Relay a message via the meta server for the target session.

  `NULL'-terminated string: Name of the game application that owns the session
  8 bytes: The big-endian encoding of the target session oid
  2 bytes: The big-endian encoding of the length of the message; the message
    bytes follow
*/

#define GZOCHID_SESSION_PROTOCOL_RELAY_MESSAGE_FROM 0x64

/*
  Relay via the metaserver a request to join the target session to the target 
  channel.
  
  `NULL'-terminated string: Name of the game application that owns the channel.
  8 bytes: The big-endian encoding of the target channel oid
  8 bytes: The big-endian encoding of the target session oid
 */

#define GZOCHID_CHANNEL_PROTOCOL_RELAY_JOIN_FROM 0x70

/*
  Relay via the metaserver a request to remove the target session from the 
  target channel.

  `NULL'-terminated string: Name of the game application that owns the channel.
  8 bytes: The big-endian encoding of the target channel oid
  8 bytes: The big-endian encoding of the target session oid
 */

#define GZOCHID_CHANNEL_PROTOCOL_RELAY_LEAVE_FROM 0x72

/*
  Relay via the metaserver a message for all servers to be broadcast over the
  target channel.

  `NULL'-terminated string: Name of the game application that owns the channel.
  8 bytes: The big-endian encoding of the target channel oid
  2 bytes: The big-endian encoding of the length of the message; the message 
    bytes follow
 */

#define GZOCHID_CHANNEL_PROTOCOL_RELAY_MESSAGE_FROM 0x74

/*
  Notifies all servers via the metaserver to close their local representations
  of the target channel.

  `NULL'-terminated string: Name of the game application that owns the channel.
  8 bytes: The big-endian encoding of the target channel oid
*/

#define GZOCHID_CHANNEL_PROTOCOL_RELAY_CLOSE_FROM 0x76

/*
  Notifies the metaserver that an application server is ready to begin serving
  an application.

  `NULL'-terminated string: Name of the game application that is ready to be
  served.
 */

#define GZOCHID_TASK_PROTOCOL_START_APP 0x80

/*
  Notifies the metaserver that an application server which has previously been
  selected as the task resubmission leader for an application has finished 
  resubmitting the application's durable tasks.
  
  `NULL'-terminated string: Name of the game application that is ready to be
  served.
 */

#define GZOCHID_TASK_PROTOCOL_COMPLETE_RESUBMISSION 0x82

/*
  Notifies the metaserver that a durable task has been submitted by the 
  originating node.

  `NULL'-terminated string: Name of the game application to which the task
  belongs.
  8 bytes: The big-endian encoding of the durable task handle oid.
*/

#define GZOCHID_TASK_PROTOCOL_SUBMIT_TASK 0x84

/*
  Notifies the metaserver that a durable task has completed on the originating
  node.

  `NULL'-terminated string: Name of the game application to which the task
  belongs.
  8 bytes: The big-endian encoding of the durable task handle oid.
*/

#define GZOCHID_TASK_PROTOCOL_TASK_COMPLETED 0x86

/*
  Notifies the metaserver that a durable task has been canceled: It should not
  run again on the node to which it is currently mapped and should not be 
  mapped to any other nodes.

  `NULL'-terminated string: Name of the game application to which the task
  belongs.
  8 bytes: The big-endian encoding of the durable task handle oid.
*/

#define GZOCHID_TASK_PROTOCOL_CANCEL_TASK 0x88

/* The following opcodes are for messages sent from the server to the client. */

/*
  1 byte: Meta protocol version. (0x02)
  `NULL'-terminated string: gzochi-metad admin server base URL
    or 1 `NULL' byte if the meta server is not running an admin web console
 */

#define GZOCHID_META_PROTOCOL_LOGIN_RESPONSE 0x11

/*
  Describes a block of oids reserved for the client. Format:

  `NULL'-terminated string: Name of the requesting game application
  8 bytes: The big-endian encoding of the first oid
  2 bytes: The big-endian encoding of the length of the block in oids
*/

#define GZOCHID_DATA_PROTOCOL_OIDS_RESPONSE 0x50

/* Contains the serialized object data stored at a particular oid. See 
   `gzochid_data_protocol_value_response_write' below for format details. */

#define GZOCHID_DATA_PROTOCOL_VALUE_RESPONSE 0x51

/* Contains the object id bound to a particular name. See 
   `gzochid_data_protocol_key_response_write' below for format details. */

#define GZOCHID_DATA_PROTOCOL_NEXT_KEY_RESPONSE 0x52

/* 
  Directs the target server to disconnect the specified client session.
   
  `NULL'-terminated string: Name of the game application that owns the session
  8 bytes: The big-endian encoding of the target session oid
*/

#define GZOCHID_SESSION_PROTOCOL_RELAY_DISCONNECT_TO 0x63

/*
  Relays a message to the session connected to the target server.
  
  `NULL'-terminated string: Name of the game application that owns the session
  8 bytes: The big-endian encoding of the target session oid
  2 bytes: The big-endian encoding of the length of the message; the message 
    bytes follow
*/

#define GZOCHID_SESSION_PROTOCOL_RELAY_MESSAGE_TO 0x65

/*
  Directs the target server to join the specified session to the specified
  channel.

  `NULL'-terminated string: Name of the game application that owns the channel.
  8 bytes: The big-endian encoding of the target channel oid
  8 bytes: The big-endian encoding of the target session oid
*/

#define GZOCHID_CHANNEL_PROTOCOL_RELAY_JOIN_TO 0x71

/*
  Directs the target server to remove the specified session from the specified
  channel.

  `NULL'-terminated string: Name of the game application that owns the channel.
  8 bytes: The big-endian encoding of the target channel oid
  8 bytes: The big-endian encoding of the target session oid
*/

#define GZOCHID_CHANNEL_PROTOCOL_RELAY_LEAVE_TO 0x73

/*
  Directs the target server to publish the specified message on the specified
  channel.

  `NULL'-terminated string: Name of the game application that owns the channel.
  8 bytes: The big-endian encoding of the target channel oid
  2 bytes: The big-endian encoding of the length of the message; the message 
    bytes follow
 */

#define GZOCHID_CHANNEL_PROTOCOL_RELAY_MESSAGE_TO 0x75

/*
  Directs the target server to close the specified channel.

  `NULL'-terminated string: Name of the gane application that owns the channel.
  8 bytes: The big-endian encoding of the target channel oid
 */

#define GZOCHID_CHANNEL_PROTOCOL_RELAY_CLOSE_TO 0x77

/*
  Directs the target server to resubmit all durable tasks for an application.
  
  `NULL'-terminated string: Name of the game application for which durable 
  tasks should be resubmitted.
 */

#define GZOCHID_TASK_PROTOCOL_RESUBMIT_TASKS 0x81

/*
  Notifies the target server that all durable tasks have been resubmitted. 
  Following receipt of this message, client connections may be accepted and
  transactions and tasks may be submitted to the metaserver.

  `NULL'-terminated string: Name of the game application that is allowed to 
  start.
*/

#define GZOCHID_TASK_PROTOCOL_APP_STARTED 0x83

/*
  Notifies the target server to add the specified task id to its local task
  queue.

  `NULL'-terminated string: Name of the game application to which the target
  durable task belongs.
  8 bytes: The big-endian encoding of the target task oid
*/ 

#define GZOCHID_TASK_PROTOCOL_ASSIGN_TASK 0x85

/*
  Notifies the target server to remove the specified task id from its local 
  task queue.

  `NULL'-terminated string: Name of the game application to which the target
  durable task belongs.
  8 bytes: The big-endian encoding of the target task oid
*/

#define GZOCHID_TASK_PROTOCOL_UNASSIGN_TASK 0x87

#endif /* GZOCHID_META_PROTOCOL_H */
