/**
 *
 * @file     bmpctrl.c
 * @author   Aleix Conchillo Flaque <aleix@member.fsf.org>
 * @date     Thu Aug 07, 2003 21:16
 * @brief    Beep Media Player control functions
 *
 * $Id: bmpctrl.c,v 1.4 2004/12/30 00:24:11 aleix Exp $
 *
 * @if copyright
 *
 * Copyright (C) 2003, 2004 Aleix Conchillo Flaque
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * @endif
 */

#include "playerctrl.h"

#include "callbacks.h"
#include "interface.h"

#include <bmp/beepctrl.h>

#include <stdlib.h>
#include <math.h>
#include <time.h>


/* This function has been copied from the gkrellmms plugin. */
void
player_start(XMMSData *xmms)
{
    static gchar *msg_error = _("Unable to launch Beep Media Player.");
    static gchar *exec_cmd = "beep-media-player &";
    gint timer;
    time_t lt;

    if (player_is_running(xmms))
    {
        return;
    }

    system(exec_cmd);

    timer = time(&lt);
    while (!player_is_running(xmms) && ((time(&lt) - timer) < 4))
    {
        usleep(0);
    }

    if (!player_is_running(xmms))
    {
        gui_show_error(xmms, msg_error);
    }
}

gboolean
player_is_running(XMMSData *xmms)
{
    return xmms_remote_is_running(xmms->session);
}

gboolean
player_is_playing(XMMSData *xmms)
{
    return xmms_remote_is_playing(xmms->session);
}

gboolean
player_is_paused(XMMSData *xmms)
{
    return xmms_remote_is_paused(xmms->session);
}

gboolean
player_is_repeat(XMMSData *xmms)
{
    return xmms_remote_is_repeat(xmms->session);
}

gboolean
player_is_shuffle(XMMSData *xmms)
{
    return xmms_remote_is_shuffle(xmms->session);
}

gboolean
player_is_main_win(XMMSData *xmms)
{
    return xmms_remote_is_main_win(xmms->session);
}

gboolean
player_is_pl_win(XMMSData *xmms)
{
    return xmms_remote_is_pl_win(xmms->session);
}

gboolean
player_is_eq_win(XMMSData *xmms)
{
    return xmms_remote_is_eq_win(xmms->session);
}

gint
player_get_output_time(XMMSData *xmms)
{
    return xmms_remote_get_output_time(xmms->session);
}

gint
player_get_playlist_pos(XMMSData *xmms)
{
    return xmms_remote_get_playlist_pos(xmms->session);
}

gchar*
player_get_playlist_title(XMMSData *xmms, gint pos)
{
    return xmms_remote_get_playlist_title(xmms->session, pos);
}

gint
player_get_playlist_time(XMMSData *xmms, gint pos)
{
    return xmms_remote_get_playlist_time(xmms->session, pos);
}

void
player_quit(XMMSData *xmms)
{
    if (player_is_running(xmms))
    {
        xmms_remote_quit(xmms->session);
    }
}

void
player_prev(XMMSData *xmms)
{
    if (player_is_running(xmms))
    {
        xmms_remote_playlist_prev(xmms->session);
    }
    gui_update(xmms);
}

void
player_play_pause(XMMSData *xmms)
{
    if (!player_is_running(xmms))
    {
        player_start(xmms);
    }

    if (player_is_running(xmms))
    {
        if (player_is_playing(xmms)
            && !player_is_paused(xmms))
        {
            xmms_remote_pause(xmms->session);
        }
        else
        {
            xmms_remote_play(xmms->session);
        }
    }

    gui_update(xmms);
}

void
player_stop(XMMSData *xmms)
{
    if (player_is_running(xmms))
    {
        xmms_remote_stop(xmms->session);
    }
    gui_update(xmms);
}

void
player_next(XMMSData *xmms)
{
    if (player_is_running(xmms))
    {
        xmms_remote_playlist_next(xmms->session);
    }
    gui_update(xmms);
}

void
player_eject(XMMSData *xmms)
{
    if (player_is_running(xmms))
    {
        xmms_remote_eject(xmms->session);
    }
    gui_update(xmms);
}

void
player_seek(XMMSData *xmms, gdouble t)
{
    if (player_is_running(xmms))
    {
        gint pos;
        gint new_time;
        gint total_time;

        pos = player_get_playlist_pos(xmms);
        total_time = player_get_playlist_time(xmms, pos);
        new_time = ceil(total_time * t) - 1;
        xmms_remote_jump_to_time(xmms->session, new_time);
    }
    gui_update(xmms);
}

void
player_repeat(XMMSData *xmms, gboolean active)
{
    if (player_is_running(xmms) && (player_is_repeat(xmms) != active))
    {
        xmms_remote_toggle_repeat(xmms->session);
    }
}

void
player_shuffle(XMMSData *xmms, gboolean active)
{
    if (player_is_running(xmms) && (player_is_shuffle(xmms) != active))
    {
        xmms_remote_toggle_shuffle(xmms->session);
    }
}

void
player_main_win(XMMSData *xmms, gboolean toggle)
{
  /* BMP quits when hiding main window */
#if 0
    if (player_is_running(xmms))
    {
        xmms_remote_main_win_toggle(xmms->session, toggle);
    }
#endif
}

void
player_pl_win(XMMSData *xmms, gboolean toggle)
{
    if (player_is_running(xmms))
    {
        xmms_remote_pl_win_toggle(xmms->session, toggle);
    }
}

void
player_eq_win(XMMSData *xmms, gboolean toggle)
{
    if (player_is_running(xmms))
    {
        xmms_remote_eq_win_toggle(xmms->session, toggle);
    }
}

void
player_show_hide(XMMSData *xmms)
{
    if (player_is_running(xmms))
    {
        gboolean show_hide;

        show_hide = !(player_is_main_win(xmms) | player_is_pl_win(xmms)
                      | player_is_eq_win(xmms));

        player_main_win(xmms, show_hide);
        player_pl_win(xmms, show_hide);
        player_eq_win(xmms, show_hide);
    }
}

void
player_preferences(XMMSData *xmms)
{
    if (player_is_running(xmms))
    {
        xmms_remote_show_prefs_box(xmms->session);
    }
}

void
player_lower_volume(XMMSData *xmms)
{
    gint new_volume;

    if (player_is_running(xmms))
    {
        new_volume = xmms_remote_get_main_volume(xmms->session) - 3;
        if(new_volume < 0)
        {
            new_volume = 0;
        }
        xmms_remote_set_main_volume(xmms->session, new_volume );
    }
}

void
player_raise_volume(XMMSData *xmms)
{
    gint new_volume;

    if (player_is_running(xmms))
    {
        new_volume = xmms_remote_get_main_volume(xmms->session) + 3;
        if(new_volume < 0)
        {
            new_volume = 0;
        }
        xmms_remote_set_main_volume(xmms->session, new_volume);
    }
}
