;; test-aa-tree.scm - unit tests for aa-tree.scm
;; Copyright (C) 2017  Christopher Howard

;; This program is free software: you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program.  If not, see <http://www.gnu.org/licenses/>.

(define-module (test-aa-tree)
  #:use-module (aa-tree)
  #:use-module (oop goops)
  #:use-module (srfi srfi-1)
  #:use-module (srfi srfi-64))

(test-begin "test-aa-tree")

(test-equal
    (let ((tree
           (fold (lambda (kv t) (aa-insert t (car kv) (cdr kv))) #nil
                 '(("e" . 5)
                   ("b" . 2)
                   ("q" . 17)
                   ("h" . 8)
                   ("m" . 14)
                   ("e" . update)
                   ("f" . 6)))))
      (fold (lambda (k l) (cons (aa-search tree k) l)) '()
            (list "m" "h" "f" "e" "b" "q" "t")))
  '(#nil (17) (2) (update) (6) (8) (14)))

(test-equal
    (let* ((tree1
            (fold (lambda (kv t) (aa-insert t (car kv) (cdr kv))) #nil
                  '((4 . "four")
                    (8 . "eight")
                    (2 . "two")
                    (3 . "three"))))
           (tree2
            (fold (lambda (k t) (aa-delete t k)) tree1
                  '(8 2))))
      (fold (lambda (k l) (cons (aa-search tree2 k) l)) '()
            (list 3 2 4 8)))
  '(#nil ("four") #nil ("three")))

(test-equal
    (call-with-output-string
      (lambda (p)
        (aa-map-keys
         (lambda (k) (display k p))
         (aa-insert (aa-insert (aa-insert #nil "b" 2) "e" 5) "a" 1))))
  "abe")

(test-equal
    (call-with-output-string
      (lambda (p)
        (aa-map-values
         (lambda (v) (display v p))
         (aa-insert (aa-insert (aa-insert #nil "b" 2) "e" 5) "a" 1))))
  "125")

(test-equal
    (aa-to-list (aa-insert (aa-insert (aa-insert #nil "e" 5) "a" 1) "b" 2))
  '(("a" . 1) ("b" . 2) ("e" . 5)))

(test-equal
    (let* ((tree1
            (aa-insert
             (aa-insert
              (aa-insert #nil 3 "three") 1 "one") 5 "five"))
           (tree2 (aa-insert tree1 (list 4) "four")))
      (aa-search tree2 4))
  (list "four"))

(test-equal
    (let* ((tree1
            (aa-insert
             (aa-insert
              (aa-insert #nil 3 "three") 1 "one") 5 "five"))
           (tree2 (aa-insert tree1 4
                             (aa-insert
                              (aa-insert
                               (aa-insert #nil "b" 2) "f" 6) "c" 3)))
           (tree3 (aa-insert tree2 (list 4 "e") 5)))
      (aa-search (car (aa-search tree3 4)) "e"))
  (list 5))

(test-equal
    (let* ((tree1
            (aa-insert
             (aa-insert
              (aa-insert #nil 3 "three") 1 "one")
             5
             (aa-insert
              (aa-insert
               (aa-insert #nil "c" 3) "a" 1)
              "b"
              (aa-insert
               (aa-insert
                (aa-insert #nil 'b 2) 'e 5) 'd 4))))
           (tree2
            (aa-insert tree1 (list 5 "b" 'h) 8)))
      (aa-search (car (aa-search (car (aa-search tree2 5)) "b")) 'h))
  '(8))

(test-assert
    (let* ((tree1
            (aa-insert
             (aa-insert
              (aa-insert #nil 3 "three") 1 "one") 5 "five"))
           (tree2 (aa-delete tree1 (list 1))))
      (and
       (equal? (aa-search tree2 1) #nil)
       (equal? (aa-search tree2 5) '("five"))
       (equal? (aa-search tree2 3) '("three")))))

(test-assert
    (let* ((tree1
            (aa-insert
             (aa-insert
              (aa-insert #nil 3 "three") 1 "one") 5 "five"))
           (tree2 (aa-insert tree1 4
                             (aa-insert
                              (aa-insert
                               (aa-insert #nil "b" 2) "f" 6) "c" 3)))
           (tree3 (aa-delete tree2 (list 4 "f"))))
      (and
       (equal? (aa-search (car (aa-search tree3 4)) "b") '(2))
       (equal? (aa-search (car (aa-search tree3 4)) "f") #nil)
       (equal? (aa-search (car (aa-search tree3 4)) "c") '(3)))))

(test-assert
    (let* ((tree1
            (aa-insert
             (aa-insert
              (aa-insert #nil 3 "three") 1 "one")
             5
             (aa-insert
              (aa-insert
               (aa-insert #nil "c" 3) "a" 1)
              "b"
              (aa-insert
               (aa-insert
                (aa-insert #nil 'b 2) 'e 5) 'd 4))))
           (tree2
            (aa-delete tree1 (list 5 "b" 'b))))
      (and
       (equal?
        (aa-search (car (aa-search (car (aa-search tree2 5)) "b")) 'd) '(4))
       (equal?
        (aa-search (car (aa-search (car (aa-search tree2 5)) "b")) 'b) #nil)
       (equal?
        (aa-search (car (aa-search (car (aa-search tree2 5)) "b")) 'e) '(5)))))

(test-equal
    (let* ((tree1
            (aa-insert
             (aa-insert
              (aa-insert #nil 3 "three") 1 "one")
             5
             (aa-insert
              (aa-insert
               (aa-insert #nil "c" 3) "a" 1)
              "b"
              (aa-insert
               (aa-insert
                (aa-insert #nil 'b 2) 'e 5) 'd 4)))))
      (aa-search tree1 (list 5 "b" 'e)))
  '(5))

(test-equal
    (let* ((tree1
            (aa-insert
             (aa-insert
              (aa-insert #nil 3 "three") 1 "one")
             5
             (aa-insert
              (aa-insert
               (aa-insert #nil "c" 3) "a" 1)
              "b"
              (aa-insert
               (aa-insert
                (aa-insert #nil 'b 2) 'e 5) 'd 4)))))
      (aa-search tree1 (list 5 "q" 'e)))
  #nil)

(define-class <coin> ()
  (side #:init-value #t #:init-keyword #:side #:getter get-side))

(define-method (equal? (a <coin>) (b <coin>))
  (let ((x (get-side a))
        (y (get-side b)))
    (or (and x y)
        (and (not x)
             (not y)))))

(define-method (< (a <coin>) (b <coin>))
  (and (not (get-side a))
       (get-side b)))

(test-assert
  (let ((c1 (make <coin> #:side #t))
        (c2 (make <coin> #:side #f))
        (c3 (make <coin> #:side #t)))
    (let ((tree1
           (aa-insert
            (aa-insert
             (aa-insert #nil c1 "head1") c2 "tail1") c3 "head2")))
      (and (equal? (aa-search tree1 c1) '("head2"))
           (equal? (aa-search tree1 c2) '("tail1"))))))

(test-end "test-aa-tree")
