# Copyright (c) 2009 Ludwig Ortmann <ludwig@spline.de>
#
# This file is part of gtranscribe.
#
# gtranscribe is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# gtranscribe is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with gtranscribe.  If not, see <http://www.gnu.org/licenses/>.

import sys, time

try:
    import pygst
    pygst.require("0.10")
    import gst
except:
    print "you need at least pygst 0.10"
    sys.exit(1)

if gst.plugin_load_by_name("scaletempo") is None:
    print "you need the scaletempo gstreamer plugin"
    print "it is usually part of the gst-plugins-bad package"
    #sys.exit(1)

class scaletempo_sink(gst.Bin):
    def __init__(self):
        gst.Bin.__init__(self)
        scaletempo = gst.element_factory_make('scaletempo', 'scaletempo')
        audioconvert    = gst.element_factory_make('audioconvert', 'audioconvert')
        audioresample   = gst.element_factory_make('audioresample', 'audioresample')
        autoaudiosink  = gst.element_factory_make('autoaudiosink', 'autoaudiosink')

        self.add(scaletempo, audioconvert, audioresample,
                autoaudiosink)
        gst.element_link_many(scaletempo, audioconvert, audioresample,
                autoaudiosink)

        pad = scaletempo.get_pad('sink')
        self.add_pad(gst.GhostPad('sink', pad));
        

class player:
    def __init__(self):

        if gst.plugin_load_by_name("scaletempo") is None:
            self.has_scaletempo = False
        else:
            self.has_scaletempo = True

        self.time_format = gst.Format(gst.FORMAT_TIME)
        self.duration = None
        self.speed = 1.0
        self.last_position = 0

        # prepare pipelines
        self.playbin = gst.element_factory_make('playbin', 'playbin')
        if self.has_scaletempo:
            self.stsink = scaletempo_sink()
        self.regularsink = gst.element_factory_make('autoaudiosink',
                'autoaudiosink')

        # activate scaletempo sink
        if self.has_scaletempo:
            self.playbin.set_property('audio-sink', self.stsink)
        else:
            self.playbin.set_property('audio-sink', self.regularsink)

        # ready =)
        self.playbin.set_state(gst.STATE_READY)

    def toggle_scaletempo(self, scale):
        if not self.has_scaletempo:
            return
        isplaying = self.is_playing()
        position = self.get_position_seconds()
        self.playbin.set_state(gst.STATE_NULL)
        if scale:
            self.playbin.set_property('audio-sink', self.stsink)
        else:
            self.playbin.set_property('audio-sink', self.regularsink)
        if isplaying:
            self.playbin.set_state(gst.STATE_PLAYING)
        else:
            self.playbin.set_state(gst.STATE_PAUSED)
        time.sleep(0.2)
        self.set_position(position)

    def load(self, file):
        self.playbin.set_property("uri", "file://" + file)
        self.playbin.set_state(gst.STATE_READY)
        self.last_position = 0
        self.duration = None

    def play(self):
        self.playbin.set_state(gst.STATE_PLAYING)
        self.set_speed(self.speed)

    def pause(self):
        self.playbin.set_state(gst.STATE_PAUSED)

    def stop(self):
        self.playbin.set_state(gst.STATE_READY)
        self.last_position = 0
        self.pause()

    def is_playing(self):
        return gst.STATE_PLAYING in self.playbin.get_state()

    def set_speed(self, speed):
        self.speed = speed
        self.playbin.seek(
                self.speed,
                self.time_format,
                gst.SEEK_FLAG_FLUSH | gst.SEEK_FLAG_ACCURATE,
                gst.SEEK_TYPE_SET, self.get_position(),
                gst.SEEK_TYPE_NONE, 0)

    def seek_rel(self, seconds):
        abs = self.get_position_seconds() + seconds
        self.set_position(abs)

    def set_position(self, seconds):
        if seconds < 0:
            seconds = 0
        else:
            seconds = min(self.get_length_seconds(), seconds)
        self.last_position = seconds*1000000000
        self.playbin.seek(
                self.speed,
                self.time_format,
                gst.SEEK_FLAG_FLUSH | gst.SEEK_FLAG_ACCURATE,
                gst.SEEK_TYPE_SET, seconds*1000000000,
                gst.SEEK_TYPE_NONE, 0)
    
    def get_position(self):
        try:
            pos_int = self.playbin.query_position(self.time_format)[0]
            self.last_position = pos_int
        except:
            pos_int = self.last_position
        return pos_int
    
    def get_position_seconds(self):
        return self.get_position()/1000000000

    def get_length(self):
        if self.duration is None:
            try:
                dur_int = self.playbin.query_duration(self.time_format)[0]
                self.duration = dur_int
            except gst.QueryError:
                return 1000000000
            return dur_int
        else:
            return self.duration
    
    def get_length_seconds(self):
        return self.get_length()/1000000000

