/*  GTKtalog.
 *  Copyright (C) 1999-2000  Mathieu VILLEGAS
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <gnome.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <zlib.h>

#include "folder.h"
#include "loadcat.h"
#include "interface.h"

#ifdef CATALOG2
#include "load_oldcat.h"
#endif
#ifdef CATALOG3
#include "loadcat3gz.h"
#endif

GString *
read_gstring (gzFile gzfh)
{
  gint l;
  GString *r;
  gchar *s;

  gzread (gzfh, &l, sizeof (gint));
  l = GINT_FROM_LE (l);
  if (l)
    {
      s = (gchar *) g_malloc (sizeof (gchar) * (l + 1));
      gzread (gzfh, s, sizeof (gchar) * l);
      s[l] = 0;
      r = g_string_new (s);
      free (s);
    }
  else
    {
      r = NULL;
    }

  return (r);
}

FILE_DATA *
read_filedata (gzFile gzfh)
{
  FILE_DATA *d;
  guint32 le32;
  guint16 le16;
  guint16 mask;
  d = (FILE_DATA *) g_malloc (sizeof (FILE_DATA));
  d->name = read_gstring (gzfh);
  gzread (gzfh, &le32, sizeof (guint32));
  d->taille = GUINT32_FROM_LE (le32);
  gzread (gzfh, &le16, sizeof (guint16));
  d->type = GUINT16_FROM_LE (le16);
  gzread (gzfh, &le32, sizeof (guint32));
  d->date = GUINT32_FROM_LE (le32);
  gzread (gzfh, &le16, sizeof (guint16));
  mask = GUINT16_FROM_LE (le16);
  if (mask & FDMASK_CATEGORY)
    gzread (gzfh, &le16, sizeof (guint16));
  else
    le16 = 0;
  d->categorie = GUINT16_FROM_LE (le16);
  if (mask & FDMASK_DESCRIPTION)
    gzread (gzfh, &le16, sizeof (guint16));
  else
    le16 = 0;
  d->description = GUINT16_FROM_LE (le16);
  if (mask & FDMASK_INFORMATION)
    d->information = read_gstring (gzfh);
  else
    d->information = NULL;
  if (mask & FDMASK_MIME)
    d->mime = read_gstring (gzfh);
  else
    d->mime = NULL;
  d->a_parent_node_is_vfs = d->type;
  return (d);
}

FILE_DATA *
new_filedata_with_only_name (gzFile gzfh)
{
  FILE_DATA *d;
  guint le;
  d = (FILE_DATA *) g_malloc (sizeof (FILE_DATA));
  d->name = read_gstring (gzfh);
  gzread (gzfh, &le, sizeof (guint));
  d->type = GUINT_FROM_LE (le);
  d->taille = 0;
  d->mime = NULL;
  d->date = 0;
  d->categorie = 0;
  d->description = 0;
  d->information = NULL;
  d->a_parent_node_is_vfs = IS_VFS;
  return (d);
}

void
load_folder_vfs (FOLDER * racine, GNode * prev_node, gzFile gzcatalog)
{
  gchar id = CAT_TREE_0;
  FILE_DATA *fd;
  gzread (gzcatalog, &id, sizeof (gchar));
  while (id != CAT_TREE_END)
    {
      switch (id)
	{
	case CAT_TREE_0:
	  fd = new_filedata_with_only_name (gzcatalog);
	  prev_node = g_node_append_data (prev_node->parent, fd);
	  fd->node = prev_node;
	  g_ptr_array_add (racine->datas, fd);
	  break;
	case CAT_TREE_UP:
	  prev_node = prev_node->parent;
	  break;
	case CAT_TREE_DOWN:
	  gzread (gzcatalog, &id, sizeof (gchar));
	  if (id != CAT_TREE_UP)
	    {
	      g_assert (id == CAT_TREE_0);
	      fd = new_filedata_with_only_name (gzcatalog);
	      prev_node = g_node_append_data (prev_node, fd);
	      fd->node = prev_node;
	      g_ptr_array_add (racine->datas, fd);
	    }
	  break;
	}
      gzread (gzcatalog, &id, sizeof (gchar));
    }
}

gboolean
set_vfs_extended_fd (GNode * gn, gpointer data)
{
  FILE_DATA *fd = get_file_data_from_gnode (gn);
  fd->a_parent_node_is_vfs = IS_VFS_EXTENDED;
  return (FALSE);
}

gboolean
update_vfs_fd (GNode * gn, gpointer data)
{
  if (is_vfs_extended (gn))
    {
      g_node_traverse (gn, G_PRE_ORDER, G_TRAVERSE_ALL, -1,
		       set_vfs_extended_fd, NULL);
    }
  if (is_vfs_any (gn))
    {
      FILE_DATA *fd = get_file_data_from_gnode (gn);
      fd->a_parent_node_is_vfs = IS_FILE;
    }
  return (FALSE);
}

void
load_folder_tree (FOLDER * racine, gzFile gzcatalog)
{
  gchar id = CAT_TREE_0;
  GNode *prev_node;
  FILE_DATA *fd;

  prev_node = racine->tree;

  gzread (gzcatalog, &id, sizeof (gchar));
  while (id != CAT_TREE_END)
    {
      switch (id)
	{
	case CAT_TREE_0:
	  fd = read_filedata (gzcatalog);
	  prev_node = g_node_append_data (prev_node->parent, fd);
	  fd->node = prev_node;
	  g_ptr_array_add (racine->datas, fd);
	  if (fd->type == IS_VFS)
	    load_folder_vfs (racine, prev_node, gzcatalog);
	  break;
	case CAT_TREE_UP:
	  prev_node = prev_node->parent;
	  break;
	case CAT_TREE_DOWN:
	  gzread (gzcatalog, &id, sizeof (gchar));
	  if (id != CAT_TREE_UP)
	    {
	      g_assert (id == CAT_TREE_0);
	      fd = read_filedata (gzcatalog);
	      prev_node = g_node_append_data (prev_node, fd);
	      fd->node = prev_node;
	      g_ptr_array_add (racine->datas, fd);
	      if (fd->type == IS_VFS)
		load_folder_vfs (racine, prev_node, gzcatalog);
	    }
	  break;
	}
      gzread (gzcatalog, &id, sizeof (gchar));
    }
  g_node_traverse (racine->tree, G_PRE_ORDER, G_TRAVERSE_ALL, -1,
		   update_vfs_fd, NULL);
}

GPtrArray *
load_gptrarray_of_gstrings (gzFile gzfh)
{
  GPtrArray *gpa;
  gint i;
  gint nb_datas = 0;

  gpa = g_ptr_array_new ();
  gzread (gzfh, &nb_datas, sizeof (gint));
  nb_datas = GINT_FROM_LE (nb_datas);
  i = GINT_FROM_LE (i);
  for (i = 0; i < nb_datas; i++)
    {
      g_ptr_array_add (gpa, read_gstring (gzfh));
    }
  return (gpa);
}

int
load_all_from_file (char *filename, FOLDER * racine, gboolean all)
{
  char fileformat[MAX_STRING_LENGTH];
  gint l, le;
#ifdef CATALOG2
  gint result;
#endif
  int catalog;
  gzFile gzcatalog = NULL;
  FILE_DATA *fd;
  gint i;
  guint16 mask;
  guint16 le16;

/* Test if the file exists */
  if ((g_file_test (filename, G_FILE_TEST_ISFILE)) != TRUE)
    {
      ERROR_DIALOG (_("File does not exist or is not a standard file"),
		    main_window);
      return (-1);
    }

/* Opens the file */
  if ((catalog = open (filename, O_RDONLY)) == -1)
    {
      ERROR_DIALOG (_("Can't open file."), main_window);
      return (-1);
    }

/* Test if the length is big enough */
  l = lseek (catalog, 0, SEEK_END);
  if (l < 9)
    {
      close (catalog);
      ERROR_DIALOG (_("File is not a GTKtalog file (Too Small!)."),
		    main_window);
      return (-1);
    }

/* Add the filename to the 'recent files' submenu */
  add_file_menu (filename);

/* Test if this is an older version of gtktalog
 * Older versions of gtktalog begin with "gtktalog"
 */
  l = lseek (catalog, 0, SEEK_SET);
  read (catalog, fileformat, sizeof (gchar) * 9);
  fileformat[9] = 0;
  if (strcmp (fileformat, "gtktalog2") == 0)
    {
#ifdef CATALOG2
      /* Try to load an gtktalog v2 */
      gzcatalog = gzdopen (catalog, "rb");
      result = load_oldfolder_tree (racine, gzcatalog, all);
      gzclose (gzcatalog);
      WARNING_DIALOG (_
		      ("Warning: You are trying to load an a catalog saved in an old format.\n"
		       "This will work now, but the code for loading an old catalog will not be maintained.\n"
		       "It should not stay in gtktalog for a long time, "
		       "so do not forget to save your converted catalogs.\n"
		       "However, a converter exists and will be maintained in case "
		       "you had forgotten to convert a catalog"),
main_window);
      if (result)
	{
	  ERROR_DIALOG (_
			("The loading of this old catalog has failed.\nEnsure this is a good catalog and\nEmail the authors specifying the problem"),
main_window);
	  return (0);
	}

      for (i = 0; i < racine->datas->len; i++)
	{
	  fd = g_ptr_array_index (racine->datas, i);
	  fd->id = i;
	}
      racine->is_modified = FALSE;
      return (0);
#else
      close (catalog);
      /* FIXME: Remove this dialog box when the code is correct for links */
      ERROR_DIALOG (_
		    ("File seems to have been generated with a version older "
		     "than 0.1.0 of GTKtalog. Formats are incompatible.\n"
		     "Try `./configure --enable-catalog2' then recompile gtktalog. "
		     "Or use the converter!"), main_window);
      return (-1);

#endif
    }
  else
    {
      /* File format prior to (at least 0.0.18) */
      fileformat[8] = 0;
      if (strcmp (fileformat, "gtktalog") == 0)
	{
	  ERROR_DIALOG (_
			("File seems to have been generated with a version older "
			 "than 0.0.18 (at least) of GTKtalog. Formats are incompatibles. "
			 "Send a mail to the authors to know what to do."),
main_window);
	  return (-1);
	}
    }

  /* Test if this is a catalog in v3 format. */
  l = lseek (catalog, 0, SEEK_SET);
  read (catalog, &le, sizeof (gint));
  l = GINT_FROM_LE (le);

  if ((l > 20) || (l < 5))
    {
      close (catalog);
      ERROR_DIALOG (_("File is not a GTKtalog file."), main_window);
      return (-1);
    }
  read (catalog, &fileformat, sizeof (gchar) * l);
  fileformat[l] = 0;

  if ((!strcmp (fileformat, "gtktalog 3"))
      || (!strcmp (fileformat, "gtktalog 3gz")))
    {
#ifdef CATALOG3

      WARNING_DIALOG (_
		      ("Warning: You are trying to load an a catalog saved in an old format.\n"
		       "This will work now, but the code for loading an old catalog will not be maintained.\n"
		       "It should not stay in gtktalog for a long time, "
		       "so do not forget to save your converted catalogs."),
main_window);
#else
      close (catalog);
      /* FIXME: Remove this dialog box when the code is correct for links */
      ERROR_DIALOG (_
		    ("File seems to have been generated with a version older "
		     "than 0.10.1 of GTKtalog. Formats are incompatibles.\n"
		     "Try `./configure --enable-catalog3' then recompile gtktalog "),
main_window);
      return (-1);
#endif
    }

#ifdef CATALOG3
  if ((strcmp (fileformat, "gtktalog 3"))
      && (strcmp (fileformat, "gtktalog 3gz"))
      && (strcmp (fileformat, "gtktalog 4.0")))
#else
  if (strcmp (fileformat, "gtktalog 4.0"))
#endif
    {
      close (catalog);
      ERROR_DIALOG (_("File is not a GTKtalog file (v4 file format)."),
		    main_window);
      return (-1);
    }

/* This is a gtktalog format, and read the file.
 */
  gzcatalog = gzdopen (catalog, "rb");
#ifdef CATALOG3
  mask = CATMASK_DESCRIPTION | CATMASK_CATEGORY;
  if (!strcmp (fileformat, "gtktalog 4.0"))
    {
      gzread (gzcatalog, &le16, sizeof (guint16));
      mask = GUINT16_FROM_LE (le16);

    }
#else
  gzread (gzcatalog, &le16, sizeof (guint16));
  mask = GUINT16_FROM_LE (le16);
#endif
  if (mask & CATMASK_DESCRIPTION)
    racine->descriptions = load_gptrarray_of_gstrings (gzcatalog);
  if (mask & CATMASK_CATEGORY)
    racine->categories = load_gptrarray_of_gstrings (gzcatalog);
  if (all == TRUE)
    {
#ifdef CATALOG3
      if ((!strcmp (fileformat, "gtktalog 3"))
	  || (!strcmp (fileformat, "gtktalog 3gz")))
	{
	  load_folder_tree_3gz (racine, gzcatalog);
	}
      else
#endif
	load_folder_tree (racine, gzcatalog);
    }
  gzclose (gzcatalog);
  if (all == TRUE)
    {
      for (i = 0; i < racine->datas->len; i++)
	{
	  fd = g_ptr_array_index (racine->datas, i);
	  fd->id = i;
	}
    }
  racine->is_modified = FALSE;
  if (racine->catalog_filename)
    g_string_free (racine->catalog_filename, TRUE);
  racine->catalog_filename = g_string_new (filename);
  racine->catalog_filename_is_valid = TRUE;
  return (0);
}

int
load_cat_from_file (char *filename, FOLDER * racine)
{
  return (load_all_from_file (filename, racine, TRUE));
}

int
load_only_categories_from_file (char *filename, FOLDER * racine)
{
  return (load_all_from_file (filename, racine, FALSE));
}
