/*  GTKtalog.
 *  Copyright (C) 1999  Mathieu VILLEGAS
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <gnome.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <zlib.h>
#if defined(USE_PTHREADS)
#    include <pthread.h>
#endif

#include "savecat.h"
#include "config_common.h"
#include "categories.h"
#include "exit.h"
#include "folder.h"
#include "thread_utils.h"
#include "progressbar.h"

static gzFile gzcatalog;
#if defined(USE_PTHREADS)
static pthread_t thread1;
#endif

void
write_gstring (GString * g, int fh)
{
  gint i = 0;
  if (g == NULL)
    {
      write (fh, &i, sizeof (gint));
    }
  else
    {
      if (g->len == 0)
	{
	  write (fh, &i, sizeof (gint));
	}
      else
	{
	  i = GINT_TO_LE (g->len);
	  write (fh, &i, sizeof (gint));
	  write (fh, (g->str), sizeof (gchar) * g->len);
	}
    }
}

void
gzwrite_gstring (GString * g, gzFile fh)
{
  gint i = 0;
  if (g == NULL)
    {
      gzwrite (fh, &i, sizeof (gint));
    }
  else
    {
      if (g->len == 0)
	{
	  gzwrite (fh, &i, sizeof (gint));
	}
      else
	{
	  i = GINT_TO_LE (g->len);
	  gzwrite (fh, &i, sizeof (gint));
	  gzwrite (fh, (g->str), sizeof (gchar) * g->len);
	}
    }
}

static void write_vfsnode (GNode * gn, gpointer data);
static void
write_subvfs (GNode * gn)
{
  gchar id;

  id = CAT_TREE_DOWN;
  gzwrite (gzcatalog, &id, sizeof (gchar));
  g_node_children_foreach (gn, G_TRAVERSE_ALL, write_vfsnode, NULL);
  id = CAT_TREE_UP;
  gzwrite (gzcatalog, &id, sizeof (gchar));
}

static void
write_vfsnode (GNode * gn, gpointer data)
{

  gchar id = CAT_TREE_0;
  guint le;
  FILE_DATA *fd = get_file_data_from_gnode (gn);
  gulong pb;

  gzwrite (gzcatalog, &id, sizeof (gchar));
  gzwrite_gstring (fd->name, gzcatalog);
  le = GUINT_TO_LE (fd->type);
  gzwrite (gzcatalog, &le, sizeof (guint));
  pb = progress_getCurrent (GNOMEAPPBAR_MAIN) + 1;
  progress_setCurrent (pb, GNOMEAPPBAR_MAIN);
#if !defined(USE_PTHREADS)
  gui_update ();
#endif
  if ((is_file (gn) == FALSE) && (is_vfs (gn) == FALSE))
    write_subvfs (gn);
}

void
write_filedata (FILE_DATA * d, gzFile fh)
{
  guint32 le32;
  guint16 le16, le16_category, le16_description;
  guint16 mask;
  gchar id;
  gulong pb;

  g_assert (d != NULL);
  gzwrite_gstring (d->name, fh);
  le32 = GUINT32_TO_LE (d->taille);
  gzwrite (fh, &le32, sizeof (guint32));
  le16 = GUINT16_TO_LE (d->type);
  gzwrite (fh, &le16, sizeof (guint16));
  le32 = GUINT32_TO_LE (d->date);
  gzwrite (fh, &le32, sizeof (guint32));
  mask = 0;
  /* Due to a bug in older versions, some catalogs may be bugged. So gtktalog has to
   * check that descriptions and categories are NULL if this is a link
   */
  if (is_link (d->node->parent))
    le16_category = 0;
  else
    le16_category = GUINT16_TO_LE (d->categorie);
  if (is_link (d->node->parent))
    le16_description = 0;
  else
    le16_description = GUINT16_TO_LE (d->description);
  if (le16_category)
    mask = mask | FDMASK_CATEGORY;
  if (le16_description)
    mask = mask | FDMASK_DESCRIPTION;
  if (d->information)
    {
      if (d->information->len)
	mask = mask | FDMASK_INFORMATION;
    }
  if (d->mime)
    {
      if (d->mime->len)
	mask = mask | FDMASK_MIME;
    }
  le16 = GUINT16_TO_LE (mask);
  gzwrite (fh, &le16, sizeof (guint16));

  if (le16_category)
    gzwrite (fh, &le16_category, sizeof (guint16));
  if (le16_description)
    gzwrite (fh, &le16_description, sizeof (guint16));
  if (d->information)
    {
      if (d->information->len)
	gzwrite_gstring (d->information, fh);
    }
  if (d->mime)
    {
      if (d->mime->len)
	gzwrite_gstring (d->mime, fh);
    }
  pb = progress_getCurrent (GNOMEAPPBAR_MAIN) + 1;
  progress_setCurrent (pb, GNOMEAPPBAR_MAIN);
#if !defined(USE_PTHREADS)
  gui_update ();
#endif
  if (d->type == IS_VFS)
    {
      write_subvfs (d->node);
      id = CAT_TREE_END;
      gzwrite (gzcatalog, &id, sizeof (gchar));
    }
}

void write_gnode (GNode * gn, gpointer data);
void
write_subdir (GNode * gn)
{
  gchar id;

  id = CAT_TREE_DOWN;
  gzwrite (gzcatalog, &id, sizeof (gchar));
  g_node_children_foreach (gn, G_TRAVERSE_ALL, write_gnode, NULL);
  id = CAT_TREE_UP;
  gzwrite (gzcatalog, &id, sizeof (gchar));
}

void
write_gnode (GNode * gn, gpointer data)
{

  gchar id = CAT_TREE_0;
  FILE_DATA *fd = get_file_data_from_gnode (gn);

  gzwrite (gzcatalog, &id, sizeof (gchar));
  write_filedata (fd, gzcatalog);
  if ((is_file (gn) == FALSE) && (is_vfs (gn) == FALSE))
    write_subdir (gn);
}

typedef struct
{
  int catalog;
  FOLDER *racine;
  char *filename;
}
_SAVE_ARGS_;

void *
_save_cat_to_file_ (void *arg)
{
  gint i;
  gchar id;
  gint nb_datas = 0;
  gint le;
//  gzFile gzcatalog;
  GString *header;
  gchar mode[] = "wb9";
  _SAVE_ARGS_ *sa = arg;
  int catalog = sa->catalog;
  FOLDER *racine = sa->racine;
  char *filename = sa->filename;
  gulong pb;
  guint16 mask;
  guint16 le16;

  /* First, remove unused categories if the user asked for it */
  if (my_config->save_only_used_categories)
    {
      remove_unused_categories (racine);
    }

  /* Header */
  header = g_string_new ("gtktalog 4.0");
  write_gstring (header, catalog);
  g_string_free (header, TRUE);

  /* reopen with zlib */
  g_assert ((my_config->compression_level >= 0)
	    && (my_config->compression_level <= 9));
  sprintf (mode, "wb%d", my_config->compression_level);
  gzcatalog = gzdopen (catalog, mode);

/* Write the mask.
 */
  mask = 0;
  if (racine->descriptions->len)
    mask |= CATMASK_DESCRIPTION;
  if (racine->categories->len)
    mask |= CATMASK_CATEGORY;

  le16 = GUINT16_TO_LE (mask);
  gzwrite (gzcatalog, &le16, sizeof (guint16));

  /* descriptions */
  if ((nb_datas = racine->descriptions->len))
    {
      le = GINT_TO_LE (nb_datas);
      gzwrite (gzcatalog, &le, sizeof (gint));
      for (i = 0; i < nb_datas; i++)
	{
	  gzwrite_gstring (g_ptr_array_index (racine->descriptions, i),
			   gzcatalog);
	  pb = progress_getCurrent (GNOMEAPPBAR_MAIN) + 1;
	  progress_setCurrent (pb, GNOMEAPPBAR_MAIN);
#if !defined(USE_PTHREADS)
	  gui_update ();
#endif

	}
    }

  /* categories */
  if ((nb_datas = racine->categories->len))
    {
      le = GINT_TO_LE (nb_datas);
      gzwrite (gzcatalog, &le, sizeof (gint));
      for (i = 0; i < nb_datas; i++)
	{
	  gzwrite_gstring (g_ptr_array_index (racine->categories, i),
			   gzcatalog);
	  pb = progress_getCurrent (GNOMEAPPBAR_MAIN) + 1;
	  progress_setCurrent (pb, GNOMEAPPBAR_MAIN);
#if !defined(USE_PTHREADS)
	  gui_update ();
#endif
	}
    }

  /* Tree structure */
  write_subdir (racine->tree);
  id = CAT_TREE_END;
  gzwrite (gzcatalog, &id, sizeof (gchar));

  gzclose (gzcatalog);
  chmod (filename, (S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH));

  racine->is_modified = FALSE;
  if (racine->catalog_filename)
    g_string_free (racine->catalog_filename, TRUE);
  racine->catalog_filename = g_string_new (filename);
  racine->catalog_filename_is_valid = TRUE;
  g_free (filename);

  set_thread_status_to_ended ();
#if defined(USE_PTHREADS)
  pthread_exit (NULL);
#endif
  return (NULL);
}

int
save_cat_to_file (char *filename, FOLDER * racine)
{
  static _SAVE_ARGS_ _save_args_;
  guint32 nb_dirs = 0, nb_files = 0;
  guint32 nb_desc = 0, nb_cat = 0;
  guint32 nb_disks = 0;


  if (racine->catalog_filename_is_valid == FALSE)
    {
      if ((g_file_test (filename, G_FILE_TEST_ISFILE)) == TRUE)
	{
	  if (yes_no_dialog
	      (_("File exists"),
	       _("File already exists.\nOverwrite it?")) != 0)
	    return (-1);
	  else
	    unlink (filename);
	}
    }
  else
    {
      if ((g_file_test (filename, G_FILE_TEST_ISFILE)) == TRUE)
	unlink (filename);
    }

  if (
      (_save_args_.catalog =
       open (filename, O_CREAT | O_WRONLY | O_TRUNC),
       (S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH)) == -1)
    {
      return (-1);
    }

  folder_statistics (racine, racine->tree, &nb_desc, &nb_cat,
		     &nb_disks, NULL, &nb_dirs, NULL, &nb_files, NULL, NULL,
		     NULL);
  progress_setTotal (nb_desc + nb_cat + nb_dirs + nb_files + nb_disks,
		     GNOMEAPPBAR_MAIN);

  set_thread_status_to_running ();
  progress_timeout (NULL);

  _save_args_.racine = racine;
  _save_args_.filename = g_strdup (filename);

#if defined(USE_PTHREADS)
  if (pthread_create (&thread1, NULL, _save_cat_to_file_, &_save_args_) < 0)
    {
      reinit_thread_status ();
      ERROR_DIALOG (_("Thread failed.\nCheck your number of threads."),
		    main_window);
    }
  else
    {
      thread_gtk_timeout_with_progress_bar (GINT_TO_POINTER(GNOMEAPPBAR_MAIN));
    }
#else
  thread_gtk_timeout_with_progress_bar (GINT_TO_POINTER(GNOMEAPPBAR_MAIN));
  _save_cat_to_file_ (&_save_args_);
#endif
  return (0);
}
