/*  GTKtalog.
 *  Copyright (C) 1999-2000  Mathieu VILLEGAS
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#define _USE_BSD
#include <config.h>
#include <gnome.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/resource.h>
#include <sys/wait.h>
#include <stdlib.h>
#include <dirent.h>
#include <unistd.h>
#include <time.h>
#include <stdio.h>
#include <string.h>
#if defined(USE_PTHREADS)
#    include <pthread.h>
#endif
#include <signal.h>

#include "addisk.h"
#include "config_common.h"
#include "fastaddisk.h"
#include "folder.h"
#include "interface.h"
#include "compare.h"
#include "io.h"
#include "progressbar.h"
#include "vfs.h"
#include "thread_utils.h"

#define READ   0
#define WRITE 1

#define ADDED_DISK_OK	1
#define ADDED_DISK_NOK	2
#define ADDED_DISK_DELETE	3

static char *local_foldername;
static char *local_diskname;
static gboolean flag_to_umount;
static gboolean flag_to_mount;
static gint added_disk_is_ok;

static GNode *local_gn;

static gboolean is_message_to_display;
static GString *message_to_display;

static gboolean vfs_scan_nok;

gint
spawn_proc (gchar * path, gint narg, gchar ** arg)
{
  gchar **arg2;
  int i, pid;
  GString *gstr = g_string_new (arg[0]);

  if (test_plugin_existence (gstr))
    {
      g_string_free (gstr, TRUE);
      pid = fork ();

      if (pid != 0)
	return pid;

      /* Child */

      arg2 = g_new0 (gchar *, (narg + 1));
      for (i = 0; i < narg; i++)
	arg2[i] = arg[i];
      arg2[narg] = NULL;

      return execvp (arg2[0], arg2);
    }
  else
    {
      g_string_free (gstr, TRUE);
      return (-1);
    }
}

guint
du_s (gchar * path)
{
  int p[2];
  static gchar buffer[1024];
  gchar *argsdu[4];
  gchar *b2;
  ssize_t len;
  int pid;
  if (pipe (p) == 0)
    {
      if ((pid = fork ()) == 0)
	{
	  close (1);
	  dup (p[1]);
	  /* FIXME: Stderr should be closed too or redirected
	   * in order not to have error messages on the console
	   */
	  argsdu[0] = g_strdup ("du");
#if defined(OS_LINUX)
	  argsdu[1] = g_strdup ("-sb");
#elif defined(OS_FREEBSD) || defined(OS_NETBSD)
	  argsdu[1] = g_strdup ("-s");
#endif
	  argsdu[2] = path;
	  argsdu[3] = NULL;
	  execvp (argsdu[0], argsdu);
	}
      else
	{
	  close (0);
	  dup (p[0]);
	  close (p[1]);
	  len = read (0, buffer, 1023);
	  buffer[len] = 0;
	  b2 = buffer;
	  while ((b2[0] >= '0') && (b2[0] <= '9'))
	    b2++;
	  b2[0] = 0;
	  wait4 (pid, NULL, 0, 0);
	  close (p[0]);
	  /* FIXME: STDIN should be reopened to the real STDIN. */
	}
    }
  return (atoi (buffer));
}

void
init_message_to_display ()
{
  is_message_to_display = FALSE;
}

void
display_message_if_necessary ()
{
  if (is_message_to_display)
    {
      is_message_to_display = FALSE;
      gnome_dialog_run_and_close (GNOME_DIALOG
				  (gnome_warning_dialog_parented
				   (message_to_display->str,
				    GTK_WINDOW (main_window))));
      g_string_free (message_to_display, TRUE);
    }

}

void
set_message_to_display (GString * message)
{
  message_to_display = message;
  is_message_to_display = TRUE;
}

gboolean eject_disk (CONFIG * my_config)
{
  int status, pid;
  char *args[2];

  seteuid (getuid ());
  args[0] = my_config->eject_prog->str;
  args[1] = escapespaces (my_config->mount_point->str);
  pid = spawn_proc (NULL, 2, args);
  if (pid == -1)
    {
      set_message_to_display (g_string_new
			      (_
			       ("Could not execute eject program, check your config")));
      return (FALSE);
    }

  wait4 (pid, &status, 0, NULL);
  if (WIFEXITED (status) == 0)
    {
      set_message_to_display (g_string_new
			      (_("Eject did not exit normally")));
      return (FALSE);
    }
  if (WEXITSTATUS (status) != 0)
    {
      set_message_to_display (g_string_new (_("Eject failed! Disk in use?")));
      return (FALSE);
    }
  return (TRUE);
}

gboolean mount_disk (CONFIG * my_config)
{
#if defined(OS_LINUX)
  int pid, status;
  char *args[2];
  int tries = my_config->mount_retry;

  seteuid (getuid ());
  args[0] = my_config->mount->str;
  args[1] = escapespaces (my_config->mount_point->str);

  while (tries >= 0)
    {
      pid = spawn_proc (NULL, 2, args);
      if (pid == -1)
	{
	  set_message_to_display (g_string_new
				  (_
				   ("Could not execute mount program, check your config")));
	  return (FALSE);
	}

      wait4 (pid, &status, 0, NULL);
      if (WIFEXITED (status) == 0)
	{
	  set_message_to_display (g_string_new
				  (_("Mount did not exit normally")));
	  return (FALSE);
	}
      if (WEXITSTATUS (status) == 0)
	{
	  return (TRUE);
	}
      tries--;
      if (tries >= 0)
	sleep (MOUNT_RETRY_TIMEOUT);
    }

  set_message_to_display (g_string_new (_("Mount failed! Disk in use?")));
  return (FALSE);
#elif defined(OS_FREEBSD) || defined(OS_NETBSD)
  int status;
  struct iso_args args;

  args.export.ex_flags = MNT_EXRDONLY;
  args.fspec = "/dev/cd0a";	// Will be changed
  args.export.ex_root = -2;
  status =
    mount (MOUNT_CD9660, my_config->mount_point->str, MNT_RDONLY, &args);
  /* error saved in errno variable, do something with it */
  if (status == 0)
    return (TRUE);
  else
    return (FALSE);
#endif
}

gboolean umount_disk (CONFIG * my_config)
{
#if defined(OS_LINUX)
  int pid, status;
  char *args[2];

  seteuid (getuid ());
  args[0] = my_config->umount->str;
  args[1] = escapespaces (my_config->mount_point->str);
  pid = spawn_proc (NULL, 2, args);
  if (pid == -1)
    {
      set_message_to_display (g_string_new
			      (_
			       ("Could not execute umount program, check your config")));
      return (FALSE);
    }

  wait4 (pid, &status, 0, NULL);
  if (WIFEXITED (status) == 0)
    {
      set_message_to_display (g_string_new
			      (_("Umount did not exit normally")));
      return (FALSE);
    }
  if (WEXITSTATUS (status) != 0)
    {
      set_message_to_display (g_string_new
			      (_("Umount failed! Disk in use?")));
      return (FALSE);
    }
  return (TRUE);
#elif defined(OS_FREEBSD) || defined(OS_NETBSD)
  int status;

  status = unmount (my_config->mount_point->str, NULL);
  if (status == 0)
    return (TRUE);
  else
    return (FALSE);
#endif
}

void
umount_and_eject_if_necessary ()
{
  if (flag_to_umount == TRUE)
    {
      /* umount the drive and eject it */
      if (my_config->eject_disk == TRUE)
	{
	  if ((my_config->use_supermount == TRUE)
	      || (my_config->use_automount == TRUE))
	    eject_disk (my_config);
	  else if (umount_disk (my_config) == TRUE)
	    eject_disk (my_config);
	}
    }

}

void
do_not_forget_to_umount (gboolean gb)
{
  flag_to_umount = gb;
}

void
do_not_forget_to_mount (gboolean gb)
{
  flag_to_mount = gb;
}

void
set_added_disk (gint gb)
{
  added_disk_is_ok = gb;
}

void
init_vfs_scan_status (void)
{
  vfs_scan_nok = FALSE;
}

gint my_timeout (gpointer data)
{
  FOLDER *racine = data;
/*  FILE_DATA *fd;*/
  display_message_if_necessary ();
  if (get_thread_status () == THREAD_SCAN_RUNNING_ENDED)
    {
      if (added_disk_is_ok == ADDED_DISK_OK)
	{
	  gtk_clist_freeze (GTK_CLIST (racine->ctree));
	  change_name (local_gn, local_diskname);
	  change_information (local_gn, local_foldername);
	  g_node_traverse (local_gn,
			   G_PRE_ORDER, G_TRAVERSE_ALL, -1,
			   add_gnode_to_ctree, racine);
	  gtk_clist_thaw (GTK_CLIST (racine->ctree));
	  update_tree (racine);
	}
      else if (added_disk_is_ok == ADDED_DISK_DELETE)
	{
	  suppress_dir ((GNode *) local_gn);
	}
      g_free (local_diskname);
      g_free (local_foldername);
/************** If umount and eject have to be in the main thread, remove the comments ********/
//umount_and_eject_if_necessary();
/*********************************************************************************************/
      progress_setCurrent (100, GNOMEAPPBAR_MAIN);
      progress_timeout (GINT_TO_POINTER (GNOMEAPPBAR_MAIN));

      reinit_thread_status (GNOMEAPPBAR_MAIN);
      if (vfs_scan_nok)
	{
	  WARNING_DIALOG (_
			  ("At least one VFS could not be read successfully"),
			  main_window);
	}
    }
  else
    {
      progress_timeout (GINT_TO_POINTER (GNOMEAPPBAR_MAIN));
      gtk_timeout_add (500, my_timeout, racine);
    }
  return (0);
}

void
set_local_disk_name (gchar * s)
{
  local_diskname = g_strdup (s);
  return;
}

void
set_local_folder_name (gchar * s)
{
  local_foldername = g_strdup (s);
  return;
}

gchar *
launch_information_plugin (gchar * file, MIME_EXTENSIONS * m)
{
  char buff[250];
  int fd[2];
  int b;
  gint pid;
  gint status;
  GString *information;
  gchar *ret;
  gchar **args;
  gchar *cmdline;
  gint nb_args;
  gchar **args_ptr;

  if (compare ("^[ \t]*$", m->prog->str, COMPARE_REGEX, FALSE))
    {
      if (pipe (fd) == -1)
	{
	  g_error ("Couldn't create a pipe");
	  return (NULL);
	}
      switch (pid = fork ())
	{
	case 0:		/*fils1 */
	  close (fd[READ]);
	  close (1);
	  dup (fd[WRITE]);
	  close (fd[WRITE]);
	  if (compare ("^[ \t]*$", m->arg->str, COMPARE_REGEX, FALSE))
	    {

	      cmdline = g_strconcat (m->prog->str, " ", m->arg->str, NULL);
	      args = g_strsplit (cmdline, " ", 0);
	      nb_args = 0;
	      args_ptr = args;
	      while (args_ptr[0] != NULL)
		{
		  nb_args++;
		  args_ptr++;
		}
	      args =
		(gchar **) g_realloc (args, sizeof (gchar *) * (nb_args + 2));
	      args[nb_args] = g_strdup (file);
	      args[nb_args + 1] = NULL;

	      execvp (args[0], args);
	    }
	  else
	    {
	      execlp (m->prog->str, m->prog->str, file, NULL);
	    }
	case -1:		/*erreur */
	  g_error (_("Could not fork\n"));
	  exit (3);
	default:
	  close (fd[WRITE]);
	  close (0);
	  dup (fd[READ]);
	  close (fd[READ]);
	  information = g_string_new ("");
	  b = read (0, buff, 249 * sizeof (char));
	  while (b != 0)
	    {
	      buff[b] = '\0';
	      information = g_string_append (information, buff);
	      b = read (0, buff, 249 * sizeof (char));
	    }
	  wait4 (pid, &status, 0, NULL);
	  ret = g_strdup (information->str);
	  g_string_free (information, TRUE);
	  return (ret);

	}
    }
  else
    {
      return (NULL);
    }

}

gchar *
read_information (gchar * file, gchar * mime)
{
  MIME_EXTENSIONS *m;
  gchar *information_string = NULL;
  if (mime)
    {
      if ((m = find_extension (mime)))
	{
	  information_string = launch_information_plugin (file, m);
	}
    }
  if (!information_string)
    {
      if (my_config->use_default_information_plugin)
	{
	  m = my_config->default_information_plugin;
	  if (m->state == TRUE)
	    {
	      information_string = launch_information_plugin (file, m);
	    }
	}
    }
  return (information_string);
}

gboolean is_dir_openable (char *dir_name)
{
  DIR *dir;

  if ((dir = opendir (dir_name)) == NULL)
    return (FALSE);
  closedir (dir);
  return (TRUE);
}

gboolean
create_folder_tree (FOLDER * racine, GNode * parent, const gchar * path,
		    gboolean getInformation, gint testvfs)
/* Stocks the directory (ans subdirs) into a FOLDER structure
 * folder is the FOLDER directory where the directory should be stored
 *   WARNING: folder should not be NULL!!!
 * path is the full path of the directory
 */
{
  DIR *dir;
  struct dirent *fichier;
  struct stat fileinfo;
  gchar *file;
  GList *gl;
  GList *new_dirs = NULL;
  GNode *gn;
  GString *my_path;
  gchar *infostring;
  static gulong totalDataBlocks = 0;
  gchar *mime;
  gpointer scan_it;

  if (progress_needReset ())
    {
      progress_reset ();
      totalDataBlocks = 0;
    }

  my_path = g_string_new (path);

  my_path = g_string_append_c (my_path, '/');

  /*  kill (my_config->current_pid, SIGUSR2); */

  if ((dir = opendir (my_path->str)) == NULL)
    {
      /* FIXME: put an English message inside a dialog box */
      printf ("Can't read the directory: %s", my_path->str);
      return (FALSE);
    }
  while ((fichier = readdir (dir)) != NULL)
    {
      file = g_strconcat (my_path->str, fichier->d_name, NULL);
      if ((strcmp (fichier->d_name, ".") != 0)
	  && (strcmp (fichier->d_name, "..") != 0))
	{

	  if (lstat (file, &fileinfo) == -1)
	    {
	      gn = folder_add (racine, parent, fichier->d_name, NULL,
			       IS_UNREADABLE, NULL, 0, 0, 0, 0);
	    }
	  /* This is a directory */
	  else if (S_ISLNK (fileinfo.st_mode))
	    {
	      char tmp[1024];
	      char tmp2[2048];

	      memset (tmp, 0, 1024);

	      if (readlink (file, tmp, 1023) == -1)
		{
		  printf
		    ("Problem with symbolic link %s, we forget it!\n", file);
		}
	      else
		{
		  sprintf (tmp2, "%s -> %s", fichier->d_name, tmp);
		  folder_add_link (racine, parent, tmp, tmp2, NULL,
				   fileinfo.st_size, fileinfo.st_ctime, 0, 0);
		}
	    }
	  else if (S_ISDIR (fileinfo.st_mode))
	    {

	      if (is_dir_openable (file))
		{
		  gn =
		    folder_add (racine, parent, fichier->d_name, NULL,
				IS_DIR, NULL, fileinfo.st_size,
				fileinfo.st_ctime, 0, 0);
		  new_dirs = g_list_append (new_dirs, gn);
		}
	    }

	  else
	    {
	      /* This is a file */

	      /* Progress Bar */
	      totalDataBlocks += fileinfo.st_size;
	      progress_setCurrent (totalDataBlocks, GNOMEAPPBAR_MAIN);

	      mime = NULL;
	      if (my_config->scan_mime_types)
		{
		  mime = g_strdup (gnome_mime_type_from_magic (file));
		}
	      if (!mime)
		mime = g_strdup (gnome_mime_type_or_default (file, NULL));
	      if (mime)
		{
		  scan_it =
		    g_tree_lookup (my_config->mime_to_skip_during_a_scan,
				   mime);
		}
	      else
		{
		  scan_it = NULL;
		}
	      if (GPOINTER_TO_INT (scan_it) != 1)
		{
		  if (getInformation)
		    {
		      /* use a plugin to get informations about the file */
		      infostring = read_information (file, mime);
		    }
		  else
		    {
		      infostring = NULL;
		    }
		  gn =
		    folder_add (racine, parent, fichier->d_name, infostring,
				IS_FILE, mime, fileinfo.st_size,
				fileinfo.st_ctime, 0, 0);
		  if (mime)
		    g_free (mime);
		  if (infostring)
		    g_free (infostring);
		  if (testvfs != TEST_VFS_NO)
		    {
		      if (vfs_read (gn, fichier->d_name, testvfs, racine))
			vfs_scan_nok = TRUE;
		    }
		}
	    }
	}
      free (file);
#if !defined(USE_PTHREADS)
      gui_update ();
#endif

    }
  closedir (dir);
  while (new_dirs)
    {
      gn = (GNode *) new_dirs->data;
      file = g_strconcat (path, "/", folder_get_name (gn), NULL);
      if (create_folder_tree (racine, gn, file, getInformation, testvfs))
	vfs_scan_nok = TRUE;
      free (file);
      gl = new_dirs;
      new_dirs = g_list_remove_link (new_dirs, new_dirs);
      g_list_free_1 (gl);
    }
  return (vfs_scan_nok);
}

gboolean check_mounted (CONFIG * my_config)
{
  FILE *fp;
  char line[1024];
  char s1[1024];
  char s2[1024];
  char s3[1024];
  int rc;

#ifdef OS_LINUX
  fp = fopen ("/proc/mounts", "r");
  if (fp == NULL)
    ERROR_DIALOG (_("Unable to open /proc/mounts\n"), main_window);

  while (fgets (line, sizeof (line), fp) != 0)
    {
      rc = sscanf (line, "%1023s %1023s %1023s", s1, s2, s3);
      if ((rc >= 3) && (strcmp (s2, my_config->mount_point->str) == 0))
	{
	  fclose (fp);
	  return TRUE;
	}
    }
  fclose (fp);
  return FALSE;
#else
/* FIXME: in many cases, this is a bug as we do not know if the CD is mounted
 * or not.
 * If you are porting gtktalog to another platform, adapt this function to
 * your platform and register that platform in configure.in
 * Do not forget to send me (Read the AUTHORS file to know who) the patch ;-)
 */
  return FALSE;
#endif
}

void *
start_thread_scan (void *arg)
{
  FOLDER *racine = arg;
#if defined(__DEBUG__)
  GML_log (2, "start_thread_scan()\n");
#endif
  init_message_to_display ();
#if defined(USE_PTHREADS)
  pthread_detach (pthread_self ());
#endif

  /* Mount the CD if we are in fastadddisk */
  flag_to_umount = FALSE;
  if (flag_to_mount == TRUE)
    {
      if ((my_config->use_supermount == FALSE)
	  && (my_config->use_automount == FALSE))
	{
	  if (check_mounted (my_config))
	    {
	      if (my_config->warn_mount)
		{
		  /* FIXME: Error Message asking if we should continue */
		  printf
		    ("FIXME: Add error message for cd already mounted\n");
		}
	    }
	  else
	    {
	      if (mount_disk (my_config) == FALSE)
		{
		  /* Display Error */
		  printf
		    ("FIXME: Add error message for error in mounting CD\n");
		  added_disk_is_ok = ADDED_DISK_NOK;
		  set_thread_status_to_ended ();
#if defined(__DEBUG__)
		  GML_log (2, "Exiting start_thread_scan(): error\n");
#endif
#if defined(USE_PTHREADS)
		  pthread_exit (NULL);
#endif
		  return (NULL);
		}
	      else
		flag_to_umount = TRUE;
	    }
	}
      else
	flag_to_umount = TRUE;
    }

  /* Get the size of the disk to init the progress bar */
  if (my_config->use_du_s == TRUE)
    progress_setTotal (du_s (local_foldername), GNOMEAPPBAR_MAIN);
  else
    progress_setTotal (getStats (escapespaces (my_config->mount_point->str)),
		       GNOMEAPPBAR_MAIN);

  /* Launch the update of the progress bar */
/*  progress_createTimeout ();*/

  local_gn =
    folder_add (racine, racine->tree,
		local_diskname, local_foldername, IS_DISK, NULL, 0, 0, 0, 0);
#if !defined(USE_PTHREADS)
  gui_update ();
#endif
  vfs_scan_nok = create_folder_tree (racine, local_gn, local_foldername,
				     my_config->getInformation,
				     my_config->testvfs);

  umount_and_eject_if_necessary ();
  set_thread_status_to_ended ();
  racine->is_modified = TRUE;
#if defined(__DEBUG__)
  GML_dump (GML_tree_gstrings, FALSE);
  GML_log (2, "Exiting start_thread_scan(): OK\n");
#endif
#if defined(USE_PTHREADS)
  pthread_exit (NULL);
#endif
  return (NULL);
}

gchar *
escapespaces (gchar * txt)
{
  gchar *ptr, *ptmp;
  gint n;

  n = 1;
  ptr = txt;
  while (ptr[0])
    {
      if (ptr[0] == ' ')
	n++;
      n++;
      ptr++;
    }
  ptr = (gchar *) malloc (sizeof (gchar) * n);
  ptmp = ptr;
  while (txt[0])
    {
      if (txt[0] == ' ')
	{
	  ptmp[0] = '\\';
	  ptmp++;
	}
      ptmp[0] = txt[0];
      ptmp++;
      txt++;
    }
  ptmp[0] = 0;
  return (ptr);
}
