/*  GTKtalog.
 *  Copyright (C) 1999-2000  Mathieu VILLEGAS
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <gnome.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <zlib.h>

#include "folder.h"
#include "loadcat.h"

GString *
read_gstring (int fh)
{
  gint l;
  GString *r;
  gchar *s;

  read (fh, &l, sizeof (gint));
  l = GINT_FROM_LE (l);
  if (l)
    {
      s = (gchar *) g_malloc (sizeof (gchar) * (l + 1));
      read (fh, s, sizeof (gchar) * l);
      s[l] = 0;
      r = g_string_new (s);
      free (s);
    }
  else
    {
      r = NULL;
    }

  return (r);
}

GString *
gzread_gstring (gzFile gzfh)
{
  gint l;
  GString *r;
  gchar *s;

  gzread (gzfh, &l, sizeof (gint));
  l = GINT_FROM_LE (l);
  if (l)
    {
      s = (gchar *) g_malloc (sizeof (gchar) * (l + 1));
      gzread (gzfh, s, sizeof (gchar) * l);
      s[l] = 0;
      r = g_string_new (s);
      free (s);
    }
  else
    {
      r = NULL;
    }

  return (r);
}

FILE_DATA *
read_filedata (gzFile gzfh, int fh)
{
  FILE_DATA *d;
  guint32 le32;
  guint16 le16;
  guint le;

  d = (FILE_DATA *) g_malloc (sizeof (FILE_DATA));
  if (fh >= 0)
    {
      d->name = read_gstring (fh);
      read (fh, &le32, sizeof (guint32));
      d->taille = GUINT32_FROM_LE (le32);
      read (fh, &le, sizeof (guint));
      d->type = GUINT_FROM_LE (le);
      read (fh, &le32, sizeof (guint32));
      d->date = GUINT32_FROM_LE (le32);
      read (fh, &le16, sizeof (guint16));
      d->categorie = GUINT16_FROM_LE (le16);
      read (fh, &le16, sizeof (guint16));
      d->description = GUINT16_FROM_LE (le16);
      d->information = read_gstring (fh);

    }
  else
    {
      d->name = gzread_gstring (gzfh);
      gzread (gzfh, &le32, sizeof (guint32));
      d->taille = GUINT32_FROM_LE (le32);
      gzread (gzfh, &le, sizeof (guint));
      d->type = GUINT_FROM_LE (le);
      gzread (gzfh, &le32, sizeof (guint32));
      d->date = GUINT32_FROM_LE (le32);
      gzread (gzfh, &le16, sizeof (guint16));
      d->categorie = GUINT16_FROM_LE (le16);
      gzread (gzfh, &le16, sizeof (guint16));
      d->description = GUINT16_FROM_LE (le16);
      d->information = gzread_gstring (gzfh);
    }
  return (d);
}

void
load_folder_tree (FOLDER * racine, gzFile gzcatalog, int catalog)
{
  gint i = 1;
  gchar id = CAT_TREE_0;
  GNode *prev_node;
  FILE_DATA *fd;

  prev_node = racine->tree;

  if (catalog >= 0)
    read (catalog, &id, sizeof (gchar));
  else
    gzread (gzcatalog, &id, sizeof (gchar));
  while (id != CAT_TREE_END)
    {
      switch (id)
	{
	case CAT_TREE_0:
	  prev_node =
	    g_node_append_data (prev_node->parent, GUINT_TO_POINTER (i++));
	  fd = read_filedata (gzcatalog, catalog);
	  fd->node = prev_node;
	  g_ptr_array_add (racine->datas, fd);
	  break;
	case CAT_TREE_UP:
	  prev_node = prev_node->parent;
	  break;
	case CAT_TREE_DOWN:
	  if (catalog >= 0)
	    read (catalog, &id, sizeof (gchar));
	  else
	    gzread (gzcatalog, &id, sizeof (gchar));
	  if (id != CAT_TREE_UP)
	    {
	      g_assert (id == CAT_TREE_0);
	      prev_node =
		g_node_append_data (prev_node, GUINT_TO_POINTER (i++));
	      fd = read_filedata (gzcatalog, catalog);
	      fd->node = prev_node;
	      g_ptr_array_add (racine->datas, fd);
	    }
	  break;
	}
      if (catalog >= 0)
	read (catalog, &id, sizeof (gchar));
      else
	gzread (gzcatalog, &id, sizeof (gchar));
    }
}

GPtrArray *
load_gptrarray_of_gstrings (gzFile gzfh, int fh)
{
  GPtrArray *gpa;
  gint i;
  gint nb_datas = 0;

  gpa = g_ptr_array_new ();
  if (fh >= 0)
    read (fh, &nb_datas, sizeof (gint));
  else
    gzread (gzfh, &nb_datas, sizeof (gint));
  nb_datas = GINT_FROM_LE (nb_datas);
  i = GINT_FROM_LE (i);
  for (i = 0; i < nb_datas; i++)
    {
      if (fh >= 0)
	g_ptr_array_add (gpa, read_gstring (fh));
      else
	g_ptr_array_add (gpa, gzread_gstring (gzfh));
    }
  return (gpa);
}

int
load_cat_from_file (char *filename, FOLDER * racine)
{
  char fileformat[MAX_STRING_LENGTH];
  gint l, le;
  int catalog;
  gzFile gzcatalog;

/* Test if the file exists */
  if ((g_file_test (filename, G_FILE_TEST_ISFILE)) != TRUE)
    {
      ERROR_DIALOG ("File does not exist or is not a standard file",
		    main_window);
      return (-1);
    }

/* Opens the file */
  if ((catalog = open (filename, O_RDONLY)) == -1)
    {
      ERROR_DIALOG ("Can't open file.", main_window);
      return (-1);
    }

/* Test if the length is big enough */
  l = lseek (catalog, 0, SEEK_END);
  if (l < 8)
    {
      close (catalog);
      ERROR_DIALOG ("File is not a GTKtalog file (v3 file format).",
		    main_window);
      return (-1);


    }

/* Test if this is an older version of gtktalog
 * Older versions of gtktalog begin with "gtktalog"
 */
  l = lseek (catalog, 0, SEEK_SET);
  read (catalog, fileformat, sizeof (gchar) * 8);
  fileformat[8] = 0;
  if (strcmp (fileformat, "gtktalog") == 0)
    {
      close (catalog);
      ERROR_DIALOG
	("File is seems to have been generated with a version older "
	 "than 0.1.0 of GTKtalog. Formats are incompatibles. "
	 "Send a mail to the authors to know what to do.", main_window);
      return (-1);
    }


/* Test if this is a catalog in v3 format. */
  l = lseek (catalog, 0, SEEK_SET);
  read (catalog, &le, sizeof (gint));
  l = GINT_FROM_LE (le);

  if ((l > 20) || (l < 5))
    {
      close (catalog);
      ERROR_DIALOG ("File is not a GTKtalog file.", main_window);
      return (-1);
    }
  read (catalog, &fileformat, sizeof (gchar) * l);
  fileformat[l] = 0;
  if ((!strcmp (fileformat, "gtktalog 3"))
      && (!strcmp (fileformat, "gtktalog 3gz")))
    {
      close (catalog);
      ERROR_DIALOG ("File is not a GTKtalog file (v3 file format).",
		    main_window);
      return (-1);
    }

/* This is a gtktalog format. Test if it is compressed or not,
 * and read the file.
 */
  if (strcmp (fileformat, "gtktalog 3"))
    {
      gzcatalog = gzdopen (catalog, "rb");
      racine->descriptions = load_gptrarray_of_gstrings (gzcatalog, -1);
      racine->categories = load_gptrarray_of_gstrings (gzcatalog, -1);
      load_folder_tree (racine, gzcatalog, -1);
      gzclose (gzcatalog);
    }
  else
    {
      racine->descriptions = load_gptrarray_of_gstrings (NULL, catalog);
      racine->categories = load_gptrarray_of_gstrings (NULL, catalog);
      load_folder_tree (racine, NULL, catalog);
      close (catalog);

    }
  racine->is_modified = FALSE;
  if (racine->catalog_filename)
    g_string_free (racine->catalog_filename, TRUE);
  racine->catalog_filename = g_string_new (filename);
  racine->catalog_filename_is_valid = TRUE;
  return (0);
}
