/*  GTKtalog.
 *  Copyright (C) 1999-2000  Mathieu VILLEGAS
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <gnome.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <stdlib.h>
#include <dirent.h>
#include <unistd.h>
#include <time.h>
#include <stdio.h>
#include <fnmatch.h>
#include <string.h>
#include <pthread.h>
#include <signal.h>

#include "addisk.h"
#include "config_common.h"
#include "fastaddisk.h"
#include "folder.h"
#include "interface.h"
#include "compare.h"
#include "io.h"
#include "progressbar.h"

#ifdef LIBC5
#define SIGUSR1 SIGSTKFLT
#define SIGUSR2 SIGUNUSED
#endif

#define READ   0
#define WRITE 1

static char *local_foldername;
static char *local_diskname;
static gboolean flag_to_umount;
static gboolean added_disk_is_ok;

static gint _thread_status;
static GNode *local_gn;


gint spawn_proc (gchar * path, gint narg, gchar ** arg)
{
  gchar **arg2;
  int i, pid = fork ();

  if (pid != 0)
    return pid;

  /* Child */

  arg2 = g_new0 (gchar *, (narg + 1));
  for (i = 0; i < narg; i++)
    arg2[i] = arg[i];
  arg2[narg] = NULL;

  return execvp (arg2[0], arg2);
}

guint du_s (gchar * path)
{
  int p[2];
  static gchar buffer[1024];
  gchar *argsdu[4];
  gchar *b2;
  ssize_t len;
  int pid;
  if (pipe (p) == 0)
    {
      if ((pid = fork ()) == 0)
	{
	  close (1);
	  dup (p[1]);
	  /* FIXME: Stderr should be closed too or redirected
	   * in order not to have error messages on the console
	   */
	  argsdu[0] = g_strdup ("du");
	  argsdu[1] = g_strdup ("-sk");
	  argsdu[2] = path;
	  argsdu[3] = NULL;
	  printf ("du -s\n");
	  execvp (argsdu[0], argsdu);
	}
      else
	{
	  close (0);
	  dup (p[0]);
	  close (p[1]);
	  len = read (0, buffer, 1023);
	  buffer[len] = 0;
	  b2 = buffer;
	  while ((b2[0] >= '0') && (b2[0] <= '9'))
	    b2++;
	  b2[0] = 0;
	  wait4 (pid, NULL, 0, 0);
	  close (p[0]);
	  /* FIXME: STDIN should be reopened to the real STDIN. */
	}
    }
  return (atoi (buffer) * 1024);
}

gboolean
eject_disk (CONFIG * my_config)
{
  int status, pid;
  char *args[2];

  seteuid (getuid ());
  args[0] = my_config->eject_prog->str;
  args[1] = my_config->mount_point->str;
  pid = spawn_proc (NULL, 2, args);
  if (pid == -1)
    {
      ERROR_DIALOG ("Could not execute eject program, check your config",
		    main_window);
      return (FALSE);
    }

  wait4 (pid, &status, 0, NULL);
  if (WIFEXITED (status) == 0)
    {
      ERROR_DIALOG ("Eject did not exit normally", main_window);
      return (FALSE);
    }
  if (WEXITSTATUS (status) != 0)
    {
      ERROR_DIALOG ("Eject failed! Disk in use?", main_window);
      return (FALSE);
    }
  return (TRUE);
}

gboolean
mount_disk (CONFIG * my_config)
{
  int pid, status;
  char *args[2];

  seteuid (getuid ());
  args[0] = my_config->mount->str;
  args[1] = my_config->mount_point->str;
  pid = spawn_proc (NULL, 2, args);
  if (pid == -1)
    {
      ERROR_DIALOG ("Could not execute mount program, check your config",
		    main_window);
      return (FALSE);
    }

  wait4 (pid, &status, 0, NULL);
  if (WIFEXITED (status) == 0)
    {
      ERROR_DIALOG ("Mount did not exit normally", main_window);
      return (FALSE);
    }
  if (WEXITSTATUS (status) != 0)
    {
      ERROR_DIALOG ("Mount failed! Disk in use?", main_window);
      return (FALSE);
    }
  return (TRUE);
}

gboolean
umount_disk (CONFIG * my_config)
{
  int pid, status;
  char *args[2];

  seteuid (getuid ());
  args[0] = my_config->umount->str;
  args[1] = my_config->mount_point->str;
  pid = spawn_proc (NULL, 2, args);
  if (pid == -1)
    {
      ERROR_DIALOG ("Could not execute umount program, check your config",
		    main_window);
      return (FALSE);
    }

  wait4 (pid, &status, 0, NULL);
  if (WIFEXITED (status) == 0)
    {
      ERROR_DIALOG ("Umount did not exit normally", main_window);
      return (FALSE);
    }
  if (WEXITSTATUS (status) != 0)
    {
      ERROR_DIALOG ("Umount failed! Disk in use?", main_window);
      return (FALSE);
    }
  return (TRUE);
}

void
umount_and_eject_if_necessary ()
{
  if (flag_to_umount == TRUE)
    {
      /* umount the drive and eject it */
      if (my_config->eject_disk == TRUE)
	{
	  if (my_config->use_supermount == TRUE)
	    eject_disk (my_config);
	  else if (umount_disk (my_config) == TRUE)
	    eject_disk (my_config);
	}
    }

}
gint get_thread_status ()
{
  return (_thread_status);
}

void
reinit_thread_status ()
{
  _thread_status = THREAD_SCAN_RUNNING_NO;
}

void
set_thread_status_to_running ()
{
  _thread_status = THREAD_SCAN_RUNNING_YES;
}

void
set_thread_status_to_ended ()
{
  _thread_status = THREAD_SCAN_RUNNING_ENDED;
}

void
do_not_forget_to_umount (gboolean gb)
{
  flag_to_umount = gb;
}

void
set_added_disk (gboolean gb)
{
  added_disk_is_ok = gb;
}

gint
my_timeout (gpointer data)
{
  FOLDER *racine = data;
  if (get_thread_status () == THREAD_SCAN_RUNNING_ENDED)
    {
      if (added_disk_is_ok == TRUE)
	{
	  change_name (local_gn, local_diskname);
	  change_information (local_gn, local_foldername);
	  g_node_traverse (local_gn,
			   G_PRE_ORDER, G_TRAVERSE_ALL, -1,
			   add_gnode_to_ctree, racine);

	  update_tree (racine);
	}
      else
	{
	  suppress_dir ((GNode *) local_gn);
	}
      g_free (local_diskname);
      g_free (local_foldername);
/************** If umount and eject have to be in the main thread, remove the comments ********/
umount_and_eject_if_necessary();
/*********************************************************************************************/
      reinit_thread_status ();
    }
  else
    {
      gtk_timeout_add (1000, my_timeout, racine);
    }
  return (0);
}

void
set_local_disk_name (gchar * s)
{
  local_diskname = g_strdup (s);
  return;
}

void
set_local_folder_name (gchar * s)
{
  local_foldername = g_strdup (s);
  return;
}

gchar *
read_information (gchar * file)
{
  char buff[250];
  int fd[2];
  int b;
  gint pid;
  gint status;
  GString *information;
  gchar *ret;
  EXTENSIONS *e;

  if ((e = find_extension (file)))
    {
      /* Launch the viewer */
      if (compare ("^[ \t]*$", e->prog->str, TRUE, FALSE))
	{
	  if (pipe (fd) == -1)
	    {
	      g_error ("Couldn't create a pipe");
	      return (NULL);
	    }
	  switch (pid = fork ())
	    {
	    case 0:		/*fils1 */
	      close (fd[READ]);
	      close (1);
	      dup (fd[WRITE]);
	      close (fd[WRITE]);
	      if (compare ("^[ \t]*$", e->viewerarg->str, TRUE, FALSE))
		{
		  execlp (e->prog->str, e->prog->str, e->arg, file, NULL);
		}
	      else
		{
		  execlp (e->prog->str, e->prog->str, file, NULL);
		}
	    case -1:		/*erreur */
	      g_error (_("Could not fork\n"));
	      exit (3);
	    default:
	      close (fd[WRITE]);
	      close (0);
	      dup (fd[READ]);
	      close (fd[READ]);
	      information = g_string_new ("");
	      b = read (0, buff, 249 * sizeof (char));
	      while (b != 0)
		{
		  buff[b] = '\0';
		  information = g_string_append (information, buff);
		  b = read (0, buff, 249 * sizeof (char));
		}
	      wait4 (pid, &status, 0, NULL);
	      ret = g_strdup (information->str);
	      g_string_free (information, TRUE);
	      return (ret);

	    }
	}
      else
	{
	  return (NULL);
	}
    }
  else
    {

      return (g_strdup (gnome_mime_type (file)));
    }
}

void
create_folder_tree (FOLDER * racine, GNode * parent, const gchar * path,
		    gboolean getInformation)
/* Stocks the directory (ans subdirs) into a FOLDER structure
 * folder is the FOLDER directory where the directory should be stored
 *   WARNING: folder should not be NULL!!!
 * path is the full path of the directory
 */
{
  DIR *dir;
  struct dirent *fichier;
  struct stat fileinfo;
  gchar *file;
  GList *gl;
  GList *new_dirs = NULL;
  GNode *gn;
  GString *my_path;
  gchar *infostring;
  static gulong totalDataBlocks = 0;

  if (progress_needReset ())
    {
      progress_reset ();
      totalDataBlocks = 0;
    }

  my_path = g_string_new (path);

  my_path = g_string_append_c (my_path, '/');

  /*  kill (my_config->current_pid, SIGUSR2); */

  if ((dir = opendir (my_path->str)) == NULL)
    {
      printf ("ne peut lister le repertoire courant: %s", my_path->str);
      return;
    }

  while ((fichier = readdir (dir)) != NULL)
    {
      file = g_strconcat (my_path->str, fichier->d_name, NULL);
      if ((strcmp (fichier->d_name, ".") != 0)
	  && (strcmp (fichier->d_name, "..") != 0))
	{

	  if (lstat (file, &fileinfo) == -1)
	    {
	      printf ("file not found: %s\n", fichier->d_name);
	    }
	  /* This is a directory */
	  if (S_ISLNK (fileinfo.st_mode))
	    {
	      char tmp[1024];
	      char tmp2[2048];

	      memset (tmp, 0, 1024);

	      if (readlink (file, tmp, 1023) == -1)
		{
		  printf
		    ("Problem with symbolic link %s, we forget it!\n", file);
		}
	      else
		{
		  sprintf (tmp2, "%s -> %s", fichier->d_name, tmp);
		  folder_add_link (racine, parent, tmp, tmp2, NULL,
				   fileinfo.st_size, fileinfo.st_ctime, 0, 0);
		}
	    }
	  else if (S_ISDIR (fileinfo.st_mode))
	    {

	      gn =
		folder_add (racine, parent, fichier->d_name, NULL,
			    IS_DIR, fileinfo.st_size,
			    fileinfo.st_ctime, 0, 0);
	      new_dirs = g_list_append (new_dirs, gn);
	    }

	  else
	    {
	      /* This is a file */

	      /* Progress Bar */
	      totalDataBlocks += fileinfo.st_size;
	      progress_setCurrent (totalDataBlocks);

	      if (getInformation)
		{
		  /* use a plugin to get informations about the file */
		  infostring = read_information (file);
		}
	      else
		{
		  infostring = NULL;
		}
	      folder_add (racine, parent, fichier->d_name, infostring,
			  IS_FILE, fileinfo.st_size, fileinfo.st_ctime, 0, 0);
	      if (infostring)
		g_free (infostring);
	    }
	}
      free (file);
    }
  closedir (dir);

  while (new_dirs)
    {
      gn = (GNode *) new_dirs->data;
      file = g_strconcat (path, "/", folder_get_name (gn), NULL);
      create_folder_tree (racine, gn, file, getInformation);
      free (file);
      gl = new_dirs;
      new_dirs = g_list_remove_link (new_dirs, new_dirs);
      g_list_free_1 (gl);
    }

}


void *
start_thread_scan (void *arg)
{
  FOLDER *racine = arg;

  pthread_detach (pthread_self ());

  local_gn =
    folder_add (racine, racine->tree,
		local_diskname, local_foldername, IS_DISK, 0, 0, 0, 0);

  create_folder_tree (racine, local_gn, local_foldername,
		      my_config->getInformation);

//  umount_and_eject_if_necessary ();
  set_thread_status_to_ended ();
  racine->is_modified = TRUE;
  pthread_exit (NULL);
  return (NULL);
}
