/*
   This file is part of the XXCalc Library - version 3.2
   Copyright (C)  2002, 2003, 2004, 2005, 2006, 2007, 2008, 2009, 2010,
   2011, 2012, 2013    Ivano Primi ( ivprimi@libero.it )    

   The XXCalc Library is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   The XXCalc library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include<stdlib.h>
#include<string.h>
#include<limits.h>		/* For LONG_MAX */
#include"varlist.h"
#include"heapsort.h"
#ifdef DMALLOC
#include <dmalloc.h>
#endif

static xx_vartype *
newvar (const char *name, c_omplex value)
{
  xx_vartype *pvar;
  str_s_ize lname = strlen (name);

  pvar = (xx_vartype *) malloc (sizeof (xx_vartype));
  if (!pvar)
    return NULL;
  else
    pvar->name = (char *) malloc ((lname + 1) * sizeof (char));
  if (!pvar->name)
    {
      free ((void *) pvar);	/* To avoid mem-leaks */
      return NULL;
    }
  else
    {
      strcpy (pvar->name, name);
      pvar->value = value;
    }
  return pvar;
}

static void
delvar (xx_vartype * pvar)
{
  pvar->value = c_convert (0);
  free ((void *) pvar->name);
  free ((void *) pvar);
}

void
xx_vl_init (xx_varlist * l)
{
  l->ppvar = NULL;
  l->nblocks = l->length = 0;
}

int
xx_vl_add (xx_varlist * l, const char *name, c_omplex value)
{
  xx_vartype *pvar, **tmp;
  ui_nteger i;

  if( l->length == 0 ) /* The list pointed to by L is empty! */
    {
      l->nblocks = 1;
      l->ppvar = (xx_vartype**) malloc (XX_VL_BLOCK_SIZE * sizeof(xx_vartype*));
      if( !l->ppvar )
	return 0;
      else
	{
	  for(i=0; i<XX_VL_BLOCK_SIZE; l->ppvar[i++] = NULL);
	  pvar = newvar(name, value);
	  if( !pvar )
	    return 0;
	  else
	    {
	      l->ppvar[0] = pvar;
	      l->length++;
	    }
	}
    }
  else /* l->length != 0 */
    {
      if( l->length / XX_VL_BLOCK_SIZE == l->nblocks )
	{
	  tmp = (xx_vartype**) malloc (l->length * sizeof(xx_vartype*));
	  if( !tmp )
	    return 0;
	  else
	    {
	      for(i=0 ; i<l->length; tmp[i] = l->ppvar[i], i++);
	      free((void*)l->ppvar);
	    }
	  l->nblocks++;
	  l->ppvar = (xx_vartype**) malloc (l->nblocks * XX_VL_BLOCK_SIZE * sizeof(xx_vartype*));
	  if( !l->ppvar )
	    {
	      for(i=0; i < l->length; i++)
		delvar(tmp[i]);
	      free((void*)tmp); /* To avoid memory-leaks */
	      return 0;
	    }
	  else
	    {
	      for(i=0; i < l->length; l->ppvar[i] = tmp[i], i++);
	      for(i = l->length; i < l->nblocks * XX_VL_BLOCK_SIZE; l->ppvar[i++] = NULL);
	      free((void*)tmp);
	    }
	}
      pvar = newvar(name, value);
      if(!pvar)
	return 0;
      else
	{
	  l->ppvar[l->length] = pvar;
	  l->length++;
	}
    } /* end if(l->length == 0) */
  return 1;
}

int
xx_vl_ins (xx_varlist * l, const char *name, c_omplex value)
{
  xx_vartype *pvar, **tmp;
  i_nteger i, j;
  int cmpres;

  if (l->length > LONG_MAX)
    {
      fprintf (stderr,
	       "Fatal Error in \"%s\", line %u: list too long(xx_vl_ins)\n\n",
	       __FILE__, __LINE__);
      return 0;
    }
  else if( (i = xx_vl_getpos(*l, name)) >= 0 )
    {
      l->ppvar[i]->value = value;
      return 1;
    }
  else /* The element must be effectively inserted */
    {
      if( l->length == 0 ) /* The list pointed to by L is empty! */
	{
	  l->nblocks = 1;
	  l->ppvar = (xx_vartype**) malloc (XX_VL_BLOCK_SIZE * sizeof(xx_vartype*));
	  if( !l->ppvar )
	    return 0;
	  else
	    {
	      for(i=0; i<XX_VL_BLOCK_SIZE; l->ppvar[i++] = NULL);
	      pvar = newvar(name, value);
	      if( !pvar )
		return 0;
	      else
		{
		  l->ppvar[0] = pvar;
		  l->length++;
		}
	    }
	}
      else /* l->length != 0 */
	{
	  if( l->length / XX_VL_BLOCK_SIZE == l->nblocks )
	    {
	      tmp = (xx_vartype**) malloc (l->length * sizeof(xx_vartype*));
	      if( !tmp )
		return 0;
	      else
		{
		  for(i=0 ; i<l->length; tmp[i] = l->ppvar[i], i++);
		  free((void*)l->ppvar);
		}
	      l->nblocks++;
	      l->ppvar = (xx_vartype**) malloc (l->nblocks * XX_VL_BLOCK_SIZE * sizeof(xx_vartype*));
	      if( !l->ppvar )
		{
		  for(i=0; i < l->length; i++)
		    delvar(tmp[i]);
		  free((void*)tmp); /* To avoid memory-leaks */
		  return 0;
		}
	      else
		{
		  for(i = 0; i < l->length; l->ppvar[i] = tmp[i], i++);
		  for(i = l->length; i < l->nblocks * XX_VL_BLOCK_SIZE; l->ppvar[i++] = NULL);
		  free((void*)tmp);
		}
	    }
	  pvar = newvar(name, value);
	  if(!pvar)
	    return 0;
	  else
	    {
	      for(i=0 ; i < l->length; i++)
		{
		  cmpres = strcmp (pvar->name, l->ppvar[i]->name);
		  if(  cmpres < 0 )
		    break;
		}
	      for(j = l->length-1; j>=i; l->ppvar[j+1] = l->ppvar[j], j--);
	      l->ppvar[i] = pvar;
	      l->length++;
	    }
	} /* end if(l->length == 0) */
      return 1; /* If you arrive here the element has been inserted */
    }
}

int xx_vl_sort(xx_varlist l)
{
  if (l.length == 0  ||  l.length > LONG_MAX)
    return 0;
  else
    {
      xx_vheapsort (l.ppvar, (long)l.length);
      return 1;
    }
}

int xx_vl_merge(xx_varlist* l1, xx_varlist l2)
{
  ui_nteger i=0, j=0;
  xx_varlist tmp;
  int cmpres;

  xx_vl_init(&tmp);
  while(i < l1->length && j < l2.length)
    {
      cmpres = strcmp (l1->ppvar[i]->name, l2.ppvar[j]->name);
      if( cmpres == 0 )
	{
	  if( !xx_vl_add (&tmp, l2.ppvar[j]->name, l2.ppvar[j]->value) )
	    {
	      xx_vl_destroy (&tmp);
	      return 0;
	    }
	  else
	    {
	      i++;
	      j++;
	    }
	}
      else if( cmpres < 0 )
	{
	  if( !xx_vl_add (&tmp, l1->ppvar[i]->name, l1->ppvar[i]->value) )
	    {
	      xx_vl_destroy (&tmp);
	      return 0;
	    }
	  else
	    i++;
	}
      else
	{
	  if( !xx_vl_add (&tmp, l2.ppvar[j]->name, l2.ppvar[j]->value) )
	    {
	      xx_vl_destroy (&tmp);
	      return 0;
	    }
	  else
	    j++;
	}
    } /* end while */
  for(; i<l1->length; i++)
    {
      if( !xx_vl_add (&tmp, l1->ppvar[i]->name, l1->ppvar[i]->value) )
	{
	  xx_vl_destroy (&tmp);
	  return 0;
	}
    }
  for(; j<l2.length; j++)
    {
      if( !xx_vl_add (&tmp, l2.ppvar[j]->name, l2.ppvar[j]->value) )
	{
	  xx_vl_destroy (&tmp);
	  return 0;
	}
    }
  xx_vl_destroy (l1);
  l1->ppvar   = tmp.ppvar;
  l1->length  = tmp.length;
  l1->nblocks = tmp.nblocks;
  /* tmp should not be destroyed! */
  return 1;
}

i_nteger
xx_vl_getpos (xx_varlist l, const char* name)
{
  i_nteger begin, end, i;
  int cmpres;

  if (l.length > LONG_MAX)
    {
      fprintf (stderr,
	       "Fatal Error in \"%s\", line %u: list too long(xx_vl_getpos)\n\n",
	       __FILE__, __LINE__);
      return -1;
    }
  if( !name ) /* If no name is given then return -1, */
    return -1;
  else /* otherwise initialize! */
    {
      begin = 0;
      end = (i_nteger) l.length - 1;
    }
  while (begin <= end)
    {
      i = (begin + end) / 2;
      cmpres = strcmp (name, l.ppvar[i]->name);
      if (cmpres == 0)
	return i;
      else if (cmpres < 0)
	end = i - 1;
      else
	begin = i + 1;
    }
  return -1; 
}

i_nteger
xx_vl_getpos_r (xx_varlist l, const char* name, i_nteger from, i_nteger to)
{
  i_nteger begin, end, i;
  int cmpres;

  if( !name || from < 0 || to < 0 || (ui_nteger)to >= l.length )
    return -1;
  else
    {
      begin = from;
      end = to;
      while (begin <= end)
	{
	  i = (begin + end) / 2;
	  cmpres = strcmp (name, l.ppvar[i]->name);
	  if (cmpres == 0)
	    return i;
	  else if (cmpres < 0)
	    end = i - 1;
	  else
	    begin = i + 1;
	}
      return -1;
    }
}

void
xx_vl_getrange (xx_varlist l, const char* prefix, i_nteger* from, i_nteger* to)
{
  str_s_ize lpref; /* length of the prefix */
  i_nteger begin, end, i;
  int cmpres;

  *from = *to = -1;
  if (l.length > LONG_MAX)
    {
      fprintf (stderr,
	       "Fatal Error in \"%s\", line %u: list too long(xx_vl_getrange)\n\n",
	       __FILE__, __LINE__);
      return;
    }
  if( !prefix )
    return;
  else
    lpref = strlen (prefix);
  /* First we must set correctly *FROM */
  begin = 0;
  end = (i_nteger) l.length - 1;
  while (begin <= end)
    {
      i = (begin + end) / 2;
      cmpres = strncmp (prefix, l.ppvar[i]->name, lpref);
      if (cmpres == 0)
	{
	  *from = i;
	  begin = 0;
	  end = i-1;
	}
      else if (cmpres < 0)
	end = i - 1;
      else
	begin = i + 1;
    }
  /* Now is the turn of *TO */
  begin = 0;
  end = (i_nteger) l.length - 1;
  while (begin <= end)
    {
      i = (begin + end) / 2;
      cmpres = strncmp (prefix, l.ppvar[i]->name, lpref);
      if (cmpres == 0)
	{
	  *to = i;
	  begin = i+1;
	  end = l.length - 1;
	}
      else if (cmpres < 0)
	end = i - 1;
      else
	begin = i + 1;
    }
}

const char*
xx_vl_at (xx_varlist l, ui_nteger pos, c_omplex* z)
{
  if( pos >= l.length )
    {
      if( (z) )
	*z = c_convert (0);
      return NULL;
    }
  else
    {
      if( (z) )
	*z = l.ppvar[pos]->value;
      return (const char*) l.ppvar[pos]->name;
    }
}

int
xx_vl_assign(xx_varlist l, ui_nteger pos, c_omplex z)
{
  if( pos >= l.length )
    return 0;
  else
    {
      l.ppvar[pos]->value = z;
      return 1;
    }
}

int
xx_vl_del (xx_varlist * l, const char *name)
{
  i_nteger i, j;

  if( (i = xx_vl_getpos (*l, name)) >= 0 )
    {
      delvar( l->ppvar[i] );
      for(j = i+1; j< l->length; l->ppvar[j-1] = l->ppvar[j], j++);
      l->ppvar[l->length-1] = NULL;
      l->length--;
      if( l->length % XX_VL_BLOCK_SIZE == 0 )
	{
	  xx_vartype** tmp;

	  if( l->length == 0 )
	    {
	      free((void*)l->ppvar);
	      l->ppvar = NULL;
	      l->nblocks = 0;
	    }
	  else
	    {
	      tmp = (xx_vartype**) malloc(l->length * sizeof(xx_vartype*));
	      if( !tmp )
		return 0;
	      else
		{
		  for(j=0 ; j < l->length; tmp[j] = l->ppvar[j], j++);
		  free((void*)l->ppvar);
		}
	      l->nblocks--;
	      l->ppvar = (xx_vartype**) malloc (l->nblocks * XX_VL_BLOCK_SIZE * sizeof(xx_vartype*));
	      if( !l->ppvar )
		{
		  for(j=0; j < l->length; j++)
		    delvar(tmp[j]);
		  free((void*)tmp); /* To avoid memory-leaks */
		  return 0;
		}
	      else
		{
		  for(j = 0; j < l->length; l->ppvar[j] = tmp[j], j++);
		  free((void*)tmp);
		}
	    }
	} /* end if(l->length % XX_VL_BLOCK_SIZE == 0) */
      return 1;
    } /* end if( (i=...) >= 0 ) */
  else
    return 0;
}

void
xx_vl_destroy (xx_varlist * l)
{
  ui_nteger i;

  if( !l || !l->ppvar )
    return;
  for(i=0 ; i < l->length; i++)
    delvar (l->ppvar[i]);
  free ((void*) l->ppvar);
  l->ppvar = NULL;
  l->length = l->nblocks = 0;
}

const char *
xx_vl_run (const xx_varlist * l, c_omplex * pz)
{
  const char *name;
  static ui_nteger length = 0, i = 0;
  static const xx_varlist* vl = NULL;

  if ((l))
    {
      vl = l;
      i = 0;
      length = vl->length;
    }
  if ( i == length )
    return NULL;
  else
    {
      *pz = vl->ppvar[i]->value;
      name = (const char *) vl->ppvar[i++]->name;
      return name;
    }
}

int
xx_vl_isempty (xx_varlist l)
{
  return (!l.ppvar);
}

ui_nteger
xx_vl_len (xx_varlist l)
{
  return l.length;
}

ui_nteger
xx_vl_nblocks (xx_varlist l)
{
  return l.nblocks;
}

ui_nteger
xx_vl_cap (xx_varlist l)
{
  return l.nblocks * XX_VL_BLOCK_SIZE;
}

int
xx_vl_set (xx_varlist l, const char *name, c_omplex value)
{
  i_nteger i;

  if( (i = xx_vl_getpos (l, name)) >= 0 )
    {
      l.ppvar[i]->value = value;
      return 1;
    }
  else
    return 0;
}

c_omplex
xx_vl_get (xx_varlist l, const char *name, int *yesno)
{
  i_nteger i;

  if( (i = xx_vl_getpos (l, name)) >= 0 )
    {
      *yesno = 1;
      return l.ppvar[i]->value;
    }
  else
    {
      *yesno = 0;
      return c_convert (0);
    }
}

/* xx_vl_load() is designed in a such way that it keeps */
/* the original order of the elements of the array.     */
xx_varlist
xx_vl_load (const xx_vartype * pvar, ui_nteger length)
{
  xx_varlist l;
  ui_nteger i;

  xx_vl_init (&l);
  for (i = 1; i <= length; i++)
    {
      if (!xx_vl_add (&l, pvar[length - i].name, pvar[length - i].value))
	{
	  xx_vl_destroy (&l);
	  break;
	}
    }
  return l;
}
